/*
 * Mesh.cpp
 *
 * Copyright (C) 1999 Stephen F. White
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (see the file "COPYING" for details); if 
 * not, write to the Free Software Foundation, Inc., 675 Mass Ave, 
 * Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#ifndef FLT_MAX 
# include <float.h>
#endif
#include "stdafx.h"

#include "Mesh.h"
#include "Face.h"
#include "List.h"
#include "MFColor.h"
#include "MFInt32.h"
#include "MFVec2f.h"
#include "MFVec3f.h"
#include "Matrix.h"

//#define DRAW_NORMALS 1

Mesh::Mesh(MFVec3f *vertices, MFInt32 *coordIndex,
	   MFVec3f *normals, MFInt32 *normalIndex,
	   MFColor *colors, MFInt32 *colorIndex,
	   MFVec2f *texCoords, MFInt32 *texCoordIndex,
	   float creaseAngle, bool ccw, bool solid,
           bool normalPerVertex)
{
    _vertices = vertices;	vertices->ref();
    _normals = normals;		if (normals) normals->ref();
    _colors = colors;		if (colors) colors->ref();
    _texCoords = texCoords;	if (texCoords) texCoords->ref();

    _coordIndex = coordIndex;	coordIndex->ref();
    _normalIndex = normalIndex ? normalIndex : coordIndex;
    _normalIndex->ref();
    _colorIndex = colorIndex ? colorIndex : coordIndex;
    _colorIndex->ref();
    _texCoordIndex = texCoordIndex ? texCoordIndex : coordIndex;
    _texCoordIndex->ref();
    _creaseAngle = creaseAngle;
    _ccw = ccw;
    _solid = solid;
    _normalPerVertex = normalPerVertex;
    buildFaces();
    generateFaceNormals();
    if (!_normals) {
	smoothNormals();
    }
}

Mesh::~Mesh()
{
    _vertices->unref();
    if (_normals) _normals->unref();
    if (_colors) _colors->unref();
    if (_texCoords) _texCoords->unref();

    _coordIndex->unref();
    _normalIndex->unref();
    _colorIndex->unref();
    _texCoordIndex->unref();

    for (int i = 0; i < _numFaces; i++) {
	delete _faces[i];
    }
    delete [] _faces;
}

void
Mesh::buildFaces()
{
    int		start = 0;
    int		n = _coordIndex->getSize();
    const int  *c = _coordIndex->getValues();
    int		i, numFaces = 0;

    for (i = 0; i < n; i++) {
	if (c[i] == -1) {
	    if (i - start > 0) {
		numFaces++;
	    }
	    start = i + 1;
	}
    }
    if ((i != 0) && (i != 1))
        if ((c[i-1] != -1) && (c[i-2] != -1))
            numFaces++;
    if (numFaces!=0)
       _faces = new Face *[numFaces];
    else
       _faces = new Face *[1];
    _numFaces = numFaces;
    numFaces = 0;
    start = 0;
    for (i = 0; i < n; i++) {
	if (c[i] == -1) {
	    if (i - start > 0) {
		_faces[numFaces] = new Face(i - start, start);
		numFaces++;
	    }
	    start = i + 1;
	}
    }
    if (numFaces < _numFaces)
        _faces[numFaces] = new Face(n - start, start);
}

void
Mesh::draw()
{
    if (_vertices->getSize()==0)
       return;       

    const int  *texCoordIndex = _texCoordIndex->getValues();
    const int  *colorIndex = _colorIndex->getValues();
    const int  *normalIndex = _normalIndex->getValues();
    const int  *coordIndex = _coordIndex->getValues();
    const float *vertices = _vertices->getValues();
    const float *normals = _normals->getValues();
    const float *texCoords = _texCoords ? _texCoords->getValues() : NULL;
    const float *colors = _colors ? _colors->getValues() : NULL;
    int		i, j;

    if (!_ccw) glFrontFace(GL_CW);
    if (_solid) {
	glEnable(GL_CULL_FACE);
    } else {
	glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_TRUE);
    }
    if (_colors) {
	glEnable(GL_COLOR_MATERIAL);
	glColorMaterial(GL_FRONT_AND_BACK, GL_DIFFUSE);
    }
    for (i = 0; i < _numFaces; i++) {
	int		offset = _faces[i]->getOffset();
	int		numVertices = _faces[i]->getNumVertices();
	glBegin(GL_POLYGON);
	for (j = offset; j < offset + numVertices; j++) {
	    if (texCoords) {
		glTexCoord2fv(texCoords + texCoordIndex[j] * 2);
	    }
	    if (colors) {
		glColor3fv(colors + colorIndex[j] * 3);
	    }
            if (normals) {
	        glNormal3fv(normals + normalIndex[j] * 3);
            }
	    glVertex3fv(vertices + coordIndex[j] * 3);
	}
	glEnd();
    }
#ifdef DRAW_NORMALS
    glDisable(GL_LIGHTING);
    glBegin(GL_LINES);
    for (i = 0; i < _numFaces; i++) {
	int		offset = _faces[i]->getOffset();
	int		numVertices = _faces[i]->getNumVertices();
	for (j = offset; j < offset + numVertices; j++) {
	    Vec3f	v1 = vertices + coordIndex[j] * 3;
	    Vec3f	v2 = v1 + Vec3f(normals + normalIndex[j] * 3) * 0.5f;
	    glVertex3f(v1.x, v1.y, v1.z);
	    glVertex3f(v2.x, v2.y, v2.z);
	}
    }
    glEnd();
    glEnable(GL_LIGHTING);
#endif
    if (_colors) {
	glDisable(GL_COLOR_MATERIAL);
    }
    if (_solid) {
	glDisable(GL_CULL_FACE);
    } else {
	glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_FALSE);
    }
    if (!_ccw) glFrontFace(GL_CCW);
}

class Edge {
public:
		Edge(int p1, int p2, int f)
		{ pos1 = p1, pos2 = p2; face = f; }
    int		pos1;
    int		pos2;
    int		face;
};

typedef List<Edge *> EdgeList;

void
Mesh::generateFaceNormals()
{
    int			i;
    int			n = _coordIndex->getSize();
    const int	       *coordIndex = _coordIndex->getValues();
    const float	       *vertices = _vertices->getValues();
    Face       * const *face = _faces;
    int                 indFaces;

    indFaces = 0;
    if (_vertices->getSize()!=0)
        for (i = 0; i < n; face++) {
            indFaces++;
            if (indFaces > _numFaces) 
                break;
            Vec3f c1(vertices + coordIndex[i++] * 3);
            Vec3f c2(vertices + coordIndex[i++] * 3);
            Vec3f c3(vertices + coordIndex[i++] * 3);
	    Vec3f v1 = c1 - c2;
	    Vec3f v2 = c3 - c2;
	    Vec3f normal = _ccw ? v2.cross(v1) : v1.cross(v2);
	    normal.normalize();
	    (*face)->setNormal(normal);
	    while (i < n && coordIndex[i] != -1) i++;
	    while (i < n && coordIndex[i] == -1) i++;
        }
}

static Edge *
findEdge(const int *coordIndex, EdgeList::Iterator *i, int vertex)
{
    for(; i != NULL; i = i->next()) {
	Edge *e = i->item();
	if (coordIndex[e->pos2] == vertex) return e;
    }
    return NULL;
}

void
Mesh::smoothNormals()
{
    int				i;
    EdgeList::Iterator         *j;
    Array<Vec3f>		normals;
    float			cosAngle = (float) cos(_creaseAngle);
    int				numFaces = 0;
    EdgeList		       *edgeLists = new EdgeList[_vertices->getSize()];
    const int		       *coordIndex = _coordIndex->getValues();
    int				nCoords = _coordIndex->getSize();
    int				nVerts = _vertices->getSize() / 3;
    int				start = 0;
    Array<int>		        normalIndex;

    _normalIndex->unref();

    if (_vertices->getSize()==0)
       return;       

    for (i = 0; i < nCoords; i++) {
	int v = coordIndex[i];
	if (v == -1) {
	    if (i - start > 0) {
		numFaces++;
	    }
	    start = i + 1;
	} else {
	    int pos2;
	    if (i == nCoords - 1 || coordIndex[i+1] == -1) {
		pos2 = start;
	    } else {
		pos2 = i+1;
	    }
  	    edgeLists[v].append(new Edge(i, pos2, numFaces));
	}
	normalIndex[i] = -1;
    }

    for (i = 0; i < nVerts; i++) {
	for (j = edgeLists[i].first(); j != NULL; j = j->next()) {
	    Edge       *e = j->item();
	    const Vec3f	&refNormal = _faces[e->face]->getNormal();
	    int		v2 = coordIndex[e->pos2];
	    Edge       *f = findEdge(coordIndex, edgeLists[v2].first(), i);
	    if (f) {
		const Vec3f &otherNormal = _faces[f->face]->getNormal();
		if (refNormal.dot(otherNormal) > cosAngle) {
		    // this edge is smooth
		    int	    i1 = normalIndex[e->pos1];
		    int	    i2 = normalIndex[f->pos2];
		    if (i1 == -1 && i2 == -1) {
			// create a new normal
			int	    index = normals.size();
			normals[index] = refNormal + otherNormal;
			normalIndex[e->pos1] = index;
			normalIndex[f->pos2] = index;
		    } else if (i1 == -1 && i2 != -1) {
			// use v2's normal
			normals[i2] += refNormal;
			normalIndex[e->pos1] = i2;
		    } else if (i1 != -1 && i2 == -1) {
			// use v1's normal
			normals[i1] += otherNormal;
			normalIndex[f->pos2] = i1;
		    } else {
			// they're both specified, so combine them in place
			normals[i1] += normals[i2];
			normals[i2] = normals[i1];
		    }
		}
	    }
	}
    }
    // cleanup:  vertices without normals get the face normal
    for (i = 0; i < nVerts; i++) {
	for (j = edgeLists[i].first(); j != NULL; j = j->next()) {
	    Edge	*e = j->item();
	    if (normalIndex[e->pos1] == -1) {
		int index = normals.size();
		normals[index] = _faces[e->face]->getNormal();
		normalIndex[e->pos1] = index;
	    }
	    delete e;
	}
    }
    for (int k = 0; k < normals.size(); k++) {
	normals[k].normalize();
    }
    _normals = new MFVec3f((float *) normals.extractData(), normals.size() * 3);
    int *n = new int[normalIndex.size() + 1];
    for (i = 0; i < normalIndex.size(); i++)
        n[i] = normalIndex[i];
    int nsize = normalIndex.size();
    if (n[nsize-1] != -1) {
        n[nsize-1] = -1;
        nsize++;
    }
    _normalIndex = new MFInt32(n, nsize);
    _normals->ref();
    _normalIndex->ref();
    delete [] edgeLists;
}

static int
compareFace(const void *f1, const void *f2)
{
    float	min1 = (*(Face **) f1)->getMinZ();
    float	max1 = (*(Face **) f1)->getMaxZ();
    float	min2 = (*(Face **) f2)->getMinZ();
    float	max2 = (*(Face **) f2)->getMaxZ();

    if (max1 < min2) return 1;
    if (max2 < min1) return -1;
    return 0;
}

void
Mesh::sort()
{
    const int  *coordIndex = _coordIndex->getValues();
    const float *vertices = _vertices->getValues();
    int		i, j;
    Matrix	matrix;

    if (_vertices->getSize()==0)
       return;       

    glGetFloatv(GL_MODELVIEW_MATRIX, (GLfloat *) matrix);

    for (i = 0; i < _numFaces; i++) {
	int	offset = _faces[i]->getOffset();
	int	numVertices = _faces[i]->getNumVertices();
	Vec3f   v(0.0f, 0.0f, 0.0f);
	float	min = FLT_MAX;
	float	max = 0.0f;
	for (j = 0; j < numVertices; j++) {
	    v = -(matrix * Vec3f(vertices + coordIndex[offset + j] * 3));
	    if (v.z > max) max = v.z;
	    if (v.z < min) min = v.z;
	}
	_faces[i]->setMinZ(min);
	_faces[i]->setMaxZ(max);
    }
    qsort(_faces, _numFaces, sizeof(Face *), compareFace);
}
