/*
 * NodeIndexedFaceSet.cpp
 *
 * Copyright (C) 1999 Stephen F. White
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (see the file "COPYING" for details); if 
 * not, write to the Free Software Foundation, Inc., 675 Mass Ave, 
 * Cambridge, MA 02139, USA.
 */

// NodeIndexedFaceSet::generateTextureCoordinates() based on 
// Polyrep.c of FreeWRL
/*******************************************************************
 Copyright (C) 1998 Tuomas J. Lukka
 Copyright (C) 2002 John Stewart, CRC Canada.
 DISTRIBUTED WITH NO WARRANTY, EXPRESS OR IMPLIED.
 See the GNU Library General Public License (file COPYING in the distribution)
 for conditions of use and redistribution.
*********************************************************************/

#include <stdio.h>
#include <string.h>
#include <assert.h>
#include "stdafx.h"

#include "NodeIndexedFaceSet.h"
#include "Proto.h"
#include "Scene.h"
#include "FieldValue.h"
#include "Node.h"
#include "Mesh.h"
#include "Face.h"
#include "Vec3f.h"
#include "NodeColor.h"
#include "NodeCoordinate.h"
#include "NodeNormal.h"
#include "NodeTextureCoordinate.h"

ProtoIndexedFaceSet::ProtoIndexedFaceSet(Scene *scene)
  : Proto(scene, "IndexedFaceSet")
{
    addEventIn(MFINT32, "set_colorIndex");
    addEventIn(MFINT32, "set_coordIndex");
    addEventIn(MFINT32, "set_normalIndex");
    addEventIn(MFINT32, "set_texCoordIndex");

    color.set(
          addExposedField(SFNODE, "color", new SFNode(NULL), NODE_COLOR));
    coord.set(
          addExposedField(SFNODE, "coord", new SFNode(NULL), NODE_COORDINATE));
    normal.set(
          addExposedField(SFNODE, "normal", new SFNode(NULL), NODE_NORMAL));
    texCoord.set(
          addExposedField(SFNODE, "texCoord", new SFNode(NULL), 
                          NODE_TEXTURE_COORDINATE));
    ccw.set(
          addField(SFBOOL, "ccw", new SFBool(true)));
    colorIndex.set(
          addField(MFINT32, "colorIndex", new MFInt32(), new SFInt32(-1)));
    colorPerVertex.set(
          addField(SFBOOL, "colorPerVertex", new SFBool(true)));
    convex.set(
          addField(SFBOOL, "convex", new SFBool(true)));
    coordIndex.set(
          addField(MFINT32, "coordIndex", new MFInt32(), new SFInt32(-1)));
    creaseAngle.set(
          addField(SFFLOAT, "creaseAngle", new SFFloat(0.0), 
                   new SFFloat(0.0f)));
    normalIndex.set(
          addField(MFINT32, "normalIndex", new MFInt32(), new SFInt32(-1)));
    normalPerVertex.set(
          addField(SFBOOL, "normalPerVertex", new SFBool(true)));
    solid.set(
          addField(SFBOOL, "solid", new SFBool(true)));
    texCoordIndex.set(
          addField(MFINT32, "texCoordIndex", new MFInt32(), new SFInt32(-1)));
}

Node *
ProtoIndexedFaceSet::create(Scene *scene)
{ 
    return new NodeIndexedFaceSet(scene, this); 
}

NodeIndexedFaceSet::NodeIndexedFaceSet(Scene *scene, Proto *def)
  : Node(scene, def)
{
    _mesh = NULL;
    _meshDirty = true;
}

NodeIndexedFaceSet::~NodeIndexedFaceSet()
{
    delete _mesh;
}

void
NodeIndexedFaceSet::setField(int index, FieldValue *value)
{
    _meshDirty = true;
    Node::setField(index, value);
}

void
NodeIndexedFaceSet::draw()
{
    if (_meshDirty) {
	createMesh();
	_meshDirty = false;
    }

    if (!_mesh) return;

    _mesh->sort();
    _mesh->draw();
}

MFVec3f *
NodeIndexedFaceSet::getCoordinates() 
{
    Node       *ncoord = coord()->getValue();
    if (ncoord == NULL)
        return NULL;
    else
        return ((NodeCoordinate *)ncoord)->point();
}

MFInt32 *
NodeIndexedFaceSet::getCoordIndex()
{
    return coordIndex();
}

MFInt32 *
NodeIndexedFaceSet::getColorIndex()
{
    return colorIndex();
}

MFInt32 *
NodeIndexedFaceSet::getNormalIndex()
{
    return normalIndex();
}

MFVec2f *
NodeIndexedFaceSet::getTextureCoordinates()
{
    Node *ntexCoord = texCoord()->getValue();
    if (ntexCoord == NULL)
        return NULL;
    else
        return ((NodeTextureCoordinate *)ntexCoord)->point();
}

MFInt32 *
NodeIndexedFaceSet::getTexCoordIndex()
{
    return texCoordIndex();;
}

void
NodeIndexedFaceSet::createMesh()
{
    Node       *coord = ((SFNode *) getField(coord_Index(),true))->getValue();
//    bool	bcolorPerVertex = colorPerVertex()->getValue();
//    bool	bconvex = convex()->getValue();
    MFInt32    *colorIndex = getColorIndex();
    MFInt32    *coordIndex = getCoordIndex();
    MFInt32    *normalIndex = getNormalIndex();
    MFInt32    *texCoordIndex = getTexCoordIndex();
   
    if (!coord || ((NodeCoordinate *) coord)->point()->getType() != MFVEC3F)
	return;

    MFVec3f    *coords = ((NodeCoordinate *)coord)->point();
    MFVec3f    *normals = NULL;
    MFColor    *colors = NULL;
    MFVec2f    *texCoords = NULL;

    if (normal()->getValue())
        if (normal()->getValue()->getType() == NODE_NORMAL)
	    normals = ((NodeNormal *)(normal()->getValue()))->vector();
    
    if (color()->getValue()) 
        if (color()->getValue()->getType() == NODE_COLOR) 
	    colors = ((NodeColor *)(color()->getValue()))->color();
    
    if (texCoord()->getValue()) 
        if (texCoord()->getValue()->getType() == NODE_TEXTURE_COORDINATE)
	    texCoords = ((NodeTextureCoordinate *)(texCoord()->getValue()))
                         ->point();
    
    if (colorIndex->getSize() != coordIndex->getSize()) {
	colorIndex = coordIndex;
    }
    if (texCoordIndex->getSize() != coordIndex->getSize()) {
	texCoordIndex = coordIndex;
    }
    if (!texCoord()->getValue()) {
	texCoords = generateTextureCoordinates(coords, texCoordIndex);
    }
    if (normalIndex->getSize() != coordIndex->getSize()) {
	normalIndex = coordIndex;
    }
    if (_mesh)
        delete _mesh;
    _mesh = new Mesh(coords, coordIndex, normals, normalIndex, colors, 
                     colorIndex, texCoords, texCoordIndex,
		     creaseAngle()->getValue(), ccw()->getValue(), 
                     solid()->getValue(), normalPerVertex()->getValue());
}

MFVec2f*		
NodeIndexedFaceSet::generateTextureCoordinates(MFVec3f *points, MFInt32* cindex)
{
    MFVec2f *texcoords = NULL;
    if (!points)
        return texcoords;
    int npoints = points->getSFSize(); 
    int ntexcoords = 0;
    int i;

    if (npoints > 0) {
        float* tcoords = new float[2*npoints];
        for (i = 0; i < npoints * 2; i++)
            tcoords[i] = 0;

	/* texture generation points... */

        float minVals[3];
        float maxVals[3];

        // find first valid point to initialise 
        bool initialised = false;
	for (i = 0; i < cindex->getSize(); i++) {
            int ind = ((SFInt32 *)cindex->getSFValue(i))->getValue();
            if ((ind >= 0) && (ind < npoints)) {
                const float *point = points->getValue(ind);
		if (point) {
                    for (int j = 0; j < 3; j++) {
                        minVals[j] = point[j];
                        maxVals[j] = point[j];
                    }
                initialised = true;
                break;
                } 
            } 
        }

        if (initialised == false)
            return NULL;

	/* generate default texture mapping */
	for (i = 0; i < cindex->getSize(); i++) {
            int ind = ((SFInt32 *)cindex->getSFValue(i))->getValue();
            if ((ind >= 0) && (ind < npoints)) {
                const float *point = points->getValue(ind);
		if (point) {
                    for (int j = 0; j < 3; j++) {
                        if (minVals[j] > point[j]) 
                            minVals[j] = point[j];
                        if (maxVals[j] < point[j]) 
                            maxVals[j] = point[j];
                    }
                } 
            } 
        }

	float Ssize = 0.0;
	float Tsize = 0.0;
	int Sindex = 0;
	int Tindex = 0;

        /* find the S,T mapping. */
        float Xsize = maxVals[0]-minVals[0]; 
        float Ysize = maxVals[1]-minVals[1];
        float Zsize = maxVals[2]-minVals[2];

        if ((Xsize >= Ysize) && (Xsize >= Zsize)) {
            /* X size largest */
            Ssize = Xsize;
            Sindex = 0;
            if (Ysize >= Zsize) {
                Tsize = Ysize;
                Tindex = 1;
            } else {
                Tsize = Zsize;
                Tindex = 2;
            }
        } else if ((Ysize >= Xsize) && (Ysize >= Zsize)) {
            /* Y size largest */
            Ssize = Ysize;
            Sindex = 1;
            if (Xsize >= Zsize) {
                Tsize = Xsize;
                Tindex = 0;
            } else {
                Tsize = Zsize;
                Tindex = 2;
            }
        } else {
            /* Z is the largest */
            Ssize = Zsize;
            Sindex = 2;
            if (Xsize >= Ysize) {
                Tsize = Xsize;
                Tindex = 0;
            } else {
                Tsize = Ysize;
                Tindex = 1;
            }
        }

        for( i = 0; i < cindex->getSize(); i++) {
            int ind = ((SFInt32 *)cindex->getSFValue(i))->getValue();
            if ((ind >= 0) && (ind < npoints)) {
                const float *point = points->getValue(ind);
		if (point) {
                    /* temporary place for X,Y,Z */
                    float XYZ[] = {0.0, 0.0, 0.0};

	            XYZ[0] = point[0]; 
                    XYZ[1] = point[1]; 
                    XYZ[2] = point[2];
                       
                    // default textures 
                    // we want the S values to range from 0..1, 
                    //     and the T values to range from 0..S/T
                    tcoords[2*ind]     = (XYZ[Sindex] - minVals[Sindex])/Ssize;
                    tcoords[2*ind + 1] = (XYZ[Tindex] - minVals[Tindex])/Ssize;
                }
            }
        }
        texcoords = new MFVec2f(tcoords, 2*npoints);
    }
    return texcoords;
}

MFVec3f *
NodeIndexedFaceSet::getSmoothNormals(void)
{
    MFVec3f *v = _mesh->getNormals();
    
    if (v != NULL) {
        v =  new MFVec3f((float *)((MFVec3f *)v->copy())->getValues(), 
                         v->getSize());
    }
    return v;
}

MFInt32 *
NodeIndexedFaceSet::getSmoothNormalIndex(void)
{
    MFInt32 *i = _mesh->getNormalIndex();
    
    if (i != NULL) {
        i =  new MFInt32((int *)((MFInt32 *)i->copy())->getValues(), 
                         i->getSize());
    }
    return i;
}

