/* 
 *    wmStock-0.10 (C) 1999 Matt Fischer (mfischer@umr.edu)
 * 
 *      based on...
 *
 *      wmWeather-1.30 (C) 1999 Mike Henderson (mghenderson@lanl.gov)
 * 
 *
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; either version 2, or (at your option)
 *      any later version.
 *
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License
 *      along with this program (see the file COPYING); if not, write to the
 *      Free Software Foundation, Inc., 59 Temple Place - Suite 330, 
 *      Boston, MA  02111-1307, USA
 *
 */

#include "wmstock.h"

#include "wmstock_master.xpm"
#include "wmstock_mask.xbm"

char            ticker[8];
int		TimeLoop;

/* FOR PRODUCTION */
int             ForceUpdate = 1;
int             DontDeleteOldQuotes = 0;
int             Offline = 0;
int             ReDrawData = 0;
int             CheckMarkets = 0;
long            UpdateDelay,
                DownloadDelay,
                TimeToNext,
		TimeToSleep;
int             ReverseOrder;
char		WebBrowser[128];
//struct stat asf;

/* Long options equivalences. */
static struct option long_options[] =
{
    {"delay", required_argument, NULL, 'd'},
    {"dldelay", required_argument, NULL, 'l'},
    {"nonuke", no_argument, NULL, 'u'},
    {"time2next", required_argument, NULL, 't'},
    {"reverse", no_argument, NULL, 'r'},
    {"version", no_argument, NULL, 'v'},
    {"newversion", no_argument, NULL, 'e'},
    {"help", no_argument, NULL, 'h'},
    {"browser", required_argument, NULL, 'b'},
    {"opentime", required_argument, NULL, 'o'},
    {0, 0, 0, 0}
};

/* needed in ParseCMDLine */
static int      ck_atol(char const *str, long *out);

ticker_struct   temp_ts;
node           *curr_t;
node           *disp_t;

int             num_quotes = 0;

char            DirName[256];
long 		regalarm=0;		
int             xfiledesc;
fd_set          fds;			/* set of file descriptors */
XEvent          event;

/* stuff needed for market open/close timings */
struct tm mopen, mclose;
struct tm curr, midnight;
time_t curr_secs, mclose_secs, mopen_secs, midnight_secs;
unsigned int mopen_hr, mopen_mn, mclose_hr, mclose_mn;
struct timeval tv;

int 
main(int argc, char *argv[])
{
    char            LabelColor[30] = "#79bdbf";
    char            NegativeColor[30] = "#ff0000";
    char            DataColor[30] = "#ffbf50";
    char            BackColor[30] = "#181818";
    char            UnusedColor[30] = "#c5a6ff";

    int             errorno;
    char            tmpfilenam[256],
                    FileName[32];
    int             tempret;
    struct stat     buf;	/* auxiliary only */
    long	    altime;
    int		    initloop=1;

    signal(SIGALRM, mySignalHandler);	/* register my SIG ALARM handler */

    head = tail = NULL;		/* init the list */
    WebBrowser[0]='\0';
    mopen.tm_hour = mopen.tm_min = -1;
    mclose.tm_hour = mclose.tm_min = -1;


/* Parse any command line arguments. */
    ParseCMDLine(argc, argv);

#ifdef MLF_DEBUG
    fprintf(stderr, "Update: %ld\tDownload: %ld\tTimeToNext: %ld\t\
	Reverse: %i\tCheckMarkets: %d\n", UpdateDelay, DownloadDelay, TimeToNext, ReverseOrder, CheckMarkets);
#endif

    sprintf(DirName, "%s", (getenv("HOME")));	/* set the filename */
    strcat(DirName, "/.wmstock");

/* check if ~/.wmstock exists */
    if (stat(DirName, &buf)) {
    /* something went wrong when trying to access .wmstock */
	if (errno == ENOENT) {
	/* file doesn't exist => create it! */
	    if (mkdir(DirName, S_IRWXU) < 0) {
		perror("Error while creating ~/.wmstock");
		exit(1);
	    }
	} else {
	/* Other error - complain... */
	    perror("Error while trying to access ~/.wmstock");
	    exit(1);
	}
    }
    curr_t = head;
    while (curr_t != NULL) {
	if ((!Offline) && (!DontDeleteOldQuotes)) {
	/* delete the old data file */
	    sprintf(tmpfilenam, "%s/%s.dat", DirName,
		    curr_t->ts.ticker);
#ifdef MLF_DEBUG
	    fprintf(stderr, "deleting: %s\n", tmpfilenam);
#endif
	    tempret = unlink(tmpfilenam);
	    if (tempret < 0 && errno != ENOENT) {
		perror("delete");
	    /* no error if file didn't exist in the first place */
		fprintf(stderr, "Error while trying to delete %s\n",
			tmpfilenam);
	    }
	}
	num_quotes++;
	curr_t->ts.oldtime.tv_sec = 0;
	curr_t->ts.oldtime.tv_usec = 0;
	curr_t->ts.newtime.tv_sec = 0;
	curr_t->ts.newtime.tv_usec = 0;
	curr_t->ts.timediff = 0;
	curr_t = curr_t->next;
    }

    if (num_quotes == 1) {
	TimeToNext = 0;
    }
/* 
 * *  Do the window opening in 2 stages. After the initXwindow() call, *
 * we know what the Depth of the Display is. We can then pick an 
 * appropriate  XPM to use. I.e. may want to use one that has fewer 
 * colors to make the App work  better on a low-color 8-bit display. */
    initXwindow(argc, argv);
    openXwindow(argc, argv, wmstock_master, wmstock_mask_bits,
		wmstock_mask_width, wmstock_mask_height, BackColor,
		LabelColor, NegativeColor, DataColor, UnusedColor);

    ClearWindow();
    RedrawWindow(); 
    drawstring("DOWNLOADING", 6, 128, 6);
    RedrawWindow();

/* get the file descriptor for the connect to the X server */
    xfiledesc = XConnectionNumber(display);
    FD_ZERO(&fds);		/* empties the set */
    FD_SET(xfiledesc, &fds);

    curr_t = head;


    while (1) {

#ifdef MLF_DEBUG
	fprintf(stderr, "Current ticker: %s\n", curr_t->ts.ticker);
#endif

	TimeToSleep=(TimeToNext*1000000)/100;
        if (!initloop) 
	{  
		for(TimeLoop=0;TimeLoop<100;TimeLoop++) 
			{ usleep(TimeToSleep); RedrawWindow(); } 
	}

	initloop = 0;

	curr_t->ts.oldtime.tv_sec = curr_t->ts.newtime.tv_sec;
	gettimeofday(&curr_t->ts.newtime, 0);
	curr_t->ts.timediff += curr_t->ts.newtime.tv_sec -
	    curr_t->ts.oldtime.tv_sec;

	sprintf(FileName, "%s/%s.dat", DirName, curr_t->ts.ticker);

	if (head->next != NULL) {	/* list has more than 1 ticker */
	    ReDrawData = 1;	/* so we need to show the new ticker */
	}

	if ((ForceUpdate) || (curr_t->ts.timediff > UpdateDelay)) {
#ifdef MLF_DEBUG
	    if (!ForceUpdate) {
		printf("Update from time! %li\n", curr_t->ts.timediff);
	    }
#endif

	    if (!Offline && are_markets_open()) {
		download_data(curr_t->ts.ticker);
		if ((errorno = get_data_from_file(argv[0], FileName)) < 0) {
		    display_error(errorno);
		    ReDrawData = 0;
		}
	    } 
		/* online, but markets closed */
	    else if (!Offline && !are_markets_open()) {

		/* try to open the file, if not there, download */
		if ((errorno = get_data_from_file(argv[0], FileName)) < 0) {
#ifdef MLF_DEBUG
	printf("markets closed, but downloading anyway\n");
#endif
			download_data(curr_t->ts.ticker);
			if ((errorno = get_data_from_file(argv[0], FileName)) < 0) {
			    display_error(errorno);
			    ReDrawData = 0;
			}
		}

	    }
		else {
		if ((errorno = get_data_from_file(argv[0], FileName)) < 0) {
		    display_error(errorno);
		    ReDrawData = 0;
		}
	    }

	    gettimeofday(&curr_t->ts.oldtime, 0);
	    curr_t->ts.newtime.tv_sec = curr_t->ts.oldtime.tv_sec;
	    curr_t->ts.timediff = 0;

	    if (errorno >= 0) {	/* if no errors! */
		ReDrawData = 1;
		ForceUpdate = 0;
	    }
	}
    /* Draw window.  */
	if (ReDrawData) {
	    ClearWindow();
	    display_data(curr_t->ts);	/* paste up the data */
	    disp_t = curr_t;

	/* Make changes visible */
	    RedrawWindow();
	    ReDrawData = 0;
	}
	if (num_quotes == 0) {
	    fprintf(stderr, "I am out of ticker symbols!  All invalid symbols have\n");
	    fprintf(stderr, "been removed from the list and there is nothing left!\n");
	    exit(-1);
	}

	initloop=1;
	if (curr_t->next == NULL)	/* advance in the list */
	    curr_t = head;
	else
	    curr_t = curr_t->next;

	altime = TimeToNext < UpdateDelay ? TimeToNext : UpdateDelay;

	if (regalarm)
	{
		/* negative or zero alarm will be very bad */
		if (altime <= 0)
			altime = 1;
		alarm(altime);
#ifdef MLF_DEBUG
	fprintf(stderr, "alarm set for %ld seconds\n", altime);
#endif
	}
	else
	{	
		if (initloop)
		{
#ifdef MLF_DEBUG
	fprintf(stderr, "set a short alarm\n");
#endif
			alarm (9);	/* alarm for the initial loop */
		}
		else 
		{
			/* set a short alarm since we are on the initial loop */
			alarm(1);
		}
	}

	regalarm = 1;		/* next time, set a regular alarm */

	while (1) {

	    if (select(xfiledesc + 1, &fds, NULL, NULL, NULL) == -1) {
		break;		/* my alarm went off */
	    }
	/* restore the FD set */
	    FD_ZERO(&fds);	/* empties the set */
	    FD_SET(xfiledesc, &fds);

	    if (FD_ISSET(xfiledesc, &fds)) {
	    /* someone clicked something */
		while (XPending(display)) {
		    XNextEvent(display, &event);
		    switch (event.type) {
		    case Expose:
			RedrawWindow();
			break;
		    case ButtonPress:
			ButtonPressEvent(&event.xbutton);
			break;
		    /* 
		     * case KeyPress: KeyPressEvent(&event.xkey); break; */
		    case ButtonRelease:
			break;
		    case EnterNotify:
			XSetInputFocus(display, PointerRoot, RevertToParent, CurrentTime);
			break;
		    case LeaveNotify:
			XSetInputFocus(display, PointerRoot, RevertToParent, CurrentTime);
			break;
		    }
		}
	    }
	}			/* while (1) */

    }				/* while (1) */

}				/* main() */



/* ParseCMDLine() */
void 
ParseCMDLine(int argc, char *argv[])
{
    int             opt;
    int             ShowVersion = 0;
    int             ShowHelp = 0;
    char            *p;
    char            command[256];
    void            print_usage();
    regex_t         re;
    char *pattern[] = { "[0-9]+:", ":[0-9]+"  };
    int error, i, eflag, j;
    regmatch_t pm;
    char *match, *tempptr;

    UpdateDelay = DEFAULT_UPDATEDELAY;
    DownloadDelay = DOWNLOAD_DELAY;	/* DEFAULT = 0 */
    TimeToNext = DEFAULT_TIME_BETWEEN;
    ReverseOrder = 0;

    while ((opt = getopt_long(argc, argv, "d:el:b:o:rt:uvh",
			      long_options, NULL)) != EOF)
	switch (opt) {
	case 'd':
	    if (optarg) {
		if (ck_atol(optarg, &UpdateDelay)) {
		    fprintf(stderr, "I don't understand the delay time you have entered (%s).\n", optarg);
		    exit(1);
		}
	    }
	    break;
	case 't':
	    if (optarg) {
		if (ck_atol(optarg, &TimeToNext)) {
		    fprintf(stderr, "I don't understand the time2next time you have entered (%s).\n", optarg);
		    exit(1);
		}
	    }
	    break;
	case 'l':
	    if (optarg) {
		if (ck_atol(optarg, &DownloadDelay)) {
		    fprintf(stderr, "I don't understand the dldelay time you have entered (%s).\n", optarg);
		    exit(1);
		}
	    }
	    break;

	case 'b':
	/* specify web browser */
	    if (optarg)
		{
		    if ( (strlen (optarg)) < (sizeof (WebBrowser)) )  {
			/* make sure we have enough space to do this */
				strcpy (WebBrowser, optarg);
			}
		    else {
			fprintf(stderr, "Path for web browser was too long.  Program will continue, but web access will be disabled.\n");
			}
		}
	    else 
		{
		 fprintf(stderr, "-b requires an argument.  The web browser function will not be activated.\n");
		}
	    break;

	case 'o':
	    if (optarg)
		{
		CheckMarkets = 1;
	        /* XXX do a proper malloc here */
		match = malloc (10);
		tempptr = optarg;

		/* try to match both cases h:mm, hh:mm */
		for (i=0; i<2; i++)
		{
			(void) regcomp (&re, pattern[i], REG_EXTENDED);
			eflag=0;
			optarg = tempptr;

		/* find both matches */
		for (j=0; j<2; j++)
		{
			error=regexec (&re, optarg, 1, &pm, eflag);
	
		if (error ==0)
		{

                        if (i==0)
                        {
                                /* XXX - check to make sure we dont overflow match */
                                /* skip the first char since its a : */
                                (void) strncpy (match, (optarg + pm.rm_so), (pm.rm_eo - pm.rm_so - 1));
				if (j==0)
					mopen_hr=atoi(match);
				else
					mclose_hr=atoi(match);

		       }
                        else
                        {
                                /* XXX - check to make sure we dont overflow match */
                                /* skip the first char since its a : */
                                (void) strncpy (match, (optarg + pm.rm_so + 1), (pm.rm_eo - pm.rm_so));
				if (j==0)
					mopen_mn=atoi(match);
				else
					mclose_mn=atoi(match);
                        }
			/* advance the string */
			optarg += (pm.rm_eo-1);
			eflag = REG_NOTBOL;

                }	/* if (error == 0) */
	   }

        }

#ifdef MLF_DEBUG
        printf ("open_hour: %d\topen_min: %d\n", mopen.tm_hour, mopen.tm_min);
        printf ("close_hour: %d\tclose_min: %d\n", mclose.tm_hour, mclose.tm_min);
#endif

	regfree(&re);
        free (match);
        match=NULL;

	/* make sure everything was initialized */
	if ((mopen_hr == -1) || (mclose_hr == -1) || (mopen_mn == -1) || (mclose_mn == -1))
	{
		fprintf(stderr, "I didnt find an open and a close time.  Please enter the time as hh:mm-hh:mm\n");
		exit(1);
		
	}

	/* compare the times to keep some sanity... (lets reuse i & j) */
	i = mopen_mn + (mopen_hr *60);
	j = mclose_mn + (mclose_hr *60);

	if (i>j)
	{
		fprintf(stderr, "You gave me a market close before the open.  I'm confused.  Remember to use 24 hour time!\n");
		exit(1);
	}
	
	

	     	}

	   else 
		{
		fprintf(stderr, "-o requires that you pass it a time in the format hh:mm\n");
		}
	    break;
	
	case 'r':
	/* reverse the display order for the ticker */
	    ReverseOrder = 1;
	    break;
	case 'e':
	/* check the website for a new version */
	    sprintf(command, "getquote -newversion %s", WMSTOCK_VERSION);
	    system(command);
	    exit(0);
	case 'u':
	/* Don't delete the old quotes from ~/.wmstock */
	    DontDeleteOldQuotes = 1;
	    break;
	case 'v':
	/* Show version info */
	    ShowVersion = 1;
	    break;
	case 'h':
	/* Show usage info */
	    ShowHelp = 1;
	    break;
	default:
	    print_usage();
	    exit(1);
	    break;
	}			/* switch ... */

    if (ShowVersion) {
	printf("wmstock version %s, released on %s\n\n",
	       WMSTOCK_VERSION, RELEASE_DATE);
	printf("Copyright (C) 1999 Matt Fischer <me@mattfischer.com>\n");
	printf("The newest release is available from http://www.mattfischer.com/wmstock\n\n");
	printf("This is free software; see the source for copying conditions. There is NO\n");
	printf("warranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n\n");
	exit(0);
    }
    if (ShowHelp) {
	print_usage();
	exit(0);
    }
/* Get TickerSymbols */
    if (optind < argc) {
	while (optind < argc) {
	    strcpy(ticker, argv[optind++]);

	    for (p = ticker; *p; p++) {		/* convert to UC */
		*p = toupper(*p);
	    }

	    strcpy(temp_ts.ticker, ticker);

	    if (ReverseOrder) {
		insert_pre(temp_ts);	/* insert into the list of tickers 
					 */
	    } else {
		insert_last(temp_ts);
	    }
#ifdef MLF_DEBUG
	    fprintf(stderr, "ticker: %s\n", temp_ts.ticker);
#endif
	}			/* while (optind...) */
    } else {
	print_usage();
	exit(1);
    }				/* if (optind...) */

}				/* ParseCMDLine() */


void display_data(struct ticker_struct tick)
{
    int             chr,
                    lc,
                    q;
    char            tempchar[2],
                    tempvar[128];
    double          divisor,
                    val;
    int             preflag,
                    slashflag,
                    getrow;

/* Clear window.  */
    copyXPMArea(5, 69, 54, 54, 5, 5);

/* Paste the stock ticker symbol */
    q = 0;
    lc = 0;

#ifdef MLF_DEBUG
    fprintf(stderr, "drawing::ticker: %s\n", tick.ticker);
#endif

    drawstring(tick.ticker, 7, 128, 6);

/* Paste up last ticker price */
    if (tick.last_price != -99999.99) {
	sprintf(tempvar, "%.6g", tick.last_price);
	q = lc = 0;
	while (tempvar[lc] != '\0') {
	    chr = (int) (tempvar[lc] - 48);
	    if (chr == -2) {
		copyXPMArea(10 * 5 + 67, 57, 4, 6, 25 + q, 15);
		q += 4;
	    } else {
		copyXPMArea(chr * 5 + 66, 57, 5, 6, 25 + q, 15);
		q += 5;
	    }
	    lc++;
	}

    }
/* post the price change */
    if ((strcmp(tick.change, "N/A") != 0) && (tick.change[0] != '\0')) {	/* make sure its not N/A */

#ifdef MLF_DEBUG
    /* fprintf (stderr,"tick.change: %s\n",tick.change); */
	printf("tick.change: %s\n", tick.change);
#endif

	lc = 0;
	q = 0;

	if (tick.change[0] == '-') {	/* if its negative, we show it as
					 * red */
	    getrow = 64;
	} else {
	    getrow = 57;
	}

	while (tick.change[lc] != '\0') {
	    chr = (int) tick.change[lc];

	    if ((chr <= 57) && (chr >= 48)) {	/* number */
		chr -= 48;
		copyXPMArea(chr * 5 + 66, getrow, 5, 6, 25 + q, 24);
	    } else if (chr == 45) {	/* - */
		copyXPMArea(65, 71, 5, 6, 25 + q, 24);
	    } else if (chr == 43) {	/* + */
		copyXPMArea(71, 71, 5, 6, 25 + q, 24);
	    } else if (chr == 46) {	/* . */
		chr = chr - 45 + 9;
		copyXPMArea(chr * 5 + 66, getrow, 5, 6, 25 + q, 24);
	    }
	    if (tick.change[lc] == '.') {
		q += 4;
	    } else {
		q += 5;
	    }

	    lc++;

	}

    }
/* draw the last trade date */

    if ((strcmp(tick.last_trade_date, "N/A\n") != 0) && (tick.last_trade_date[0] != '\0')) {	/* make sure its not N/A */

#ifdef MLF_DEBUG
	fprintf(stderr, "tick.ltd: %s\n", tick.last_trade_date);
#endif

	lc = 0;
	q = 0;
	slashflag = 0, preflag = 0;
	while ((tick.last_trade_date[lc] != '\0') && (slashflag == 0)) {
	    chr = (int) tick.last_trade_date[lc];

	    if ((chr <= 57) && (chr >= 48)) {	/* number */
		chr -= 48;
		copyXPMArea(chr * 5 + 66, 57, 5, 6, 30 + q, 33);
		q += 5;
	    } else if (chr == 47) {	/* / */
		if (preflag == 1) {
		    slashflag = 1;
		} else {
		    copyXPMArea(112, 34, 5, 6, 30 + q, 33);
		    preflag = 1;
		    q += 4;
		}
	    }
	    lc++;
	}
    }
/* draw the last trade time */

    if ((strcmp(tick.last_trade_time, "N/A\n") != 0) && (tick.last_trade_time[0] != '\0')) {	/* make  sure its not N/A */

#ifdef MLF_DEBUG
	fprintf(stderr, "tick.ltt: %s\n", tick.last_trade_time);
#endif

	lc = 0;
	q = 0;
	while (tick.last_trade_time[lc] != '\0') {
	    chr = (int) tick.last_trade_time[lc];

	    if ((chr <= 57) && (chr >= 48)) {	/* number */
		chr -= 48;
		copyXPMArea(chr * 5 + 66, 57, 5, 6, 30 + q, 41);
	    } else if (chr == 58) {	/* : */
		chr = chr - 58 + 10;
		copyXPMArea(chr * 5 + 2, 135, 5, 6, 30 + q, 41);
	    } else {		/* letter */
		chr -= 65;
		copyXPMArea(chr * 5 + 2, 128, 5, 6, 30 + q, 41);
	    }

	    if (tick.last_trade_time[lc] == ':') {
		q += 3;
	    } else {
		q += 5;
	    }
	    lc++;
	}
    }
    if (tick.volume != 0) {
#ifdef MLF_DEBUG
	fprintf(stderr, "tick.volume: %li\n", tick.volume);
#endif

	lc = 0;
	q = 0;
	divisor = 10;
	val = tick.volume;

	while (val > 1.0) {
	    val = val / divisor;
	    lc++;
	}

	val = tick.volume;
	if (lc >= 12) {
	    strcpy(tempchar, "B");
	    val = val / 1e9;
	} else if ((lc <= 9) && (lc > 6)) {
	    strcpy(tempchar, "M");
	    val = val / 1e6;
	} else if ((lc <= 6) && (lc > 3)) {
	    strcpy(tempchar, "K");
	    val = val / 1e3;
	} else {
	    strcpy(tempchar, "");
	}

	sprintf(tempvar, "%.4g", val);
    /* gcvt(val,4,tempvar); */
	strcat(tempvar, tempchar);

	lc = 0;
	q = 0;

	while (tempvar[lc] != '\0') {
	    chr = (int) tempvar[lc];

	    if ((chr <= 57) && (chr >= 48)) {	/* number */
		chr -= 48;
		copyXPMArea(chr * 5 + 66, 57, 5, 6, 25 + q, 49);
		q += 5;
	    } else if (chr == 46) {	/* . */
		chr = chr - 45 + 9;
		copyXPMArea(chr * 5 + 66, 57, 5, 6, 25 + q, 49);
		q += 3;
	    } else {		/* letter */
		chr -= 65;
		copyXPMArea(chr * 5 + 2, 128, 5, 6, 25 + q, 49);
		q += 5;
	    }

	    lc++;
	}
    }
}


void 
print_usage()
{
    printf("\n wmstock version: %s\n", WMSTOCK_VERSION);
    printf("----------------------------------------------------------------------------\n");
    printf(" usage: wmstock [OPTIONS]... StockSymbol [StockSymbol ...]\n");
    printf("-------------------------------- Options: ----------------------------------\n");
    printf(" -h, --help             Print this message\n");
    printf(" -v, --version          Print some version info\n");
    printf(" -e, --newversion       Checks the website for a new version\n");
    printf(" -b, --browser=BROWSER  Sets the web browser for the program to use.\n");
    printf(" -d, --delay=DELAY      Delay between updates in seconds(default=300)\n");
    printf(" -l, --dldelay=DELAY    Approx time in seconds to download/parse the stock\n");
    printf("                           data (default=0)\n");
    printf("                          Note: This is deprecated...  You should not need\n");
    printf("                                to use this unless you have problems!\n");
    printf(" -u, --nonuke           Don't delete previously downloaded quotes.\n");
    printf("                          Note: When using this, wmstock will show old\n");
    printf("                                data onstartup!\n");
    printf(" -t, --time2next=DELAY  Amount of time waited before moving to the next\n");
    printf("                          ticker symbol. (default=10s)\n");
    printf("                          Note: This option is ignored if only one\n");
    printf("                                StockSymbol is given!\n");
    printf(" -r, --reverse          Reverse the order that the quotes are displayed\n");
    printf(" -o, --open=TIME        Times during which the market is open.\n");
    printf("				see manpage for details\n");
    printf("-----------------------------------------------------------------------------\n\n");
    printf(" Example usage: wmstock --delay=10 --open=8:30-16:00 --time2next=20 FDX OV HWP\n");

}

int 
get_data_from_file(char process_name[], char FileName[])
{
    FILE           *fp;
    int             tempv;
    char            data[256],
                    tmpfilenam[256];

#ifdef MLF_DEBUG
    fprintf(stderr, "get_data_from_file::filename: %s", FileName);
#endif

    if ((fp = fopen(FileName, "r")) != NULL) {
#ifdef MLF_DEBUG
	fprintf(stderr, " opened successfully.\n");
#endif

	if (fgets(data, sizeof(data), fp) == NULL) {
	    fprintf(stderr, "%s::There was an error while reading data from %s.\n \
				Are you connected to the Internet?\n", process_name, FileName);

	    sprintf(tmpfilenam, "%s/%s", DirName, FileName);
	/* remove the file */
#ifdef MLF_DEBUG
	    fprintf(stderr, " removing %s\n", data);
#endif
	    tempv = unlink(tmpfilenam);
	    if (tempv < 0) {
		fprintf(stderr, "Error while trying to delete %s in %s\n", tmpfilenam, process_name);
	    }
	    reset_ticker(curr_t->ts);
	    return -1;
	}
	fgets(data, sizeof(data), fp);
	strcpy(curr_t->ts.company_name, data);

	fgets(data, sizeof(data), fp);
	curr_t->ts.last_price = atof(data);

	fgets(data, sizeof(data), fp);
	strcpy(curr_t->ts.last_trade_date, data);

	fgets(data, sizeof(data), fp);
	strcpy(curr_t->ts.last_trade_time, data);

	fgets(data, sizeof(data), fp);
	strcpy(curr_t->ts.change, data);

	fgets(data, sizeof(data), fp);
	strcpy(curr_t->ts.percent_change, data);

	fgets(data, sizeof(data), fp);
	curr_t->ts.volume = atol(data);

	fgets(data, sizeof(data), fp);
	curr_t->ts.average_volume = atol(data);

	fgets(data, sizeof(data), fp);
	strcpy(curr_t->ts.bid, data);

	fgets(data, sizeof(data), fp);
	strcpy(curr_t->ts.ask, data);

	fgets(data, sizeof(data), fp);
	curr_t->ts.previous_close = atof(data);

	fgets(data, sizeof(data), fp);
	curr_t->ts.open = atof(data);

	fgets(data, sizeof(data), fp);
	strcpy(curr_t->ts.days_range, data);

	fgets(data, sizeof(data), fp);
	strcpy(curr_t->ts.fifty_two_week_range, data);

	fgets(data, sizeof(data), fp);
	curr_t->ts.eps = atof(data);

	fgets(data, sizeof(data), fp);
	curr_t->ts.pe_ratio = atof(data);

	fgets(data, sizeof(data), fp);
	strcpy(curr_t->ts.div_date, data);

	fgets(data, sizeof(data), fp);
	curr_t->ts.div_per_share = atof(data);

	fgets(data, sizeof(data), fp);
	curr_t->ts.div_yield = atof(data);

	fgets(data, sizeof(data), fp);
	strcpy(curr_t->ts.market_cap, data);

#ifdef MLF_DEBUG
	printf("Company Name: %s\nLast Price: %f\nChange: %s\n", curr_t->ts.company_name, curr_t->ts.last_price, curr_t->ts.change);
#endif

	fclose(fp);

    /* if these 3 important things occur we *probably* dont have any data */
	if ((curr_t->ts.previous_close == 0) &&
	    (strcmp(curr_t->ts.last_trade_date, "N/A")) &&
	    (strcmp(curr_t->ts.company_name, curr_t->ts.ticker))) {
	    reset_ticker(curr_t->ts);
	    return -2;
	} else
	    return 0;
    } else {
	reset_ticker(curr_t->ts);

#ifdef MLF_DEBUG
	fprintf(stderr, " failed to open.\n");
#endif

	return -1;
    }
}

void 
download_data(char ticker[])
{
	int wstatus;
	int cpid;
    char            command[256];

#ifdef MLF_DEBUG
    fprintf(stderr, "Downloading Data...");
#endif

/* Execute Perl script to grab the stock quote */
    sprintf(command, "getquote %s", ticker);
//    system(command);

if ((cpid=fork())==0)		/* child needs to exec the script */
{

#ifdef MLF_DEBUG
	fprintf(stderr,"about to exec getquote\n");
#endif

	execlp("getquote","getquote",ticker,NULL);
}

while (!(waitpid(cpid, &wstatus, WNOHANG)))
{
    FD_ZERO(&fds);              /* empties the set */
    FD_SET(xfiledesc, &fds);
         if (FD_ISSET(xfiledesc, &fds)) {
            /* someone clicked something */
                while (XPending(display)) {
                    XNextEvent(display, &event);
                    switch (event.type) {
                    case Expose:
#ifdef MLF_DEBUG
        fprintf(stderr, "Expose\n");
#endif
                        RedrawWindow();
                        break;
                    case ButtonPress:
                        ButtonPressEvent(&event.xbutton);
                        break;
                    /* 
                     * case KeyPress: KeyPressEvent(&event.xkey); break; */
                    case ButtonRelease:
                        break;
                    case EnterNotify:
                        XSetInputFocus(display, PointerRoot, RevertToParent, CurrentTime);
                        break;
                    case LeaveNotify:
                        XSetInputFocus(display, PointerRoot, RevertToParent, CurrentTime);
                        break;
                    }
	}
}


}


#ifdef MLF_DEBUG
    fprintf(stderr, "...finished\n");
#endif

    ForceUpdate = 1;
}

void 
reset_ticker(struct ticker_struct tick)
{
    strcpy(tick.ticker, "ERROR");
    tick.company_name[0] = '\0';
    tick.last_trade_date[0] = '\0';
    tick.last_trade_time[0] = '\0';
    tick.change[0] = '\0';
    tick.percent_change[0] = '\0';
    tick.bid[0] = '\0';
    tick.ask[0] = '\0';
    tick.days_range[0] = '\0';
    tick.fifty_two_week_range[0] = '\0';
    tick.div_date[0] = '\0';
    tick.market_cap[0] = '\0';

    tick.last_price = -99999.99;
    tick.previous_close = 0.0;
    tick.open = 0.0;
    tick.eps = 0.0;
    tick.pe_ratio = 0.0;
    tick.div_per_share = 0.0;
    tick.div_yield = 0.0;

    tick.volume = 0;
    tick.average_volume = 0;

    tick.oldtime.tv_sec = 0;
    tick.oldtime.tv_usec = 0;
    tick.newtime.tv_sec = 0;
    tick.newtime.tv_usec = 0;
    tick.timediff = 0;
}

void 
ButtonPressEvent(XButtonEvent * xev)
{
    char cmdline[256];
    char URL[80];

    if ((xev->button == Button1) && (xev->type == ButtonPress)) {

#ifdef MLF_DEBUG
	fprintf(stderr, "1 Left Click Detected.\n");
#endif

	if (curr_t->next == NULL)	/* advance in the list */
	    curr_t = head;
	else
	    curr_t = curr_t->next;
	display_data(curr_t->ts);	/* paste up the data */
	disp_t = curr_t;

    /* Make changes visible */
	RedrawWindow();
    }

    if ((xev->button == Button3) && (xev->type == ButtonPress)) {

#ifdef MLF_DEBUG
	fprintf(stderr, "1 right Click Detected.\n");
#endif

	sprintf(URL,"\"http://quote.yahoo.com/q?s=%s",disp_t->ts.ticker);
	strcat(URL,"&d=1d\"");

#ifdef MLF_DEBUG
	fprintf(stderr, "URL: %s\n", URL);
#endif

#ifdef MLF_DEBUG
	fprintf(stderr, "WebBrowser: %s\n", WebBrowser);
#endif

	if (WebBrowser[0]!='\0')
	{
		sprintf(cmdline, "%s %s", WebBrowser, URL);
		if (!fork())
		{
			if (system(cmdline) < 0)
			{
				fprintf (stderr, "Could not start your browser (%s).\n", WebBrowser);
			}
			exit(0);

		}
		
	}

    }
	
}

void 
display_error(int errornum)
{

#ifdef MLF_DEBUG
    fprintf(stderr, "Error number %i occured when trying to open the data file.\n", errornum);
#endif
/* bad ticker symbol */
    if (errornum == -2) {
	ClearWindow();
	RedrawWindow();
	drawstring("BAD TICKER", 6, 128, 6);
	RedrawWindow();

    /* delete the bad ticker symbol */
	if (delete(curr_t) == 1) {
#ifdef MLF_DEBUG
	    fprintf(stderr, "Symbol Deleted!\n");
#endif
	    num_quotes--;
	    regalarm=0;		/* set a short alarm next time */
	} else {
	    fprintf(stderr, "Delete of ticker %s failed.\n", curr_t->ts.ticker);
	    exit(-1);
	}
	RedrawWindow();
    }
/* no stock data available */
    else if (errornum == -1) {
	ClearWindow();
	RedrawWindow();
	drawstring("NO DATA", 6, 128, 6);
	RedrawWindow();
	ForceUpdate = 0;	/* force the update */
    }
    return;
}

/* Convert STR to a positive integer, storing the result in *OUT. If STR
 * is not a valid integer, return -1 (otherwise 0). Taken from GNU
 * grep-2.3 */

static int 
ck_atol(char const *str, long *out)
{
    char const     *p;
    for (p = str; *p; p++)
	if (*p < '0' || *p > '9')
	    return -1;

    *out = atol(optarg);
    return 0;
}

/* ignore the signal since its an alarm and will be handled by the loop */
void 
mySignalHandler(int signal_number)
{
    signal(SIGALRM, mySignalHandler); /* register my SIG ALARM handler again*/
    return;
}

/* draw a string to the screen, needs to be more robust */
void 
drawstring(char *str, int xoffset, int yrow, int ypos)
{
    int             lc = 0;
    int             q = 0;
    int             chr;

    while (str[lc] != '\0') {
	chr = (int) (str[lc] - 65);
	copyXPMArea(chr * 5 + 2, yrow, 5, 6, xoffset + q, ypos);
	q += 5;
	lc++;
    }
}

/* test to see if the markets are open */
int are_markets_open()
{
	char temptime[200];

	/* we dont care, so they are open */
	if (CheckMarkets != 1)
		return 1;

	 // get date
        if (gettimeofday (&tv, NULL) == -1)
        {
                fprintf (stderr, "failed to get the current date/time.\n");
                exit(1);
        }

        curr = *localtime(&tv.tv_sec);
        curr_secs = mktime (&curr);

        /* find midnight */
        midnight = *localtime(&tv.tv_sec);
        midnight.tm_hour=0;
        midnight.tm_min=0;
        midnight.tm_sec=0;
        midnight_secs = mktime (&midnight);

	mopen = *localtime(&midnight_secs);
	mopen.tm_hour = mopen_hr;
	mopen.tm_min  = mopen_mn;

	strcpy(temptime, asctime (&mopen));

#ifdef TIME_DEBUG
	printf("mopen: %s\n",temptime);	
#endif

        mopen_secs = mktime (&mopen);

        mclose = *localtime(&midnight_secs);
	mclose.tm_hour = mclose_hr;
	mclose.tm_min  = mclose_mn;

	strcpy(temptime, asctime (&mclose));
#ifdef TIME_DEBUG
	printf("mclose: %s\n",temptime);	
#endif

        mclose_secs = mktime (&mclose);

#ifdef TIME_DEBUG
        printf ("current secs: %ld\n", curr_secs);
        printf ("market open secs: %ld\n", mopen_secs);
        printf ("market close secs: %ld\n", mclose_secs);
#endif

        if ( (curr_secs < mclose_secs) && (curr_secs > mopen_secs) )
        {
                //download quotes, markets open!
#ifdef TIME_DEBUG
        printf ("market is open!\n");
#endif
		return 1;
        }
	
#ifdef TIME_DEBUG
        printf ("market is closed!\n");
#endif
	
	return 0;
}

