/*
    Xbaby - A game of skill.  Try to find the babies under your windows.
    
    Copyright 1991 by J.T. Anderson

    jta@locus.com
    
    This program may be freely distributed provided that all
    copyright notices are retained.

    To build:
      cc -o xbaby baby.c -lX11 [-lsocketorwhatever] [-lm] [-l...]

    Dedicated to Greg McFarlane.   (gregm@otc.otca.oz.au)
    
    Squish option contributed by Rick Petkiewizc (rick@locus.com)
    
    Virtual root code adapted from patch sent by Colin Rafferty who
    borrowed it from Tom LaStrange.  Several other folks sent similar
    fixes.

    Converted (perverted?) from roaches to babies by Michael Lawley
*/

/* @(#)xbaby.c	1.5 4/2/91 11:53:31 */

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>
#include <X11/Xatom.h>

#include <stdio.h>
#include <math.h>
#include <signal.h>

#if __STDC__
#include <stdlib.h>
#else
#include <malloc.h>
long strtol();
double strtod();
char *getenv();
#endif

/*	If strtod not available, use below */

#ifdef Encore
double	atof();
# define strtod(s, p)	atof(s)
#endif

char Copyright[] = "Xbaby\nCopyright 1991 M.J.Lawley\nBased on Xbaby by J.T. Anderson";

#include "babymap.h"

typedef unsigned long Pixel;
typedef int ErrorHandler();

#define SCAMPER_EVENT	(LASTEvent + 1)

#define bpIndex(baby)	(((baby->intX ^ baby->intY) + (baby->steps >> 1)) & 0x01)

#if !defined(GRAB_SERVER)
#define GRAB_SERVER	0
#endif

Display *display;
int screen;
Window rootWin;
unsigned int display_width, display_height;
int center_x, center_y;
GC gc;
GC gutsGC;
char *display_name = NULL;
Pixel black, white;

int done = 0;
int eventBlock = 0;
int errorVal = 0;
Bool squishbaby = False;
Pixmap squishMap;
Bool squishWinUp = False;

typedef struct baby {
    BabyMap *bp[2];
    int index;
    float x;
    float y;
    int intX;
    int intY;
    int hidden;
    int turnLeft;
    int steps;
} Baby;

Baby *babies;
int maxbabies = 10;
int curbabies = 0;
float babySpeed = 10.0;

Region rootVisible = NULL;

void Usage();
void SigHandler();
void Addbaby();
void Movebaby();
void Drawbabies();
void CoverRoot();
int CalcRootVisible();
int MarkHiddenbabies();
Pixel AllocNamedColor();
Window FindRootWindow();

void
main(ac, av)
int ac;
char *av[];
{
    XGCValues xgcv;
    int ax;
    char *arg;
    BabyMap *bp[2];
    int rx;
    float angle;
    XEvent ev;
    char *babyColor = NULL;
    char *gutsColor = NULL;
    int nVis;
    int needCalc;
    Window squishWin;
    XSetWindowAttributes xswa;
    int	i;


    
    /*
       Process command line options.
    */
    for (ax=1; ax<ac; ax++) {
	arg = av[ax];
	if (strcmp(arg, "-display") == 0) {
	    display_name = av[++ax];
	}
	else if (strcmp(arg, "-rc") == 0) {
	    babyColor = av[++ax];
	}
	else if (strcmp(arg, "-speed") == 0) {
	    babySpeed = strtod(av[++ax], (char **)NULL);
	}
	else if (strcmp(arg, "-babies") == 0) {
	    maxbabies = strtol(av[++ax], (char **)NULL, 0);
	}
	else if (strcmp(arg, "-squish") == 0) {
	    squishbaby = True;
	}
	else if (strcmp(arg, "-rgc") == 0) {
	    gutsColor = av[++ax];
	}
	else {
	    Usage();
	}
    }

    srand((int)time((long *)NULL));
    
    /*
       Catch some signals so we can erase any visible babies.
    */
    signal(SIGKILL, SigHandler);
    signal(SIGINT, SigHandler);
    signal(SIGTERM, SigHandler);
    signal(SIGHUP, SigHandler);

    display = XOpenDisplay(display_name);
    if (display == NULL) {
	if (display_name == NULL) display_name = getenv("DISPLAY");
	(void) fprintf(stderr, "%s: cannot connect to X server %s\n", av[0],
	    display_name ? display_name : "(default)");
	exit(1);
    }

    screen = DefaultScreen(display);
    rootWin = FindRootWindow();
    black = BlackPixel(display, screen);
    white = WhitePixel(display, screen);

    if (!babyColor)
    {
	if (DefaultDepth(display, screen) > 1)
	    babyColor = "pink";
	else
	    babyColor = "white";
    }
    if (!gutsColor)
    {
	if (DefaultDepth(display, screen) > 1)
	    gutsColor = "red";
	else
	    gutsColor = "white";
    }

    display_width = DisplayWidth(display, screen);
    display_height = DisplayHeight(display, screen);
    center_x = display_width / 2;
    center_y = display_height / 2;
    
    /*
       Create baby pixmaps at several orientations.
    */
    for (ax=0; ax<360; ax+=BABY_ANGLE) {
	rx = ax / BABY_ANGLE;
	angle = rx * 0.261799387799;
	for (i=0; i < 2; i++)
	{
	    bp[i] = &BabyPix[i][rx];
	    bp[i]->pixmap = XCreateBitmapFromData(display, rootWin,
		bp[i]->BabyBits, bp[i]->width, bp[i]->height);
	    bp[i]->sine = sin(angle);
	    bp[i]->cosine = cos(angle);
	}
    }

    /*
      Create the squished pixmap
    */
    if (squishbaby) {
    	squishMap = XCreateBitmapFromData(display, rootWin,
		squish_bits, squish_width, squish_height);
    }

    babies = (Baby *)malloc(sizeof(Baby) * maxbabies);

    gc = XCreateGC(display, rootWin, 0L, &xgcv);
    XSetForeground(display, gc, AllocNamedColor(babyColor, white));
    XSetFillStyle(display, gc, FillStippled);

    if (squishbaby && gutsColor != NULL) {
        gutsGC = XCreateGC(display, rootWin, 0L, &xgcv);
        XSetForeground(display, gutsGC, AllocNamedColor(gutsColor, black));
        XSetFillStyle(display, gutsGC, FillStippled);
    }
    else
	gutsGC = gc;
    
    while (curbabies < maxbabies)
	Addbaby();
    
    XSelectInput(display, rootWin, ExposureMask | SubstructureNotifyMask);

    if (squishbaby) {
	xswa.event_mask = ButtonPressMask;
	xswa.override_redirect = True;
	squishWin = XCreateWindow(display, rootWin, 0, 0,
			display_width, display_height, 0,
			CopyFromParent, InputOnly, CopyFromParent,
			CWOverrideRedirect | CWEventMask, &xswa);
	XLowerWindow(display, squishWin);
    }
    
    needCalc = 1;
    while (!done) {
	if (XPending(display)) {
	    XNextEvent(display, &ev);
	}
	else {
	    if (needCalc) {
		needCalc = CalcRootVisible();
	    }
	    if (needCalc)
		nVis = 0;
	    else
	    	nVis = MarkHiddenbabies();
	    if (nVis) {
		ev.type = SCAMPER_EVENT;
		if (!squishWinUp && squishbaby) {
		    XMapWindow(display, squishWin);
		    squishWinUp = True;
		}
	    }
	    else {
		if (squishWinUp && squishbaby) {
		    XUnmapWindow(display, squishWin);
		    squishWinUp = False;
		}
		if (needCalc == 0)
		    Drawbabies();
		eventBlock = 1;
		XNextEvent(display, &ev);
		eventBlock = 0;
	    }
	}
	
	switch (ev.type) {
	    
	    case SCAMPER_EVENT:
		for (rx=0; rx<curbabies; rx++) {
		    if (!babies[rx].hidden)
			Movebaby(rx);
		}
		Drawbabies();
		XSync(display, False);
		break;
		
	    case UnmapNotify:
		if (ev.xunmap.window != squishWin)
		    needCalc = 1;
		break;

	    case MapNotify:
	    case Expose:
	    case ConfigureNotify:
		needCalc = 1;
		break;
	
	    case ButtonPress:
		checkSquish(&ev);
		break;
		
	}
    }
    
    CoverRoot();
    
    XCloseDisplay(display);
}

#define USEPRT(msg) fprintf(stderr, msg)

void
Usage()
{
    USEPRT("Usage: xbaby [options]\n\n");
    USEPRT("Options:\n");
    USEPRT("       -display displayname\n");
    USEPRT("       -rc      babycolor\n");
    USEPRT("       -babies numbabies\n");
    USEPRT("       -speed   babyspeed\n");
    USEPRT("       -squish\n");
    USEPRT("       -rgc     babygutscolor\n");
    
    exit(1);
}

void
SigHandler()
{
       
    /*
       If we are blocked, no babies are visible and we can just bail
       out.  If we are not blocked, then let the main procedure clean
       up the root window.
    */
    if (eventBlock) {
	XCloseDisplay(display);
	exit(0);
    }
    else {
	done = 1;
    }
}

/*
   Find the root or virtual root window.
*/
Window
FindRootWindow()
{
    Window rootWin;
    Window realRoot;
    Atom swmVroot;
    Window rootReturn, parentReturn, *children;
    unsigned int numChildren;
    int cx;
    Atom actualType;
    Atom actualFormat;
    unsigned long nItems;
    unsigned long bytesAfter;
    Window *newRoot;
    
    /*
       Get real root window.
    */
    realRoot = rootWin = RootWindow(display, screen);
    
    /*
       Get atom for virtual root property.  If the atom doesn't
       exist, we can assume the corresponding property does not
       exist. 
    */
    swmVroot = XInternAtom(display, "__SWM_VROOT", True);
    
    if (swmVroot == None)
	return rootWin;
    
    /*
       Run children of root, looking for a virtual root.
    */
    XQueryTree(display, rootWin, &rootReturn, &parentReturn, 
		    &children, &numChildren);
    for (cx=0; cx<numChildren; cx++) {
	newRoot = NULL;
	nItems = 0;
	if (XGetWindowProperty(display, children[cx], swmVroot, 0L, 1L, False,
	    XA_WINDOW, &actualType, (int *)&actualFormat, &nItems,
	    &bytesAfter,(unsigned char **) &newRoot) == Success && actualFormat != None) {
		if (nItems >= 1) {
		    rootWin = *newRoot;
		}
		if (newRoot) {
		    XFree(newRoot);
		}
	}
	if (rootWin != realRoot) break;
    }
    XFree(children);
    
    return rootWin;
}

/*
   Generate random integer between 0 and maxVal-1.
*/
int
RandInt(maxVal)
int maxVal;
{
	return rand() % maxVal;
}

/*
   Check for baby completely in specified rectangle.
*/
int
BabyInRect(baby, rx, ry, x, y, width, height)
Baby *baby;
int rx;
int ry;
int x;
int y;
unsigned int width;
unsigned int height;
{
    int	i = bpIndex(baby);

    if (rx < x) return 0;
    if ((rx + baby->bp[i]->width) > (x + width)) return 0;
    if (ry < y) return 0;
    if ((ry + baby->bp[i]->height) > (y + height)) return 0;
    
    return 1;
}

/*
   Check for baby overlapping specified rectangle.
*/
int
babyOverRect(baby, rx, ry, x, y, width, height)
Baby *baby;
int rx;
int ry;
int x;
int y;
unsigned int width;
unsigned int height;
{
    int	i = bpIndex(baby);

    if (rx >= (x + width)) return 0;
    if ((rx + baby->bp[i]->width) <= x) return 0;
    if (ry >= (y + height)) return 0;
    if ((ry + baby->bp[i]->height) <= y) return 0;
    
    return 1;
}

/*
   Give birth to a baby.
*/
void
Addbaby()
{
    Baby *r;
    
    if (curbabies < maxbabies) {
	r = &babies[curbabies++];
	r->index = RandInt(BABY_HEADINGS);
	r->bp[0] = &BabyPix[0][r->index];
	r->bp[1] = &BabyPix[1][r->index];
	r->x = RandInt(display_width - r->bp[0]->width);
	r->y = RandInt(display_height - r->bp[0]->height);
	r->intX = -1;
	r->intY = -1;
	r->hidden = 0;
	r->steps = RandInt(200);
	r->turnLeft = RandInt(100) >= 50;
    }
}

/*
   Turn a baby.
*/
void
Turnbaby(baby)
Baby *baby;
{
    int i = bpIndex(baby);

    if (baby->index != (baby->bp[i] - BabyPix[i])) return;

    if (baby->turnLeft) {
	baby->index += (RandInt(30) / 10) + 1;
	if (baby->index >= BABY_HEADINGS)
	    baby->index -= BABY_HEADINGS;
    }
    else {
	baby->index -= (RandInt(30) / 10) + 1;
	if (baby->index < 0)
	    baby->index += BABY_HEADINGS;
    }
}

/*
   Move a baby.
*/
void
Movebaby(rx)
int rx;
{
    Baby *baby;
    Baby *r2;
    float newX;
    float newY;
    int i, ii;
    
    baby = &babies[rx];
    i = bpIndex(baby);

    newX = baby->x + (babySpeed * baby->bp[i]->cosine);
    newY = baby->y - (babySpeed * baby->bp[i]->sine);
    
    if (BabyInRect(baby, (int)newX, (int)newY, 
			    0, 0, display_width, display_height)) {
	
	baby->x = newX;
	baby->y = newY;

	if (baby->steps-- <= 0) {
	    Turnbaby(baby);
	    baby->steps = RandInt(200);
	}

	for (ii=rx+1; ii<curbabies; ii++) {
	    r2 = &babies[ii];
	    i = bpIndex(r2);
	    if (babyOverRect(baby, (int)newX, (int)newY,
		r2->intX, r2->intY, r2->bp[i]->width, r2->bp[i]->height)) {
	
		Turnbaby(baby);
	    }
	}
    }
    else {
	Turnbaby(baby);
    }
}
    
/*
   Draw all babies.
*/
void
Drawbabies()
{
    Baby *baby;
    int rx, i;
    
    for (rx=0; rx<curbabies; rx++) {
	baby = &babies[rx];
	i = bpIndex(baby);
	
	if (baby->intX >= 0 && baby->bp[i] != NULL) {
	    XClearArea(display, rootWin, baby->intX, baby->intY,
		baby->bp[i]->width, baby->bp[i]->height, False);
	}
    }
    
    for (rx=0; rx<curbabies; rx++) {
	baby = &babies[rx];
	i = bpIndex(baby);
	
	if (!baby->hidden) {
	    baby->intX = baby->x;
	    baby->intY = baby->y;
	    baby->bp[i] = &BabyPix[i][baby->index];
    
	    XSetStipple(display, gc, baby->bp[i]->pixmap);
	    XSetTSOrigin(display, gc, baby->intX, baby->intY);
	    XFillRectangle(display, rootWin, gc,
		baby->intX, baby->intY, baby->bp[i]->width, baby->bp[i]->height);
	}
	else {
	    baby->intX = -1;
	}
    }
}

/*
   Cover root window to erase babies.
*/
void
CoverRoot()
{
    XSetWindowAttributes xswa;
    long wamask;
    Window babyWin;
    
    xswa.background_pixmap = ParentRelative;
    xswa.override_redirect = True;
    wamask = CWBackPixmap | CWOverrideRedirect;
    babyWin = XCreateWindow(display, rootWin, 0, 0,
		    display_width, display_height, 0, CopyFromParent,
		    InputOutput, CopyFromParent, wamask, &xswa);
    XLowerWindow(display, babyWin);
    XMapWindow(display, babyWin);
    XFlush(display);
}    

#if !GRAB_SERVER

int
babyErrors(dpy, err)
Display *dpy;
XErrorEvent *err;
{
    errorVal = err->error_code;
    
    return 0;
}

#endif /* GRAB_SERVER */

/*
   Calculate Visible region of root window.
*/
int
CalcRootVisible()
{
    Region covered;
    Region visible;
    Window *children;
    int nChildren;
    Window dummy;
    XWindowAttributes wa;
    int wx;
    XRectangle rect;
    int winX, winY;
    unsigned int winHeight, winWidth;
    unsigned int borderWidth;
    unsigned int depth;
    
    /*
       If we don't grab the server, the XGetWindowAttribute or XGetGeometry
       calls can abort us.  On the other hand, the server grabs can make for
       some annoying delays.
    */
#if GRAB_SERVER
    XGrabServer(display);
#else
    XSetErrorHandler(babyErrors);
#endif

    /*
       Get children of root.
    */
    XQueryTree(display, rootWin, &dummy, &dummy, &children,(unsigned int *) &nChildren);
    
    /*
       For each mapped child, add the window rectangle to the covered
       region.
    */
    covered = XCreateRegion();
    for (wx=0; wx<nChildren; wx++) {
	if (XEventsQueued(display, QueuedAlready)) {
	    XDestroyRegion(covered);
	    return 1;
	}
	errorVal = 0;
	XGetWindowAttributes(display, children[wx], &wa);
	if (errorVal) continue;
	if (wa.class == InputOutput && wa.map_state == IsViewable) {
	    XGetGeometry(display, children[wx], &dummy, &winX, &winY,
		&winWidth, &winHeight, &borderWidth, &depth);
	    if (errorVal) continue;
	    rect.x = winX;
	    rect.y = winY;
	    rect.width = winWidth + (borderWidth * 2);
	    rect.height = winHeight + (borderWidth * 2);
	    XUnionRectWithRegion(&rect, covered, covered);
	}
    }
    XFree(children);

#if GRAB_SERVER
    XUngrabServer(display);
#else
    XSetErrorHandler((ErrorHandler *)NULL);
#endif
    
    /*
       Subtract the covered region from the root window region.
    */
    visible = XCreateRegion();
    rect.x = 0;
    rect.y = 0;
    rect.width = display_width;
    rect.height = display_height;
    XUnionRectWithRegion(&rect, visible, visible);
    XSubtractRegion(visible, covered, visible);
    XDestroyRegion(covered);
    
    /*
       Save visible region globally.
    */
    if (rootVisible)
	XDestroyRegion(rootVisible);
    rootVisible = visible;
    
    
    /*
       Mark all babies visible.
    */
    for (wx=0; wx<curbabies; wx++) 
	babies[wx].hidden = 0;

    return 0;
}

/*
   Mark hidden babies.
*/
int
MarkHiddenbabies()
{
    int rx, i;
    Baby *r;
    int nVisible;
    
    nVisible = 0;
    for (rx=0; rx<curbabies; rx++) {
	r = &babies[rx];
	i = bpIndex(r);
	
	if (!r->hidden) {
	    if (r->intX > 0 && XRectInRegion(rootVisible, r->intX, r->intY,
			    r->bp[i]->width, r->bp[i]->height) == RectangleOut)
		r->hidden = 1;
	    else
		nVisible++;
	}
    }
    
    return nVisible;
}

/*
   Allocate a color by name.
*/
Pixel
AllocNamedColor(colorName, dfltPix)
char *colorName;
Pixel dfltPix;
{
	Pixel pix;
	XColor scrncolor;
	XColor exactcolor;

	if (XAllocNamedColor(display, DefaultColormap(display, screen),
		colorName, &scrncolor, &exactcolor)) {
		pix = scrncolor.pixel;
	}
	else {
		pix = dfltPix;
	}

	return pix;
}

/*
 *	squishCheck - Check to see if we have to squish any babies.
 */
checkSquish(buttonEvent)
XButtonEvent *buttonEvent;
{
    int x, y;
    int i;
    int rx;
    Baby *r;
/* */
    x = buttonEvent->x;
    y = buttonEvent->y;

    for (rx=0; rx<curbabies; rx++) {
	r = &babies[rx];
	if (r->bp[0] == NULL) continue;

	if (x > r->intX &&
	    x < (r->intX + r->bp[0]->width) &&
	    y > r->intY &&
	    y < (r->intY + r->bp[0]->height)) {
	    XSetStipple(display, gutsGC, squishMap);
	    XSetTSOrigin(display, gutsGC, r->intX, r->intY);
	    XFillRectangle(display, rootWin, gutsGC,
		r->intX, r->intY, squish_width, squish_height);

	    /*
	     * Delete the baby
	     */
	    for (i = rx; i < curbabies - 1; i++)
	        babies[i] = babies[i + 1];
	
	    curbabies--;
	    rx--;
	}
    }

    return;
}
