/*
 * Copyright (c) 1999, 2000, 2001, 2002 Seth Kingsley.  All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgment:
 *    This product includes software developed by Seth Kingsley and
 *    contributors.
 * 4. Neither the author's name, nor any of the contributors names may be
 *    used to endorse or promote products derived from this software without
 *    specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include    <X11/Xlib.h>
#include    <X11/Xlibint.h>
#include    <X11/Xproto.h>
#include    <X11/Xatom.h>
#include    <X11/Intrinsic.h>
#include    <X11/Xmu/WinUtil.h>
#include    <X11/Shell.h>
#include    <X11/StringDefs.h>
#include    <X11/extensions/xtraplib.h>
#include    <X11/extensions/xtraplibp.h>

#include    <esd.h>

#include    <sys/types.h>
#include    <sys/queue.h>
#include    <sys/time.h>
#include    <sys/stat.h>

#include    <unistd.h>
#include    <stdlib.h>
#include    <fcntl.h>
#include    <math.h>
#include    <pwd.h>
#include    <errno.h>
#include    <assert.h>
#include    <err.h>
#include    <sysexits.h>
#include    <dirent.h>

#ifndef lint
static const char rcsid[] =
	"$Id: xbelld.c,v 1.7 2002/10/27 00:28:36 sethk Exp $";
#endif /* !lint */

#define XskRFilename	"XskRFilename"
#define XskRKBytes	"XskRKBytes"
#define XskRPercent	"XskRPercent"

#define APP_CLASS	"XTrapBell"
#define AUDIO_SUFS	{"aiff", "au", "wav"}
#define DEF_CLASS	"default"
#define NCLI_HASH	256
#define NWIN_UNIT	4
#define CACHE_UNIT	1024

#define SET_EXIT(bd)	do { \
	    (bd)->bd_done = True; \
	    XtNoticeSignal((bd)->bd_fatsid); \
	} while (0)

#define CLIENT_ID(xid)	(((xid) >> _bd.bd_idshift) & _bd.bd_idmask)

#define _hash_size(h)	((h).h_size)

struct node
{
    void    *n_data;
    TAILQ_ENTRY(node) n_link;
};

struct hash
{
    u_long  h_size;
    TAILQ_HEAD(hash_bucket, node) *h_buckets;
    u_int   (*h_hash)(const void *_data);
    Boolean (*h_cmp)(const void *_data1, const void *_data2);
    void    (*h_free)(void *_data);
};

struct sample_desc;

struct sound_desc
{
    String		sd_class;
    char		*sd_fname;
    struct sample_desc	*sd_samp;
};

struct client_desc
{
    u_int   cd_id;
    String  cd_class;
    u_int   cd_nwin;
    Window  *cd_winlist;
};

struct sample_desc
{
    int			sd_sampid;
    size_t		sd_size;
    u_int		sd_refcnt; /* XXX: unused */
    time_t		sd_lastuse;
    struct sound_desc	*sd_sound;
    LIST_ENTRY(sample_desc) sd_link;
};

static struct belld_data
{
    /* Resource values */
    Boolean		bd_verb,
			bd_nofork;
    float		bd_vol;
    int			bd_cache;
    Boolean		bd_multi;
    String		bd_pidfn,
			bd_dir,
			bd_esdsvr;
    /* Private data */
    Display		*bd_disp;
    XErrorHandler	bd_oxerr;
    u_int		bd_idshift,
			bd_idmask,
			bd_myid;
    XtAppContext	bd_xtac;
    XETC		*bd_xetc;
    XtSignalId		bd_fatsid;
    int			bd_esdfd;
    Bool		bd_done;
    int			bd_opitch;
    struct hash		bd_clihash,
			bd_sndhash;
    struct sound_desc	*bd_defsnd;
    size_t		bd_smpsize;
    LIST_HEAD(bd_smplist, sample_desc) bd_smplist;
}   _bd;

struct sound_file
{
    char	    *sf_fname;
    AFfilehandle    sf_afh;
    AFfileoffset    sf_size;
    esd_format_t    sf_efmt;
    float	    sf_frlen;
    double	    sf_rate;
};

enum action {ACT_ADD, ACT_RM};

static void
_node_destroy(struct hash *hp, struct node *np)
{
    if (hp->h_free)
	hp->h_free(np->n_data);
    XtFree((char *)np);
}

static Boolean
_is_prime(u_int n)
{
    u_int   d, maxdiv = sqrt(n);

    assert((n & 0x1) == 0x1);
    for (d = 3; d < maxdiv; d+=2)
	if (n % d == 0)
	    return False;
    return True;
}

static void
_hash_setup(struct hash *hp)
{
    u_int   i;

    if ((hp->h_size & 0x1) == 0x0)
	++hp->h_size;
    while (_is_prime(hp->h_size) == False)
	hp->h_size+= 2;
    hp->h_buckets =
	(struct hash_bucket *)XtMalloc(sizeof(struct hash_bucket) *
				    hp->h_size);
    for (i = 0; i < hp->h_size; ++i)
	TAILQ_INIT(&(hp->h_buckets[i]));
}

static void
_hash_init(struct hash *hp,
	u_long size,
	u_int (*hash)(const void *_data),
	Boolean (*cmp)(const void *_data1, const void *_data2),
	void (*free)(void *_data))
{
    hp->h_size = size;
    _hash_setup(hp);
    hp->h_hash = hash;
    hp->h_cmp = cmp;
    hp->h_free = free;
}

static void
_hash_clear(struct hash *hp)
{
    u_int   i;

    if (!hp->h_size)
	return;
    for (i = 0; i < hp->h_size; ++i)
	if (!TAILQ_EMPTY(&(hp->h_buckets[i])))
	{
	    struct node	*np, *nnp;

	    for (np = TAILQ_FIRST(&(hp->h_buckets[i])); np; np = nnp)
	    {
		nnp = TAILQ_NEXT(np, n_link);
		TAILQ_REMOVE(&(hp->h_buckets[i]), np, n_link);
		_node_destroy(hp, np);
	    }
	}
}

static void
_hash_destroy(struct hash *hp)
{
    _hash_clear(hp);
    XtFree((char *)hp->h_buckets);
}

static void
_hash_resize(struct hash *hp, u_long size)
{
    assert(size != hp->h_size);
    if (hp->h_size)
    {
	_hash_clear(hp);
	XtFree((char *)hp->h_buckets);
    }
    hp->h_size = size;
    _hash_setup(hp);
}

static Boolean
_hash_nfind(struct hash *hp,
	struct hash_bucket **hbpp,
	struct node **npp,
	const void *data)
{
    u_int   i;

    if (!hp->h_size)
	return NULL;
    i = hp->h_hash(data) % hp->h_size;
    if (!TAILQ_EMPTY(&(hp->h_buckets[i])))
    {
	struct node *np;

	TAILQ_FOREACH(np, &(hp->h_buckets[i]), n_link)
	    if (hp->h_cmp(data, np->n_data) == True)
	    {
		if (hbpp)
		    *hbpp = &(hp->h_buckets[i]);
		*npp = np;
		return True;
	    }
    }
    return False;
}

static void *
_hash_find(struct hash *hp, const void *data)
{
    struct node	*np;

    if (_hash_nfind(hp, NULL, &np, data) == True)
	return np->n_data;
    return NULL;
}

static Boolean
_hash_insert(struct hash *hp, void *data)
{
    struct node	*np;

    if (_hash_find(hp, data))
	return False;
    np = XtNew(struct node);
    np->n_data = data;
    TAILQ_INSERT_TAIL(&(hp->h_buckets[hp->h_hash(data) % hp->h_size]),
	    np,
	    n_link);
    return True;
}

static void
_hash_remove(struct hash *hp, void *data)
{
    struct node		*np;
    struct hash_bucket	*hbp;

    if (_hash_nfind(hp, &hbp, &np, data) == True)
    {
	TAILQ_REMOVE(hbp, np, n_link);
	_node_destroy(hp, np);
    }
}

/* String Hash function from P.J. Weinberger C Compiler */
static u_int
_sdesc_hash(const void *data)
{
    char    *s = ((const struct sound_desc *)(data))->sd_class;
    u_int   val = 0;

    while (*s)
    {
	u_int	tmp;

	val = (val << 4) + *s;
	if ((tmp = (val & 0xf0000000)))
	{
	    val = val ^ (tmp >> 24);
	    val = val ^ tmp;
	}
	++s;
    }
    return val;
}

static Boolean
_sdesc_cmp(const void *data1, const void *data2)
{
    return (!strcmp(((const struct sound_desc *)(data1))->sd_class,
		((const struct sound_desc *)(data2))->sd_class)) ?
	True : False;
}

static void
_sdesc_free(void *data)
{
    struct sound_desc	*sdp = (struct sound_desc *)data;

    XtFree(sdp->sd_class);
    XtFree(sdp->sd_fname);
    XtFree((char *)sdp);
}

static struct client_desc *
_cdesc_new(u_int id, String class)
{
    struct client_desc	*cdp = XtNew(struct client_desc);

    cdp->cd_id = id;
    cdp->cd_class = XtNewString(class);
#ifdef DEBUG
    warnx("DEBUG: adding client id %d => %s",
	    cdp->cd_id,
	    cdp->cd_class);
#endif /* DEBUG */
    cdp->cd_nwin = 0;
    return cdp;
}

static void
_cdwin_addrm(Window w, enum action act)
{
    struct client_desc	cd, *cdp;
    u_int		i;

    cd.cd_id = CLIENT_ID(w);
    if (!(cdp = _hash_find(&(_bd.bd_clihash), &cd)))
    {
	if (act == ACT_RM)
	    return;
	else /* act == ACT_ADD */
	{
	    XClassHint	ch;

	    if (XGetClassHint(_bd.bd_disp, w, &ch))
	    {
		cdp = _cdesc_new(cd.cd_id, ch.res_class);
		_hash_insert(&(_bd.bd_clihash), cdp);
		XFree(ch.res_class);
		XFree(ch.res_name);
	    }
	    else
	    {
#ifdef DEBUG
		warnx("phantom window %08lx\n", w);
#endif /* DEBUG */
		return;
	    }
	}
    }
    for (i = 0; i < cdp->cd_nwin; ++i)
    {
	if (w == cdp->cd_winlist[i])
	{
	    if (act == ACT_ADD)
#ifdef DEBUG
		warnx("DEBUG: window %08lx already in winlist of "
			"client %d",
			w,
			cd.cd_id);
#else
		;
#endif /* DEBUG */
	    else
	    {
#ifdef DEBUG
		warnx("DEBUG: removing %08lx from winlist for client "
			"%d",
			w,
			cd.cd_id);
#endif /* DEBUG */
		bcopy(cdp->cd_winlist + i,
			cdp->cd_winlist + (i - 1),
			cdp->cd_nwin - i);
		if (cdp->cd_nwin == 1)
		    _hash_remove(&(_bd.bd_clihash), cdp);
		else if ((cdp->cd_nwin % NWIN_UNIT) == 1)
		    cdp->cd_winlist = (Window *)XtRealloc(
			    (char *)cdp->cd_winlist,
			    cdp->cd_nwin - 1);
		--cdp->cd_nwin;
	    }
	    return;
	}
    }
    if (act == ACT_ADD)
    {
	if (!cdp->cd_nwin)
	    cdp->cd_winlist = (Window *)XtMalloc(sizeof(Window) *
		    NWIN_UNIT);
	if (!(cdp->cd_nwin % NWIN_UNIT))
	    cdp->cd_winlist = (Window *)XtRealloc(
		    (char *)cdp->cd_winlist,
		    sizeof(Window) * (cdp->cd_nwin + NWIN_UNIT));
#ifdef DEBUG
	warnx("DEBUG: adding %08lx to winlist for client %d",
		w,
		cd.cd_id);
#endif /* DEBUG */
	cdp->cd_winlist[cdp->cd_nwin++] = w;
	XSelectInput(_bd.bd_disp, w, StructureNotifyMask);
    }
}

static u_int
_cdesc_hash(const void *data)
{
    return ((const struct client_desc *)(data))->cd_id;
}

static Boolean
_cdesc_cmp(const void *data1, const void *data2)
{
    return (((const struct client_desc *)(data1))->cd_id ==
	    ((const struct client_desc *)(data2))->cd_id) ?
	True : False;
}

static void
_cdesc_free(void *data)
{
    struct client_desc	*cdp = (struct client_desc *)data;

#ifdef DEBUG
    warnx("DEBUG: removing client %d => %s", cdp->cd_id, cdp->cd_class);
#endif /* DEBUG */
    XtFree(cdp->cd_class);
    if (cdp->cd_nwin)
	XtFree((char *)cdp->cd_winlist);
    XtFree((char *)cdp);
}

static Boolean
_win_hasprop(Window w, Atom prop)
{
    Atom    type;
    int	    fmt;
    u_long  items, bytes;
    u_char  *props;

    XGetWindowProperty(_bd.bd_disp,
	    w,
	    prop,
	    0, 0,
	    False,
	    AnyPropertyType,
	    &type, &fmt,
	    &items, &bytes,
	    &props);
    return (type != AnyPropertyType) ? True : False;
}

static Boolean
_find_client(Window w, Boolean desc)
{
    if (_win_hasprop(w, XA_WM_COMMAND) && _win_hasprop(w, XA_WM_CLASS))
	_cdwin_addrm(w, ACT_ADD);
    if (desc == True)
    {
	Window	junk;
	u_int	ncw;
	Window	*cws;

	XQueryTree(_bd.bd_disp, w, &junk, &junk, &cws, &ncw);
	if (ncw)
	{
	    u_int   i;

	    for (i = 0; i < ncw; ++i)
		if (_find_client(cws[i], False) == True)
		{
		    XFree(cws);
		    return True;
		}
	    for (i = 0; i < ncw; ++i)
		if (_find_client(cws[i], True) == True)
		{
		    XFree(cws);
		    return True;
		}
	}
    }
    return False;
}

static void
_chash_init(void)
{
    u_int   si;

    _hash_init(&(_bd.bd_clihash),
	    NCLI_HASH,
	    _cdesc_hash,
	    _cdesc_cmp,
	    _cdesc_free);
    for (si = 0; si < (u_int)ScreenCount(_bd.bd_disp); ++si)
    {
	Window	rwin = RootWindow(_bd.bd_disp, si);
	Window	junk;
	Window	*cws;
	u_int	ncw;
	u_int	wi;

	XSelectInput(_bd.bd_disp, rwin, SubstructureNotifyMask);
	XQueryTree(_bd.bd_disp, rwin, &junk, &junk, &cws, &ncw);
	if (ncw)
	{
	    for (wi = 0; wi < ncw; ++wi)
		_find_client(cws[wi], True);
	    XFree(cws);
	}
    }
}

static void
_aferr(long err, const char *msg)
{
    warnx("%s", msg);
}

static void
_samp_free(struct sample_desc *sdp)
{
#ifdef DEBUG
    warnx("DEBUG: destroying sample %d => %s",
	    sdp->sd_sampid,
	    sdp->sd_sound->sd_fname);
#endif /* DEBUG */
    esd_sample_free(_bd.bd_esdfd, sdp->sd_sampid);
    _bd.bd_smpsize-= sdp->sd_size;
    sdp->sd_sound->sd_samp = NULL;
    XtFree((char *)sdp);
}

static void
_samp_play(struct sample_desc *sdp, u_short vol)
{
    if (_bd.bd_verb == True)
	warnx("playing sample id %d (%s) from cache, volume %d",
		sdp->sd_sampid,
		sdp->sd_sound->sd_fname,
		vol);
    esd_set_default_sample_pan(_bd.bd_esdfd, sdp->sd_sampid, vol, vol);
    time(&(sdp->sd_lastuse));
    if (esd_sample_play(_bd.bd_esdfd, sdp->sd_sampid) < 0)
	warnx("could not play sample %s", sdp->sd_sound->sd_fname);
}

static void
_sfile_close(struct sound_file *sfp)
{
    afCloseFile(sfp->sf_afh);
    XFree(sfp->sf_fname);
}

static Boolean
_sfile_open(const char *fname, struct sound_file *sfp)
{
    char    *emsg;
    int	    affmt, afwid, afchns;

    sfp->sf_fname = XtMalloc(strlen(_bd.bd_dir) +
	    1 +
	    strlen(fname) +
	    1);
    sprintf(sfp->sf_fname, "%s/%s", _bd.bd_dir, fname);
    if ((sfp->sf_afh = afOpenFile(sfp->sf_fname,
		    "r",
		    AF_NULL_FILESETUP)) == AF_NULL_FILEHANDLE)
	return False;
    afGetSampleFormat(sfp->sf_afh, AF_DEFAULT_TRACK, &affmt, &afwid);
    if (afwid != 8 && afwid != 16)
    {
	emsg = "unsupported sample size";
	goto err;
    }
    afchns = afGetChannels(sfp->sf_afh, AF_DEFAULT_TRACK);
    if (afchns != 1 && afchns != 2)
    {
	emsg = "unsupported number of channels";
	goto err;
    }
    sfp->sf_size = afGetTrackBytes(sfp->sf_afh, AF_DEFAULT_TRACK);
    switch(afwid)
    {
	case 8: sfp->sf_efmt = ESD_BITS8; break;
	case 16: sfp->sf_efmt = ESD_BITS16; break;
	default: assert(0);
    }
    switch (afchns)
    {
	case 1: sfp->sf_efmt |= ESD_MONO; break;
	case 2: sfp->sf_efmt |= ESD_STEREO; break;
	default: assert(0);
    }
    sfp->sf_efmt |= ESD_STREAM | ESD_PLAY;
    sfp->sf_rate = afGetRate(sfp->sf_afh, AF_DEFAULT_TRACK);
    sfp->sf_frlen = afGetFrameSize(sfp->sf_afh, AF_DEFAULT_TRACK, 1);
    return True;

err:
    _sfile_close(sfp);
    warnx("%s: %s", sfp->sf_fname, emsg);
    return False;
}

static void
_sfile_rew(struct sound_file *sfp)
{
    if (afTellFrame(sfp->sf_afh, AF_DEFAULT_TRACK))
	afSeekFrame(sfp->sf_afh, AF_DEFAULT_TRACK, 0);
}

static void
_sfile_play(struct sound_file *sfp, u_short vol)
{
    int	efd;

    if (_bd.bd_verb == True)
	warnx("playing file %s, volume %d", sfp->sf_fname, vol);
    if ((efd = esd_play_stream(sfp->sf_efmt,
		    sfp->sf_rate,
		    _bd.bd_esdsvr,
		    sfp->sf_fname)) > 0)
    {
	_sfile_rew(sfp);
	esd_set_stream_pan(_bd.bd_esdfd, efd, vol, vol);
	esd_send_file(efd, sfp->sf_afh, sfp->sf_frlen);
	esd_close(efd);
    }
}

static void
_scache_clear(void)
{
    struct sample_desc	*sdp;

    if (!_bd.bd_cache)
	return;
    sdp = LIST_FIRST(&(_bd.bd_smplist));
    while (sdp)
    {
	struct sample_desc  *nsdp;
	nsdp = LIST_NEXT(sdp, sd_link);
	LIST_REMOVE(sdp, sd_link);
	_samp_free(sdp);
	sdp = nsdp;
    }
    _bd.bd_smpsize = 0;
}

static Boolean
_scache_free(size_t size)
{
    if (size > _bd.bd_cache)
	return False;
    while ((_bd.bd_cache - _bd.bd_smpsize) < size)
    {
	struct sample_desc  *sdp, *dsdp = NULL;

	LIST_FOREACH(sdp, &(_bd.bd_smplist), sd_link)
	    if (!dsdp || sdp->sd_lastuse < dsdp->sd_lastuse)
		dsdp = sdp;
	LIST_REMOVE(dsdp, sd_link);
	_samp_free(dsdp);
    }
    return True;
}

static void
_scache_add(struct sound_file *sfp, struct sound_desc *sdp)
{
    struct sample_desc	*sadp;

    if (_scache_free(sfp->sf_size) == False)
	return;
    sadp = XtNew(struct sample_desc);
    if ((sadp->sd_sampid = esd_sample_cache(_bd.bd_esdfd,
		    sfp->sf_efmt,
		    sfp->sf_rate,
		    sfp->sf_size,
		    sdp->sd_fname)) > 0)
    {
#ifdef DEBUG
	warnx("caching sample %d => %s",
		sadp->sd_sampid,
		sdp->sd_fname);
#endif /* DEBUG */
	_sfile_rew(sfp);
	esd_send_file(_bd.bd_esdfd, sfp->sf_afh, sfp->sf_frlen);
	if (esd_confirm_sample_cache(_bd.bd_esdfd) != sadp->sd_sampid)
	{
	    warnx("could not cache sample %s", sdp->sd_fname);
	    XtFree((char *)sadp);
	    return;
	}
	sadp->sd_size = sfp->sf_size;
	sadp->sd_lastuse = ~(time_t)0;
	sadp->sd_sound = sdp;
	sdp->sd_samp = sadp;
	LIST_INSERT_HEAD(&(_bd.bd_smplist), sadp, sd_link);
    }
    else
	XtFree((char *)sadp);
}

static void
_play(struct sound_desc *sdp, float per)
{
    u_short vol;

    if (_bd.bd_multi == False)
	esd_lock(_bd.bd_esdfd);
    if (per == 0.0)
	per = _bd.bd_vol;
    if (per < 0.0)
	per = -per;
    else
	per*= _bd.bd_vol;
    if (!(vol = per * ESD_VOLUME_BASE))
    {
#ifdef DEBUG
	warnx("muted sample, skipping");
#endif /* DEBUG */
	return;
    }
    if (sdp->sd_samp)
	_samp_play(sdp->sd_samp, vol);
    else
    {
	struct sound_file   sf;

	if (_sfile_open(sdp->sd_fname, &sf) == True)
	{
	    _sfile_play(&sf, vol);
	    _scache_add(&sf, sdp);
	    _sfile_close(&sf);
	}
    }
    if (_bd.bd_multi == False)
	esd_unlock(_bd.bd_esdfd);
}

static void
_cb_bell(XETC *tc, XETrapDatum *data, XPointer *unused)
{
    xBellReq		*breq = (xBellReq *)&(data->u.req);
    struct client_desc	cd, *cdp;
    struct sound_desc	sd, *sdp;

#ifdef __GNUC__
    (void)tc;
    (void)unused;
#endif /* __GNUC__ */
    sd.sd_class = NULL;
    assert(XETrapHeaderIsRequest(&(data->hdr)));
    assert(breq->reqType == X_Bell);
    cd.cd_id = XETrapGetHeaderClient(&(data->hdr));
    if (cd.cd_id == _bd.bd_myid)
	return;
    cdp = (struct client_desc *)_hash_find(&(_bd.bd_clihash), &cd);
#ifdef DEBUG
	warnx("DEBUG: trapped bell request from client %d (%s), volume "
		"%d",
		cd.cd_id,
		(cdp) ? cdp->cd_class : "???",
		breq->percent);
#endif /* DEBUG */
    if (cdp)
	sd.sd_class = cdp->cd_class;
    if (sd.sd_class &&
	    (sdp = (struct sound_desc *)_hash_find(&(_bd.bd_sndhash),
						   (void *)&sd)))
	_play(sdp, (float)breq->percent / 100);
    else if (_bd.bd_defsnd)
	_play(_bd.bd_defsnd, (float)breq->percent / 100);
    else
    {
	XKeyboardControl    kc;

	if (_bd.bd_verb == True)
	    warnx("no matching audio file; synthesizing XBell request");
	kc.bell_pitch = _bd.bd_opitch;
	XChangeKeyboardControl(_bd.bd_disp, KBBellPitch, &kc);
	XBell(_bd.bd_disp, breq->percent);
	kc.bell_pitch = 0;
	XChangeKeyboardControl(_bd.bd_disp, KBBellPitch, &kc);
    }
}

#if 0
static Boolean
_is_root(Window w)
{
    u_int   i;

    for (i = 0; i < ScreenCount(_bd.bd_disp); ++i)
	if (w == RootWindow(_bd.bd_disp, i))
	    return True;
    return False;
}
#endif /* 0 */

static void
_cb_strmod(XEvent ev)
{
    if (ev.type == CreateNotify &&
	    (_win_hasprop(ev.xcreatewindow.window,
			  XA_WM_COMMAND) == False &&
	     _win_hasprop(ev.xcreatewindow.window,
		 XA_WM_CLASS) == False))
	return;
#ifdef DEBUG
    warnx("DEBUG: trapped %s event, window %08lx",
	    XEEventIDToString(ev.type, _bd.bd_xetc),
	    ev.xcreatewindow.window);
#endif /* DEBUG */
    _cdwin_addrm(ev.xcreatewindow.window, ev.type == CreateNotify ?
	    ACT_ADD : ACT_RM);
}

static Boolean
_rescan(void)
{
    int		    dfd;
    char	    *dbuf;
    struct dirent   *dep;
    struct stat	    st;
    int		    nb;
    u_int	    nent,
		    i;

    if (_bd.bd_verb == True)
	warnx("searching audio dir %s", _bd.bd_dir);
    if ((dfd = open(_bd.bd_dir, O_RDONLY)) == -1)
    {
	warn("cannot open audio directory %s", _bd.bd_dir);
	return False;
    }
    if (fstat(dfd, &st) == -1)
    {
	warn("cannot stat audio directory %s", _bd.bd_dir);
	return False;
    }
    dbuf = XtMalloc(st.st_size);
    if ((nb = getdents(dfd, dbuf, st.st_size)) == -1)
    {
	warn("error reading directory %s", _bd.bd_dir);
	return False;
    }
    close(dfd);
    for (nent = 0, dep = (struct dirent *)dbuf;
	    (char *)dep - dbuf < nb;
	    ++nent, (char *)dep+= dep->d_reclen)
	;
    _hash_resize(&(_bd.bd_sndhash), nent * 2);
    if (_bd.bd_defsnd)
    {
	_sdesc_free(_bd.bd_defsnd);
	_bd.bd_defsnd = NULL;
    }
    for (dep = (struct dirent *)dbuf;
	    (char *)dep - dbuf < nb;
	    (char *)dep+= dep->d_reclen)
    {
	char		    *suf;
	static const char   *sufs[] = AUDIO_SUFS;

	if (dep->d_name[0] == '.')
	    continue;
	if (!(suf = strrchr(dep->d_name, '.')))
	    continue;
	++suf;
	for (i = 0; i < XtNumber(sufs); ++i)
	    if (!strcasecmp(suf, sufs[i]))
	    {
		char		    *bfn;
		struct sound_desc   *sdp;

		bfn = strrchr(dep->d_name, '/');
		bfn = (bfn) ? bfn + 1 : dep->d_name;
		sdp = XtNew(struct sound_desc);
		sdp->sd_class = XtMalloc((suf - bfn) + 1);
		bcopy(bfn, sdp->sd_class, suf - bfn);
		sdp->sd_class[suf - bfn - 1] = '\0';
		sdp->sd_fname = XtNewString(bfn);
		sdp->sd_samp = NULL;
#ifdef DEBUG
		warnx("DEBUG: adding audio file %s => %s",
			sdp->sd_class,
			sdp->sd_fname);
#endif /* DEBUG */
		if (!strcmp(sdp->sd_class, DEF_CLASS))
		    _bd.bd_defsnd = sdp;
		else if (_hash_insert(&(_bd.bd_sndhash),
			    sdp) == False && _bd.bd_verb == True)
		    warnx("already found file for class %s",
			    sdp->sd_class);
		continue;
	    }
    }
    if (_bd.bd_defsnd)
    {
	struct sound_file   sf;

	if (_sfile_open(_bd.bd_defsnd->sd_fname, &sf) == True)
	{
	    _scache_add(&sf, _bd.bd_defsnd);
	    _sfile_close(&sf);
	}
    }
    XtFree(dbuf);
    return True;
}

static void
_catch_hup(int sig)
{
#ifdef __GNUC__
    (void)&sig;
#endif /* __GNUC__ */
    if (_bd.bd_verb == True)
	warnx("caught Hangup -- rescanning audio dir");
    _scache_clear();
    if (_rescan() == False)
	SET_EXIT(&_bd);
}

static void
_cb_null(XtPointer unused, XtSignalId *id)
{
#ifdef __GNUC__
    (void)unused;
    (void)id;
#endif /* __GNUC__ */
}

static void
_catch_fatal(int sig)
{
    if (_bd.bd_verb == True)
	warnx("caught deadly signal (%s) -- shutting down",
		sys_siglist[sig]);
    SET_EXIT(&_bd);
}

static Boolean
_xsk_conv_str2fn(Display *dpy,
	XrmValue *args, Cardinal *argc,
	XrmValue *from, XrmValue *to,
	XtPointer *data)
{
    char    *src = (char *)from->addr;

#ifdef __GNUC__
    (void)args;
    (void)argc;
    (void)data;
#endif /* __GNUC__ */
    if (*src == '~')
    {
	char	*hd;
	size_t	hdl, srcl;
	char	*dest;

	++src;
	if (*src == '/')
	{
	    ++src;
	    if (!(hd = getenv("HOME")))
	    {
		XtAppErrorMsg(XtDisplayToApplicationContext(dpy),
			"conversionError",
			"string",
			"XskError",
			"Unset environment variable: HOME",
			NULL, 0);
		return False;
	    }
	}
	else
	{
	    char	    *un, *es;
	    Boolean	    freeun = False;
	    struct passwd   *pwp;

	    if ((es = strchr(src, '/')))
	    {
		size_t	unl = es - src;

		un = XtMalloc(unl + 1);
		freeun = True;
		bcopy(src, un, unl);
		un[unl] = '\0';
	    }
	    else
		un = src;
	    if (!(pwp = getpwnam(un)))
	    {
		char	msg[256];

		snprintf(msg, sizeof(msg), "Unknown user: %s", un);
		XtAppErrorMsg(XtDisplayToApplicationContext(dpy),
			"conversionError",
			"string",
			"XskError",
			msg,
			NULL, 0);
		return False;
	    }
	    src+= strlen(un);
	    if (freeun)
		XtFree(un);
	    hd = pwp->pw_dir;
	}
	hdl = strlen(hd);
	srcl = strlen(src);
	to->size = hdl + 1 + srcl + 1;
	*(String *)(to->addr) = dest = XtMalloc(to->size);
	bcopy(hd, dest, hdl);
	dest[hdl] = '/';
	bcopy(src, dest + hdl + 1, srcl + 1);
    }
    else
	*(String *)(to->addr) = XtNewString(from->addr);
    return True;
}

static Boolean
_xsk_conv_str2kb(Display *dpy,
	XrmValue *args, Cardinal *argc,
	XrmValue *from, XrmValue *to,
	XtPointer *data)
{
    String  str = from->addr;
    size_t  *valp = (size_t *)to->addr;
    u_long  size;
    char    *se;

    size = strtoul(str, &se, 10);
    if (se - str < strlen(str))
	return False;
    *valp = size * 1024;
    return True;
}

static Boolean
_xsk_conv_str2per(Display *dpy,
	XrmValue *args, Cardinal *argc,
	XrmValue *from, XrmValue *to,
	XtPointer *data)
{
    String  str = from->addr;
    float   *valp = (float *)to->addr;
    u_long  per;
    char    *se;

    per = strtoul(str, &se, 10);
    if (se - str < strlen(str))
	return False;
    *valp = per / 100;
    return True;
}

static void
_xtrap_while(XtAppContext xtac, XETC *tc, Bool *done)
{
    XEvent	ev;
    XtInputMask	mask;

    while (*done == False)
    {
	mask = XETrapAppPending(xtac);
	if (mask & XtIMXEvent)
	{
	    XtAppNextEvent(xtac, &ev);
	    switch (ev.type)
	    {
		case CreateNotify:
		case DestroyNotify:
		    _cb_strmod(ev);
		    break;
		default:
		    XETrapDispatchEvent(&ev, tc);
	    }
	}
	else if (mask & (XtIMTimer | XtIMAlternateInput))
	    XtAppProcessEvent(_bd.bd_xtac,
		    (XtIMTimer | XtIMAlternateInput));
	else
	    XETrapWaitForSomething(_bd.bd_xtac);
    }
}

static int
_xerror(Display *dpy, XErrorEvent *xep)
{
    if (xep->error_code == BadWindow)
    {
	char	xerr[64 + 1];

	XGetErrorText(dpy, xep->error_code, xerr, sizeof(xerr));
#ifdef DEBUG
	warnx("dropping XError: %s: %s",
		XERequestIDToString(xep->request_code, _bd.bd_xetc),
		xerr);
#endif /* DEBUG */
	return 0;
    }
    else
	return _bd.bd_oxerr(dpy, xep);
}

int
main(int ac, char *av[])
{
    Widget		top_w;
    float		dper = 0.75;
    XtResource		rsrcs[] =
    {
	{
	    "verbose", "Verbose",
	    XtRBoolean, sizeof(Boolean),
	    XtOffsetOf(struct belld_data, bd_verb),
	    XtRImmediate,
#ifdef DEBUG
		(XtPointer)True
#else
		(XtPointer)False
#endif /* DEBUG */
	},
	{
	    "dontFork", "DontFork",
	    XtRBoolean, sizeof(Boolean),
	    XtOffsetOf(struct belld_data, bd_nofork),
	    XtRImmediate,
#ifdef DEBUG
		(XtPointer)True
#else
		(XtPointer)False
#endif /* DEBUG */
	},
	{
	    "volume", "Volume",
	    XskRPercent, sizeof(float),
	    XtOffsetOf(struct belld_data, bd_vol),
	    XskRPercent, &dper
	},
	{
	    "cacheSize", "CacheSize",
	    XskRKBytes, sizeof(size_t),
	    XtOffsetOf(struct belld_data, bd_cache),
	    XtRImmediate, (XtPointer)(512 * 1024)
	},
	{
	    "multiplexAudio", "MultiplexAudio",
	    XtRBoolean, sizeof(Boolean),
	    XtOffsetOf(struct belld_data, bd_multi),
	    XtRImmediate, (XtPointer)True
	},
	{
	    "pidFile", "PidFile",
	    XskRFilename, sizeof(String),
	    XtOffsetOf(struct belld_data, bd_pidfn),
	    XtRString, ""
	},
	{
	    "audioDir", "AudioDir",
	    XskRFilename, sizeof(String),
	    XtOffsetOf(struct belld_data, bd_dir),
	    XtRString, "~/.xbells/"
	},
	{
	    "esoundServer", "EsoundServer",
	    XtRString, sizeof(String),
	    XtOffsetOf(struct belld_data, bd_esdsvr),
	    XtRString, ""
	}
    };
    XrmOptionDescRec	opts[] =
    {
	{"+verbose", ".verbose", XrmoptionNoArg, "False"},
	{"-verbose", ".verbose", XrmoptionNoArg, "True"},
	{"+nofork", ".dontFork", XrmoptionNoArg, "False"},
	{"-nofork", ".dontFork", XrmoptionNoArg, "True"},
	{"-volume", ".volume", XrmoptionSepArg, NULL},
	{"-cache", ".cacheSize", XrmoptionSepArg, NULL},
	{"+multi", ".multiplexAudio", XrmoptionNoArg, "False"},
	{"-multi", ".multiplexAudio", XrmoptionNoArg, "True"},
	{"-pidfile", ".pidFile", XrmoptionSepArg, NULL},
	{"-dir", ".audioDir", XrmoptionSepArg, NULL},
	{"-esdserver", ".esoundServer", XrmoptionSepArg, NULL}
    };
    XKeyboardState	ks;
    XKeyboardControl	kc;
    ReqFlags		rf;
    EventFlags		ef;
    int			fsigs[] = {SIGINT, SIGTERM, SIGQUIT, SIGABRT};
    u_int		i;
    FILE		*pidfp = NULL;

    top_w = XtVaOpenApplication(&_bd.bd_xtac,
	    "XTrapBell",
	    opts, XtNumber(opts),
	    &ac, av,
	    NULL,
	    applicationShellWidgetClass,
	    NULL);
    if (ac > 1)
    {
	u_int	ai;

	for (ai = 1; ai < ac; ++ai)
	    if (*av[ai] == '-' || *av[ai] == '+')
	    {
		u_int	oi;
		char	*msg = "illegal option";

		for (oi = 0; oi < XtNumber(opts); ++oi)
		{
		    if (!strcmp(av[ai], opts[oi].option))
		    {
			assert(opts[oi].argKind == XrmoptionSepArg);
			msg = "missing argument for option";
			break;
		    }
		}
		warnx("%s -- %s", msg, av[ai]);
	    }
	fprintf(stderr, "Usage: %s [(+|-)verbose] [(+|-)nofork] "
		"[-volume <volume>] [-pidfile <pid_file>] "
		"[-dir <audio_dir>] [-esdserver <esound_server>] "
		"[toolkit options]\n",
		getprogname());
	return EX_USAGE;
	
    }
    XtAppSetTypeConverter(_bd.bd_xtac,
	    XtRString, XskRFilename,
	    _xsk_conv_str2fn,
	    NULL, 0,
	    XtCacheByDisplay,
	    NULL);
    XtAppSetTypeConverter(_bd.bd_xtac,
	    XtRString, XskRKBytes,
	    _xsk_conv_str2kb,
	    NULL, 0,
	    XtCacheAll,
	    NULL);
    XtAppSetTypeConverter(_bd.bd_xtac,
	    XtRString, XskRPercent,
	    _xsk_conv_str2per,
	    NULL, 0,
	    XtCacheAll,
	    NULL);
    XtVaGetApplicationResources(top_w,
	    &_bd,
	    rsrcs, XtNumber(rsrcs),
	    NULL);
    if (*_bd.bd_pidfn)
    {
	char	ps[10 + 1];

	if (_bd.bd_verb == True)
	    warnx("writing PID to file %s", _bd.bd_pidfn);
	snprintf(ps, sizeof(ps), "%d", getpid());
	if (!(pidfp = fopen(_bd.bd_pidfn, "w")) ||
		fputs(ps, pidfp) == EOF ||
		fflush(pidfp) == EOF ||
		unlink(_bd.bd_pidfn) == -1)
	    warn("error writing PID file %s", _bd.bd_pidfn);
    }
    _bd.bd_disp = XtDisplay(top_w);
    _bd.bd_idshift = 0;
    while ((1 << _bd.bd_idshift) & _bd.bd_disp->resource_mask)
	++_bd.bd_idshift;
    _bd.bd_idmask = ~(_bd.bd_disp->resource_mask) >> _bd.bd_idshift;
    _bd.bd_myid = CLIENT_ID(_bd.bd_disp->resource_base);
#ifdef DEBUG
    XSynchronize(_bd.bd_disp, True);
    warnx("DEBUG: myid = %d", _bd.bd_myid);
    warnx("DEBUG: idshift = %d, idmask = %#x",
	    _bd.bd_idshift,
	    _bd.bd_idmask);
#endif /* DEBUG */
    _bd.bd_oxerr = XSetErrorHandler(_xerror);
    afSetErrorHandler(_aferr);
    if (_bd.bd_nofork == False)
	daemon(1,
#ifdef DEBUG
		1
#else
		(_bd.bd_verb == True)
#endif /* DEBUG */
		);
    if (_bd.bd_verb == True)
	warnx("connected to display \"%s\"",
		DisplayString(_bd.bd_disp));
    if (!(_bd.bd_xetc = XECreateTC(_bd.bd_disp, 0, NULL)))
	err(1, "could not initialize DEC-XTRAP extension");
    if ((_bd.bd_esdfd =
		esd_open_sound((*_bd.bd_esdsvr) ? _bd.bd_esdsvr : NULL))
	    < 0)
    {
	XEFreeTC(_bd.bd_xetc);
	XCloseDisplay(_bd.bd_disp);
	errx(1, "cannot connect to esound server \"%s\"",
		_bd.bd_esdsvr);
    }
    _chash_init();
    _hash_init(&(_bd.bd_sndhash),
	    0,
	    _sdesc_hash,
	    _sdesc_cmp,
	    _sdesc_free);
    if (_bd.bd_cache)
    {
	_bd.bd_smpsize = 0;
	LIST_INIT(&(_bd.bd_smplist));
    }
    if (_rescan() == False)
    {
	esd_close(_bd.bd_esdfd);
	XEFreeTC(_bd.bd_xetc);
	XCloseDisplay(_bd.bd_disp);
	return 1;
    }
    _bd.bd_fatsid = XtAppAddSignal(_bd.bd_xtac, _cb_null, NULL);
    XGetKeyboardControl(_bd.bd_disp, &ks);
    if ((_bd.bd_opitch = ks.bell_pitch))
    {
	if (_bd.bd_verb == True)
	    warnx("setting bell_pitch to 0");
	kc.bell_pitch = 0;
	XChangeKeyboardControl(_bd.bd_disp, KBBellPitch, &kc);
    }
#ifdef DEBUG
    warnx("DEBUG: setting max XETrap packet size to %d",
	    XETrapMinPktSize);
#endif /* DEBUG */
    XETrapSetMaxPacket(_bd.bd_xetc, True, XETrapMinPktSize);
    bzero(rf, sizeof(ReqFlags));
#ifdef DEBUG
    warnx("DEBUG: trapping requests of type X_Bell (%d)", X_Bell);
#endif /* DEBUG */
    BitTrue(rf, X_Bell);
    XETrapSetRequests(_bd.bd_xetc, True, rf);
    XEAddRequestCB(_bd.bd_xetc, X_Bell, _cb_bell, NULL);
    bzero(ef, sizeof(EventFlags));
    XETrapSetEvents(_bd.bd_xetc, True, ef);
    _bd.bd_done = False;
    signal(SIGHUP, _catch_hup);
    for (i = 0; i < XtNumber(fsigs); ++i)
    {
	signal(fsigs[i], _catch_fatal);
	siginterrupt(fsigs[i], 1);
    }
    XEStartTrapRequest(_bd.bd_xetc);
    _xtrap_while(_bd.bd_xtac, _bd.bd_xetc, &_bd.bd_done);
    XEStopTrapRequest(_bd.bd_xetc);
    _hash_destroy(&(_bd.bd_clihash));
    _hash_destroy(&(_bd.bd_sndhash));
    esd_close(_bd.bd_esdfd);
    XEFreeTC(_bd.bd_xetc);
    if (_bd.bd_opitch)
    {
	if (_bd.bd_verb == True)
	    warnx("resetting bell_pitch to %d", _bd.bd_opitch);
	kc.bell_pitch = _bd.bd_opitch;
	XChangeKeyboardControl(_bd.bd_disp, KBBellPitch, &kc);
    }
    if (_bd.bd_verb == True)
	warnx("shutting down connection to display \"%s\"",
		DisplayString(_bd.bd_disp));
    XCloseDisplay(_bd.bd_disp);
    if (pidfp)
	fclose(pidfp);
    return EX_OK;
}

