/*
 * $Id: xcmd.c,v 1.8 1998/03/27 03:47:33 jmknoble Exp $
 *
 * xcmd.c
 * by Jim Knoble <jmknoble@pobox.com>
 * Copyright  1997 Jim Knoble
 * 
 * Derived from portions of:
 *      - xwit-3.2 by Mark Martin and David DiGiacomo
 * 
 *      - xwininfo from XFree86-3.2, by the X Consortium 
 *        and the XFree86 Project
 * 
 * Disclaimer:
 * 
 * The software is provided "as is", without warranty of any kind,
 * express or implied, including but not limited to the warranties of
 * merchantability, fitness for a particular purpose and
 * noninfringement. In no event shall the author(s) be liable for any
 * claim, damages or other liability, whether in an action of
 * contract, tort or otherwise, arising from, out of or in connection
 * with the software or the use or other dealings in the software.
 */
/**********************************************************************/

#include <X11/Xos.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xproto.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
/* #include <sys/time.h> */

#include "dsimple.h"
#include "xcmd.h"

/* terminal program commands */
static char *trmpgm_def = TRMPGM_DEF;
static char *trmpgm_rxvt = TRMPGM_RXVT;
static char *trmpgm_xterm = TRMPGM_XTERM;
static char *trmpgm_envvar = TRMPGM_ENVVAR;

/* environment variable for specifying action to raise window with */
static char *fndwdw_envvar = FNDWDW_ENVVAR;

/* root window for given display */
static Window rotwdw; 

/* T=>we found a window to act on */
static int winfndflg;

/* message to display when help requested
 * or error in calling syntax
 */
static char *hlpmsg = 
"  usage: %s [options] {--cmd | --xcmd} <command>\n"
"\n"
"  execute <command>, with the option of finding and raising a prior\n"
"  instance instead of running a new one.  if `--cmd' is specified,\n"
"  execute <command> in a terminal window; if `--xcmd' is specified,\n"
"  execute <command> in its own window.\n"
"\n"
"  general options:\n"
"    -c or --cmd <command>          command to run in a terminal window\n"
"    -d or --display <display>      X display to use\n"
"    -h or --help                   print this help\n"
"    -x or --xcmd <command>         command to run in its own window\n"
"\n"
"  find options:\n"
"    -fc or --find-class <text>     class of window to raise\n"
"    -fr or --find-resource <text>  resource name of window to raise\n"
"    -fs or --find-substr <text>    title-substring to raise\n"
"    -ft or --find-title <text>     title of window to raise\n"
"\n"
"  terminal window options (for --cmd):\n"
"    -t or --title <text>           title of window\n"
"    -i or --icon <text>            title of icon\n"
"    -n or --name <text>            name to retrieve resources under\n"
"    -g or --geometry <geometry>    geometry of terminal window\n"
"    -s or --scroll                 display scrollbar in terminal window\n"
"    +s or +scroll                  no scrollbar in terminal window\n"
"    -o or --opt <options>          options to pass to terminal program\n"
"    -rt or --rxvt                  use rxvt as terminal program\n"
"    -xt or --xterm                 use xterm as terminal program\n"
"\n"
"  raise options:\n"
"    -ir or --iconify-and-raise     first iconify, then raise windows we find\n"
"    -or or --only-raise            only raise windows we find (default)\n"
"\n"
"  environment variables:\n"
"    %s (default terminal program to use; `%s' if not set)\n"
"    %s (`0'/`no'/`false' => only raise windows we find [default];\n"
"                  `1'/`yes'/`true' => iconify window, then raise)\n"
"\n"
;

/* display usage information, then exit.
 * note: called by dsimple.c code, must be global 
 */
int usage(void)
{
   static char vsnstr[] = "$Revision: 1.8 $";
   char *vsnbeg, *vsnend;

   for (vsnbeg = vsnstr; *vsnbeg; vsnbeg++)
    {
       if (' ' == *vsnbeg)
        {
           break;
        }
    }
   if (*vsnbeg)
    {
       for (vsnend = ++vsnbeg; *vsnend; vsnend++)
        {
           if (' ' == *vsnend)
            {
               *vsnend = 0;
               break;
            }
        }
       fprintf(HLPMSGFILHND, "\n  %s version %s\n\n",
               program_name, vsnbeg);
    }
   fprintf(HLPMSGFILHND, hlpmsg, program_name, 
           trmpgm_envvar, trmpgm_def, 
           fndwdw_envvar);
   exit(HLPRTNCOD);
} /* usage() */


/* Raise the given window */
WDWFNC_RTN_TYP RaiseWindow(Window wdw)
{
   winfndflg = 1;

   XMapRaised(dpy, wdw);
   XSync(dpy, False);
} /* RaiseWindow() */


/* Iconify the given window */
WDWFNC_RTN_TYP IconifyWindow(Window wdw)
{
   winfndflg = 1;

   /* first iconify the window */
#if iconify_by_sending_client_message
   static XClientMessageEvent event;
   
   if (event.type == 0) 
    {
       event.type = ClientMessage;
# ifdef XA_WM_CHANGE_STATE
       event.message_type = XA_WM_CHANGE_STATE;
# else
       event.message_type =
          XInternAtom(dpy, "WM_CHANGE_STATE", True);
       if (event.message_type == 0)
          Fatal_Error("no WM_CHANGE_STATE atom");
# endif
       event.format = 32;
       event.data.l[0] = IconicState;
    }
   event.window = wdw;
   if (0 == XSendEvent(dpy, rotwdw, (Bool) False,
                       SubstructureRedirectMask | SubstructureNotifyMask,
                       (XEvent *) &event
                       )
       ) 
    {
       Fatal_Error("send event failed");
    }
#else /* iconify_by_sending_client_message */
   if (0 == XIconifyWindow(dpy, wdw, screen)) 
    {
       Fatal_Error("iconify failed");
    }
#endif /* iconify_by_sending_client_message */
   XSync(dpy, False);
} /* IconifyWindow() */


/* first iconify, then raise the given window */
WDWFNC_RTN_TYP IconifyAndRaiseWindow(Window wdw)
{
   IconifyWindow(wdw);
   /* give the X server a chance to do what we asked */
   usleep(WATIVL);
   RaiseWindow(wdw);
} /* IconifyAndRaiseWindow */


/* only raise the window */
WDWFNC_RTN_TYP OnlyRaiseWindow(Window wdw)
{
   RaiseWindow(wdw);
} /* OnlyRaiseWindow */


/* Find windows with a particular property
 * containing a particular value, and
 * execute a function for each matching window
 */
void FindWindowsForProperty(/* window to start at */
                            Window wdw,
                            /* value of property to look for */
                            char *ppyvlu,
                            /* kind of property to look for */
                            WDWPPY_TYP ppytyp,
                            /* function to execute when a window found */
                            WDWFNC_TYP wdwfnc)
{
   Window dmyrotwdw;    /* dummy: root window of display */
   Window dmyprnwdw;    /* dummy: parent of given window */
   unsigned int numchd; /* number of child windows */
   Window *chdlst;      /* list of child windows */
   int i;
   int mchflg;          /* T=>found a window matching the property */
   char *curttl;        /* current window title */
   XClassHint clshnt;   /* class hints for getting class and resource names */
   
   if (XQueryTree(dpy, wdw, &dmyrotwdw, &dmyprnwdw, &chdlst, &numchd))
    {
       for (i = (int)numchd - 1; i >= 0; i--) 
        {
           mchflg = 0;
           curttl = NULL;
           
           if (chdlst[i]) 
            {
               switch (ppytyp) 
                {
                 case WDWPPY_CLS:
                   if (XGetClassHint(dpy, chdlst[i], &clshnt)) 
                    {
                       if (clshnt.res_class) 
                        {
                           mchflg = (0 == strcmp(clshnt.res_class, ppyvlu));
                           XFree(clshnt.res_class);
                        }
                    }
                   break;
                 case WDWPPY_RSC:
                   if (XGetClassHint(dpy, chdlst[i], &clshnt)) 
                    {
                       if (clshnt.res_name) 
                        {
                           mchflg = (0 == strcmp(clshnt.res_name, ppyvlu));
                           XFree(clshnt.res_name);
                        }
                    }
                   break;
                 case WDWPPY_TTL:
                   if (XFetchName(dpy, chdlst[i], &curttl))
                    {
                       if (curttl)
                        {
                           mchflg = (0 == strcmp(curttl, ppyvlu));
                           XFree(curttl);
                        }
                    }
                   break;
                 case WDWPPY_SUBSTR:
                   if (XFetchName(dpy, chdlst[i], &curttl))
                    {
                       if (curttl)
                        {
                           mchflg = (NULL != strstr(curttl, ppyvlu));
                           XFree(curttl);
                        }
                    }
                   break;
                 case WDWPPY_NON:
                   Fatal_Error("This shouldn't happen.");
                   break;
                } /* switch (ppytyp) */
               if (mchflg)
                {
                   wdwfnc(chdlst[i]);
                }
               FindWindowsForProperty(chdlst[i], ppyvlu, ppytyp, wdwfnc);
            } /* if (chldlst[i]) */
        } /* for(i...) */
    } /* if (XQueryTree(...)) */
   if (chdlst) 
    {
       XFree((char *)chdlst);
    }
} /* FindWindowsForProperty() */


/* Close the current display */
void CloseDisplay(void)
{
   XSync(dpy, False);
   (void) XCloseDisplay(dpy);
} /* CloseDisplay() */


/* allocate enough extra memory to append <src> to <dst>,
 * and append it
 */
char *StringAppend(char **dst, char *src)
{
   *dst = realloc(*dst, 1 + strlen(*dst) + strlen(src));
   *dst = strcat(*dst, src);
   return (*dst);
} /* StringAppend */


/* Execute the given shell command */
int ExecuteCommand(char *cmd)
{
   /* T=>execution succeeded */
   int sts;
   /* command buffer */
   char *cmdbuf;
   char *err_prefix;
   
   cmdbuf = strdup(SHL_EXECMD);
   cmdbuf = StringAppend(&cmdbuf, cmd);
   
   sts = execl(SHL_NAM, SHL_NAM, SHL_CMDOPT, cmdbuf, NULL);
   /* If execl() returns, there was an error. */
   err_prefix = strdup(program_name);
   StringAppend(&err_prefix, ": execl");
   perror(err_prefix);
   sts = 0;

   free(err_prefix);
   free(cmdbuf);
   return (sts);
} /* ExecuteCommand() */


/* Run a command inside a terminal window */
int RunCommandInWindow(/* command to execute */
                       char *cmd,
                       /* terminal program info */
                       TRMWDWINF_TYP trmwdwinf)
{
   int sts;
   char *cmdbuf;

   cmdbuf = strdup("");
   
   cmdbuf = StringAppend(&cmdbuf, trmwdwinf.trmpgm);
   if (NULL != trmwdwinf.ttl)
    {
       cmdbuf = StringAppend(&cmdbuf, TRMCMD_TTLOPT);
       cmdbuf = StringAppend(&cmdbuf, trmwdwinf.ttl);
       cmdbuf = StringAppend(&cmdbuf, TRMCMD_QUO);
    }
   if (NULL != trmwdwinf.icnnam)
    {
       cmdbuf = StringAppend(&cmdbuf, TRMCMD_ICNNAMOPT);
       cmdbuf = StringAppend(&cmdbuf, trmwdwinf.icnnam);
       cmdbuf = StringAppend(&cmdbuf, TRMCMD_QUO);
    }
   if (NULL != trmwdwinf.rscnam)
    {
       cmdbuf = StringAppend(&cmdbuf, TRMCMD_RSCNAMOPT);
       cmdbuf = StringAppend(&cmdbuf, trmwdwinf.rscnam);
       cmdbuf = StringAppend(&cmdbuf, TRMCMD_QUO);
    }
   if (NULL != trmwdwinf.geo)
    {
       cmdbuf = StringAppend(&cmdbuf, TRMCMD_GEOOPT);
       cmdbuf = StringAppend(&cmdbuf, trmwdwinf.geo);
    }
   if (FLG_NONSET != trmwdwinf.srlflg)
    {
       if (trmwdwinf.srlflg)
        {
           cmdbuf = StringAppend(&cmdbuf, TRMCMD_SRLOPT);
        }
       else
        {
           cmdbuf = StringAppend(&cmdbuf, TRMCMD_NONSRLOPT);
        }
    }
   if (NULL != trmwdwinf.opt)
    {
       cmdbuf = StringAppend(&cmdbuf, trmwdwinf.opt);
    }
   cmdbuf = StringAppend(&cmdbuf, TRMCMD_EXEOPT);
   cmdbuf = StringAppend(&cmdbuf, cmd);
   
   CloseDisplay();
   sts = ExecuteCommand(cmdbuf);
   
   free(cmdbuf);
   return (sts);
} /* RunCommandInWindow() */


/* Run the given command;
 * if the terminal program info so indicates,
 * run it in a terminal window,
 * otherwise simply execute the plain command
 */
int RunCommand(/* command to execute */
               char *cmd,
               /* terminal program info */
               TRMWDWINF_TYP trmwdwinf)
{
   int sts;
   
   if (NULL == trmwdwinf.trmpgm)
    {
       CloseDisplay();
       sts = ExecuteCommand(cmd);
    }
   else
    {
       sts = RunCommandInWindow(cmd, trmwdwinf);
    }
   return (sts);
} /* RunCommand() */


static int MatchOption(/* key to match against */
                       char *key,
                       /* length of key to match */
                       int lng, 
                       /* number of arguments required */
                       int nargs,
                       /* count of available arguments */
                       int *argc, 
                       /* available arguments */
                       char **argv)
{
   /* T=>option matched */
   int mchflg;
   
   if (lng) 
    {
       mchflg = (0 == strncmp(key, *argv, lng));
    }
   else
    {
      mchflg = (0 == strcmp(key, *argv));
    }
   if (mchflg) 
    {
       if (argc[0] <= nargs) 
        {
           Fatal_Error("option %s needs %d argument%s",
                       key,
                       nargs,
                       (nargs > 1 ? "s" : "")
                       );
        }
       argc[0] -= nargs;
    }
   return mchflg;
} /* MatchOption() */


/* Local error handler to deal with BadWindow errors when
 * the window-manager makes icons go away.
 */
int LocalErrorHandler(Display *dpy, XErrorEvent *errevt)
{
   /* Do nothing; we really don't care
    * about windows we can't find.
    */
   ;
} /* LocalErrorHandler */


int main(int argc, char **argv)
{
   int *pargc = &argc;
   
   /* command to execute */
   char *cmd = NULL;

   /* T=>execute command in a terminal window */
   int trmwdwflg = 0;

   /* window property to look for */
   char *ppyvlu = NULL;
   /* type of property to look for */
   WDWPPY_TYP ppytyp = WDWPPY_NON;

   /* stuff to set up for executing command in a terminal window */
   TRMWDWINF_TYP trmwdwinf =
    {
       NULL, NULL, NULL, NULL, NULL, NULL, FLG_NONSET
    };
   
   /* default raise window action (no command-line option, no envariable) */
   WDWFNC_TYP *fndwdw_deffnc = &OnlyRaiseWindow;

   /* function to use to raise a window we find */
   WDWFNC_TYP *wdwfnc = NULL;
   
   /* value of environment variable for method of raising window */
   char *fndwdw_envvar_vlu = getenv(fndwdw_envvar);

   /* program_name is defined and required by some routines
    * in dsimple.c
    */
   program_name = argv[0] + strlen(argv[0]);
   while ((argv[0] != program_name) && ('/' != program_name[-1]))
      program_name--;
   
   Setup_Display_And_Screen(pargc, argv);
   
   while (argv++, --argc > 0) 
    {
       /* argv[0] = next argument */
       /* argc = # of arguments left */
       if (MatchOption("-h", 0, 0, pargc, argv) ||
           MatchOption("--help", 0, 0, pargc, argv)
           ) 
        {
           usage();
        }
       else if (MatchOption("-c", 0, 1, pargc, argv) ||
                MatchOption("-e", 0, 1, pargc, argv) ||
                MatchOption("--cmd", 0, 1, pargc, argv)
                ) 
        {
           cmd = *(++argv);
           trmwdwflg = 1;
        }
       else if (MatchOption("-x", 0, 1, pargc, argv) ||
                MatchOption("--xcmd", 0, 1, pargc, argv)
                ) 
        {
           cmd = *(++argv);
           trmwdwflg = 0;
        }
       else if (MatchOption("-fc", 0, 1, pargc, argv) ||
                MatchOption("--find-class", 0, 1, pargc, argv)
                ) 
        {
           ppyvlu = *(++argv);
           ppytyp = WDWPPY_CLS;
        }
       else if (MatchOption("-fr", 0, 1, pargc, argv) ||
                MatchOption("--find-resource", 0, 1, pargc, argv)
                ) 
        {
           ppyvlu = *(++argv);
           ppytyp = WDWPPY_RSC;
        }
       else if (MatchOption("-fs", 0, 1, pargc, argv) ||
                MatchOption("--find-substr", 0, 1, pargc, argv)
                ) 
        {
           ppyvlu = *(++argv);
           ppytyp = WDWPPY_SUBSTR;
        }
       else if (MatchOption("-ft", 0, 1, pargc, argv) ||
                MatchOption("--find-title", 0, 1, pargc, argv)
                ) 
        {
           ppyvlu = *(++argv);
           ppytyp = WDWPPY_TTL;
        }
       else if (MatchOption("-t", 0, 1, pargc, argv) ||
                MatchOption("--title", 0, 1, pargc, argv)
                ) 
        {
           trmwdwinf.ttl = *(++argv);
        }
       else if (MatchOption("-i", 0, 1, pargc, argv) ||
                MatchOption("--icon", 0, 1, pargc, argv)
                ) 
        {
           trmwdwinf.icnnam = *(++argv);
        }
       else if (MatchOption("-n", 0, 1, pargc, argv) ||
                MatchOption("--name", 0, 1, pargc, argv)
                ) 
        {
           trmwdwinf.rscnam = *(++argv);
        }
       else if (MatchOption("-g", 0, 1, pargc, argv) ||
                MatchOption("--geometry", 0, 1, pargc, argv) ||
                MatchOption("-geometry", 0, 1, pargc, argv)
                ) 
        {
           trmwdwinf.geo = *(++argv);
        }
       else if (MatchOption("-s", 0, 0, pargc, argv) ||
                MatchOption("--scroll", 0, 0, pargc, argv)
                ) 
        {
           trmwdwinf.srlflg = 1;
        }
       else if (MatchOption("+s", 0, 0, pargc, argv) ||
                MatchOption("+scroll", 0, 0, pargc, argv)
                ) 
        {
           trmwdwinf.srlflg = 0;
        }
       else if (MatchOption("-o", 0, 1, pargc, argv) ||
                MatchOption("--opt", 0, 1, pargc, argv)
                ) 
        {
           trmwdwinf.opt = *(++argv);
        }
       else if (MatchOption("-rt", 0, 0, pargc, argv) ||
                MatchOption("--rxvt", 0, 0, pargc, argv)
                ) 
        {
           trmwdwinf.trmpgm = trmpgm_rxvt;
        }
       else if (MatchOption("-xt", 0, 0, pargc, argv) ||
                MatchOption("--xterm", 0, 0, pargc, argv)
                ) 
        {
           trmwdwinf.trmpgm = trmpgm_xterm;
        }
       else if (MatchOption("-ir", 0, 0, pargc, argv) ||
                MatchOption("--iconify-and-raise", 0, 0, pargc, argv)
                )
        {
           wdwfnc = &IconifyAndRaiseWindow;
        }
       else if (MatchOption("-or", 0, 0, pargc, argv) ||
                MatchOption("--only-raise", 0, 0, pargc, argv)
                )
        {
           wdwfnc = &OnlyRaiseWindow;
        }
       else
        {
          usage();
        }
    } /* (argv++, --argc > 0) */
   
   /*************************/
   /* Begin sanity checking */

   /* error if no command specified */
   if (NULL == cmd) 
    {
       Fatal_Error("no command to execute; use `--help' for usage info.");
    }
   /* if we're executing a command in a window,
    * set up the appropriate values,
    * otherwise complain if terminal-window stuff is set
    */
   if (trmwdwflg)
    {
       /* if no title set, use resource name if set */
       if (NULL == trmwdwinf.ttl)
        {
           if (NULL != trmwdwinf.rscnam)
            {
               trmwdwinf.ttl = trmwdwinf.rscnam;
            }
           else
            {
               /* otherwise, use command name */
               trmwdwinf.ttl = cmd;
            }
        }
       /* if no icon name set, but title is, use window title */
       if ((NULL == trmwdwinf.icnnam) &&
           (NULL != trmwdwinf.ttl)
           )
        {
           trmwdwinf.icnnam = trmwdwinf.ttl;
        }
       /* if no terminal program specified on command line,
        * check our environment variable for one
        */
       if (NULL == trmwdwinf.trmpgm)
        {
           trmwdwinf.trmpgm = getenv(trmpgm_envvar);
           if (NULL == trmwdwinf.trmpgm)
            {
               /* if no environment variable is set,
                * use the default terminal program
                */
               trmwdwinf.trmpgm = trmpgm_def;
            }
        }
    }
   else /* trmwdwflg */
    {
       char *trmwdwppy = NULL;
       
       if (NULL != trmwdwinf.ttl) 
        {
           trmwdwppy = "title";
        }
       else if (NULL != trmwdwinf.icnnam)
        {
           trmwdwppy = "icon name";
        }
       else if (NULL != trmwdwinf.rscnam)
        {
           trmwdwppy = "resource name";
        }
       else if (NULL != trmwdwinf.geo)
        {
           trmwdwppy = "terminal geometry";
        }
       else if (NULL != trmwdwinf.opt)
        {
           trmwdwppy = "extra options";
        }
       else if (FLG_NONSET != trmwdwinf.srlflg)
        {
           trmwdwppy = "scrollbar";
        }
       if (NULL != trmwdwppy)
        {
           Fatal_Error("you can't specify %s with `--xcmd'", trmwdwppy);
        }
    } /* trmwdwflg */
   
   /* check to see if an environment variable is set
    * telling us how we should raise windows; the
    * command line overrides the environment variable.
    */
   if (NULL == wdwfnc)
    {
       if (NULL != fndwdw_envvar_vlu)
        {
           if (0 == strcmp(fndwdw_envvar_vlu, ENVVAR_ZROVLU) ||
               0 == strcmp(fndwdw_envvar_vlu, ENVVAR_NONVLU) ||
               0 == strcmp(fndwdw_envvar_vlu, ENVVAR_FLSVLU)
               )
            {
               wdwfnc = &OnlyRaiseWindow;
            }
           else if (0 == strcmp(fndwdw_envvar_vlu, ENVVAR_ONEVLU) ||
                    0 == strcmp(fndwdw_envvar_vlu, ENVVAR_YESVLU) ||
                    0 == strcmp(fndwdw_envvar_vlu, ENVVAR_TRUVLU)
                    )
            {
               wdwfnc = &IconifyAndRaiseWindow;
            }
           else
            {
               Fatal_Error("%s set to invalid value `%s'; "
                           "use `--help' for usage info.",
                           fndwdw_envvar, fndwdw_envvar_vlu);
            }
        }
       else
        {
           wdwfnc = fndwdw_deffnc;
        }
    }
   else
    {
       /* do nothing; wdwfnc set using command-line switch */
       ;
    }
   
   /* End Sanity checking */
   /***********************/
   
   /* if user didn't request us to look for a window,
    * just execute the specified command
    */
   if (WDWPPY_NON == ppytyp) 
    {
       RunCommand(cmd, trmwdwinf);
    }
   else
    {
       /* otherwise, starting with the root window, raise every
        * matching window.  we need to use a local error handler,
        * though, since some window managers (e.g., WindowMaker)
        * create & destroy icon windows on the fly, and we end up with
        * BadWindow error messages from the default X error handler.
        */
       XSetErrorHandler(&LocalErrorHandler);
       rotwdw = DefaultRootWindow(dpy);
       FindWindowsForProperty(rotwdw, ppyvlu, ppytyp, *wdwfnc);
       /* if we couldn't find a window,
        * execute the specified command
        */
       if (!winfndflg) 
        {
           RunCommand(cmd, trmwdwinf);
        }
    }
   
   /* close our connection to the display and exit */
   CloseDisplay();
   exit(!winfndflg);
   
} /* main() */

/**********************************************************************/
/*
 * $Log: xcmd.c,v $
 * Revision 1.8  1998/03/27 03:47:33  jmknoble
 * bugfix: execl() wasn't getting called with NULL as last argument
 * (how did it ever work?).
 *
 * Revision 1.7  1998/01/04 03:54:56  jmknoble
 * Moved log to bottom of file.
 *
 * Revision 1.6  1998/01/04 03:53:52  jmknoble
 * Some window managers (notably WindowMaker) create and destroy
 * icon windows on the fly, and so if we've deiconified a window, and then
 * we encounter its window id further down the list of the root window's
 * children, we get BadWindow errors.  Therefore, we install our own
 * X error handler to ignore the error and keep processing.
 * Also, now we default to raising the window only instead of
 * iconifying it, then raising.
 * Minor code reformatting.
 *
 * Revision 1.5  1997/08/11 05:21:13  jmknoble
 * added command-line options to choose whether to
 * iconify-and-raise or raise-only when we
 * find a window; added environment variable
 * to specify same behavior.
 *
 * Revision 1.4  1997/05/30 04:34:00  jmknoble
 * added default terminal program to --help display
 *
 * Revision 1.3  1997/05/30 00:50:19  jmknoble
 * replaced '--cmd --in-window' with orthogonal '--cmd' or '--xcmd';
 * added '--geometry' option;
 * fixed handling of environment variable;
 * some small bit of reformatting.
 *
 * Revision 1.2  1997/05/29 10:01:52  jmknoble
 * added better error message when invoked with no arguments
 *
 * Revision 1.1  1997/05/29 09:42:08  jmknoble
 * Initial revision
 *
 */
