/*****************************************************************************/
/* XMangekyou - a kaleidoscope for X                                         */
/* Programed by Sakai Hiroaki.                                               */
/*                                                                           */
/* Copyright (c) 1998 Sakai Hiroaki.                                         */
/* All Rights Reserved.                                                      */
/*                                                                           */
/* XMangekyou is a kaleidoscope. You can see many, many, many patterns,      */
/* but, care for yourself from 'trip' filling!                               */
/*                                                                           */
/* XMangekyou is a free software. You may copy or distribute the original    */
/* XMangekyou freely. But, you may not modify or distribute the original     */
/* XMangekyou without permission of the author.                              */
/*****************************************************************************/

/*****************************************************************************/
/* Include Header Files.                                                     */
/*****************************************************************************/

#include "init.h"
#include "icon.xpm"

/*****************************************************************************/
/* Set Default Values.                                                       */
/*****************************************************************************/

void SetDefaultValues()
{
  xmangekyou.degree = 0;
  xmangekyou.lines = (Line *)NULL;
  xmangekyou.circles = (Circle *)NULL;
  xmangekyou.source_pixmaps = (SourcePixmap *)NULL;
  xmangekyou.copy_number = -1;
  xmangekyou.copy_points = (DoublePoint *)NULL;
  xmangekyou.copy_points_rotated = (DoublePoint *)NULL;
  xmangekyou.animation_data_filename = DEFAULT_ANIMATION_DATA_FILENAME;
  xmangekyou.animation_flag = OFF;
  xmangekyou.animation_fix_number = NO_ANIMATION_FIX;

  xmangekyou.width = DEFAULT_WIDTH;
  xmangekyou.height = DEFAULT_HEIGHT;
  xmangekyou.root_flag = DEFAULT_ROOT_FLAG;
  xmangekyou.quit_flag = DEFAULT_QUIT_FLAG;
  xmangekyou.lines_number = DEFAULT_LINES_NUMBER;
  xmangekyou.circles_number = DEFAULT_CIRCLES_NUMBER;
  xmangekyou.line_thickness = DEFAULT_LINE_THICKNESS;
  xmangekyou.circle_size = DEFAULT_CIRCLE_SIZE;
  xmangekyou.size = DEFAULT_SIZE;
  xmangekyou.speed = DEFAULT_SPEED;
  xmangekyou.change = DEFAULT_CHANGE;
  xmangekyou.wait = DEFAULT_WAIT;
}

/*****************************************************************************/
/* Initializing Display.                                                     */
/*****************************************************************************/

void CreateCanvas(int * argc, char ** argv)
{
  Window root;
  int x, y, border, depth;
  int n;
  Arg args[10];

  n = 0;
  XtSetArg(args[n], XtNwidth , xmangekyou.width ); n++;
  XtSetArg(args[n], XtNheight, xmangekyou.height); n++;
  xmangekyou.toplevel = XtAppInitialize(&(xmangekyou.app_context),
					"XMangekyou", NULL, 0,
					argc, argv, NULL, args, n);
  xmangekyou.display = XtDisplay(xmangekyou.toplevel);
  xmangekyou.screen = DefaultScreen(xmangekyou.display);
  xmangekyou.colormap = DefaultColormap(xmangekyou.display, xmangekyou.screen);
  xmangekyou.root_window = XRootWindow(xmangekyou.display, xmangekyou.screen);
  xmangekyou.depth = DefaultDepth(xmangekyou.display, xmangekyou.screen);
  xmangekyou.canvas = XtCreateManagedWidget("", labelWidgetClass,
					    xmangekyou.toplevel, NULL, 0);
  n = 0;
  XtSetArg(args[n], XtNbackground, &(xmangekyou.background_color)); n++;
  XtGetValues(xmangekyou.toplevel, args, n);

  if (xmangekyou.root_flag == ON) {
    xmangekyou.toplevel_window = xmangekyou.root_window;
    xmangekyou.canvas_window = xmangekyou.root_window;
    xmangekyou.width  = DisplayWidth(xmangekyou.display, xmangekyou.screen);
    xmangekyou.height = DisplayHeight(xmangekyou.display, xmangekyou.screen);
  } else {
    XtRealizeWidget(xmangekyou.toplevel);
    xmangekyou.toplevel_window = XtWindow(xmangekyou.toplevel);
    xmangekyou.canvas_window = XtWindow(xmangekyou.canvas);
    n = 0;
    XtSetArg(args[n], XtNwidth , &(xmangekyou.width )); n++;
    XtSetArg(args[n], XtNheight, &(xmangekyou.height)); n++;
    XtGetValues(xmangekyou.toplevel, args, n);
  }

  while (*argc > 1) {
    fprintf(stderr, "ERROR : Unknown Parameter : %s\n", argv[1]);
    DeleteParameter(argc, argv, 1);
  }
}

void CreateGC()
{
  unsigned long int black;

  black = BlackPixel(xmangekyou.display, xmangekyou.screen);
  xmangekyou.gc_source_clear =
    XCreateGC(xmangekyou.display, xmangekyou.toplevel_window, 0, NULL);
  xmangekyou.gc_source_hex_clear =
    XCreateGC(xmangekyou.display, xmangekyou.toplevel_window, 0, NULL);
  xmangekyou.gc_canvas_clear =
    XCreateGC(xmangekyou.display, xmangekyou.toplevel_window, 0, NULL);
  xmangekyou.gc_copy_source_canvas =
    XCreateGC(xmangekyou.display, xmangekyou.toplevel_window, 0, NULL);
  xmangekyou.gc_copy_canvas_canvas =
    XCreateGC(xmangekyou.display, xmangekyou.toplevel_window, 0, NULL);
  XSetForeground(xmangekyou.display, xmangekyou.gc_source_clear,
		 xmangekyou.background_color);
  XSetBackground(xmangekyou.display, xmangekyou.gc_source_clear,
		 xmangekyou.background_color);
  XSetForeground(xmangekyou.display, xmangekyou.gc_source_hex_clear, black);
  XSetBackground(xmangekyou.display, xmangekyou.gc_source_hex_clear, black);
  XSetForeground(xmangekyou.display, xmangekyou.gc_canvas_clear, black);
  XSetBackground(xmangekyou.display, xmangekyou.gc_canvas_clear, black);
  XSetFillRule(xmangekyou.display, xmangekyou.gc_source_hex_clear,
	       EvenOddRule);
  XSetFunction(xmangekyou.display, xmangekyou.gc_copy_source_canvas, GXor);
  XSetFunction(xmangekyou.display, xmangekyou.gc_copy_canvas_canvas, GXcopy);
}

void InitializeDisplay(int * argc, char ** argv)
{
  CreateCanvas(argc, argv);
  CreateGC();
}

/*****************************************************************************/
/* Initializing Mangekyou.                                                   */
/*****************************************************************************/

int MakeHexCopyData(int r)
{
  int i, j, k, a, n;
  DoublePoint dp, dv, csp, csv, p, v;

  n = 3*r*r + 3*r + 1;
  xmangekyou.copy_points = (DoublePoint *)malloc(sizeof(DoublePoint) * n);
  xmangekyou.copy_points_rotated =
    (DoublePoint *)malloc(sizeof(DoublePoint) * n);
  xmangekyou.copy_points[0].x = xmangekyou.copy_points[0].y = 0.0;
  a = 1;
  dv.x = SQRT_3;
  dv.y = 0.0;
  for (i = 1; i <= r; i++) {
    dp.x = SQRT_3 * i;
    dp.y = 0.0;
    for (j = 0; j < 6; j++) {
      CalCosSin(64 *  60*j,      &csp); CalRotate(dp, csp, &p);
      CalCosSin(64 * (60*j+120), &csv); CalRotate(dv, csv, &v);
      for (k = 0; k < i; k++) {
	xmangekyou.copy_points[a  ].x = p.x + (v.x * k);
	xmangekyou.copy_points[a++].y = p.y + (v.y * k);
      }
    }
  }
  return (n);
}

void InitializeDrawRoutine1(int r)
{
  int i;

  xmangekyou.copy_end = xmangekyou.copy_max = MakeHexCopyData(r);
  xmangekyou.copy_number = 1;
  xmangekyou.source_pixmaps = (SourcePixmap *)malloc(sizeof(SourcePixmap) * 2);
  xmangekyou.source_pixmaps[0].width.x = xmangekyou.size * 2 + 3;
  xmangekyou.source_pixmaps[0].width.y = xmangekyou.size * 2 + 3;
  xmangekyou.source_pixmaps[1].width.x = xmangekyou.width;
  xmangekyou.source_pixmaps[1].width.y = xmangekyou.height;
  xmangekyou.source_pixmaps[0].size = xmangekyou.size;
  for (i = 0; i < 2; i++) {
    xmangekyou.source_pixmaps[i].center.x =
      xmangekyou.source_pixmaps[i].width.x / 2;
    xmangekyou.source_pixmaps[i].center.y =
      xmangekyou.source_pixmaps[i].width.y / 2;
    xmangekyou.source_pixmaps[i].pixmap =
      XCreatePixmap(xmangekyou.display, xmangekyou.toplevel_window,
		    xmangekyou.source_pixmaps[i].width.x,
		    xmangekyou.source_pixmaps[i].width.y, xmangekyou.depth);
  }
#ifdef XMANGEKYOU_DEBUG
  fprintf(stderr, "Draw routine 1 is selected.\n");
#endif
}

void InitializeDrawRoutine2(double canvas_size)
{
  int i, a, n;
  double tmp;

  MakeHexCopyData(2);
  tmp = log(canvas_size / (0.5 * SQRT_3 * xmangekyou.size)) / log(2.0);
  if (tmp < 0) n = 0; else n = (int)tmp + 1;
  xmangekyou.copy_max = 19;
  if ((n % 2) == 1) xmangekyou.copy_end = 7; else xmangekyou.copy_end = 19;
  xmangekyou.copy_number = (n + 1) / 2;
  xmangekyou.source_pixmaps =
    (SourcePixmap *)malloc(sizeof(SourcePixmap)*(xmangekyou.copy_number + 1));
  a = 1;
  for (i = 0; i <= xmangekyou.copy_number; i++) {
    if (i == xmangekyou.copy_number) {
      xmangekyou.source_pixmaps[i].width.x = xmangekyou.width;
      xmangekyou.source_pixmaps[i].width.y = xmangekyou.height;
    } else {
      xmangekyou.source_pixmaps[i].width.x = a * xmangekyou.size * 2 + 3;
      xmangekyou.source_pixmaps[i].width.y = a * xmangekyou.size * 2 + 3;
    }
    xmangekyou.source_pixmaps[i].size = a * xmangekyou.size;
    xmangekyou.source_pixmaps[i].center.x =
      xmangekyou.source_pixmaps[i].width.x / 2;
    xmangekyou.source_pixmaps[i].center.y =
      xmangekyou.source_pixmaps[i].width.y / 2;
    xmangekyou.source_pixmaps[i].pixmap =
      XCreatePixmap(xmangekyou.display, xmangekyou.toplevel_window,
		    xmangekyou.source_pixmaps[i].width.x,
		    xmangekyou.source_pixmaps[i].width.y,
		    xmangekyou.depth);
    a *= 4;
  }
#ifdef XMANGEKYOU_DEBUG
  fprintf(stderr, "Draw routine 2 is selected.\n");
#endif
}

void InitializeMangekyou()
{
  int r;
  double canvas_size;

  canvas_size = 0.5 * sqrt((double)xmangekyou.width  * xmangekyou.width +
			   (double)xmangekyou.height * xmangekyou.height)+2.0;
  r = (int)((canvas_size-0.5 * xmangekyou.size)/(1.5 * xmangekyou.size)+1.0);
  if (r < 11) InitializeDrawRoutine1(r);
  else InitializeDrawRoutine2(canvas_size);
}

/*****************************************************************************/
/* Initializing Pieces.                                                      */
/*****************************************************************************/

void InitializeLines()
{
  int i;

  xmangekyou.lines = (Line *)malloc(sizeof(Line) * xmangekyou.lines_number);
  for (i = 0; i < xmangekyou.lines_number; i++) {
    xmangekyou.lines[i].point_1.x = (double)(rand() % xmangekyou.size);
    xmangekyou.lines[i].point_1.y = (double)(rand() % xmangekyou.size) -
      (xmangekyou.size / 2);
    xmangekyou.lines[i].point_2.x = (double)(rand() % xmangekyou.size);
    xmangekyou.lines[i].point_2.y = (double)(rand() % xmangekyou.size) -
      (xmangekyou.size / 2);
    xmangekyou.lines[i].gc = XCreateGC(xmangekyou.display,
				       xmangekyou.toplevel_window, 0, NULL);
    XSetForeground(xmangekyou.display, xmangekyou.lines[i].gc,
		   GetColor(xmangekyou.display, xmangekyou.colormap,
			    color_name[rand() % COLORS_NUMBER]));
    XSetLineAttributes(xmangekyou.display, xmangekyou.lines[i].gc,
		       (rand() % xmangekyou.line_thickness),
		       LineSolid, CapButt ,JoinMiter);
  }
}

void InitializeCircles()
{
  int i;

  xmangekyou.circles =
    (Circle *)malloc(sizeof(Circle) * xmangekyou.circles_number);
  for (i = 0; i < xmangekyou.circles_number; i++) {
    xmangekyou.circles[i].r = (rand() % xmangekyou.circle_size);
    xmangekyou.circles[i].center.x = (double)(rand() % xmangekyou.size);
    xmangekyou.circles[i].center.y = (double)(rand() % xmangekyou.size) -
      (xmangekyou.size / 2);
    xmangekyou.circles[i].gc = XCreateGC(xmangekyou.display,
					 xmangekyou.toplevel_window, 0, NULL);
    XSetForeground(xmangekyou.display, xmangekyou.circles[i].gc,
		   GetColor(xmangekyou.display, xmangekyou.colormap,
			    color_name[rand() % COLORS_NUMBER]));
    XSetArcMode(xmangekyou.display, xmangekyou.circles[i].gc, ArcPieSlice);
  }
}

void InitializePieces()
{
  srand((unsigned)time(NULL));
  InitializeLines();
  InitializeCircles();
}

/*****************************************************************************/
/* Addition of EventHandler                                                  */
/*****************************************************************************/

void PushKey(Widget w, XtPointer p, XEvent * event, Boolean * dispatch)
{
  XMangekyou * xmp;

  xmp = (XMangekyou *)p;
  if (event->type == KeyPress) {
    if (XKeycodeToKeysym(event->xkey.display, event->xkey.keycode, 0) == XK_q)
      Quit();
    if (XKeycodeToKeysym(event->xkey.display, event->xkey.keycode, 0) == XK_a)
      SetAnimation();
    if (XKeycodeToKeysym(event->xkey.display, event->xkey.keycode, 0) == XK_p)
      AnimationPictureStart();
    if (XKeycodeToKeysym(event->xkey.display, event->xkey.keycode, 0) == XK_d)
      DumpAnimationData();
  }
#ifdef XMANGEKYOU_DEBUG
  fprintf(stderr, "Key Press.\n");
#endif
}

void ExposeCanvas(Widget w, XtPointer p, XEvent * event, Boolean * dispatch)
{
  XMangekyou * xmp;

  xmp = (XMangekyou *)p;
  if (event->type == Expose) {
    XSetWindowBackgroundPixmap(xmangekyou.display, xmangekyou.canvas_window,
	       xmangekyou.source_pixmaps[xmangekyou.copy_number].pixmap);
    XClearWindow(xmangekyou.display, xmangekyou.canvas_window);
    /*
    XCopyArea(xmangekyou.display,
	      xmangekyou.source_pixmaps[xmangekyou.copy_number].pixmap,
	      xmangekyou.canvas_window, xmangekyou.gc_copy_canvas_canvas,
	      event->xexpose.x, event->xexpose.y,
	      event->xexpose.width, event->xexpose.height,
	      event->xexpose.x, event->xexpose.y);
	      */
  }
#ifdef XMANGEKYOU_DEBUG
  fprintf(stderr, "Expose.\n");
#endif
}

void ResizeCanvas()
{
  int i;
  int n = 0;
  Arg args[10];

  XtSetArg(args[n], XtNwidth, &(xmangekyou.width)); n++;
  XtSetArg(args[n], XtNheight, &(xmangekyou.height)); n++;
  XtGetValues(xmangekyou.toplevel, args, n);
  XResizeWindow(xmangekyou.display, xmangekyou.canvas_window,
		xmangekyou.width, xmangekyou.height);
  for (i = 0; i <= xmangekyou.copy_number; i++)
    XFreePixmap(xmangekyou.display, xmangekyou.source_pixmaps[i].pixmap);
  free(xmangekyou.source_pixmaps);
  free(xmangekyou.copy_points);
  free(xmangekyou.copy_points_rotated);
  InitializeMangekyou();
#ifdef XMANGEKYOU_DEBUG
  fprintf(stderr, "Resize.\n");
#endif
}

void CanvasEvent(Widget w, XtPointer p, XEvent * event, Boolean * dispatch)
{
  XMangekyou * xmp;

  xmp = (XMangekyou *)p;
  if (event->type == ConfigureNotify) ResizeCanvas();
  else if (event->type == ClientMessage) {
    if ( (event->xclient.message_type == xmangekyou.atom_1) &&
	 (event->xclient.data.l[0] == xmangekyou.atom_2) ) Quit();
  }
#ifdef XMANGEKYOU_DEBUG
  fprintf(stderr, "Structure Notify Event.\n");
#endif
}

void AddEvent()
{
  XtAddEventHandler(xmangekyou.toplevel, ExposureMask, False, ExposeCanvas,
		    &(xmangekyou));
  if (xmangekyou.root_flag == OFF) {
    XtAddEventHandler(xmangekyou.toplevel, KeyPressMask, False, PushKey,
		      &(xmangekyou));
    XtAddEventHandler(xmangekyou.toplevel, StructureNotifyMask, True,
		      CanvasEvent, &(xmangekyou));
  }
}

/*****************************************************************************/
/* Setting Icon.                                                             */
/*****************************************************************************/

void SetIcon()
{
  XIconSize * is;
  int i, j, n;
  int width_match = OFF;
  int height_match = OFF;
  XWMHints wmh;

  xmangekyou.icon_pixmap =
    MakePixmapFromData(xmangekyou.display, xmangekyou.root_window,
		       XMANGEKYOU_ICON,
		       &(xmangekyou.icon_width), &(xmangekyou.icon_height));

  wmh.flags = IconPixmapHint;
  wmh.icon_pixmap = xmangekyou.icon_pixmap;
  XSetWMHints(xmangekyou.display, xmangekyou.toplevel_window, &wmh);
}

/*****************************************************************************/
/* Setting Window Attributes.                                                */
/*****************************************************************************/

void SetWindowAttributes()
{
  XWMHints wmhints;

  if (xmangekyou.root_flag == OFF) {
    XtSetSensitive(xmangekyou.toplevel, True);
    XtSetSensitive(xmangekyou.canvas, True);

    /*
    XSelectInput(xmangekyou.display, xmangekyou.canvas_window,
		 KeyPressMask | ExposureMask);
    XSelectInput(xmangekyou.display, xmangekyou.toplevel_window,
    StructureNotifyMask);
    */

    wmhints.input = True;
    XSetWMHints(xmangekyou.display, xmangekyou.toplevel_window, &wmhints);
    xmangekyou.atom_1 = XInternAtom(xmangekyou.display, "WM_PROTOCOLS", False);
    xmangekyou.atom_2 = XInternAtom(xmangekyou.display, "WM_DELETE_WINDOW",
				    False);
    XSetWMProtocols(xmangekyou.display, xmangekyou.toplevel_window,
		    &(xmangekyou.atom_2), 1);
    XStoreName(xmangekyou.display, xmangekyou.toplevel_window, "XMangekyou");
    XSetIconName(xmangekyou.display, xmangekyou.toplevel_window, "XMangekyou");
    SetIcon();
  }
}

/*****************************************************************************/
/* End of File.                                                              */
/*****************************************************************************/
