/*****************************************************************************/
/* XMangekyou - a kaleidoscope for X                                         */
/* Programed by Sakai Hiroaki.                                               */
/*                                                                           */
/* Copyright (c) 1998 Sakai Hiroaki.                                         */
/* All Rights Reserved.                                                      */
/*                                                                           */
/* XMangekyou is a kaleidoscope. You can see many, many, many patterns,      */
/* but, care for yourself from 'trip' filling!                               */
/*                                                                           */
/* XMangekyou is a free software. You may copy or distribute the original    */
/* XMangekyou freely. But, you may not modify or distribute the original     */
/* XMangekyou without permission of the author.                              */
/*****************************************************************************/

/*****************************************************************************/
/* Include Header File.                                                      */
/*****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <signal.h>
#include <X11/Xlib.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/keysym.h>
#include <X11/xpm.h>
#include <X11/Xaw/Label.h>

#include "configure.h"
#include "xmangekyou.h"
#include "anime.h"
#include "cal.h"
#include "graph.h"
#include "init.h"
#include "quit.h"
#include "string.h"
#include "time.h"

/*****************************************************************************/
/* Moving Pieces.                                                            */
/*****************************************************************************/

void MovePieces()
{
  int i;
  DoublePoint dominant, cs, move;
  dominant.x = 1.0;
  dominant.y = 0.0;
  CalCosSin(- xmangekyou.degree, &cs);
  CalRotate(dominant, cs, &move);
  move.x *= 0.000001 * xmangekyou.change;
  move.y *= 0.000001 * xmangekyou.change;
  for (i = 0; i < xmangekyou.lines_number; i++) {
    xmangekyou.lines[i].point_1.x += move.x * (rand()%1000);
    xmangekyou.lines[i].point_1.y += move.y * (rand()%1000);
    xmangekyou.lines[i].point_2.x += move.x * (rand()%1000);
    xmangekyou.lines[i].point_2.y += move.y * (rand()%1000);
  }
  for (i = 0; i < xmangekyou.circles_number; i++) {
    xmangekyou.circles[i].center.x += move.x * (rand()%1000);
    xmangekyou.circles[i].center.y += move.y * (rand()%1000);
  }
}

/*****************************************************************************/
/* Drawing Mangekyou.                                                        */
/*****************************************************************************/

SixPoints CalHexPoints(DoublePoint p1, DoublePoint value_000,
		       DoublePoint value_120, DoublePoint value_240)
{
  SixPoints p;
  DoublePoint p2;
  p2.x = - p1.x;
  p2.y =   p1.y;
  CalRotate(p1, value_000, &(p.point[0]));
  CalRotate(p2, value_000, &(p.point[1]));
  CalRotate(p1, value_120, &(p.point[2]));
  CalRotate(p1, value_240, &(p.point[3]));
  CalRotate(p2, value_120, &(p.point[4]));
  CalRotate(p2, value_240, &(p.point[5]));
  return (p);
}

void ClearPixmaps()
{
  int i;

  XFillRectangle(xmangekyou.display, xmangekyou.source_pixmaps[0].pixmap,
		 xmangekyou.gc_source_clear, 0, 0,
		 xmangekyou.source_pixmaps[0].width.x,
		 xmangekyou.source_pixmaps[0].width.y);
  for (i = 1; i <= xmangekyou.copy_number; i++)
    XFillRectangle(xmangekyou.display, xmangekyou.source_pixmaps[i].pixmap,
		   xmangekyou.gc_canvas_clear, 0, 0,
		   xmangekyou.source_pixmaps[i].width.x,
		   xmangekyou.source_pixmaps[i].width.y);
}

void DrawPieces(DoublePoint value_000, DoublePoint value_120,
		DoublePoint value_240)
{
  int i, j, n, sx, sy;
  SixPoints points_1, points_2;

  n = xmangekyou.lines_number;
  if (n < xmangekyou.circles_number) n = xmangekyou.circles_number;
  sx = xmangekyou.source_pixmaps[0].center.x;
  sy = xmangekyou.source_pixmaps[0].center.y;
  for (i = 0; i < n; i++) {
    if (i < xmangekyou.lines_number) {
      points_1 = CalHexPoints(xmangekyou.lines[i].point_1,
			      value_000, value_120, value_240);
      points_2 = CalHexPoints(xmangekyou.lines[i].point_2,
			      value_000, value_120, value_240);
      for (j = 0; j < 6; j++)
	XDrawLine(xmangekyou.display, xmangekyou.source_pixmaps[0].pixmap,
		  xmangekyou.lines[i].gc,
		  (int)points_1.point[j].x+sx, (int)points_1.point[j].y+sy,
		  (int)points_2.point[j].x+sx, (int)points_2.point[j].y+sy);
    }
    if (i < xmangekyou.circles_number) {
      points_1 = CalHexPoints(xmangekyou.circles[i].center,
			      value_000, value_120, value_240);
      for (j = 0; j < 6; j++)
	XFillArc(xmangekyou.display, xmangekyou.source_pixmaps[0].pixmap,
		 xmangekyou.circles[i].gc,
		 (int)(points_1.point[j].x - xmangekyou.circles[i].r * 0.5)+sx,
		 (int)(points_1.point[j].y - xmangekyou.circles[i].r * 0.5)+sy,
		 xmangekyou.circles[i].r, xmangekyou.circles[i].r, 0, 360*64);
    }
  }
}

void CalCopyPoints(DoublePoint d)
{
  int i;

  for (i = 0; i < xmangekyou.copy_max; i++)
    CalRotate(xmangekyou.copy_points[i], d,
	      &(xmangekyou.copy_points_rotated[i]));
}

void CalClearPoints(DoublePoint d)
{
  int i;

  for (i = 0; i < CLEAR_POINTS_NUMBER; i++) {
    CalRotate(clear_points[i], d, &(xmangekyou.clear_points_rotated[i]));
    CalRotate(clear_points_offset[i], d,
	      &(xmangekyou.clear_points_offset_rotated[i]));
  }
}

void ClearHexPoints(int a)
{
  int i;

  for (i = 0; i < CLEAR_POINTS_NUMBER; i++) {
    xmangekyou.clear_points_rotated_int[i].x =
      (int)(xmangekyou.clear_points_rotated[i].x *
	    xmangekyou.source_pixmaps[a].size +
	    xmangekyou.clear_points_offset_rotated[i].x +
	    xmangekyou.source_pixmaps[a].center.x);
    xmangekyou.clear_points_rotated_int[i].y =
      (int)(xmangekyou.clear_points_rotated[i].y *
	    xmangekyou.source_pixmaps[a].size +
	    xmangekyou.clear_points_offset_rotated[i].y +
	    xmangekyou.source_pixmaps[a].center.y);
  }
  XFillPolygon(xmangekyou.display, xmangekyou.source_pixmaps[a].pixmap,
	       xmangekyou.gc_source_hex_clear,
	       xmangekyou.clear_points_rotated_int, CLEAR_POINTS_NUMBER,
	       Complex, CoordModeOrigin);
}

void CopyPixmaps(int a)
{
  int b, i, cx, cy;

  if (a == xmangekyou.copy_number - 1) b = xmangekyou.copy_end;
  else b = xmangekyou.copy_max;
  for (i = 0; i < b; i++) {
    cx = (int)(xmangekyou.copy_points_rotated[i].x *
	       xmangekyou.source_pixmaps[a].size) +
      xmangekyou.source_pixmaps[a + 1].center.x;
    cy = (int)(xmangekyou.copy_points_rotated[i].y *
	       xmangekyou.source_pixmaps[a].size) +
      xmangekyou.source_pixmaps[a + 1].center.y;
    if ( (cx >= - xmangekyou.source_pixmaps[a].center.x) &&
	 (cx <=   xmangekyou.source_pixmaps[a].center.x +
	  xmangekyou.source_pixmaps[a+1].width.x) &&
	 (cy >= - xmangekyou.source_pixmaps[a].center.y) &&
	 (cy <=   xmangekyou.source_pixmaps[a].center.y +
	  xmangekyou.source_pixmaps[a+1].width.y) )
      XCopyArea(xmangekyou.display, xmangekyou.source_pixmaps[a].pixmap,
		xmangekyou.source_pixmaps[a+1].pixmap,
		xmangekyou.gc_copy_source_canvas, 0, 0,
		xmangekyou.source_pixmaps[a].width.x,
		xmangekyou.source_pixmaps[a].width.y,
		(int)(cx - xmangekyou.source_pixmaps[a].center.x),
		(int)(cy - xmangekyou.source_pixmaps[a].center.y));
  }
}

void DrawMangekyou()
{
  int i;
  DoublePoint value_000, value_120, value_240;

  CalCosSin(         xmangekyou.degree, &value_000);
  CalCosSin(120*64 + xmangekyou.degree, &value_120);
  CalCosSin(240*64 + xmangekyou.degree, &value_240);
  ClearPixmaps();
  DrawPieces(value_000, value_120, value_240);
  CalCopyPoints(value_000);
  CalClearPoints(value_000);

  for (i = 0; i < xmangekyou.copy_number; i++) {
    ClearHexPoints(i);
    CopyPixmaps(i);
  }

  XSetWindowBackgroundPixmap(xmangekyou.display, xmangekyou.canvas_window,
		     xmangekyou.source_pixmaps[xmangekyou.copy_number].pixmap);
  XClearWindow(xmangekyou.display, xmangekyou.canvas_window);
/*
  XCopyArea(xmangekyou.display,
	    xmangekyou.source_pixmaps[xmangekyou.copy_number].pixmap,
	    xmangekyou.canvas_window, xmangekyou.gc_copy_canvas_canvas, 0, 0,
	    xmangekyou.width, xmangekyou.height, 0, 0);
*/
  XFlush(xmangekyou.display);
}

/*****************************************************************************/
/* Main Loop.                                                                */
/*****************************************************************************/

Boolean MainMangekyou(caddr_t p)
{
  XMangekyou * xmp;

  xmp = (XMangekyou *)p;
  DrawMangekyou();
  if (xmangekyou.animation_flag == ON) AnimationPieces();
  else MovePieces();
  xmangekyou.degree += xmangekyou.speed;
  if ( xmangekyou.degree >  360*64 ) xmangekyou.degree -= 360*64;
  if ( xmangekyou.degree < -360*64 ) xmangekyou.degree += 360*64;
  CheckAnimation();
  usleep(xmangekyou.wait);
  if (xmangekyou.quit_flag == ON) Quit();
  return (False);
}

void InterruptHandler()
{
  DrawMangekyou();
  Quit();
}

void InitializeSignal()
{
  signal(SIGINT, InterruptHandler);
}

main(int argc, char ** argv)
{
  SetDefaultValues();
  ReadParameters(&argc, argv);
  InitializeDisplay(&argc, argv);
  InitializeMangekyou();
  InitializePieces();
  MakeAnimationData();
  AddEvent();
  SetWindowAttributes();
  XtAppAddWorkProc(xmangekyou.app_context, (XtWorkProc)MainMangekyou,
		   &xmangekyou);
  InitializeSignal();
  XtAppMainLoop(xmangekyou.app_context);
}

/*****************************************************************************/
/* End of File.                                                              */
/*****************************************************************************/
