/* XRacer (C) 1999 Richard W.M. Jones.
 * $Id: game.c,v 1.18 1999/09/20 20:01:42 rich Exp $
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "xracer.h"
#include "globals.h"
#include "xracer-net.h"
#include "sound.h"
#include "samples.h"

struct game game;

/* Strings and fonts used in display_game_over screen. */
static void *font_large;
static const char *final_position = "Final position";
static int final_position_len;
static const char *race_time = "Race time";
static int race_time_len;
static const char *best_lap_time = "Best lap time";
static int best_lap_time_len;
static const char *average_lap_time = "Average lap time";
static int average_lap_time_len;

/* Initialize game structure. */
#define DEFAULT_LAPS_NUMBER 3
void
game_init ()
{
  int i;

  game.countdown_time = current_time;

  if (mode == NETGAME_MODE)
    game.state = GAME_WAITING;
  else
    game.state = GAME_COUNTING;

  for (i = 0; i < 10; ++i)
    game.lap_time[i] = -1;

  if( game.nr_laps == 0)
	  game.nr_laps = DEFAULT_LAPS_NUMBER ; 

  font_large = find_font ("crillee", 48);
  log_assert (font_large != 0);
  final_position_len = get_text_width (font_large, final_position);
  race_time_len = get_text_width (font_large, race_time);
  best_lap_time_len = get_text_width (font_large, best_lap_time);
  average_lap_time_len = get_text_width (font_large, average_lap_time);
}

/* Format time into a printable string. */
static const char *
printable_time (double t)
{
  static char buffer[6];
  int n;
  if (t < 0 || t > 999.9) t = 999.9;
  n = t * 10;
  sprintf (buffer, "%03u.%u", n/10, n%10);
  return buffer;
}

/* This function is called whenever we pass over the start line. */
void
game_trigger_start ()
{
}

#define NOT_TOO_OFTEN(cr) { static double last[8]; if (current_time - last[cr] < 0.5) return; last[cr] = current_time; }

/* This function is called whenever we go over a fast (one arrow) spot. */
void
game_trigger_fast (int cr)
{
  NOT_TOO_OFTEN (cr);

  /* Set thrust to max. */
  pilot[cr].thrust = craft[cr].max_thrust;
}

/* This function is called whenever we go over a faster (two arrows) spot. */
void
game_trigger_faster (int cr)
{
  NOT_TOO_OFTEN (cr);

  /* Multiply speed. */
  pilot[cr].faster *= craft[cr].faster_mult;

  /* Play faster sound. */
  if (cr == local_player_nr && sound_state == SOUND_STATE_ENABLED)
    sound_play_sample (SAMPLE_FASTER);
}

/* This function is called whenever we go over a powerup spot. */
void
game_trigger_powerup (int cr)
{
  NOT_TOO_OFTEN (cr);

  /* Choose a random powerup and give it to the pilot. */
  if (pilot[cr].powerup == NO_POWERUP)
    {
      pilot[cr].powerup = (random () % MAX_POWERUP) + 1;

      /* Play powerup sound. */
      if (cr == local_player_nr && sound_state == SOUND_STATE_ENABLED)
	sound_play_sample (SAMPLE_POWERUP);
    }
}

#define SHIELD_POWERUP_AMOUNT 0.4

/* This function is called whenever a player uses a weapon or powerup. */
void
game_fire (int cr)
{
  int powerup = pilot[cr].powerup;

  if (powerup != NO_POWERUP)
    {
      /* Use up the powerup. */
      pilot[cr].powerup = NO_POWERUP;

      /* What is it? */
      switch (powerup)
	{
	case POWERUP_FASTER:
	  /* Multiply speed. */
	  pilot[cr].faster *= craft[cr].faster_mult;

	  /* Play faster sound. */
	  if (cr == local_player_nr && sound_state == SOUND_STATE_ENABLED)
	    sound_play_sample (SAMPLE_FASTER);
	  break;

	case POWERUP_ENERGY:
	  /* Fix shield energy. */
	  pilot[cr].shield += SHIELD_POWERUP_AMOUNT;
	  if (pilot[cr].shield > 1) pilot[cr].shield = 1;
	  break;

	case POWERUP_SHIELD:
	  /* Enable shield. */
	  pilot[cr].has_external_shield = 1;
	  pilot[cr].external_shield_start_time = current_time;
	  break;

	case POWERUP_AUTOPILOT:
	  /* Enable autopilot. */
	  pilot[cr].has_autopilot = 1;
	  pilot[cr].autopilot_start_time = current_time;
	  break;
	}
    }
}

#define DUNK_MOMENTUM_PENALTY 3.0
#define DUNK_THRUST_PENALTY 3.0
#define DUNK_SHIELD_PENALTY 0.05

/* This function is called whenever we hit the sides or bottom. */
void
game_dunk (int cr)
{
  int p;

  /* Reduce momentum. */
  for (p = 0; p < 3; ++p)
    {
      pilot[cr].momentum[p][0] /= DUNK_MOMENTUM_PENALTY;
      pilot[cr].momentum[p][1] /= DUNK_MOMENTUM_PENALTY;
      pilot[cr].momentum[p][2] /= DUNK_MOMENTUM_PENALTY;
    }

  /* Reduce speed. */
  if (pilot[cr].thrust > craft[cr].max_thrust/DUNK_THRUST_PENALTY)
    pilot[cr].thrust = craft[cr].max_thrust/DUNK_THRUST_PENALTY;

  /* Kill go-faster points. */
  pilot[cr].faster = 1;

  NOT_TOO_OFTEN (cr);

  /* XXX Sound effect. */

  /* Reduce shield. */
  if (! pilot[cr].has_external_shield)
    {
      pilot[cr].shield -= DUNK_SHIELD_PENALTY;
      if (pilot[cr].shield < 0) pilot[cr].shield = 0;
    }
}

#define CRASH_SHIELD_PENALTY 0.02

/* This function is called whenever we hit another player. */
void
game_crash (int cr)
{
  /* Kill go-faster points. */
  pilot[cr].faster = 1;

  NOT_TOO_OFTEN (cr);

  /* XXX Sound effect. */

  /* Reduce shield. */
  if (! pilot[cr].has_external_shield)
    {
      pilot[cr].shield -= CRASH_SHIELD_PENALTY;
      if (pilot[cr].shield < 0) pilot[cr].shield = 0;
    }
}

/* This function is called whenever the player starts a new lap. */
void
game_new_lap (int cr)
{
  double t = current_time - game.start_time;
  int lap_index;

  if (cr != local_player_nr) return;

  /* Save the player's time in that lap record. */
  lap_index = pilot[local_player_nr].current_lap - 2;
  game.lap_time[lap_index] = t;
  strcpy (game.lap_time_s[lap_index], printable_time (t));

  /* Start a new lap. */
  game.start_time = current_time;
}

/* These are filled in at the end of the game for the local player. */
static int final_rank;
static char final_rank_s[10];
static int final_rank_s_len;
static double final_avg_lap_time;
static char final_avg_lap_time_s[10];
static int final_avg_lap_time_s_len;
static double final_best_lap_time;
static char final_best_lap_time_s[10];
static int final_best_lap_time_s_len;
static double final_race_time;
static char final_race_time_s[10];
static int final_race_time_s_len;

/* This function is called whenever the player ends the game. */
void
game_end_game (int cr)
{
  int i, nr_players;

  if (cr != local_player_nr) return;

  game_new_lap (cr);		/* Save lap time. */

  game.state = GAME_ENDED;

  /* Save game positions, etc. for display_game_over function below. */
  calculate_player_rank (local_player_nr, &final_rank, &nr_players);
  sprintf (final_rank_s, "%d", final_rank);
  final_rank_s_len = get_text_width (font_large, final_rank_s);

  final_race_time = 0;
  final_best_lap_time = 100000;
  for (i = 0; i < game.nr_laps; ++i)
    {
      final_race_time += game.lap_time[i];
      if (game.lap_time[i] < final_best_lap_time)
	final_best_lap_time = game.lap_time[i];
    }
  final_avg_lap_time = final_race_time / game.nr_laps;

  strcpy (final_race_time_s, printable_time (final_race_time));
  final_race_time_s_len = get_text_width (font_large, final_race_time_s);
  strcpy (final_avg_lap_time_s, printable_time (final_avg_lap_time));
  final_avg_lap_time_s_len = get_text_width (font_large, final_avg_lap_time_s);
  strcpy (final_best_lap_time_s, printable_time (final_best_lap_time));
  final_best_lap_time_s_len = get_text_width (font_large, final_best_lap_time_s);
}

/* This function is called when the panel needs to display the current time. */
const char *
game_get_time_s ()
{
  if (game.state == GAME_RUNNING)
    {
      return printable_time (current_time - game.start_time);
    }
  else
    return "000.0";
}

/* This returns true until the point where the game has
 * started. This is used in pilot.c where it holds the
 * craft onto the start line.
 */
int
game_held ()
{
  if (game.state == GAME_RUNNING) return 0;
  if (game.state == GAME_ENDED) return 1;
  if (game.state == GAME_WAITING) return 1;

  /* Time to start the game? */
  if (current_time - game.countdown_time >= 5)
    {
      game.state = GAME_RUNNING;
      game.start_time = current_time;
      log (LOG_DEBUG, "game started");
    }

  return 1;
}

int game_waiting()
{
  return (game.state == GAME_WAITING);
}

void game_begin()
{
  if (mode == NETGAME_MODE)
    net_begin_game();
  
  if (game.state == GAME_WAITING)
    game.state = GAME_COUNTING;

  game.countdown_time = current_time;
}

/* Work out what lamp to display - red, yellow, green,
 * and return:
 *  0 = no lamp
 *  1 = display lamp, but none lit up
 *  2 = red lamp
 *  3 = yellow lamp
 *  4 = green lamp
 */
int
game_lamp_stage ()
{
  int t = current_time - game.countdown_time;

  if (game.state == GAME_WAITING)
    return 2;

  if (t >= 7) return 0;
  if (t >= 5) return 4;
  if (t >= 4) return 3;
  if (t >= 3) return 2;
  return 1;
}

/* Display the game over screen and stats. */
void
display_game_over ()
{
  if (mode != TIME_TRIAL_MODE)
    {
      draw_text (font_large, final_position,
		 (refwidth - final_position_len) / 2, refheight / 9);
      draw_text (font_large, final_rank_s,
		 (refwidth - final_rank_s_len) / 2, refheight * 2 / 9);
    }

  draw_text (font_large, race_time,
	     (refwidth - race_time_len) / 2, refheight * 3 / 9);
  draw_text (font_large, final_race_time_s,
	     (refwidth - final_race_time_s_len) / 2, refheight * 4 / 9);

  draw_text (font_large, best_lap_time,
	     (refwidth - best_lap_time_len) / 2, refheight * 5 / 9);
  draw_text (font_large, final_best_lap_time_s,
	     (refwidth - final_best_lap_time_s_len) / 2, refheight * 6 / 9);

  draw_text (font_large, average_lap_time,
	     (refwidth - average_lap_time_len) / 2, refheight * 7 / 9);
  draw_text (font_large, final_avg_lap_time_s,
	     (refwidth - final_avg_lap_time_s_len) / 2, refheight * 8 / 9);
}
