/* XRacer (C) 1999 Richard W.M. Jones.
 * $Id: precipitation.c,v 1.6 1999/07/17 23:53:05 rich Exp $
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>

#include "xracer.h"

/* This file deals with displaying two types of precipitation: rain and snow */

/* Snow is modelled rather like rocks (a program which comes
 * with xscreensaver). You have lots of little snowflakes which
 * radiate out from near a point (c_x, c_y) which is somewhere on
 * or off the screen. The point moves around according to what
 * the craft is doing. If the craft goes faster, the point
 * moves down towards the centre of the screen. If the craft
 * goes slower, it moves up to beyond the top of the screen.
 * If the craft turns left, the point moves right. If the
 * craft turns right, the point moves left. In addition, the
 * speed at which snow comes out increases with speed.
 */

static GLfloat snow_colour[4] = { 1, 1, 1, 1 };

#define NR_SNOWFLAKES 500
#define MAX_HEIGHT (height*3/2)
#define MIN_HEIGHT (height/2)
#define INIT_ACCEL 0.1
#define ACCEL_SCALE (1/200.)
#define VELOCITY_SCALE (1/200.)
#define BIGGER_AGE 10
#define BIGGEST_AGE 20
#define X_SRC_RAND 100
#define Y_SRC_RAND 100

static GLfloat c_x, c_y;

struct {
  /* Equation of line this snowflake follows. */
  GLfloat m, c;
  /* Current x position. */
  GLfloat x;
  /* Speed (change in x). */
  GLfloat dx;
  /* Acceleration. */
  GLfloat d2x;
  /* Faked distance from viewer. */
  int t;
} snowflake[NR_SNOWFLAKES];

static void
new_snowflake (int i)
{
  GLfloat start_x, start_y;
  GLfloat cr_x = c_x, cr_y = c_y;
  GLfloat velocity = pilot[local_player_nr].speed * VELOCITY_SCALE;

  /* Introduce a bit of randomness into radiating point. */
  cr_x += drand48() * X_SRC_RAND - X_SRC_RAND/2;
  cr_y += drand48() * Y_SRC_RAND - Y_SRC_RAND/2;

  /* Choose random line, but only choose lines which
   * actually pass through the screen.
   */
  start_x = drand48() * width;
  start_y = drand48() * height;

  if (cr_x == start_x) start_x++; /* Avoid divide by zero error. */

  snowflake[i].x = start_x;
  snowflake[i].m = (cr_y - start_y) / (cr_x - start_x);
  snowflake[i].c = cr_y - snowflake[i].m * cr_x;

  /* Choose an initial velocity. */
  if (snowflake[i].x < cr_x)
    {
      snowflake[i].dx = - drand48() * velocity;
      snowflake[i].d2x = - INIT_ACCEL - drand48() * velocity;
    }
  else
    {
      snowflake[i].dx = drand48() * velocity;
      snowflake[i].d2x = INIT_ACCEL + drand48() * velocity;
    }

  snowflake[i].t = 0;
}

static inline void
display_snow ()
{
  int i;
  GLfloat x, y;

  glBegin (GL_POINTS);
  glColor4fv (snow_colour);

  for (i = 0; i < NR_SNOWFLAKES; ++i)
    {
      /* Draw snowflake. */
      x = snowflake[i].x;
      y = snowflake[i].m*x+snowflake[i].c;
      glVertex2f (x, y);

      /* If snowflake is close to us, make it bigger. */
      if (snowflake[i].t > BIGGER_AGE)
	{
	  glVertex2f (x+1, y);
	  glVertex2f (x+1, y+1);
	  glVertex2f (x, y+1);
	}
      if (snowflake[i].t > BIGGEST_AGE)
	{
	  glVertex2f (x+2, y);
	  glVertex2f (x+2, y+1);
	  glVertex2f (x+2, y+2);
	  glVertex2f (x+1, y+2);
	  glVertex2f (x, y+2);
	}
      snowflake[i].t++;

      /* Move snowflake. */
      snowflake[i].x += snowflake[i].dx;
      snowflake[i].dx += snowflake[i].d2x;

      /* Off screen? Generate a new one. */
      if (snowflake[i].x < 0 || snowflake[i].x >= width)
	{
	  new_snowflake (i);
	}
    }

  glEnd ();
}

static void
init_snow ()
{
  int i;

  c_x = width/2;
  c_y = MAX_HEIGHT;

  for (i = 0; i < NR_SNOWFLAKES; ++i)
    new_snowflake (i);
}

/* static GLfloat rain_colour[4] = { 0, 1, 1, 1 }; */

static inline void
display_rain ()
{
}

static void
init_rain ()
{
}

void
display_precipitation ()
{
  c_x = width/2 + width/2 * (pilot[local_player_nr].control_x / 128.);
  c_y = MAX_HEIGHT -
    ((MAX_HEIGHT - MIN_HEIGHT) * pilot[local_player_nr].speed / 2000);

  if (track->is_precipitation == RAIN)
    display_rain ();
  else
    display_snow ();
}

void
precipitation_init ()
{
  init_rain ();
  init_snow ();
}
