/* XRacer (C) 1999 Richard W.M. Jones.
 * $Id: windows.c,v 1.1 1999/09/12 13:16:25 rich Exp $
 */

/* 
 * XRacer windows sound code based on MSDN DirectSound
 * examples.
 *
 * Uses "double buffer" algorithm in conjunction with
 * a looping audio buffer.  The audio buffer
 * is fixed in size, and can only be written to by
 * updating exactly 1/2 of its size.  Each write updates
 * the half of the buffer that's currently not being
 * played by the audio device.
 *
 * Windows event notification mechanism is used to
 * signal when to update the audio buffer.  Each write
 * to the buffer blocks until a position notification
 * is received.  The new data is then copied into the
 * audio buffer.  Note write doesn't wait for the data
 * to be played.
 *
 * Position notifications are sent when the play position
 * enters the beginning of each half of the buffer.
 * The first half is written to when the play position
 * is at the beginning of the second half, and the 2nd
 * half is written to when the play position is at "0".
 *
 * Play position notification events:
 *
 *      Event 0:          Event 1:
 *
 *        -------           -------
 *        |  |XX|           |XX|  |
 *        -------           -------
 *        P  W              W  P
 *
 * This shows the 'P'lay and 'W'rite positions after
 * receiving one of the play position notification events.  
 * 'XX' is the new data written to the buffer.
 *
 */

#include <windows.h>
#include <initguid.h>
#include <windowsx.h>
#include <mmsystem.h>
#include <dsound.h>
#include "globals.h"
#include "xracer.h"
#include "sound_process.h"
#include "os_sound.h"

#define DSBNUMEVENTS 2
#define DSBFRAMESIZE SECTOR_LENGTH

LPDIRECTSOUND lpDS;
LPDIRECTSOUNDBUFFER lpDSB;
LPDIRECTSOUNDNOTIFY lpdsNotify;
DWORD rghEvt[2];
static DWORD dsb_size_bytes, dsb_channels = 2, dsb_rate = 44100, dsb_bits = 16;
static BOOL dsb_open = FALSE;

int
os_sound_open_device ()
{
  PCMWAVEFORMAT pcmwf; 
  DSBUFFERDESC dsbdesc;
  DSBPOSITIONNOTIFY rgdsbpn[2];
  HRESULT hr;

  if (dsb_open)
    return 1;

  printf("ds open ghWnd = %x\n", (unsigned int) ghWnd);

  if (DirectSoundCreate(NULL, &lpDS, NULL) == DS_OK) {
    if (IDirectSound_SetCooperativeLevel(lpDS, ghWnd, DSSCL_NORMAL) != DS_OK) {
          log_perror("can't set cooperative level of ds object");
          return -1;
    }
  }
  else {
    log_perror("can't create ds object");
    return -1;
  }

  dsb_size_bytes = 2 * DSBFRAMESIZE * dsb_channels * dsb_bits / 8;

  // Set up wave format structure. 
  memset(&pcmwf, 0, sizeof(PCMWAVEFORMAT)); 
  pcmwf.wf.wFormatTag = WAVE_FORMAT_PCM;
  pcmwf.wf.nChannels = dsb_channels; 
  pcmwf.wf.nSamplesPerSec = dsb_rate;
  pcmwf.wBitsPerSample = dsb_bits;
  pcmwf.wf.nBlockAlign = dsb_channels * dsb_bits / 8; 
  pcmwf.wf.nAvgBytesPerSec = pcmwf.wf.nSamplesPerSec * pcmwf.wf.nBlockAlign; 
  // Set up DSBUFFERDESC structure. 
  memset(&dsbdesc, 0, sizeof(DSBUFFERDESC)); // Zero it out. 
  dsbdesc.dwSize = sizeof(DSBUFFERDESC); 
  // Need default controls (pan, volume, frequency). 
  dsbdesc.dwFlags = DSBCAPS_CTRLDEFAULT | DSBCAPS_CTRLPOSITIONNOTIFY;
  dsbdesc.dwBufferBytes = dsb_size_bytes; 
  dsbdesc.lpwfxFormat = (LPWAVEFORMATEX)&pcmwf;
  // Create buffer. 
  hr = IDirectSound_CreateSoundBuffer(lpDS, &dsbdesc, &lpDSB, NULL);
  if (!SUCCEEDED(hr)) {
    log_perror("can't create dsb object");
    return -1;
  }
  // set up buffer position notification
  rghEvt[0] = CreateEvent(NULL, FALSE, FALSE, NULL);
  rghEvt[1] = CreateEvent(NULL, FALSE, FALSE, NULL);
  rgdsbpn[0].dwOffset = 0;
  rgdsbpn[0].hEventNotify = rghEvt[0];
  rgdsbpn[1].dwOffset = dsb_size_bytes / 2;
  rgdsbpn[1].hEventNotify = rghEvt[1];
  if (FAILED(IDirectSoundBuffer_QueryInterface(lpDSB, 
              (REFIID) &IID_IDirectSoundNotify, (LPVOID *) &lpdsNotify))) {
    log_perror("can't create dsb notify");
    return -1;
  }
  if (FAILED(IDirectSoundNotify_SetNotificationPositions(
             lpdsNotify, DSBNUMEVENTS, rgdsbpn))) {
    IDirectSoundNotify_Release(lpdsNotify);
    log_perror("can't set notification positions");
    return -1;
  } 
  dsb_open = TRUE;
  return 1;
}

int
os_sound_write (void *buffer, int size)
{
  LPBYTE lpbSoundData;       // our data
  DWORD dwOffset;            // write position 
  DWORD dwSoundBytes;        // size of block to copy
  // locked area of dsb
  LPVOID  lpvPtr1;           
  DWORD   dwBytes1; 
  LPVOID  lpvPtr2; 
  DWORD   dwBytes2;
  DWORD dwEvt;       // position notification event id
  MSG msg;
  HRESULT hr;

  if (size != dsb_size_bytes / 2) {
    log_perror("buffer size not equal to frame size");
    return -1;
  }

  lpbSoundData = (LPBYTE) buffer;

  dwSoundBytes = size;

  // wait for position notification
  while (TRUE) {
    dwEvt = MsgWaitForMultipleObjects(
            DSBNUMEVENTS,   // How many possible events
            rghEvt,         // Location of handles
            FALSE,          // Wait for all?
            1000,           // How long to wait
            QS_ALLINPUT);   // Any message is an event

    if (dwEvt == WAIT_TIMEOUT) {
      puts("dsb timeout");
      continue;
    }

    // handle dsb event
    dwEvt -= WAIT_OBJECT_0;
    if (dwEvt < DSBNUMEVENTS)
      break;

    // handle other windows events
    if (dwEvt == DSBNUMEVENTS) {
      puts("sound process handling windows event");
      PeekMessage(&msg, NULL, 0, 0, PM_REMOVE);
      TranslateMessage(&msg);
      DispatchMessage(&msg);
    }
  }

  // write to dsb

  // Which half of DSB to lock?
  dwOffset = dwEvt ? 0 : (dsb_size_bytes / 2);

  // Lock area of DSB.
  hr = IDirectSoundBuffer_Lock(lpDSB, dwOffset, dwSoundBytes, &lpvPtr1, 
      &dwBytes1, &lpvPtr2, &dwBytes2, 0); 
 
  // If DSERR_BUFFERLOST is returned, restore and retry lock. 
  if (DSERR_BUFFERLOST == hr) { 
    IDirectSoundBuffer_Restore(lpDSB); 
    hr =  IDirectSoundBuffer_Lock(lpDSB, dwOffset, dwSoundBytes, 
        &lpvPtr1, &dwBytes1, &lpvPtr2, &dwBytes2, 0); 
  } 
  if (SUCCEEDED(hr)) { 
    // Write to pointers. 
    CopyMemory(lpvPtr1, lpbSoundData, dwBytes1);
    // Should never get 2nd pointer!
    if (NULL != lpvPtr2) {
        CopyMemory(lpvPtr2, lpbSoundData+dwBytes1, dwBytes2); 
    } 
    // Release the data back to DirectSound. 
    hr = IDirectSoundBuffer_Unlock(lpDSB, lpvPtr1, dwBytes1, lpvPtr2, dwBytes2); 
    if (!SUCCEEDED(hr))
      return -1;
  }
  else 
    return -1;
  return 1;
} 


int
os_sound_set_sample_rate (int sample_rate)
{
  if (!dsb_open)
    return -1;
  if (sample_rate != dsb_rate) {
    dsb_rate = sample_rate;
    os_sound_close_device();
    os_sound_open_device();
  }	
  return 1;
}

int
os_sound_set_nr_channels (int nr_channels)
{
  if (!dsb_open)
    return -1;
  if (nr_channels != dsb_channels) {
    dsb_channels = nr_channels;
    os_sound_close_device();
    os_sound_open_device();
  }	
  return 1;
}

int
os_sound_set_bits (int bits)
{
  if (!dsb_open)
    return -1;
  if (bits != dsb_bits) {
    dsb_bits = bits;
    os_sound_close_device();
    os_sound_open_device();
  }	
  return 1;
}

int os_sound_enable ()
{
  // start playing buffer
  if (!SUCCEEDED(IDirectSoundBuffer_Play(lpDSB,0,0,DSBPLAY_LOOPING))) {
        log_perror("can't play dsb");
        return -1;
  }
  puts("dsb play success");
  return 1;
}

int os_sound_disable ()
{
  // stop playing buffer
  if (!SUCCEEDED(IDirectSoundBuffer_Stop(lpDSB))) {
        log_perror("can't stop dsb");
        return -1;
  }
  return 1;
}


int os_sound_close_device ()
{
  os_sound_disable();
  if (!SUCCEEDED(IDirectSoundNotify_Release(lpdsNotify)))
    return -1;
  if (!SUCCEEDED(IDirectSound_Release(lpDS)))
    return -1;
  dsb_open = FALSE;
  return 1;
}



