/* XRacer (C) 1999 Richard W.M. Jones.
 * $Id: sound_process.c,v 1.11 1999/09/12 13:16:24 rich Exp $
 */

#include "config.h"

#ifdef __CYGWIN32__
#include <windows.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <assert.h>
#include <string.h>

#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <signal.h>
#include <fcntl.h>

#ifdef HAVE_SYS_MMAN_H
#include <sys/mman.h>
#endif

#ifdef HAVE_SYS_POLL_H
#include <sys/poll.h>
#endif

#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif

#include "xracer.h"
#include "globals.h"
#include "samples.h"
#include "sound.h"
#include "sound_process.h"
#include "os_sound.h"

/* Notation:
 * <-------------------- window ------------------------------>
 * <----- sector -----><----- sector -----><----- sector ----->
 *                |
 *              frame
 */

/* The length of a window. The length of a window defines
 * the maximum length of an echo. Since we don't normally
 * want echos longer than 1/10th second, and since sample
 * rates are 11025, 22050 or 44100, the length of a window
 * should be at least 4410.
 */
#define WINDOW_LENGTH 8192

/* The length of a sector. This is how much data is written
 * to the audio device each time. Larger values are more
 * efficient, processor wise. Smaller values mean that
 * there is less latency for playing samples. WINDOW_LENGTH
 * has to be a whole multiple of SECTOR_LENGTH.
 */
#define SECTOR_LENGTH (WINDOW_LENGTH/128)

#define MAX_ECHO_LENGTH(sample_rate) ((float) WINDOW_LENGTH / (sample_rate))

/* Current window. We generate both channels. In mono mode, we mix
 * the two channels equally when generating audio data.
 */
static float window[WINDOW_LENGTH][2];
static int window_posn;

static int sample_rate, nr_channels, bits, enabled, pipe_fd, audio_fd;
static struct sound_shm *shm;

static struct sound_request request_data, *request = &request_data;

/* Device buffer. The size of this buffer is
 * SECTOR_LENGTH * nr_channels * bits/8, and it must
 * be reallocated every time one of these factors
 * is changed.
 */
static void *device_buffer = 0;
static int device_buffer_size = -1;

/* List of samples currently playing. */
struct sample_playing {
  struct sample_playing *next, *prev;
  struct sample *sample;
  int position;
};

static struct sample_playing *sp_head = 0, *sp_last = 0;

/* The samples. Note that these have to match the samples defined in
 * the <samples.h> header file.
 */
struct sample {
  float *data;
  int len;
};

static struct sample sample[NR_SAMPLES];

static const char *sample_names[NR_SAMPLES] = {
  "sounds/powerup.wav",
  "sounds/faster.wav",
  "sounds/wind.wav"
};

#if 0
/* Generate wind noise into current window. */
#define WIND_MIN -1
#define WIND_MAX 1
#define WIND_FACTOR (2./200)

static void
wind_noise_generator (int start, int len,
		      float mixer_amplitude,
		      float craft_velocity)
{
  static float cp = 0;
  float v;
  int i;

  for (i = 0; i < len; ++i)
    {
      cp += drand48 () * WIND_FACTOR * craft_velocity
	- WIND_FACTOR * craft_velocity / 2;
#if 0
      while (cp > WIND_MAX) cp -= WIND_MAX - WIND_MIN;
      while (cp < WIND_MIN) cp += WIND_MAX - WIND_MIN;
#else
      if (cp > WIND_MAX) cp = WIND_MAX;
      else if (cp < WIND_MIN) cp = WIND_MIN;
#endif
      v = cp * mixer_amplitude;
      window[start+i][0] += v;
      window[start+i][1] += v;
    }
}
#endif

/* Generate wind noise into current window. */
#define WIND_FACTOR 0.001

static void
wind_noise_generator (int start, int len,
		      float mixer_amplitude,
		      float craft_velocity)
{
  static int sample_pos = 0;
  int i, c;
  float *sample_data = sample[SAMPLE_WIND].data + sample_pos,
        *sample_end = sample[SAMPLE_WIND].data + sample[SAMPLE_WIND].len;

  for (i = 0; i < len; ++i)
    for (c = 0; c < 2; ++c)
      {
	window[start+i][c] += *sample_data++ * WIND_FACTOR * craft_velocity;
	if (sample_data == sample_end)
	  sample_data = sample[SAMPLE_WIND].data;
      }

  sample_pos = sample_data - sample[SAMPLE_WIND].data;
}

/* Zero sector values. */
static void
zero_sector (int start, int len)
{
  memset (&window[start][0], 0, sizeof (float) * len * 2);
}

#define CLAMP_MIN -32768
#define CLAMP_MAX 32767

/* Convert current sector into final sound format. We
 * perform a number of distinct operations at the same
 * time here for efficiency:
 * (1) Multiply sound by output_amplitude.
 * (2) For 8-bit sounds, divide sound by 256.
 * (3) Clamp sound to range -32768..32767 for
 *     16-bit sounds or range -128..127 for 8-bit
 *     sounds.
 * (4) Convert sound to 8 or 16 bit samples and
 *     write into the output array.
 * OUTPUT is an array of bits/8 * len * nr_channels bytes.
 */
static void
sector_to_sound (int start, int len,
		 float output_amplitude,
		 void *output)
{
  int i, c, s[2];
  float v;

  for (i = 0; i < len; ++i)
    {
      for (c = 0; c < 2; ++c)
	{
	  v = window[start+i][c];

	  v *= output_amplitude;

	  if (v < CLAMP_MIN)
	    v = CLAMP_MIN;
	  else if (v > CLAMP_MAX)
	    v = CLAMP_MAX;

	  /* Convert to an integer. */
	  s[c] = v;
	}

      if (nr_channels == 2)
	{
	  /* Stereo. */
	  if (bits == 16)
	    {
	      /* 16 bit. */
	      signed short *o = (signed short *) output;
	      *o++ = s[0];
	      *o++ = s[1];
	      output = (void *) o;
	    }
	  else
	    {
	      /* 8 bit. */
	      signed char *o = (signed char *) output;
	      *o++ = s[0] >> 8;
	      *o++ = s[1] >> 8;
	      output = (void *) o;
	    }
	}
      else
	{
	  int sm;

	  /* In mono mode, mix down both the channels. */
	  sm = (s[0] + s[1]) >> 1;

	  if (bits == 16)
	    {
	      /* 16 bit. */
	      unsigned short *o = (unsigned short *) output;
	      *o++ = sm;
	      output = (void *) o;
	    }
	  else
	    {
	      /* 8 bit. */
	      unsigned char *o = (unsigned char *) output;
	      *o++ = sm >> 8;
	      output = (void *) o;
	    }
	}
    }
}

/* Merge echo into current window.
 * ECHO_DELAY must be a multiple of 1/sample_rate.
 */
static void
merge_echo (int start, int len,
	    float echo_delay, float echo_amplitude, float mixer_amplitude)
{
  float v;
  int p, i, c;

  /* Scale echo delay so that it is measured in frames. */
  echo_delay *= sample_rate;

  p = start - echo_delay;
  if (p < 0) p += WINDOW_LENGTH;

  for (i = 0; i < len; ++i)
    {
      for (c = 0; c < 2; ++c)
	{
	  /* Pick sample. */
	  v = window[p][c];
	  p++;
	  if (p == WINDOW_LENGTH) p = 0;

	  /* Add it to current place. */
	  window[start+i][c] += echo_amplitude * mixer_amplitude * v;
	}
    }
}

/* Merge one sample into current window. */
static void
merge_one_sample (int start, int len,
		  float *sample_data, int sample_start,
		  float mixer_amplitude)
{
  int i, c;

  /* XXX This is wrong. It should take account of sample rate, etc. */

  for (i = 0; i < len; ++i)
    for (c = 0; c < 2; ++c)
      window[start+i][c]
	+= mixer_amplitude * sample_data[(sample_start+i) * 2 + c];
}

/* Merge all currently outstanding samples. */
static void
merge_samples (int start, int len)
{
  struct sample_playing *p, *p_next;
  int sample_remaining;

  for (p = sp_head; p; p = p_next)
    {
      p_next = p->next;		/* In case we delete the sample. */
      sample_remaining = p->sample->len - p->position;

      /* Play this sample. */
      merge_one_sample (start,
			sample_remaining <= len ? sample_remaining : len,
			p->sample->data, p->position,
			shm->mixer.sample_amplitude);

      p->position += len;

      /* Delete this sample? */
      if (sample_remaining <= len)
	{
	  if (p->prev)
	    p->prev->next = p->next;
	  else
	    sp_head = p->next;

	  if (p->next)
	    p->next->prev = p->prev;
	  else
	    sp_last = p->prev;
	}
    }
}

/* Reallocate device_buffer. Call this whenever
 * nr_channels or bits settings change.
 */
static void
reallocate_device_buffer ()
{
  if (device_buffer != 0) free (device_buffer);
  device_buffer = 0;
  device_buffer_size = SECTOR_LENGTH * nr_channels * bits / 8;
  device_buffer = xmalloc (device_buffer_size);
}

/* Tweak the sound device, setting sample_rate,
 * nr_channels and bits settings. Call this
 * whenever one of those settings changes.
 */
static void
tweak_sound_device ()
{
  if ((audio_fd = os_sound_set_sample_rate (sample_rate)) < 0
      || (audio_fd = os_sound_set_nr_channels (nr_channels)) < 0
      || (audio_fd = os_sound_set_bits (bits)) < 0)
    {
      exit (1);
    }
}

/* Ready next sector and write it to sound buffer. */
static void
write_sector ()
{
  zero_sector (window_posn, SECTOR_LENGTH);
  wind_noise_generator (window_posn, SECTOR_LENGTH,
                        shm->mixer.wind_noise_amplitude,
			shm->misc.craft_velocity);
  merge_samples (window_posn, SECTOR_LENGTH);
  merge_echo (window_posn, SECTOR_LENGTH,
              shm->echo.delay, shm->echo.level,
	      shm->mixer.echo_amplitude);
  sector_to_sound (window_posn, SECTOR_LENGTH,
                   shm->mixer.output_amplitude, device_buffer);

#if 0
  printf ("write to sound dev: fd = %d, device_buf = %p (size: %d bytes)\n",
	  audio_fd, device_buffer, device_buffer_size);
#endif

  if ((audio_fd = os_sound_write (device_buffer, device_buffer_size)) < 0)
    exit (1);
}

/* Start playing a sample. */
static void
start_sample (int sample_num)
{
  struct sample_playing *p = xmalloc (sizeof (struct sample_playing));

  p->prev = 0;
  p->next = sp_head;
  p->sample = &sample[sample_num];
  p->position = 0;
  sp_head = p;

  if (sp_last == 0)
    sp_last = sp_head;
}

/* Process sound request. */
static void
dispatch_sound_request ()
{
  switch (request->type)
    {
    case SOUND_REQUEST_SET_SAMPLE_RATE:
      sample_rate = request->u.i;
      tweak_sound_device ();
      break;
    case SOUND_REQUEST_SET_BITS:
      bits = request->u.i;
      reallocate_device_buffer ();
      tweak_sound_device ();
      break;
    case SOUND_REQUEST_SET_NR_CHANNELS:
      nr_channels = request->u.i;
      reallocate_device_buffer ();
      tweak_sound_device ();
      break;
    case SOUND_REQUEST_ENABLE:
      os_sound_enable ();
      enabled = 1;
      break;
    case SOUND_REQUEST_DISABLE:
      os_sound_disable ();
      enabled = 0;
      break;
    case SOUND_REQUEST_PLAY_SAMPLE:
      start_sample (request->u.i);
      break;
    default:
      fprintf (stderr, "sound process: oops! unknown type code (%d) in stream. Exiting.\n", request->type);
      exit (1);
    }
}

/* Read sound request from pipe and dispatch it. */
static void
get_sound_request ()
{
  /* Read from pipe. Let's hope that most reasonable operating
   * systems can do small transactions across the pipe
   * atomically :-)
   */
  int r = read (pipe_fd, request, sizeof *request);

  if (r == 0)               /* End of file. Parent process exited ... */
    exit (0);               /* ... and we exit too. */
  else if (r < 0)
    {
      perror ("sound process: read");
      exit (1);
    }
  else if (r != sizeof *request)
    {
      fprintf (stderr,
	       "sound process: oops! read partial request. Exiting.\n");
      exit (1);
    }

  dispatch_sound_request ();
}

#if SOUND_USE_FORK

/* Select input from pipe or output to sound device. */

static void
select_event ()
{
#ifdef HAVE_POLL

  struct pollfd fds[2];
  int r, n = enabled ? 2 : 1;

  fds[0].fd = pipe_fd;
  fds[0].events = POLLIN;
  if (n > 1)
    {
      fds[1].fd = audio_fd;
      fds[1].events = POLLOUT;
    }
  r = poll (fds, n, -1);

  if (r < 0) { perror ("sound process: poll"); return; }
  assert (r > 0);

  if (fds[0].revents)
    get_sound_request ();

  if (n > 1 && fds[1].revents)
    write_sector ();

#elif HAVE_SELECT

#define MAX(a,b) ((a)>(b) ? (a) : (b))

  fd_set rfds, wfds;
  int r, n = enabled ? 2 : 1;

  FD_ZERO (&rfds);
  FD_SET (pipe_fd, &rfds);

  FD_ZERO (&wfds);
  if (n > 1) FD_SET (audio_fd, &wfds);

  r = select (MAX (pipe_fd, audio_fd) + 1, &rfds, &wfds, NULL, NULL);

  if (r < 0) { perror ("sound process: select"); return; }
  assert (r > 0);

  if (FD_ISSET (pipe_fd, &rfds))
    get_sound_request ();

  if (n > 1 && FD_ISSET (audio_fd, &wfds))
    write_sector ();

#else
  /* I once worked on an operating system where this was true (OS-9 2.4). */
#error "you don't seem to have either poll(2) or select(2)"
#endif
}

#else /* !SOUND_USE_FORK */

static void
select_event ()
{
  int r, i;

  r = shm->num_sound_requests;

  if (r > 0)
    {
      for (i = 0; i < r; ++i)
	{
	  request = &shm->request[i];
	  dispatch_sound_request ();
	}
      shm->num_sound_requests = 0;
    }

  if (enabled)
    {
      write_sector ();
    }

#ifdef __CYGWIN32__
  Sleep (50); /* assumes sector is 100 ms */
#endif
}

#endif /* !SOUND_USE_FORK */

void
sound_process (const struct sound_process_data *data)
{
  pipe_fd = data->pipe_fd;
  shm = data->shm;
  audio_fd = data->audio_fd;
  sample_rate = data->sample_rate;
  nr_channels = data->nr_channels;
  bits = data->bits;
  enabled = data->enabled;

#if 0
  printf ("sound_process: pipe_fd = %d, shm = %p, audio_fd = %d, sample_rate = %d, nr_channels = %d, bits = %d, enabled = %d, SOUND_USE_FORK = %d\n",
	  pipe_fd, shm, audio_fd, sample_rate, nr_channels, bits, enabled, SOUND_USE_FORK);
#endif

  reallocate_device_buffer ();
  tweak_sound_device ();

  for (;;)
    {
      select_event ();
    }
}

/* Convert the samples into raw format at 44100 stereo float and load them
 * into memory.
 */
void
load_samples ()
{
  char command[1024];
  float buffer[256];
  int i, j, n, s;
  FILE *fp;

  for (i = 0; i < NR_SAMPLES; ++i)
    {
      sprintf (command, "sox %s -r 44100 -c 2 -t raw -f -s - rate", sample_names[i]);
      fp = popen (command, "r");
      sample[i].data = 0;
      sample[i].len = 0;
      while ((n = fread (buffer, 2 * sizeof (float),
			 sizeof buffer / (sizeof (float) * 2), fp)) > 0)
	{
	  /* Scale sample to range -1 to 1. */
	  for (j = 0; j < sizeof buffer / sizeof (float); ++j)
	    buffer[j] /= 32768;

	  s = sample[i].len;
	  sample[i].len += n;
#if 0
	  printf ("s = %d, n = %d, sample[i].len = %d\n",
		  s, n, sample[i].len);
#endif
	  sample[i].data = xrealloc (sample[i].data,
				     sample[i].len * 2 * sizeof (float));
	  memcpy (sample[i].data + 2 * s,
		  buffer, 2 * sizeof (float) * n);
	}
      if (ferror (fp)) { perror ("read"); exit (1); }
      pclose (fp);

      printf ("loaded sample %s, data = %p, length = %d\n",
	      sample_names[i], sample[i].data, sample[i].len);
    }
}
