/* XRacer (C) 1999 Richard W.M. Jones.
 * $Id: texture.c,v 1.8 1999/09/12 13:16:24 rich Exp $
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>

#include "xracer.h"
#include "globals.h"
#include "jpeg.h"

static GLuint generic_load_texture (const char *, const char *, int *, int *);

GLuint
load_texture (const char *filename)
{
  return generic_load_texture (filename, 0, 0, 0);
}

GLuint
load_texture_with_alpha (const char *filename, const char *filename_alpha)
{
  return generic_load_texture (filename, filename_alpha, 0, 0);
}

GLuint
load_texture_with_size (const char *filename,
			int *width, int *height)
{
  return generic_load_texture (filename, 0, width, height);
}

GLuint
load_texture_with_alpha_size (const char *filename, const char *filename_alpha,
			      int *width, int *height)
{
  return generic_load_texture (filename, filename_alpha, width, height);
}

#if 0
/* Debugging function. */
static void
dump_texture (const GLubyte *image, int width, int height)
{
  int i;

  FILE *fp = fopen ("/tmp/texture-debug.ppm", "w");
  if (fp == 0) { log_perror ("/tmp/texture-debug.ppm"); exit (1); }

  fprintf (fp, "P3 %d %d 255\n", width, height);
  for (i = 0; i < width*height; ++i)
    {
      int r = *image++;
      int g = *image++;
      int b = *image++;
      fprintf (fp, "%d %d %d\n", r, g, b);
    }
  fclose (fp);
}
#endif

/* Find a value nearest to n which is a power of two */

static GLint round_pow_2(GLint n)
{
  GLint m;

  for(m = 1 ; m < n ; m *= 2);

  if((m - n) <= (n - (m / 2)))
    return m;
  else
    return (m / 2);
} 

/* Load texture from file.
 * Because of limitations (in Mesa? or the 3DFX card?) all textures
 * have to be 128x128 or smaller.
 */
static GLuint
generic_load_texture (const char *filename, const char *filename_alpha,
		      int *width_rtn, int *height_rtn)
{
  int width, height, components, error;
  GLuint texture;
  GLubyte *image = read_JPEG_file (filename, &width, &height, &components);

  if (image == NULL)
    log_fatal ("%s: fatal error loading texture file", filename);

  log_assert (components == 3);

  /* Is there an alpha component? If so, load it. */
  if (filename_alpha)
    {
      int alpha_width, alpha_height, alpha_components, i;
      GLubyte *p, *q, *r,
	*alpha_image = read_JPEG_file (filename_alpha,
				       &alpha_width, &alpha_height,
				       &alpha_components);
      GLubyte *combined_image;

      if (alpha_image == NULL)
	log_fatal ("%s: fatal error loading texture alphamap file", filename);

      log_assert (alpha_components == 1 || alpha_components == 3);
      log_assert (alpha_width == width);
      log_assert (alpha_height == height);

      /* Allocate space for the combined image. */
      combined_image = xmalloc (sizeof (GLubyte) * 4 * width * height);

      /* Create the combined image. */
      p = image;
      q = alpha_image;
      r = combined_image;
      for (i = 0; i < width * height; ++i)
	{
	  *r++ = *p++;		/* R */
	  *r++ = *p++;		/* G */
	  *r++ = *p++;		/* B */
	  *r++ = 255 - *q;	/* A - just use red component for now. */
	  q += alpha_components;
	}

      free (image);
      free (alpha_image);
      components = 4;
      image = combined_image;
    }

  glGenTextures (1, &texture);
  glBindTexture (GL_TEXTURE_2D, texture);

  glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
  glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);

  /* glTexEnvf (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL); */
  glTexEnvf (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);

  /* Build mipmaps. */
  glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
  if(mipmapping)
  {
    glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);

    if ((error = gluBuild2DMipmaps (GL_TEXTURE_2D,
				    components == 3 ? GL_RGB : GL_RGBA,
				    width, height,
				    components == 3 ? GL_RGB : GL_RGBA,
				    GL_UNSIGNED_BYTE, image)) != 0)
    {
      log_fatal ("gluBuild2DMipmaps failed (GL error: %s)",
		 gluErrorString (error));
    }
  }
  else
  {
    GLint pow_2_width = round_pow_2(width);
    GLint pow_2_height = round_pow_2(height);

    if((pow_2_width != width) || (pow_2_height != height))
    {
      GLubyte *newimage = xmalloc(sizeof(GLubyte) * components * pow_2_width * pow_2_height);
      if((error = gluScaleImage(components == 3 ? GL_RGB : GL_RGBA,
                                width, height,
                                GL_UNSIGNED_BYTE,
                                image,
                                pow_2_width, pow_2_height,
                                GL_UNSIGNED_BYTE,
                                newimage)) != 0)
      {
        log_fatal("gluScaleImage failed (GL error: %s)",
                  gluErrorString(error));
      }
      else
      {
        free(image);
        image = newimage;
        width = pow_2_width;
        height = pow_2_height;
      }
    }

    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
    
    glTexImage2D(GL_TEXTURE_2D,
                 0,
                 components == 3 ? GL_RGB : GL_RGBA,
                 width, height,
                 0,
                 components == 3 ? GL_RGB : GL_RGBA,
                 GL_UNSIGNED_BYTE,
                 image);
  }


  free (image);

  if (width_rtn) *width_rtn = width;
  if (height_rtn) *height_rtn = height;

  return texture;
}

void
texture_init ()
{
  /*glDisable (GL_DITHER);*/
  glHint (GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);
}
