/* XRacer (C) 1999 Richard W.M. Jones.
 * $Id: xracer.h,v 1.40 1999/09/12 13:16:24 rich Exp $
 */

#ifndef __xracer_h__
#define __xracer_h__

#include "config.h"

#ifdef __CYGWIN32__
#include <windows.h>
#endif

#include <stdlib.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include <GL/glut.h>
#include <math.h>

/* Each side of a segment. */
struct side
{
  GLfloat plane[4];		/* Plane parameters of side. */
  GLfloat levitate_factor;	/* Levitation strength factor [0..1]. */
  GLfloat levitate_dist;	/* Levitation distance. */
  GLfloat levitate_damping;	/* Levitation damping. */
};

/* A track is divided into a list of segments. */
struct segment
{
  GLuint dlist;			/* Display list to draw for this segment. */
  int lookahead;		/* How many segments can we see in front. */
  GLfloat forwards[3];		/* Forwards vector. */
  GLfloat autopilot_midpoint[3]; /* Midpoint of track at enter plane. */
  GLfloat enter[4];		/* Enter plane. */
  GLfloat triggers[3][4];	/* Left, middle and right trigger planes. */
  struct side *sides;		/* List of sides of the segment. */
  int nr_sides;
  int trigger;			/* Trigger something when entering segment. */
  int *visible;			/* List of scenery objects visible from here */
  int nr_visible;
  int sky_visible;		/* Sky visible flag. */
};

/* Scenery objects around the track. */
struct scenery
{
  int is_animated;		/* Is this animated scenery? */
  int is_start_lamp;		/* Is this the start lamp? */
  union {
    struct {
      GLuint dlist;		/* Display list. */
    } not_animated;
    struct {
      int nr_frames;		/* Number of frames in animation. */
      int *frame;		/* Scenery object in each frame. */
      double *ftime;		/* Time for each frame. */
      double total_ftime;	/* Total time for all frames. */
    } animated;
  } u;
};

#define RAIN 1
#define SNOW 2

/* A track. */
struct track
{
  struct segment *segments;	/* List of segments making up track. */
  int nr_segments;
  struct scenery *scenery;	/* List of scenery objects. */
  int nr_scenery;

  /* General track parameters. */
  int is_fog;
  GLfloat fog_colour[4];
  GLfloat fog_density;
  int fog_mode;

  int is_precipitation;		/* Either 0, or RAIN or SNOW. */

  GLfloat background_fill_colour[4];

  int backdrop_tex;
  GLfloat backdrop_x_scale;
};

/* Possible triggers. */
#define TRIGGER_START           1 /* Start of lap. */
#define TRIGGER_LEFT_FAST       4 /* Go fasters and powerups. */
#define TRIGGER_LEFT_FASTER     8
#define TRIGGER_LEFT_POWERUP   16
#define TRIGGER_RIGHT_FAST     32
#define TRIGGER_RIGHT_FASTER   64
#define TRIGGER_RIGHT_POWERUP 128
/* NB. I will add other triggers here, for example crowd noises. */

/* Possible powerups. As in Wipeout, a player may only hold
 * one powerup at a time.
 */
#define NO_POWERUP              0
#define POWERUP_FASTER          1
#define POWERUP_ENERGY          2
#define POWERUP_SHIELD          3
#define POWERUP_AUTOPILOT       4
#define POWERUP_CANNON          5 /* Not implemented. */
#define POWERUP_MISSILE         6 /* Not implemented. */
#define MAX_POWERUP             4 /* Max. implemented powerup. */

/* Macro to convert radians to degrees. Why does OpenGL use degrees??? */
#define DEGREES(r) (180.*(r)/M_PI)

/* XXX This should not be a global variable. */
extern struct track *track;

/* Name of track to load next. */
extern char *track_to_load;

/* Name of currently loaded track. */
extern char *current_track_name;

#define MAX_PLAYERS 8

enum pilot_type { PILOT_NONE, PILOT_LOCAL, PILOT_ROBOT, PILOT_REMOTE };

/* Pilot positions. */
extern struct pilot {
  int pilot_type;
  char *pilot_name;
  /* Craft is modelled by three points with springs between them.
   * Point 0 = front, Point 1 = left back, Point 2 = right back.
   */
  GLfloat posn[3][3];
  /* Momentum. */
  GLfloat momentum[3][3];
  /* Forward thrust magnitude. */
  GLfloat thrust;
  /* This is a multiple > 1 if we go over a faster spot. */
  GLfloat faster;
  /* Roll magnitude (< 0 left). */
  GLfloat roll;
  /* Pitch magnitude (< 0 up). */
  GLfloat pitch;
  /* Yaw magnitude (< 0 left). */
  GLfloat yaw;
  /* Segment in which each point is placed. */
  int seg[3];
  /* Strength of shield (range 0-1). */
  GLfloat shield;
  /* External shield (powerup). */
  int has_external_shield;
  double external_shield_start_time;
  /* Autopilot (powerup). */
  int has_autopilot;
  double autopilot_start_time;
  /* Currently held powerup. */
  int powerup;
  /* Actual control position *** Do not set this *** */
  int control_x;
  /* Actual speed *** Do not set this *** */
  int speed;
  /* These are used to determine current position in the race. */
  int greatest_segment, current_lap;
} pilot[MAX_PLAYERS];

/* Parameters of spaceships. */
extern struct craft {
  GLfloat accel, brake;
  GLfloat max_thrust, faster_mult;
  GLfloat roll_div, yaw_div, pitch_div;
} craft[MAX_PLAYERS];

/* difficulty level */
extern int difficulty;
enum difficulty_level { EASY , MEDIUM , HARD }; 
/* Mode. */
extern int mode;

/* Who is the local player. */
extern int local_player_nr;

#define ARCADE_MODE 1
#define TIME_TRIAL_MODE 2
#define NETGAME_MODE 3

/* This is true if the player's own craft can be seen. */
extern int player_in_view;

/* Display wireframe only */
extern int wireframe;

/* Display segment number ?*/
extern int segments;
/* Controls. */
struct control
{
  /* General controls. */
  int screenshot;

  /* Display FPS readout */
  int show_fps;

  /* Accelerate and brake keys. */
  int accelerate, brake;

  /* Keyboard control. */
  int keyboard_flag;
  int keyboard_x, keyboard_y;
  int mouse_pitch;

  /* Mouse control. */
  int mouse_flag;
  int mouse_x, mouse_y;
  int mouse_accelerate, mouse_decelerate;
  float mouse_sensitivity;

  /* Joystick control. */
  int joystick_flag;
  int joystick_x, joystick_y;
  int joystick_button1, joystick_button2;
  int joystick_button3, joystick_button4;

  /* Autopilot control. */
  int autopilot_flag;
  int autopilot_x, autopilot_y;
};

extern struct control control[MAX_PLAYERS];

/* Origin. */
extern GLfloat origin[4];

/* ZX Spectrum colours! */
extern GLfloat colour[8][4];
extern GLfloat *white, *red, *green, *yellow;

/* Width and height of viewport. */
extern int width, height;

/* This global stores the current time in seconds. It is
 * recomputed once at the start of each frame.
 */
extern double current_time;
extern double start_time;
extern double frame_time;

/* The current frame. */
extern int frame;

#define GAME_WAITING  0
#define GAME_COUNTING 1
#define GAME_RUNNING  2
#define GAME_ENDED    3

/* This is the state of the game. */
struct game {
  int state;
  double countdown_time;	/* Time since start of countdown. */
  double start_time;		/* Time at start of this *lap*. */
  double lap_time[10];		/* Lap times. */
  char lap_time_s[10][10];	/* Lap times as formatted strings. */
  int nr_laps;			/* Total number of laps in race. */
};

extern struct game game;

/* Logging levels. */
#define LOG_DEBUG 0
#define LOG_INFO 1
#define LOG_WARNING 2
#define LOG_ERROR 3

/* Matrix function prototypes are in a separate file,
 * because we use them in other, unrelated programs.
 */
#include "matrix.h"

extern void    load_track (const char *filename);
extern void   *xmalloc (size_t n);
extern void   *xrealloc (void *p, size_t n);
extern char   *xstrdup(char *);
extern void    pilot_init (void);
extern void    pilot_update (void);
extern void    log_init (void);
extern void    log_display (void);
extern void    log_assertfail (const char *file, int line, const char *expr);
extern void    log_fatal_with_line (const char *file, int line,
				    const char *fs, ...);
extern void    log_with_line (const char *file, int line, int level,
			      const char *fs, ...);
extern void    log_perror_with_line (const char *file, int line,
				     const char *fs, ...);
extern void    log_not_impl_with_line (const char *file, int line);
extern void    take_screenshot (void);
extern void    reshape (int w, int h);
extern void    display (void);
extern void    mouse (int button, int state, int x, int y);
extern void    mouse_motion (int x, int y);
extern void    keyboard (unsigned char key, int x, int y);
extern void    special (int key, int x, int y);
extern void    keyboard_up (unsigned char key, int x, int y);
extern void    special_up (int key, int x, int y);
extern void    gidle (void);
extern void    joystick_init (void);
extern void    control_init (void);
extern void    panel_display (void);
extern void    panel_init (void);
extern void    lcd_init (void);
extern void    draw_lcd_digit (int digit, int point,
			       GLfloat x, GLfloat y, GLfloat scale);
extern void    draw_lcd_string (const char *buffer,
				GLfloat x, GLfloat y, GLfloat scale);
extern GLuint  load_texture (const char *filename);
extern GLuint  load_texture_with_alpha (const char *filename,
					const char *filename_alpha);
extern GLuint  load_texture_with_size (const char *filename,
				       int *width, int *height);
extern GLuint  load_texture_with_alpha_size (const char *filename,
					     const char *filename_alpha,
					     int *width, int *height);
extern void    texture_init (void);
extern void    recalc_time (void);
extern void    game_init (void);
extern void    game_begin(void);
extern void    game_trigger_start (void);
extern void    game_trigger_fast (int);
extern void    game_trigger_faster (int);
extern void    game_trigger_powerup (int);
extern void    game_new_lap (int cr);
extern void    game_end_game (int cr);
extern void    game_dunk (int cr);
extern void    game_crash (int cr);
extern void    game_fire (int cr);
extern const char *game_get_time_s (void);
extern int     game_held (void);
extern int     game_waiting(void);
extern int     game_lamp_stage (void);
extern void    display_game_over (void);
extern void    calculate_player_rank (int cr, int *rank, int *nr_players);
extern void    sky_init (void);
extern void    display_sky (void);
extern void    display_background (void);
extern void    lamp_init (void);
extern void    display_lamp (void);
extern void    craft_display (void);
extern void    craft_init (void);
extern void    autopilot (int);
extern void    do_initializations (void);
extern void    display_precipitation (void);
extern void    precipitation_init (void);
extern void    backdrop_init (void);
extern void    display_backdrop (void);
extern void    text_init (void);
extern void   *find_font (const char *name, int point_size);
extern void    draw_text (void *font, const char *text, int x, int y);
extern int     get_text_width (void *font, const char *text);
extern int     get_text_height (void *font);
extern void    menu_start (void);
extern void    options_menu_start (void);
extern void    display_menu_image (void);
extern int     do_menu_fade_effect (int *state, double start);
extern void    menu_text_reset (void);
extern void    menu_text_print (const char *str);
extern void    menu_text_tab (void);
extern void    menu_text_nl (void);
extern void    load_track_start (void);

/* This returns true if the game has finished. */
extern inline int
game_ended ()
{
  return (game.state == GAME_ENDED);
}

#if !defined(__STRING)
#if defined(__STDC__) && __STDC__
#define __STRING(x) #x
#else
#define __STRING(x) "x"
#endif
#endif

#define log_assert(expr) ((void) ((expr) ? 0 : (log_assertfail (__FILE__, __LINE__, __STRING (expr))), 0))
#define log_fatal(fs...) log_fatal_with_line (__FILE__, __LINE__, fs)
#define log(lvl, fs...) log_with_line (__FILE__, __LINE__, lvl, fs)
#define log_perror(fs...) log_perror_with_line (__FILE__, __LINE__, fs)
#define log_not_impl() log_not_impl_with_line (__FILE__, __LINE__)

#ifndef HAVE_VSNPRINTF
#define vsnprintf(m,s,f,a) vsprintf(m,f,a)
#endif

#ifndef __CYGWIN32__
/* Unix: Use mmap to create shared memory, fork off a
 * subprocess and use a pipe to communicate requests.
 */
#define SOUND_USE_FORK 1
#else
/* Windows: Use a threaded implementation, shared memory
 * is not required (because of threads) and use a queue
 * of request structures in memory instead of the pipe.
 */
#define SOUND_USE_FORK 0
#endif

#endif /* __xracer_h__ */
