/* XRacer (C) 1999 Richard W.M. Jones.
 * $Id: mktrackmodel.c,v 1.9 1999/07/31 16:53:26 rich Exp $
 */

/* This tool takes a track shape file (.tsf) and exports
 * it into a format suitable for use by a 3D modeller.
 * Most of this code is derived from old track-general.c.
 */

#ifdef __CYGWIN32__
#include <windows.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>

#include "config.h"

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <GL/gl.h>

#include "matrix.h"

static void trim (char *);
static GLfloat get_number (void);
static GLfloat get_angle (void);
static void file_format_error (int lineno);
static void make_track (void);

static void
usage ()
{
  fprintf (stderr,
	   "Usage:\n"
	   "  mktrackmodel [-options] inputfile\n"
	   "Options:\n"
	   "  -o outputfile  Specify name of output file\n"
	   "  -p previewfile Specify name of preview file\n"
	   "  -f format      Specify output file format\n"
	   "  -v             Verbose\n"
	   "Notes:\n"
	   "  Supported formats are: ac3d\n"
	   "  If not specified and the input filename is track.tsf, then the output\n"
	   "  file is track.ac and the preview file is track.preview.ac\n");
  exit (1);
}

/* These are the supported export formats. */
#define FORMAT_AC3D 1

static int verbose = 0;
static char *inputfile;
static char *outputfile = 0, *previewfile = 0;
static int outputformat = FORMAT_AC3D;

/* This is the structure of each part of the track definition. */
struct track_defn
{
  /* Type of section: straight or curved. */
  enum { track_defn_end, track_defn_bezier, track_defn_straight, track_defn_curve } type;
  union
  {
    /* This is used for straight bits. */
    struct {
      /* How long is this bit. */
      GLfloat length;
    } s;
    /* This is used for curved bits. */
    struct {
      /* How many radians to go round. If +ve, we go right. If -ve we go
       * left.
       */
      GLfloat angle;
      /* Radius of curve. */
      GLfloat radius;
    } c;
    struct {
      GLfloat v1;     // length of first control
      GLfloat v2[3];
      GLfloat v3[3];  // end point
    } sp;
  } u1;
  /* Now for sections which go up or down. */
  enum { track_defn_flat, track_defn_linear, track_defn_sine } contour_type;
  union
  {
    /* This is used for track sections which go up or down linearly. */
    struct {
      /* How much to go up (+ve) or down (-ve) by (in total). */
      GLfloat dist;
    } l;
    /* This is used for track sections which go up or down as a sine wave. */
    struct {
      /* Beginning and end of sine wave section (radians). */
      GLfloat angle[2];
      /* How much to go up (+ve) or down (-ve) by (in total).
       * We scale the angles so that the track always goes
       * up or down by +/- dist.
       */
      GLfloat dist;
      /* These internal fields are filled in by the make_track function below.
       */
      GLfloat scale, offset;
    } s;
  } u2;
};

static struct track_defn *track_defn = 0;

int
main (int argc, char *argv[])
{
  int c, lineno = 0;
  const char *ext = 0;
  char line [1024];
  FILE *fp;
  struct track_defn *section;
  int nr_sections = 0;

  while ((c = getopt (argc, argv, "f:o:p:v")) != -1)
    {
      switch (c)
	{
	case 'f':
	  if (strcasecmp (optarg, "ac3d") == 0)
	    outputformat = FORMAT_AC3D;
	  else
	    usage ();
	  break;
	case 'o':
	  outputfile = optarg;
	  break;
	case 'p':
	  previewfile = optarg;
	  break;
	case 'v':
	  verbose = 1;
	  break;
	default:
	  usage ();
	}
    }

  if (optind != argc - 1) usage ();

  inputfile = argv [optind++];

  switch (outputformat)
    {
    case FORMAT_AC3D: ext = ".ac"; break;
    }

  if (outputfile == 0)
    {
      char *t;

      outputfile = malloc (strlen (inputfile) + 20);
      strcpy (outputfile, inputfile);

      /* Truncate inputfile at last '.' and append appropriate
       * designator.
       */
      t = strrchr (outputfile, '.');
      if (t != 0)
	strcpy (t, ext);
      else
	strcat (outputfile, ext);
    }

  if (previewfile == 0)
    {
      char *t;

      previewfile = malloc (strlen (inputfile) + 50);
      strcpy (previewfile, inputfile);

      /* Truncate inputfile at last '.' and append appropriate
       * designator.
       */
      t = strrchr (previewfile, '.');
      if (t != 0)
	{
	  strcpy (t, ".preview");
	  strcat (t, ext);
	}
      else
	{
	  strcat (previewfile, ".preview");
	  strcat (previewfile, ext);
	}
    }

  if (verbose) printf ("mktrackmodel (C) 1999 Richard W.M. Jones\n");

  /* Parse the input file. */
  fp = fopen (inputfile, "r");
  if (fp == NULL) { perror (inputfile); exit (1); }
  while (fgets (line, sizeof line, fp))
    {
      char *t;

      lineno ++;

      /* Trim whitespace from the line. */
      trim (line);

      /* Ignore blank lines and comments. */
      if (line[0] == '#' || line[0] == '\0') continue;

      /* Allocate a track section. */
      nr_sections ++;
      track_defn = realloc (track_defn,
			    nr_sections * sizeof (struct track_defn));
      section = &track_defn[nr_sections-1];
      section->type = section->contour_type = -1;

      /* Break up the line into tokens. */
      t = strtok (line, " \t");
      while (t)
	{
	  if (strcasecmp (t, "straight") == 0)
	    {
	      if (section->type != -1) file_format_error (lineno);
	      section->type = track_defn_straight;
	      section->u1.s.length = get_number ();
	    }
	  else if (strcasecmp (t, "leftcurve") == 0)
	    {
	      if (section->type != -1) file_format_error (lineno);
	      section->type = track_defn_curve;
	      section->u1.c.angle = - get_angle ();
	      section->u1.c.radius = get_number ();
	    }
	  else if (strcasecmp (t, "rightcurve") == 0)
	    {
	      if (section->type != -1) file_format_error (lineno);
	      section->type = track_defn_curve;
	      section->u1.c.angle = get_angle ();
	      section->u1.c.radius = get_number ();
	    }
	  else if (strcasecmp (t, "bezier") == 0)
	    {
	      if (section->type != -1) file_format_error (lineno);
	      section->type = track_defn_bezier;
	      section->u1.sp.v1 = get_number ();
	      section->u1.sp.v2[0] = get_number ();
	      section->u1.sp.v2[1] = get_number ();
	      section->u1.sp.v2[2] = get_number ();
	      section->u1.sp.v3[0] = get_number ();
	      section->u1.sp.v3[1] = get_number ();
	      section->u1.sp.v3[2] = get_number ();
	    }
	  else if (strcasecmp (t, "flat") == 0)
	    {
	      if (section->contour_type != -1) file_format_error (lineno);
	      section->contour_type = track_defn_flat;
	    }
	  else if (strcasecmp (t, "linear") == 0)
	    {
	      if (section->contour_type != -1) file_format_error (lineno);
	      section->contour_type = track_defn_linear;
	      section->u2.l.dist = get_number ();
	    }
	  else if (strcasecmp (t, "sine") == 0)
	    {
	      if (section->contour_type != -1) file_format_error (lineno);
	      section->contour_type = track_defn_sine;
	      section->u2.s.angle[0] = get_angle ();
	      section->u2.s.angle[1] = get_angle ();
	      section->u2.s.dist = get_number ();
	    }
	  else
	    file_format_error (lineno);

	  t = strtok (NULL, " \t");
	}
    }

  fclose (fp);

  /* Allocate the last track section. */
  nr_sections ++;
  track_defn = realloc (track_defn, nr_sections * sizeof (struct track_defn));
  section = &track_defn[nr_sections-1];
  section->type = track_defn_end;

  /* Make the track. */
  make_track ();

  exit (0);
}

/* Trim whitespace from the beginning and end of the string. */
static void
trim (char *str)
{
  int len = strlen (str);

  while (len > 0 && 0 < *str && *str <= ' ')
    {
      memmove (str, str+1, len);
      len --;
    }

  while (len > 0 && 0 < str[len-1] && str[len-1] <= ' ')
    {
      str[len-1] = '\0';
      len --;
    }
}

static GLfloat
get_number ()
{
  const char *t = strtok (NULL, " \t");
  GLfloat f;

  if (sscanf (t, "%f", &f) == 1) return f;
  else
    {
      fprintf (stderr, "mktrackmodel: `%s' is not a number\n", t);
      exit (1);
    }
}

static GLfloat
get_angle ()
{
  const char *t = strtok (NULL, " \t");
  GLfloat f;

  if (strcasecmp (t, "-2.pi") == 0) return -2*M_PI;
  else if (strcasecmp (t, "-7.pi/4") == 0) return -7*M_PI/4;
  else if (strcasecmp (t, "-3.pi/2") == 0) return -3*M_PI/2;
  else if (strcasecmp (t, "-5.pi/4") == 0) return -5*M_PI/4;
  else if (strcasecmp (t, "-pi") == 0) return -M_PI;
  else if (strcasecmp (t, "-3.pi/4") == 0) return -3*M_PI/4;
  else if (strcasecmp (t, "-pi/2") == 0) return -M_PI/2;
  else if (strcasecmp (t, "-pi/4") == 0) return -M_PI/4;
  else if (strcasecmp (t, "pi/4") == 0) return M_PI/4;
  else if (strcasecmp (t, "pi/2") == 0) return M_PI/2;
  else if (strcasecmp (t, "3.pi/4") == 0) return 3*M_PI/4;
  else if (strcasecmp (t, "pi") == 0) return M_PI;
  else if (strcasecmp (t, "5.pi/4") == 0) return 5*M_PI/4;
  else if (strcasecmp (t, "3.pi/2") == 0) return 3*M_PI/2;
  else if (strcasecmp (t, "7.pi/4") == 0) return 7*M_PI/4;
  else if (strcasecmp (t, "2.pi") == 0) return 2*M_PI;
  else if (sscanf (t, "%f", &f) == 1) return f;
  else
    {
      fprintf (stderr, "mktrackmodel: `%s' is not an angle\n", t);
      exit (1);
    }
}

static void
file_format_error (int lineno)
{
  fprintf (stderr, "mktrackmodel: cannot parse symbol at line %d\n", lineno);
  exit (1);
}

#define VIRTBARRIER_HEIGHT   1

#define TRACK_WIDTH          20
#define TRACK_DEPTH          1

/* NB. The following have to be equal for the moment until
 * I fix the texture map coords to work properly.
 */
#define SEGMENT_LEN_STRAIGHT 8
#define SEGMENT_LEN_CURVES   8

static void make_segment (GLfloat p0x, GLfloat p0y, GLfloat p0z,
			  GLfloat p1x, GLfloat p1y, GLfloat p1z,
			  GLfloat p2x, GLfloat p2y, GLfloat p2z,
			  GLfloat p3x, GLfloat p3y, GLfloat p3z);
static void join_segments (void);

/* This is an attempt to make a ``general purpose'' track
 * generator, for simple tracks at least. These tracks
 * consist of: straight bits, and curved bits. The straight
 * bits are (you guessed it) straight. And the curved bits
 * and arcs from a circle.
 *
 * Straight sections and curved sections can also go up
 * and down, either linearly up and down or smoothly up
 * and down (the height being defined as part of a sine
 * wave).
 *
 * You define a track by making a list of sections. The
 * code in this file then builds the appropriate track
 * structure for you.
 *
 * You don't need to worry about how sections connect to
 * each other. The code here connects sections to each
 * for you.
 */

/* Compute 2D point p on circle at centre, with
 * given radius. The angle is +ve for a right
 * turn and -ve for a left turn.
 */
static void
point_on_circle (const GLfloat *v, const GLfloat *w,
		 const GLfloat *centre, GLfloat radius,
		 int is_left,
		 GLfloat angle, GLfloat *p)
{
  GLfloat v_mag, w_mag, vn[2], wn[2];

#if 0
  printf ("point_on_circle: left = %d, centre = (%g,%g) radius = %g\n"
	  "  v = (%g,%g)  w = (%g,%g)  angle = %g\n",
	  is_left, centre[0], centre[1], radius,
	  v[0], v[1], w[0], w[1], angle);
#endif

  if (is_left)
    angle = - angle;

  /* Calculate v and w magnitudes required. */
  v_mag = radius * sin (angle);
  w_mag = radius * cos (angle);

  /* Compute vectors vn and wn with required magnitudes. */
  memcpy (vn, v, 2 * sizeof (GLfloat));
  memcpy (wn, w, 2 * sizeof (GLfloat));
  normalize2d (vn, vn);
  normalize2d (wn, wn);
  vn[0] *= v_mag;
  vn[1] *= v_mag;
  wn[0] *= w_mag;
  wn[1] *= w_mag;

#if 0
  printf ("  vn = (%g,%g)  wn = (%g,%g)\n", vn[0], vn[1], wn[0], wn[1]);
#endif

  /* Compute point == centre + vn + wn. */
  if (is_left)
    {
      p[0] = centre[0] + vn[0] - wn[0];
      p[1] = centre[1] + vn[1] - wn[1];
    }
  else
    {
      p[0] = centre[0] + vn[0] + wn[0];
      p[1] = centre[1] + vn[1] + wn[1];
    }

#if 0
  printf ("  result p = (%g, %g)\n", p[0], p[1]);
#endif
}

static inline GLfloat
height_fn (int i, int j, int n, GLfloat h)
{
  GLfloat r = h, a0, a1;

  switch (track_defn[i].contour_type)
    {
    case track_defn_flat:
      break;

    case track_defn_linear:
      r += track_defn[i].u2.l.dist * j / n;
      break;

    case track_defn_sine:
      a0 = track_defn[i].u2.s.angle[0];
      a1 = track_defn[i].u2.s.angle[1];

      r += track_defn[i].u2.s.scale * (sin (a0 + (a1 - a0) * j / n)
				       + track_defn[i].u2.s.offset);

#if 0
      printf ("scale = %g, offset = %g\n",
	      track_defn[i].u2.s.scale,
	      track_defn[i].u2.s.offset);
#endif
    }

#if 0
  printf ("height_fn (i = %d, j = %d, n = %d, h = %g) = %g\n",
	  i, j, n, h, r);
#endif

  return r;
}

void
make_track ()
{
  int i;
  GLfloat p[2], v[3], h;

  /* Current position and direction. */
  p[0] = p[1] = 0;
  h = 0;              // Track level
  v[0] = 0;
  v[1] = 1;
  v[2] = 0;

  /* Precompute various values for sine contours. See height_fn. */
  for (i = 0; track_defn[i].type != track_defn_end; ++i)
    {
      if (track_defn[i].contour_type == track_defn_sine)
	{
	  GLfloat a0, a1, dist, v0, v1;

	  a0 = track_defn[i].u2.s.angle[0];
	  a1 = track_defn[i].u2.s.angle[1];
	  dist = track_defn[i].u2.s.dist;
	  v0 = sin (a0);
	  v1 = sin (a1);

	  if (v0 != v1)
	    {
	      track_defn[i].u2.s.offset = -v0;
	      track_defn[i].u2.s.scale = dist / (v1 - v0);
	    }
	  else
	    {
	      track_defn[i].u2.s.offset = 0;
	      track_defn[i].u2.s.scale = 0;
	      fprintf (stderr, "mktrackmodel: you wanted a curved contour, but you got a flat contour\n");
	    }
	}
    }

  /* Read the track definition. */
  for (i = 0; track_defn[i].type != track_defn_end; ++i)
    {
	switch ( track_defn[i].type )
	{
	case track_defn_bezier:
	{
          
	  GLfloat   w[2][3], t, cu, u, du[3], xx[2][3];
          GLdouble cp[4][3];

	  w[0][0] = -v[1];
	  w[0][1] = v[0];
          w[0][2] = 0.0;
	  /* Set magnitude of w to half track width. */
	  t = magnitude2d (&w[0][0]);
	  w[0][0] *= TRACK_WIDTH / (2*t);
	  w[0][1] *= TRACK_WIDTH / (2*t);

        
          t = sqrt(v[0]*v[0] + v[1]*v[1] + v[2]*v[2]);
          
          /* x(u)  =   u^3*x1 + 3*u^2*(1-u)*x2 + 3*u*(1-u)^2*x3 + (1-u)^3*x4 */
          /* x'(u) = 3*u^2*x1 + 3*x2*(2*u-3*u*u)   + 3*x3*(1-4*u+3*u*u)     - 3*(1-u)^2*x4 */
          cp[0][0] = xx[0][0] = p[0];
          cp[0][1] = xx[0][1] = p[1];
          cp[0][2] = xx[0][2] =    h;
          cp[1][0] = track_defn[i].u1.sp.v1 * ( v[0] / t ) + p[0];
          cp[1][1] = track_defn[i].u1.sp.v1 * ( v[1] / t ) + p[1];
          cp[1][2] = track_defn[i].u1.sp.v1 * ( v[2] / t ) + h;
          cp[2][0] = track_defn[i].u1.sp.v2[0] + p[0];
          cp[2][1] = track_defn[i].u1.sp.v2[1] + p[1];
          cp[2][2] = track_defn[i].u1.sp.v2[2];
          cp[3][0] = track_defn[i].u1.sp.v3[0] + p[0];
          cp[3][1] = track_defn[i].u1.sp.v3[1] + p[1];
          cp[3][2] = track_defn[i].u1.sp.v3[2];
          printf("%f, %f, %f \n",v[0], v[1], v[2] );
   
          u = 0.0; cu = 1.0;
          while ( u < 1.0 )
          {
             du[0] = 3.0*(cu*cu*cp[0][0] + (      2.0*cu - 3.0*cu*cu)*cp[1][0] +
                                          (1.0 - 4.0*cu + 3.0*cu*cu)*cp[2][0] -
                            u*u*cp[3][0]);
             du[1] = 3.0*(cu*cu*cp[0][1] + (      2.0*cu - 3.0*cu*cu)*cp[1][1] +
                                          (1.0 - 4.0*cu + 3.0*cu*cu)*cp[2][1] -
                          u*u*cp[3][1]);
             du[3] = 3.0*(cu*cu*cp[0][2] + (      2.0*cu - 3.0*cu*cu)*cp[1][2] +
                                           (1.0 - 4.0*cu + 3.0*cu*cu)*cp[2][2] -
                          u*u*cp[3][2]);
             
             t = magnitude(du);
             u +=  SEGMENT_LEN_CURVES / t ;
             if ( u > 1.0 )
                u = 1.0;
             cu = 1.0-u;
             /* let the compiler do optimisations */
             xx[1][0] =   cu*cu*cu*cp[0][0] + 3.0*cu*cu*u*cp[1][0] +
                        3.0*cu*u*u*cp[2][0] +       u*u*u*cp[3][0];
             xx[1][1] =   cu*cu*cu*cp[0][1] + 3.0*cu*cu*u*cp[1][1] +
                        3.0*cu*u*u*cp[2][1] +       u*u*u*cp[3][1];
             xx[1][2] =   cu*cu*cu*cp[0][2] + 3.0*cu*cu*u*cp[1][2] +
                        3.0*cu*u*u*cp[2][2] +       u*u*u*cp[3][2];

             w[1][0] =  du[1];
             w[1][1] = -du[0];
             t = magnitude2d(&w[1][0]);
       	     w[1][0] *= TRACK_WIDTH / (2*t) ;
	     w[1][1] *= TRACK_WIDTH / (2*t) ;
             
             make_segment (xx[0][0] + w[0][0], xx[0][1] + w[0][1], xx[0][2],
	                   xx[0][0] - w[0][0], xx[0][1] - w[0][1], xx[0][2],
			   xx[1][0] + w[1][0], xx[1][1] + w[1][1], xx[1][2],
			   xx[1][0] - w[1][0], xx[1][1] - w[1][1], xx[1][2]);
             xx[0][0] = xx[1][0];
             xx[0][1] = xx[1][1];
             xx[0][2] = xx[1][2];
             w[0][0]  = w[1][0];
             w[0][1]  = w[1][1];
             w[0][2]  = w[1][2];
//             printf("%f : %f, %f \n",u, xx[0][0], xx[0][1] );
          }
          v[0] = track_defn[i].u1.sp.v3[0] - track_defn[i].u1.sp.v2[0];
          v[1] = track_defn[i].u1.sp.v3[1] - track_defn[i].u1.sp.v2[1];
          v[2] = track_defn[i].u1.sp.v3[2] - track_defn[i].u1.sp.v2[2];
          p[0] = xx[0][0];
          p[1] = xx[0][1];
          h    = xx[0][2];
          printf("%f, %f, %f \n",v[0], v[1], v[2] );
	  break;
	}

      	case track_defn_straight:
	{
	  /* Straight sections. */
	  GLfloat len = track_defn[i].u1.s.length;
	  GLfloat w[2], t;
	  int n, j;

	  /* Work out how many segments to use. */
	  n = len / SEGMENT_LEN_STRAIGHT;

	  /* Compute a vector w at right angles to v. */
	  w[0] = -v[1];
	  w[1] = v[0];

	  /* Set magnitude of v to segment length. */
	  t = magnitude2d (v);
	  v[0] *= SEGMENT_LEN_STRAIGHT / t;
	  v[1] *= SEGMENT_LEN_STRAIGHT / t;

	  /* Set magnitude of w to half track width. */
	  t = magnitude2d (w);
	  w[0] *= TRACK_WIDTH / (2*t);
	  w[1] *= TRACK_WIDTH / (2*t);

	  /* Make each segment. */
	  for (j = 0; j < n; ++j)
	    {
	      make_segment (p[0] + w[0], p[1] + w[1], height_fn (i,j,n,h),
			    p[0] - w[0], p[1] - w[1], height_fn (i,j,n,h),
			    p[0] + v[0] + w[0], p[1] + v[1] + w[1],
			                              height_fn (i,j+1,n,h),
			    p[0] + v[0] - w[0], p[1] + v[1] - w[1],
			                              height_fn (i,j+1,n,h));
	      p[0] += v[0];
	      p[1] += v[1];
	    }
            
	  /* Update h. */
	  h = height_fn (i,n,n,h);
           break;
	}

      	default:
	{
	  /* Curved sections. */
	  GLfloat angle = track_defn[i].u1.c.angle;
	  GLfloat radius = track_defn[i].u1.c.radius;
	  GLfloat len;
	  GLfloat w[2], centre[2];
	  GLfloat q[2][2];
	  int n, j, is_left = angle < 0;

	  /* Compute a vector w at right angles to v. */
	  w[0] = -v[1];
	  w[1] = v[0];
	  normalize2d (w, w);

	  /* Compute the centre of the circle. If a > 0, then
	   * the centre is on the right of v. If a < 0, then
	   * the centre is on the left.
	   */
	  if (! is_left)
	    {
	      centre[0] = p[0] - w[0] * radius;
	      centre[1] = p[1] - w[1] * radius;
	    }
	  else
	    {
	      centre[0] = p[0] + w[0] * radius;
	      centre[1] = p[1] + w[1] * radius;
	    }

	  /* Work out the length of the circumference. */
	  len = radius * fabs (angle);

	  /* Work out how many segments to use. */
	  n = len / SEGMENT_LEN_CURVES;

	  /* Now calculate each segment. */
	  for (j = 0; j < n; ++j)
	    {
	      GLfloat p0[2], p1[2], p2[2], p3[2];

	      point_on_circle (v, w, centre, radius - TRACK_WIDTH/2, is_left,
			       j * angle / n, p0);
	      point_on_circle (v, w, centre, radius + TRACK_WIDTH/2, is_left,
			       j * angle / n, p1);
	      point_on_circle (v, w, centre, radius - TRACK_WIDTH/2, is_left,
			       (j+1) * angle / n, p2);
	      point_on_circle (v, w, centre, radius + TRACK_WIDTH/2, is_left,
			       (j+1) * angle / n, p3);

	      if (is_left)
		{
		  make_segment (p0[0], p0[1], height_fn (i,j,n,h),
				p1[0], p1[1], height_fn (i,j,n,h),
				p2[0], p2[1], height_fn (i,j+1,n,h),
				p3[0], p3[1], height_fn (i,j+1,n,h));
		}
	      else
		{
		  make_segment (p1[0], p1[1], height_fn (i,j,n,h),
				p0[0], p0[1], height_fn (i,j,n,h),
				p3[0], p3[1], height_fn (i,j+1,n,h),
				p2[0], p2[1], height_fn (i,j+1,n,h));
		}
	    }

	  /* Update point p and vector v. */
	  point_on_circle (v, w, centre, radius, is_left, angle, p);
	  point_on_circle (v, w, centre, radius + 1, is_left, angle, q[0]);
	  point_on_circle (v, w, centre, radius - 1, is_left, angle, q[1]);
	  if (is_left)
	    {
	      v[0] = q[1][1] - q[0][1];
	      v[1] = q[0][0] - q[1][0];
	    }
	  else
	    {
	      v[0] = q[0][1] - q[1][1];
	      v[1] = q[1][0] - q[0][0];
	    }

	  /* Update h. */
	  h = height_fn (i,n,n,h);
	}
	} // switch
    }

  /* Join up all our saved segments. */
  join_segments ();
}

/* This rather subtle little function builds a single
 * segment and appends it to the segment list.
 *
 * A segment is defined by four corner points of a
 * square. We can build all other points by just knowing
 * the four corner points.
 *
 * The four corner points, call them P0, P1, P2, P3
 * are shown in the diagram below:
 *
 *  P2           P3
 *   +-----------+
 *   |     ^     |
 *   |     | v   |
 *   |     |     |
 *   |     |     |
 *   +-----+-----+
 *  P0     p     P1
 *
 * Note that in this view, we are looking down on the
 * segment. The craft moves forwards along v, and
 * turns left towards P2 and right towards P3.
 *
 * From these points, we build the interior of the
 * track. From a side view, the interior looks like
 * this:
 *
 * ----                         ----  ^
 * P2  \                      /   P3  |
 *      \                    /        | TRACK_DEPTH
 *       ---              ---         |
 *          ---________---            V
 *
 * You should have a look at track-circle.c to
 * see how this is done. The track is divided
 * into 9 sections by lines M0 - M9:
 *
 * mb0  mb1  mb2  mb3  mb4      mb5  mb6  mb7  mb8  mb9
 * +----+----+----+----+--------+----+----+----+----+
 * |    |    |    |    |        |    |    |    |    |
 * |    |    |    |    |   40%  |    |    |    |    |
 * |    |    |    |    |        |    |    |    |    |
 * +----+----+----+----+--------+----+----+----+----+
 * ma0  ma1  ma2  ma3  ma4      ma5  ma6  ma7  ma8  ma9
 *
 * The far left and far right sections are flat.
 * The rest curve down. The section M4-M5 is the
 * base of the track and is wider than the others.
 *
 * To stop the craft coming off the track complete, there
 * are two high barriers on either side:
 *
 *      p +       + q
 *        |       |
 *        |       |
 *  ma[0] +-     -+ ma[9]
 *          \___/
 *
 * These barriers are not visible.
 *
 * Triggers are controlled by three barriers in the
 * middle of the track, looking like this:
 *
 *        r      s      t
 *        +      +      +
 *        |      |      |
 *        +------+------+
 *
 * In order to ensure that segments join up exactly,
 * we save segments we create in make_segment onto
 * a private list. At the end we call join_segments
 * which joins all the vertices from adjacent
 * segments, creates the real track segments and
 * display lists, then frees our private list.
 */
struct private_segment {
  GLfloat ma[10][3], mb[10][3];
  GLfloat mp01[3], mp23[3];
  GLfloat pa[3], pb[3], qa[3], qb[3];
  GLfloat ra[3], rb[3], sa[3], sb[3], ta[3], tb[3];
};

static struct private_segment *private_segment_list = 0;
static int nr_segments = 0;

static void
make_segment (GLfloat p0x, GLfloat p0y, GLfloat p0z,
	      GLfloat p1x, GLfloat p1y, GLfloat p1z,
	      GLfloat p2x, GLfloat p2y, GLfloat p2z,
	      GLfloat p3x, GLfloat p3y, GLfloat p3z)
{
  const GLfloat depth_factor[10] = {
    0,
    0,
    0.433833200054945,
    0.781761530259907,
    0.974890452530677,
    0.974977815437082,
    0.782006319936301,
    0.434186944818459,
    0,
    0
  };
  GLfloat p0[3] = {p0x, p0y, p0z};
  GLfloat p1[3] = {p1x, p1y, p1z};
  GLfloat p2[3] = {p2x, p2y, p2z};
  GLfloat p3[3] = {p3x, p3y, p3z};
  GLfloat mp01[3], mp23[3];
  GLfloat forwards[3], left[2][3], up[3];
  GLfloat ma[10][3], mb[10][3];
  GLfloat pa[3], pb[3], qa[3], qb[3];
  GLfloat ra[3], rb[3], sa[3], sb[3], ta[3], tb[3];
  GLfloat t;
  struct private_segment *segment;
  int i;

#if 0
  printf ("p0 = (%g, %g)  p1 = (%g, %g)  p2 = (%g, %g)  p3 = (%g, %g)\n",
	  p0x, p0y,
	  p1x, p1y,
	  p2x, p2y,
	  p3x, p3y);
#endif

  /* Compute midpoints of side P0-P1 and P2-P3. */
  midpoint (p0, p1, mp01);
  midpoint (p2, p3, mp23);

  /* Compute forwards, left, normalized up vectors. */
  forwards[0] = mp23[0] - mp01[0];
  forwards[1] = mp23[1] - mp01[1];
  forwards[2] = mp23[2] - mp01[2];
  left[0][0] = p0[0] - p1[0];
  left[0][1] = p0[1] - p1[1];
  left[0][2] = p0[2] - p1[2];
  left[1][0] = p2[0] - p3[0];
  left[1][1] = p2[1] - p3[1];
  left[1][2] = p2[2] - p3[2];
  cross_product (forwards, left[0], up);
  normalize (up, up);

  /* Compute points M*0 - M*9.
   * MA9 = P1                   MB9 = P3
   * MA8 = P1 + 7.5% * LEFT[0]  MB8 = P3 + 7.5% * LEFT[1]
   * MA7 = P1 + 15% * LEFT[0]   MB7 = P3 + 15% * LEFT[1]
   * etc.
   * and we also include a component of the up
   * vector in the middle points to get the track
   * to dip.
   */
  t = 0;
  for (i = 9; i >= 0; --i)
    {
      GLfloat depth = depth_factor[i] * TRACK_DEPTH;

      memcpy (ma[i], p1, 3 * sizeof (GLfloat));
      ma[i][0] += t * left[0][0];
      ma[i][1] += t * left[0][1];
      ma[i][2] += t * left[0][2];
      ma[i][0] -= up[0] * depth;
      ma[i][1] -= up[1] * depth;
      ma[i][2] -= up[2] * depth;

      memcpy (mb[i], p3, 3 * sizeof (GLfloat));
      mb[i][0] += t * left[1][0];
      mb[i][1] += t * left[1][1];
      mb[i][2] += t * left[1][2];
      mb[i][0] -= up[0] * depth;
      mb[i][1] -= up[1] * depth;
      mb[i][2] -= up[2] * depth;

      if (i != 5) t += 0.075; else t += 0.4;
    }

  /* Compute high barriers. */
  memcpy (pa, p0, 3 * sizeof (GLfloat));
  pa[0] += up[0] * VIRTBARRIER_HEIGHT;
  pa[1] += up[1] * VIRTBARRIER_HEIGHT;
  pa[2] += up[2] * VIRTBARRIER_HEIGHT;

  memcpy (qa, p1, 3 * sizeof (GLfloat));
  qa[0] += up[0] * VIRTBARRIER_HEIGHT;
  qa[1] += up[1] * VIRTBARRIER_HEIGHT;
  qa[2] += up[2] * VIRTBARRIER_HEIGHT;

  memcpy (pb, p2, 3 * sizeof (GLfloat));
  pb[0] += up[0] * VIRTBARRIER_HEIGHT;
  pb[1] += up[1] * VIRTBARRIER_HEIGHT;
  pb[2] += up[2] * VIRTBARRIER_HEIGHT;

  memcpy (qb, p3, 3 * sizeof (GLfloat));
  qb[0] += up[0] * VIRTBARRIER_HEIGHT;
  qb[1] += up[1] * VIRTBARRIER_HEIGHT;
  qb[2] += up[2] * VIRTBARRIER_HEIGHT;

  /* Computer trigger barriers. */
  memcpy (ra, ma[3], 3 * sizeof (GLfloat));
  ra[0] += up[0] * VIRTBARRIER_HEIGHT;
  ra[1] += up[1] * VIRTBARRIER_HEIGHT;
  ra[2] += up[2] * VIRTBARRIER_HEIGHT;

  memcpy (rb, mb[3], 3 * sizeof (GLfloat));
  rb[0] += up[0] * VIRTBARRIER_HEIGHT;
  rb[1] += up[1] * VIRTBARRIER_HEIGHT;
  rb[2] += up[2] * VIRTBARRIER_HEIGHT;

  memcpy (sa, mp01, 3 * sizeof (GLfloat));
  sa[0] += up[0] * VIRTBARRIER_HEIGHT;
  sa[1] += up[1] * VIRTBARRIER_HEIGHT;
  sa[2] += up[2] * VIRTBARRIER_HEIGHT;

  memcpy (sb, mp23, 3 * sizeof (GLfloat));
  sb[0] += up[0] * VIRTBARRIER_HEIGHT;
  sb[1] += up[1] * VIRTBARRIER_HEIGHT;
  sb[2] += up[2] * VIRTBARRIER_HEIGHT;

  memcpy (ta, ma[6], 3 * sizeof (GLfloat));
  ta[0] += up[0] * VIRTBARRIER_HEIGHT;
  ta[1] += up[1] * VIRTBARRIER_HEIGHT;
  ta[2] += up[2] * VIRTBARRIER_HEIGHT;

  memcpy (tb, mb[6], 3 * sizeof (GLfloat));
  tb[0] += up[0] * VIRTBARRIER_HEIGHT;
  tb[1] += up[1] * VIRTBARRIER_HEIGHT;
  tb[2] += up[2] * VIRTBARRIER_HEIGHT;

  /* Allocate a segment on the private list. */
  nr_segments ++;
  private_segment_list = realloc (private_segment_list,
				  nr_segments
				  * sizeof (struct private_segment));
  segment = &private_segment_list[nr_segments-1];

  /* Copy the other parameters over. */
  memcpy (segment->ma, ma, sizeof ma);
  memcpy (segment->mb, mb, sizeof mb);
  memcpy (segment->pa, pa, sizeof pa);
  memcpy (segment->pb, pb, sizeof pb);
  memcpy (segment->qa, qa, sizeof qa);
  memcpy (segment->qb, qb, sizeof qb);
  memcpy (segment->mp01, mp01, sizeof mp01);
  memcpy (segment->mp23, mp23, sizeof mp23);
  memcpy (segment->ra, ra, sizeof ra);
  memcpy (segment->rb, rb, sizeof rb);
  memcpy (segment->sa, sa, sizeof sa);
  memcpy (segment->sb, sb, sizeof sb);
  memcpy (segment->ta, ta, sizeof ta);
  memcpy (segment->tb, tb, sizeof tb);
}

static void start_output (void);
static void end_output (void);
static void start_segment (int s);
static void end_segment (int s);
static void output_track_preview (GLfloat *a, GLfloat *b,
				  GLfloat *c, GLfloat *d);
static void output_invisible_quad (GLfloat *a, GLfloat *b,
				   GLfloat *c, GLfloat *d);
static void output_striped_quad (GLfloat *a, GLfloat *b,
				 GLfloat *c, GLfloat *d, int f);
static void output_striped_edge_quad (GLfloat *a, GLfloat *b,
				      GLfloat *c, GLfloat *d, int f);
static void output_enter_quad (GLfloat *a, GLfloat *b,
			       GLfloat *c, GLfloat *d);
static void output_left_trigger_quad (GLfloat *a, GLfloat *b,
				      GLfloat *c, GLfloat *d);
static void output_mid_trigger_quad (GLfloat *a, GLfloat *b,
				     GLfloat *c, GLfloat *d);
static void output_right_trigger_quad (GLfloat *a, GLfloat *b,
				       GLfloat *c, GLfloat *d);
static void output_quad (GLfloat *a, GLfloat *b,
			 GLfloat *c, GLfloat *d, int i, int s);

static void
join_segments ()
{
  int s, ns;

  start_output ();

  for (s = 0; s < nr_segments; ++s)
    {
      struct private_segment *private_segment = &private_segment_list[s];
      int i;

      /* Join up the current and next segment.
       * XXX We should do some more checking here.
       */
      ns = s+1;
      if (ns >= nr_segments) ns = 0;
      memcpy (private_segment_list[s].mb, private_segment_list[ns].ma,
	      sizeof private_segment_list[ns].ma);
      memcpy (private_segment_list[s].pb, private_segment_list[ns].pa,
	      sizeof private_segment_list[ns].pa);
      memcpy (private_segment_list[s].qb, private_segment_list[ns].qa,
	      sizeof private_segment_list[ns].qa);

      start_segment (s);

      /* Output the track segments. */
      output_track_preview (private_segment->ma[0], private_segment->mb[0],
			    private_segment->mb[9], private_segment->ma[9]);
      output_invisible_quad (private_segment->pa, private_segment->pb,
			     private_segment->mb[0], private_segment->ma[0]);
      output_striped_edge_quad (private_segment->ma[0], private_segment->mb[0],
				private_segment->mb[1], private_segment->ma[1], 0);
      output_striped_quad (private_segment->ma[1], private_segment->mb[1],
			   private_segment->mb[2], private_segment->ma[2], 1);
      for (i = 2; i < 7; ++i)
	output_quad (private_segment->ma[i], private_segment->mb[i],
		     private_segment->mb[i+1], private_segment->ma[i+1], i, s);
      output_striped_quad (private_segment->ma[7], private_segment->mb[7],
			   private_segment->mb[8], private_segment->ma[8], 0);
      output_striped_edge_quad (private_segment->ma[8], private_segment->mb[8],
				private_segment->mb[9], private_segment->ma[9], 1);
      output_invisible_quad (private_segment->ma[9], private_segment->mb[9],
			     private_segment->qb, private_segment->qa);

      /* Output the enter plane. */
      output_enter_quad (private_segment->pa, private_segment->ma[0],
			 private_segment->ma[9], private_segment->qa);

      /* Output left, middle and right trigger planes. */
      output_left_trigger_quad (private_segment->ra, private_segment->ma[3],
				private_segment->mb[3], private_segment->rb);
      output_mid_trigger_quad (private_segment->sa, private_segment->mp01,
			       private_segment->mp23, private_segment->sb);
      output_right_trigger_quad (private_segment->ta, private_segment->ma[6],
				 private_segment->mb[6], private_segment->tb);

      end_segment (s);
    }

  end_output ();
}

static FILE *outputfp = 0, *previewfp = 0;
static int seg = -1, quad_num = -1;

static void
start_output ()
{
  outputfp = fopen (outputfile, "w");
  if (outputfp == NULL) { perror (outputfile); exit (1); }

  previewfp = fopen (previewfile, "w");
  if (previewfp == NULL) { perror (previewfile); exit (1); }

  switch (outputformat)
    {
    case FORMAT_AC3D:
      fprintf (outputfp,
	       "AC3Db\n"
	       "MATERIAL \"\" rgb 0 0 0 amb 0.2 0.2 0.2 emis 0 0 0 spec 0.5 0.5 0.5 shi 10 trans 0\n"
	       "MATERIAL \"\" rgb 1 1 0 amb 0.2 0.2 0.2 emis 0 0 0 spec 0.5 0.5 0.5 shi 10 trans 0\n"
	       "MATERIAL \"\" rgb 0.5 0.5 0.5 amb 0.2 0.2 0.2 emis 0 0 0 spec 0.5 0.5 0.5 shi 10 trans 0\n"
	       "MATERIAL \"\" rgb 0.5 0.5 0.5 amb 0.2 0.2 0.2 emis 0 0 0 spec 0.5 0.5 0.5 shi 10 trans 1\n"
	       "OBJECT world\n"
	       "kids %d\n",
	       15 * nr_segments);
      fprintf (previewfp,
	       "AC3Db\n"
	       "MATERIAL \"\" rgb 0 0 0 amb 0.2 0.2 0.2 emis 0 0 0 spec 0.5 0.5 0.5 shi 10 trans 0\n"
	       "MATERIAL \"\" rgb 1 1 0 amb 0.2 0.2 0.2 emis 0 0 0 spec 0.5 0.5 0.5 shi 10 trans 0\n"
	       "MATERIAL \"\" rgb 0.5 0.5 0.5 amb 0.2 0.2 0.2 emis 0 0 0 spec 0.5 0.5 0.5 shi 10 trans 0\n"
	       "MATERIAL \"\" rgb 0.5 0.5 0.5 amb 0.2 0.2 0.2 emis 0 0 0 spec 0.5 0.5 0.5 shi 10 trans 1\n"
	       "OBJECT world\n"
	       "kids %d\n",
	       1 * nr_segments);
      break;
    }
}

static void
end_output ()
{
  switch (outputformat)
    {
    case FORMAT_AC3D:
      break;
    }

  fclose (outputfp);
  fclose (previewfp);
}

static void
start_segment (int s)
{
  seg = s;
  quad_num = 0;

  switch (outputformat)
    {
    case FORMAT_AC3D:
      break;
    }
}

static void
end_segment (int s)
{
  seg = -1;

  switch (outputformat)
    {
    case FORMAT_AC3D:
      break;
    }
}

static void
output_invisible_quad (GLfloat *a, GLfloat *b,
		       GLfloat *c, GLfloat *d)
{
  switch (outputformat)
    {
    case FORMAT_AC3D:
      fprintf (outputfp,
	       "OBJECT poly\n"
	       "name \"t_%d_%d_i\"\n"
	       "loc 0 0 0\n"
	       "numvert 4\n"
	       "%g %g %g\n"
	       "%g %g %g\n"
	       "%g %g %g\n"
	       "%g %g %g\n"
	       "numsurf 1\n"
	       "SURF 0x20\n"
	       "mat 0\n"
	       "refs 4\n"
	       "0 0 0\n"
	       "1 0 0\n"
	       "2 0 0\n"
	       "3 0 0\n",
	       seg, quad_num,
	       a[0], a[1], a[2],
	       b[0], b[1], b[2],
	       c[0], c[1], c[2],
	       d[0], d[1], d[2]);
      break;
    }

  quad_num ++;
}

static void
output_track_preview (GLfloat *a, GLfloat *b,
		      GLfloat *c, GLfloat *d)
{
  switch (outputformat)
    {
    case FORMAT_AC3D:
      fprintf (previewfp,
	       "OBJECT poly\n"
	       "name \"pr_%d\"\n"
	       "loc 0 0 0\n"
	       "numvert 4\n"
	       "%g %g %g\n"
	       "%g %g %g\n"
	       "%g %g %g\n"
	       "%g %g %g\n"
	       "numsurf 1\n"
	       "SURF 0x22\n"
	       "mat 1\n"
	       "refs 4\n"
	       "0 0 0\n"
	       "1 0 0\n"
	       "2 0 0\n"
	       "3 0 0\n",
	       seg,
	       a[0], a[1], a[2],
	       b[0], b[1], b[2],
	       c[0], c[1], c[2],
	       d[0], d[1], d[2]);
      break;
    }

  quad_num ++;
}

static void
output_striped_quad (GLfloat *a, GLfloat *b,
		     GLfloat *c, GLfloat *d, int f)
{
  switch (outputformat)
    {
    case FORMAT_AC3D:
      fprintf (outputfp,
	       "OBJECT poly\n"
	       "name \"t_%d_%d_s\"\n"
	       "loc 0 0 0\n"
	       "numvert 4\n"
	       "%g %g %g\n"
	       "%g %g %g\n"
	       "%g %g %g\n"
	       "%g %g %g\n"
	       "numsurf 1\n"
	       "SURF 0x20\n"
	       "mat 1\n"
	       "refs 4\n",
	       seg, quad_num,
	       a[0], a[1], a[2],
	       b[0], b[1], b[2],
	       c[0], c[1], c[2],
	       d[0], d[1], d[2]);
      if (f == 0)
	fprintf (outputfp,
		 "0 0 0\n"
		 "1 1 0\n"
		 "2 1 1\n"
		 "3 0 1\n");
      else
	fprintf (outputfp,
		 "0 1 0\n"
		 "1 0 0\n"
		 "2 0 1\n"
		 "3 1 1\n");
      break;
    }

  quad_num ++;
}

static void
output_striped_edge_quad (GLfloat *a, GLfloat *b,
			  GLfloat *c, GLfloat *d, int f)
{
  switch (outputformat)
    {
    case FORMAT_AC3D:
      fprintf (outputfp,
	       "OBJECT poly\n"
	       "name \"t_%d_%d_S\"\n"
	       "loc 0 0 0\n"
	       "numvert 4\n"
	       "%g %g %g\n"
	       "%g %g %g\n"
	       "%g %g %g\n"
	       "%g %g %g\n"
	       "numsurf 1\n"
	       "SURF 0x20\n"
	       "mat 1\n"
	       "refs 4\n",
	       seg, quad_num,
	       a[0], a[1], a[2],
	       b[0], b[1], b[2],
	       c[0], c[1], c[2],
	       d[0], d[1], d[2]);
      if (f == 0)
	fprintf (outputfp,
		 "0 0 0\n"
		 "1 1 0\n"
		 "2 1 1\n"
		 "3 0 1\n");
      else
	fprintf (outputfp,
		 "0 1 0\n"
		 "1 0 0\n"
		 "2 0 1\n"
		 "3 1 1\n");
      break;
    }

  quad_num ++;
}

static void
output_enter_quad (GLfloat *a, GLfloat *b,
		   GLfloat *c, GLfloat *d)
{
  switch (outputformat)
    {
    case FORMAT_AC3D:
      fprintf (outputfp,
	       "OBJECT poly\n"
	       "name \"t_%d_%d_e\"\n"
	       "loc 0 0 0\n"
	       "numvert 4\n"
	       "%g %g %g\n"
	       "%g %g %g\n"
	       "%g %g %g\n"
	       "%g %g %g\n"
	       "numsurf 1\n"
	       "SURF 0x20\n"
	       "mat 3\n"
	       "refs 4\n"
	       "0 0 0\n"
	       "1 0 0\n"
	       "2 0 0\n"
	       "3 0 0\n",
	       seg, quad_num,
	       a[0], a[1], a[2],
	       b[0], b[1], b[2],
	       c[0], c[1], c[2],
	       d[0], d[1], d[2]);
      break;
    }

  quad_num ++;
}

static void
output_left_trigger_quad (GLfloat *a, GLfloat *b,
			  GLfloat *c, GLfloat *d)
{
  switch (outputformat)
    {
    case FORMAT_AC3D:
      fprintf (outputfp,
	       "OBJECT poly\n"
	       "name \"tr_%d_l\"\n"
	       "loc 0 0 0\n"
	       "numvert 4\n"
	       "%g %g %g\n"
	       "%g %g %g\n"
	       "%g %g %g\n"
	       "%g %g %g\n"
	       "numsurf 1\n"
	       "SURF 0x20\n"
	       "mat 3\n"
	       "refs 4\n"
	       "0 0 0\n"
	       "1 0 0\n"
	       "2 0 0\n"
	       "3 0 0\n",
	       seg,
	       a[0], a[1], a[2],
	       b[0], b[1], b[2],
	       c[0], c[1], c[2],
	       d[0], d[1], d[2]);
      break;
    }

  quad_num ++;
}

static void
output_mid_trigger_quad (GLfloat *a, GLfloat *b,
			 GLfloat *c, GLfloat *d)
{
  switch (outputformat)
    {
    case FORMAT_AC3D:
      fprintf (outputfp,
	       "OBJECT poly\n"
	       "name \"tr_%d_m\"\n"
	       "loc 0 0 0\n"
	       "numvert 4\n"
	       "%g %g %g\n"
	       "%g %g %g\n"
	       "%g %g %g\n"
	       "%g %g %g\n"
	       "numsurf 1\n"
	       "SURF 0x20\n"
	       "mat 3\n"
	       "refs 4\n"
	       "0 0 0\n"
	       "1 0 0\n"
	       "2 0 0\n"
	       "3 0 0\n",
	       seg,
	       a[0], a[1], a[2],
	       b[0], b[1], b[2],
	       c[0], c[1], c[2],
	       d[0], d[1], d[2]);
      break;
    }

  quad_num ++;
}

static void
output_right_trigger_quad (GLfloat *a, GLfloat *b,
			   GLfloat *c, GLfloat *d)
{
  switch (outputformat)
    {
    case FORMAT_AC3D:
      fprintf (outputfp,
	       "OBJECT poly\n"
	       "name \"tr_%d_r\"\n"
	       "loc 0 0 0\n"
	       "numvert 4\n"
	       "%g %g %g\n"
	       "%g %g %g\n"
	       "%g %g %g\n"
	       "%g %g %g\n"
	       "numsurf 1\n"
	       "SURF 0x20\n"
	       "mat 3\n"
	       "refs 4\n"
	       "0 0 0\n"
	       "1 0 0\n"
	       "2 0 0\n"
	       "3 0 0\n",
	       seg,
	       a[0], a[1], a[2],
	       b[0], b[1], b[2],
	       c[0], c[1], c[2],
	       d[0], d[1], d[2]);
      break;
    }

  quad_num ++;
}

static void
output_quad (GLfloat *a, GLfloat *b,
	     GLfloat *c, GLfloat *d, int i, int s)
{
  switch (outputformat)
    {
    case FORMAT_AC3D:
      {
	double s1, s2;
	static double tex_point[6] = { 1, 0.625/0.7, 0.55/0.7,
				       0.15/0.7, 0.075/0.7, 0 };

	s1 = (s % 10) / 10.;
	s2 = ((s+1) % 10) / 10.;
	if (s2 < s1) s2 += 1;

	fprintf (outputfp,
		 "OBJECT poly\n"
		 "name \"t_%d_%d_t\"\n"
		 "loc 0 0 0\n"
		 "numvert 4\n"
		 "%g %g %g\n"
		 "%g %g %g\n"
		 "%g %g %g\n"
		 "%g %g %g\n"
		 "numsurf 1\n"
		 "SURF 0x20\n"
		 "mat 2\n"
		 "refs 4\n"
		 "0 %g %g\n"
		 "1 %g %g\n"
		 "2 %g %g\n"
		 "3 %g %g\n",
		 seg, quad_num,
		 a[0], a[1], a[2],
		 b[0], b[1], b[2],
		 c[0], c[1], c[2],
		 d[0], d[1], d[2],
		 s1, tex_point[i-2],
		 s2, tex_point[i-2],
		 s2, tex_point[i-1],
		 s1, tex_point[i-1]);
	break;
      }
    }

  quad_num ++;
}
