/* XRacer (C) 1999 Richard W.M. Jones.
 * $Id: perlin.c,v 1.4 1999/07/12 12:57:07 rich Exp $
 */

/* This code was written by Richard Jones, but based on
 * the Perlin noise generator sent to me by Andrew Ford
 * <aford@daisy.uwaterloo.ca>. You can find out more about
 * Perlin noise here:
 *
 * http://freespace.virgin.net/hugo.elias/models/m_perlin.htm
 */

#ifdef __CYGWIN32__
#include <windows.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <assert.h>
#include <sys/time.h>

#include "config.h"

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

static struct layer {
  int nr_x_samples, nr_y_samples;
  double *data;
} *layer;

static int nr_layers;

static inline double
get_data (int k, int a, int b)
{
  assert (0 <= k && k < nr_layers);
  assert (0 <= a && a < layer[k].nr_x_samples+2);
  assert (0 <= b && b < layer[k].nr_y_samples+2);
  return layer[k].data [a * (layer[k].nr_y_samples+1) + b];
}

static inline void
set_data (int k, int a, int b, double v)
{
  assert (0 <= k && k < nr_layers);
  assert (0 <= a && a < layer[k].nr_x_samples+2);
  assert (0 <= b && b < layer[k].nr_y_samples+2);
  layer[k].data [a * (layer[k].nr_y_samples+1) + b] = v;
}

static inline double
get_value (int k, double x, double y)
{
  double z1, z2, c11, c21, c22, c12, scale;
  int ax, ay;

  z1 = x * layer[k].nr_x_samples;
  z2 = y * layer[k].nr_y_samples;
  ax = (int) floor (z1);
  ay = (int) floor (z2);

  z1 -= floor (z1);
  z2 -= floor (z2);

  c11 = (1 - z1) * (1 - z2);
  c21 = z1       * (1 - z2);
  c22 = z1       * z2;
  c12 = (1 - z1) * z2;
  scale = c11 + c21 + c12 + c22;
  c11 /= scale;
  c21 /= scale;
  c12 /= scale;
  c22 /= scale;
  return
      c11 * get_data (k, ax, ay)
    + c21 * get_data (k, ax+1, ay)
    + c12 * get_data (k, ax, ay+1)
    + c22 * get_data (k, ax+1, ay+1);
}

static inline double
get_total_value (double x, double y)
{
  double total = 0;
  int k;

  for (k = 0; k < nr_layers; ++k)
    total += get_value (k, x, y);

  return total;
}

static inline double
int_noise ()
{
  return ((double) (rand () % 10000)) / 10000.0;
}

static void
usage ()
{
  fprintf (stderr,
	   "perlin -n nrLayers -x nrXsamples -y nrYsamples -w width -h height -f freq -a amp\n");
  exit (1);
}

int
main (int argc, char *argv[])
{
  int c, i, j, k, nr_x_samples = -1, x, nr_y_samples = -1, y, freq = -1;
  int width = -1, height = -1;
  double amp, amplitude = -1, px, py, max, min, *image, *p, v;

  /* Parse command line arguments. */
  while ((c = getopt (argc, argv, "x:y:w:h:f:a:n:")) != -1)
    {
      switch (c)
	{
	case 'x':
	  sscanf (optarg, "%d", &nr_x_samples);
	  break;
	case 'y':
	  sscanf (optarg, "%d", &nr_y_samples);
	  break;
	case 'w':
	  sscanf (optarg, "%d", &width);
	  break;
	case 'h':
	  sscanf (optarg, "%d", &height);
	  break;
	case 'f':
	  sscanf (optarg, "%d", &freq);
	  break;
	case 'a':
	  sscanf (optarg, "%lf", &amplitude);
	  break;
	case 'n':
	  sscanf (optarg, "%d", &nr_layers);
	  break;
	default:
	  usage ();
	}
    }

  if (nr_x_samples == -1 || nr_y_samples == -1 || width == -1 || height == -1
      || freq == -1 || amplitude == -1 || nr_layers == -1)
    usage ();

  /* Generate the noise. */
  x = nr_x_samples;
  y = nr_y_samples;
  amp = amplitude;

  srand (time (NULL));

  layer = malloc (sizeof (struct layer) * nr_layers);

  for (k = 0; k < nr_layers; ++k)
    {
      layer[k].nr_x_samples = x;
      layer[k].nr_y_samples = y;

      layer[k].data = malloc (sizeof (double) * (x+2) * (y+2));
      if (layer[k].data == 0) { perror ("malloc"); exit (1); }

      for (i = 0; i < x; ++i)
	{
	  for (j = 0; j < y; ++j)
	    set_data (k, i, j, amp * int_noise ());
	  set_data (k, i, y, get_data (k, i, 0));
	  set_data (k, i, y+1, get_data (k, i, 0));
	}

      for (j = 0; j < y; ++j)
	{
	  set_data (k, x, j, get_data (k, 0, j));
	  set_data (k, x+1, j, get_data (k, 0, j));
	}

      set_data (k, x, y, get_data (k, 0, 0));
      set_data (k, x+1, y, get_data (k, 0, 0));
      set_data (k, x, y+1, get_data (k, 0, 0));
      set_data (k, x+1, y, get_data (k, 0, 0));

      x *= freq;
      y *= freq;
      amp *= amplitude;
    }

  /* Compute max. and min. over image. */
  max = -1e9;
  min = 1e9;
  p = image = malloc (sizeof (double) * width * height);
  for (y = 0; y < height; ++y)
    for (x = 0; x < width; ++x)
      {
	px = (double) x / (width-1);
	py = (double) y / (height-1);

	*p = get_total_value (px, py);
	if (*p < min) min = *p;
	if (*p > max) max = *p;
	p++;
      }

  /* Output the image. */
  printf ("P2 %d %d 255\n", width, height);
  p = image;
  for (y = 0; y < height; ++y)
    for (x = 0; x < width; ++x)
      {
	v = *p++;
	v = 255 * (v - min) / (max - min);
	printf ("%d\n", (int) floor (v));
      }

  exit (0);
}
