/* XRacer (C) 1999 Richard W.M. Jones.
 * $Id: log.c,v 1.7 1999/09/20 20:01:42 rich Exp $
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>

#include "xracer.h"

/* We store messages on a linked list. The head of the
 * linked list (MESSAGE_HEAD) is the oldest message.
 * The last message on the list (LAST_MESSAGE) is the
 * most recent message.
 */
struct message
{
  struct message *next;		/* Linked list pointer. */
  char *message;		/* The message. */
  double entry_time;		/* The time we put the message on the list. */
};

struct message *message_head = 0, *last_message = 0;
static int nr_messages = 0;

static int stderr_threshold = 0;
static int display_threshold = 0;

static void *font = 0;
static int text_height;

#define MAX_TIME 20		/* Max. time to remain on the list. */
#define MAX_MESSAGES 10		/* Max. number of messages on the list. */
#define MAX_MESSAGE_LEN 70	/* Max. message length for display. */

#define LOG_X_OFFSET 16
#define LOG_Y_OFFSET (-16 - text_height)

/* Scroll up the log messages. */
static void
scroll_messages ()
{
  struct message *m;

  log_assert (nr_messages > 0 && message_head && last_message);

  m = message_head->next;
  free (message_head->message);
  free (message_head);
  message_head = m;

  if (m == 0) last_message = 0;

  nr_messages --;
}

/* Add a message to the list. */
static void
add_message (const char *message)
{
  int len, truncate = 0;
  struct message *m;

  /* If we have run out of space on the list, scroll the
   * list up first.
   */
  if (nr_messages >= MAX_MESSAGES)
    scroll_messages ();

  m = xmalloc (sizeof (struct message));
  m->next = 0;

  /* Copy the message into another buffer. */
  len = strlen (message);
  if (len > MAX_MESSAGE_LEN)
    {
      len = MAX_MESSAGE_LEN;
      truncate = 1;
    }
  m->message = xmalloc ((len+1) * sizeof (char));
  memcpy (m->message, message, len * sizeof (char));
  m->message[len] = '\0';
  if (truncate)
    {
      m->message[len-2] = m->message[len-1] = '.';
    }

  /* Set the time that we entered this message. */
  m->entry_time = current_time;

  /* Append the message to the list. */
  if (last_message == 0)
    message_head = last_message = m;
  else
    {
      last_message->next = m;
      last_message = m;
    }

  nr_messages ++;
}

/* Convert log level number to string. */
static const char *
level_to_string (int level)
{
  static char *str[4] = { "debug", "info", "warning", "error" };
  log_assert (0 <= level && level <= 3);
  return str [level];
}

/* This function displays the actual log on screen. */
void
log_display ()
{
  int n;
  struct message *m;

  /* Throw away very old messages. */
  while (nr_messages > 0 && current_time - message_head->entry_time > MAX_TIME)
    scroll_messages ();

  if (nr_messages == 0) return;

  for (n = 0, m = message_head; m; n++, m = m->next)
    {
      draw_text (font, m->message,
		 LOG_X_OFFSET, n * text_height - LOG_Y_OFFSET);
    }
}

/* Assertion failed. */
void
log_assertfail (const char *file, int line, const char *expr)
{
  log_fatal_with_line (file, line, "assertion failed: %s", expr);
}

/* Unimplemented function called. */
void
log_not_impl_with_line (const char *file, int line)
{
  log_fatal_with_line (file, line, "function not implemented");
}

/* Fatal error. */
void
log_fatal_with_line (const char *file, int line, const char *fs, ...)
{
  va_list args;

  fprintf (stderr, "xracer:%s:%d: fatal error: ", file, line);

  va_start (args, fs);
  vfprintf (stderr, fs, args);
  va_end (args);

  fprintf (stderr, "\n");
  exit (1);
}

/* Log message. */
void
log_with_line (const char *file, int line, int level, const char *fs, ...)
{
  va_list args;
  char message [1024];

  if (level >= stderr_threshold &&
      level >= display_threshold)
    {
      va_start (args, fs);
#if HAVE_VSNPRINTF
      vsnprintf (message, sizeof message, fs, args);
#else
      vsprintf (message, fs, args);
#endif
      va_end (args);

      /* Log to std. error. */
      if (level >= stderr_threshold)
        fprintf (stderr,
                 "xracer:%s:%d: %s: %s\n",
                 file, line, level_to_string (level), message);

      /* Log message to screen. */
      if (level >= display_threshold)
        add_message (message);
    }
}

/* System error message. */
void
log_perror_with_line (const char *file, int line, const char *fs, ...)
{
  va_list args;
  char message [1024], *error_string;
  const int level = LOG_ERROR;
  int err = errno;

  if (level >= stderr_threshold &&
      level >= display_threshold)
    {
      va_start (args, fs);
#if HAVE_VSNPRINTF
      vsnprintf (message, sizeof message, fs, args);
#else
      vsprintf (message, fs, args);
#endif
      va_end (args);

      error_string = strerror (err);
      if (strlen (message) + strlen (error_string) + 2 < sizeof message)
        {
          strcat (message, ": ");
          strcat (message, error_string);
        }

      /* Log to std. error. */
      if (level >= stderr_threshold)
        fprintf (stderr,
                 "xracer:%s:%d: %s: %s\n",
                 file, line, level_to_string (level), message);

      /* Log message to screen. */
      if (level >= display_threshold)
        add_message (message);
    }
}

/* Initialize fonts. */
void
log_init ()
{
  font = find_font ("crillee", 14);
  log_assert (font != 0);

  text_height = get_text_height (font);
}
