/* XRacer (C) 1999 Richard W.M. Jones.
 * $Id: main.c,v 1.35 1999/09/12 13:16:24 rich Exp $
 */

#include "config.h"

#ifdef __CYGWIN32__
#include <windows.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <sys/time.h>

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#include "xracer.h"
#include "xracer-net.h"
#include "keyboard.h"
#include "cdmusic.h"
#include "globals.h"
#include "init.h"
#include "profile.h"
#include "os.h"
#include "sound.h"

#ifdef __CYGWIN32__
static int UnixMain(int argc, char **argv);
static LRESULT WINAPI MainWndProc( HWND, UINT, WPARAM, LPARAM );
static char *args[256];
static char args_str[2048];
#endif

static void restore_key_repeat (void);
static void deactivate_game_mode (void);

int
#ifndef __CYGWIN32__
main (int argc, char *argv [])
#else
UnixMain (int argc, char *argv [])
#endif
{
  const char *home;
  char game_mode_str[256];

  /* Check XRACER_HOME environment variable is set. */
  home = getenv ("XRACER_HOME");
  if (home == 0)
    {
      fprintf (stderr,
	       "xracer: The XRACER_HOME environment variable must be set, and must point\n"
	       "to whereever you installed the xracer data files.\n");
      exit (1);
    }
  if (chdir (home) == -1)
    {
      perror (home);
      exit (1);
    }

  game.nr_laps = 0;
  check_args (argc, argv);

  /* This disables the splash screen on 3DFX cards, and has no effect
   * for anyone else.
   */
  putenv ("FX_GLIDE_NO_SPLASH=1");

  /* Initialize PRNG. */
  srandom (time (NULL));

  /* Initialize sound. */
  if (!nosound)
  {
    if (verbose)
      printf("Initializing sound...\n");
    
    sound_init (44100, 2, 16);
  }
  else if (verbose)
    printf("Sound disabled\n");

  /* Force fullscreen mode if requested. */
  if (fullscreen) putenv ("MESA_GLX_FX=fullscreen");

  /* Initialize GLUT library. */
  if (verbose) printf ("Initializing GLUT library...\n");
  glutInit (&argc, argv);

  /* Initialize controls. */
  if (verbose) printf ("Initializing controls...\n");
  control_init ();

  /* Detect and initialize joystick. */
  if (verbose) printf ("Detecting and initializing joystick...\n");
  joystick_init ();

  /* Initialize GLUT properly. */
  glutInitDisplayMode (GLUT_DOUBLE | GLUT_RGBA | GLUT_DEPTH);
  glutInitWindowSize (width, height);
  if (fullscreen)
  {
    glutInitWindowPosition (0, 0);
    sprintf (game_mode_str, "width=%d height=%d bpp=16", width, height);
    glutEnterGameMode();
  }
  else
  {
    glutCreateWindow ("XRacer (C) 1999 Richard W.M. Jones");
  }
  glutSetKeyRepeat (GLUT_KEY_REPEAT_OFF);
  ws_fix_window();

  /* Packing used for glDrawPixels. */
  glPixelStorei (GL_UNPACK_ALIGNMENT, 1);

  /* Max. quality. */
  glHint (GL_POINT_SMOOTH_HINT, GL_NICEST);
  glHint (GL_LINE_SMOOTH_HINT, GL_NICEST);
  glHint (GL_POLYGON_SMOOTH_HINT, GL_NICEST);
  glHint (GL_FOG_HINT, GL_NICEST);

  /* Clear background each frame. */
  glClearColor (0, 0, 0, 0);

  /* Load and parse font files. */
  text_init ();

  /* Initialize logging. */
  log_init ();

  /* Enable texture mapping. */
  texture_init ();

  /* Initialize spacecraft. */
  if (verbose) printf ("initializing craft\n");
  craft_init ();

  /* Initialize precipitation. */
  if (verbose) printf ("initializing precipitation\n");
  precipitation_init ();

  /* Initialize the keyboard events */
  if (verbose) printf ("initializing keyboard\n");
  keyboard_init ();

  /* Initialize time early. */
  recalc_time ();

  /* Initialize pilot position. */
  if (verbose) printf ("initializing pilot\n");
  pilot_init ();

  /* Initialize lcd. */
  if (verbose) printf ("initializing lcd\n");
  lcd_init ();

  /* Initialize panel. */
  if (verbose) printf ("initializing panel\n");
  panel_init ();

  /* Play CD */
  play_cd();

  /* Enable network code. */
  net_init ();

  /* Start with first menu. */
  menu_start ();

  /* Start profiling. */
  PROFILE_START_CODE;

  /* Common GLUT callback functions. */
  glutReshapeFunc (reshape);

  /* Register a function to set keyboard repeat back on at exit. */
  atexit (restore_key_repeat);

  /* Register a function to switch GLUT game mode off. */
  if (fullscreen)
    atexit (deactivate_game_mode);

  /* Run GLUT. */
  glutMainLoop ();
  
  exit (0);
}

void *
xmalloc (size_t n)
{
  void *p;

  if (n == 0) n = 1;		/* Fix required for AIX. */
  p = malloc (n);
  if (p == 0) { perror ("malloc"); exit (1); }
  return p;
}

void *
xrealloc (void *p, size_t n)
{
  if (n == 0) n = 1;		/* Fix required for AIX. */

  if (p == 0)			/* For non-GNU realloc. */
    p = malloc (n);
  else
    p = realloc (p, n);
  if (p == 0) { perror ("realloc"); exit (1); }
  return p;
}

char *
xstrdup(char *p)
{
  char *r;

  if (p == NULL)
    return NULL;

  r = (char *)xmalloc(strlen(p) + 1);

  strcpy(r,p);

  return r;
}

static void
restore_key_repeat()
{
  printf ("Restoring keyboard repeat.\n");
  glutSetKeyRepeat (GLUT_KEY_REPEAT_DEFAULT);
  ws_flush_events();
}

/* Switch back from GLUT game mode */
static void
deactivate_game_mode ()
{
  glutLeaveGameMode ();
}

#ifdef __CYGWIN32__
int WINAPI
WinMain (HINSTANCE hInstance,
	 HINSTANCE hPrevInstance,
	 LPSTR cmdLine,
	 int cmdShow)
{
   WNDCLASS wc;
   MSG msg;
   HWND hWnd;
   int i;
   char *arg;


   if(!hPrevInstance )
   {
      wc.lpszClassName = "GenericAppClass";
      wc.lpfnWndProc = MainWndProc;
      wc.style = CS_OWNDC | CS_VREDRAW | CS_HREDRAW;
      wc.hInstance = hInstance;
      wc.hIcon = LoadIcon( NULL, IDI_APPLICATION );
      wc.hCursor = LoadCursor( NULL, IDC_ARROW );
      wc.hbrBackground = (HBRUSH)( COLOR_WINDOW+1 );
      wc.lpszMenuName = "GenericAppMenu";
      wc.cbClsExtra = 0;
      wc.cbWndExtra = 0;

      RegisterClass( &wc );
   }

   // create generic window for the app
   // this is not the display window
   hWnd = CreateWindow( "GenericAppClass",
      "XRACER WINDOW",
      WS_OVERLAPPEDWINDOW,
      0,
      0,
      260,
      60,
      NULL,
      NULL,
      hInstance,
      NULL
   );

   // set global handle to window
   // this handle needed for some windows calls
   ghWnd = hWnd;

   // have to show this window!
   ShowWindow( hWnd, cmdShow );

   // convert windows command line string to unix argv array
   strcpy(args_str, cmdLine);
   arg = args_str;
   i = 0;
   while (TRUE) {
     args[i++] = arg;
     arg = strstr(arg, " ");
     if (arg == NULL)
       break;
     *arg++ = (char) NULL;
   }

   return UnixMain(i,args);
}

LRESULT CALLBACK
MainWndProc (HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
   PAINTSTRUCT ps;
   HDC hDC;

   switch( msg ) {
      case WM_PAINT:
         hDC = BeginPaint(hWnd, &ps);
         EndPaint(hWnd, &ps);
         break;

      case WM_DESTROY:
         PostQuitMessage(0);
         break;

      default:
         return DefWindowProc( hWnd, msg, wParam, lParam );
   }
   return 0;
}
#endif /* __CYGWIN32__ */
