/* XRacer (C) 1999 Richard W.M. Jones.
 * $Id: matrix.h,v 1.4 1999/07/31 16:09:20 rich Exp $
 */

#ifndef __matrix_h__
#define __matrix_h__

/* Some of these functions have been inlined. I only inline
 * functions based on evidence from profiling the code as a
 * whole or where the function is so simple that the overhead
 * of calling it is larger than the inlined code. Excessive
 * inlining can itself cause performance problems, particularly
 * on modern processors which are very good at making jumps
 * function calls.
 *   - RWMJ.
 */

extern GLfloat I[16];		/* The identity matrix. */

extern inline void
identity_matrix (GLfloat *m)
{
  memcpy (m, I, sizeof (GLfloat) * 16);
}

extern void    make_rotation_matrix (GLfloat angle,
				     GLfloat x, GLfloat y, GLfloat z,
				     GLfloat *m);
extern void    matrix_multiply (const GLfloat *a, const GLfloat *b,
				GLfloat *product);

/* Multiply matrix by vector. */
extern inline void
matrix_vector_multiply (const GLfloat *m, const GLfloat *v,
			GLfloat *result)
{
  result[0] = m[0]*v[0] + m[1]*v[1] + m[2]*v[2] + m[3]*v[3];
  result[1] = m[4]*v[0] + m[5]*v[1] + m[6]*v[2] + m[7]*v[3];
  result[2] = m[8]*v[0] + m[9]*v[1] + m[10]*v[2] + m[11]*v[3];
  result[3] = m[12]*v[0] + m[13]*v[1] + m[14]*v[2] + m[15]*v[3];
}

/* Compute the magnitude of a vector. */
extern inline GLfloat
magnitude (const GLfloat *v)
{
  return sqrt (v[0]*v[0] + v[1]*v[1] + v[2]*v[2]);
}

extern GLfloat magnitude2d (const GLfloat *v);

/* Normalize a vector. */
extern inline void
normalize (const GLfloat *v, GLfloat *r)
{
  GLfloat w = magnitude (v);
  r[0] = v[0] / w;
  r[1] = v[1] / w;
  r[2] = v[2] / w;
}

extern void    normalize2d (const GLfloat *v, GLfloat *r);

extern inline void
unit_normal_to_side (GLfloat *side, GLfloat *normal)
{
  GLfloat n[3] = { side[0], side[1], side[2] };
  normalize (n, normal);
}

extern GLfloat dot_product (const GLfloat *v1, const GLfloat *v2);
extern GLfloat magnitude_in_direction (const GLfloat *v1, const GLfloat *v2);

/* Compute angle between two vectors. */
extern inline GLfloat
angle_between (const GLfloat *v1, const GLfloat *v2)
{
  return acos (magnitude_in_direction (v1, v2));
}

extern void    midpoint (const GLfloat *p1, const GLfloat *p2, GLfloat *mp);
extern void    midpoint2d (const GLfloat *p1, const GLfloat *p2, GLfloat *mp);
extern void    cross_product (const GLfloat *v, const GLfloat *w, GLfloat *r);
extern GLfloat distance (const GLfloat *p, const GLfloat *q);

/* Distance from a point to a plane. */
extern inline GLfloat
distance_point_to_plane (const GLfloat *plane, const GLfloat *point)
{
  GLfloat a = plane[0];
  GLfloat b = plane[1];
  GLfloat c = plane[2];
  GLfloat d = plane[3];
  GLfloat x = point[0];
  GLfloat y = point[1];
  GLfloat z = point[2];
  GLfloat t = (a*x + b*y + c*z + d) / - (a*a + b*b + c*c);
  GLfloat t2 = t*t;
  GLfloat dist = sqrt (t2*a*a + t2*b*b + t2*c*c);
  /* Don't lose the sign of t. */
  if (t < 0) return dist; else return -dist;
}

extern void    plane_coefficients (const GLfloat *p, const GLfloat *q,
				   const GLfloat *r, GLfloat *co);

#endif /*  __matrix_h__ */
