#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <sys/time.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>

#include "xracer.h"
#include "globals.h"
#include "os.h"
#include "xracer-net.h"

static int sock_fd = -1;
static int serv_sock_fd = -1;

/* Only used in server mode code */
static int player_fds[MAX_PLAYERS];
static struct sockaddr_in player_addrs[MAX_PLAYERS];

/* Only used in client mode code */
static struct sockaddr_in serv_addr;
static unsigned short udp_port;

/* Local stuff */
void net_init_client(void);
void net_init_server(void);
int net_add_player(int s,struct sockaddr_in from);
void net_drop_player(int i);
void net_send_join();
void net_handle_motion_packet(void);
int net_handle_packet(int plr);
void net_handle_join_request(int s,int plr);
void net_handle_join_accept(int s);
void net_handle_player_list(int s);
void net_send_motion_packet(struct sockaddr_in *a,int plr);
void net_send_packet(int s,unsigned char type,void *data,int len);
void net_send_player_lists(void);

void net_init_client(void)
{
  struct sockaddr_in addr;
  struct hostent *he;
  int tmp;

  sock_fd = socket(AF_INET,SOCK_DGRAM,IPPROTO_UDP);

  if (sock_fd < 0)
  {
    log_perror("failed to create UDP socket");
    return;
  }

  memset(&addr,0,sizeof(addr));

  addr.sin_family = AF_INET;
  addr.sin_addr.s_addr = INADDR_ANY;

  for (udp_port=5000; udp_port < 6000; udp_port++)
  {
    addr.sin_port = htons(udp_port);

    if (bind(sock_fd,(struct sockaddr *)&addr,sizeof(addr)) == 0)
      break;
  }

  if (udp_port == 6000)
  {
    log(LOG_ERROR,"no UDP ports between 5000 and 6000 free");
    close(sock_fd);
    sock_fd = -1;
    return;
  }

  serv_addr.sin_family = AF_INET;
  serv_addr.sin_port = htons(XRACER_PORT);

  he = gethostbyname(netserver);

  if (he)
    memcpy(&serv_addr.sin_addr,he->h_addr_list[0],sizeof(serv_addr.sin_addr));
  else
  {
    log(LOG_ERROR,"could not find IP address for server %s",netserver);
    close(sock_fd);
    sock_fd = -1;
    return;
  }

  log(LOG_DEBUG,"client mode, connecting to server at %s",
      inet_ntoa(serv_addr.sin_addr));

  serv_sock_fd = socket(AF_INET,SOCK_STREAM,IPPROTO_TCP);
  if (serv_sock_fd < 0)
  {
    log_perror("failed to create TCP socket");
    close(sock_fd);
    sock_fd = -1;
    return;
  }

  if (setsockopt(serv_sock_fd,IPPROTO_TCP,TCP_NODELAY,&tmp,sizeof(tmp)) < 0)
    log_perror("setsockopt");

  if (connect(serv_sock_fd,(struct sockaddr *)&serv_addr,sizeof(serv_addr)))
  {
    log_perror("could not connect to server");
    close(sock_fd);
    close(serv_sock_fd);
    sock_fd = -1;
    return;
  }

  log(LOG_DEBUG,"networking initialized, client mode");

  net_send_join();
}

void net_init_server(void)
{
  struct sockaddr_in addr;
  int t;

  sock_fd = socket(AF_INET,SOCK_DGRAM,IPPROTO_UDP);

  if (sock_fd < 0)
  {
    log_perror("failed to create UDP socket");
    return;
  }

  memset(&addr,0,sizeof(addr));

  addr.sin_family = AF_INET;
  addr.sin_port = htons(XRACER_PORT);
  addr.sin_addr.s_addr = INADDR_ANY;

  if (bind(sock_fd,(struct sockaddr *)&addr,sizeof(addr)))
  {
    log_perror("failed to bind socket to port %d",XRACER_PORT);
    close(sock_fd);
    sock_fd = -1;
    return;
  }

  memset(&serv_addr,0,sizeof(serv_addr));

  for (t=1; t<MAX_PLAYERS; t++)
    player_fds[t] = -1;

  serv_sock_fd = socket(AF_INET,SOCK_STREAM,IPPROTO_TCP);
  if (serv_sock_fd < 0)
  {
    log_perror("failed to create TCP socket");
    close(sock_fd);
    sock_fd = -1;
    return;
  }

  if (bind(serv_sock_fd,(struct sockaddr *)&addr,sizeof(addr)))
  {
    log_perror("failed to bind socket");
    close(sock_fd);
    close(serv_sock_fd);
    sock_fd = -1;
    return;
  }

  if (listen(serv_sock_fd,5))
  {
    log_perror("could not listen to incoming connections");
    close(sock_fd);
    close(serv_sock_fd);
    sock_fd = -1;
    return;
  }

  log(LOG_DEBUG,"networking initialized, server mode");
}

void net_init(void)
{
  if (mode != NETGAME_MODE)
    return;

  if (netserver)
    net_init_client();
  else
    net_init_server();
}

void net_run(void)
{
  fd_set fds;
  struct timeval tv;
  int i,j;
  static double last_packet_time = 0.0;

  if (sock_fd == -1)
    return;

  while (1)
  {
    FD_ZERO(&fds);
    FD_SET(sock_fd,&fds);
    tv.tv_sec = 0;
    tv.tv_usec = 0;

    if (select(FD_SETSIZE,&fds,NULL,NULL,&tv) < 0)
      log_perror("select");

    if (FD_ISSET(sock_fd,&fds))
      net_handle_motion_packet();
    else
      break;
  }

  FD_ZERO(&fds);
  FD_SET(serv_sock_fd,&fds);
  tv.tv_sec = 0;
  tv.tv_usec = 0;

  if (!netserver)
    for (i=1; i<MAX_PLAYERS; i++)
      if (player_fds[i] != -1)
        FD_SET(player_fds[i],&fds);

  if (select(FD_SETSIZE,&fds,NULL,NULL,&tv) < 0)
    log_perror("select");

  if (FD_ISSET(serv_sock_fd,&fds))
  {
    if (!netserver)
    {
      int tmp_fd;
      struct sockaddr_in addr;
      int tmp;

      tmp = sizeof(addr);

      tmp_fd = accept(serv_sock_fd,(struct sockaddr *)&addr,&tmp);

      if (!net_add_player(tmp_fd,addr))
        close(tmp_fd);
    }
    else
    {
      if (!net_handle_packet(0))
      {
        log(LOG_WARNING,"lost connection to server");
        close(serv_sock_fd);
        close(sock_fd);
        sock_fd = -1;
        return;
      }
    }
  }

  if (!netserver)
  {
    for (i=1; i<MAX_PLAYERS; i++)
    {
      if (player_fds[i] != -1 && FD_ISSET(player_fds[i],&fds))
      {
        if (!net_handle_packet(i))
          net_drop_player(i);
      }
    }
  }

  if (current_time - last_packet_time < PACKET_INTERVAL)
    return;

  last_packet_time = current_time;

  if (netserver)
    net_send_motion_packet(&serv_addr,local_player_nr);
  else
    for (i=1; i<MAX_PLAYERS; i++)
      for (j=0; j<MAX_PLAYERS; j++)
        if (i != j && player_fds[i] != -1 && player_fds[j] != -1)
          net_send_motion_packet(&player_addrs[i],j);
}

int net_add_player(int s,struct sockaddr_in from)
{
  int i;

  if (!game_waiting())
    return 0;
  
  for (i=1; i<MAX_PLAYERS; i++)
    if (player_fds[i] == -1)
    {
      player_fds[i] = s;
      memcpy(&player_addrs[i],&from,sizeof(from));

      return 1;
    }

  return 0;
}

void net_drop_player(int i)
{
  log(LOG_DEBUG,"%s dropped",pilot[i].pilot_name);

  close(player_fds[i]);

  player_fds[i] = -1;

  pilot[i].pilot_type = PILOT_NONE;
  free(pilot[i].pilot_name);

  net_send_player_lists();
}

void net_send_join()
{
  struct packet_join_request p;

  strcpy(p.player_name,pilot[local_player_nr].pilot_name);
  p.udp_port = udp_port;

  net_send_packet(serv_sock_fd,PACKET_JOIN_REQUEST,&p,sizeof(p));
}

void net_handle_motion_packet(void)
{
  struct packet_position_data p;

  if (read(sock_fd,&p,sizeof(p)) != sizeof(p))
  {
    log(LOG_WARNING,"partial packet");
    return;
  }

  memcpy(pilot[p.playernum].posn,p.pos,sizeof(p.pos));
  pilot[p.playernum].roll = p.roll;
  pilot[p.playernum].pitch = p.pitch;
  pilot[p.playernum].yaw = p.yaw;
}

int net_handle_packet(int plr)
{
  int s;
  unsigned char type;
  
  if (netserver)
    s = serv_sock_fd;
  else
    s = player_fds[plr];
  
  if (read(s,&type,1) < 1)
    return 0;

  switch (type)
  {
    case PACKET_JOIN_REQUEST:
      net_handle_join_request(s,plr);
      break;
    case PACKET_JOIN_ACCEPT:
      net_handle_join_accept(s);
      break;
    case PACKET_PLAYER_LIST:
      net_handle_player_list(s);
      break;
    case PACKET_BEGIN_GAME:
      game.state = GAME_COUNTING;
      game.countdown_time = current_time;
      break;
  }
  
  return 1;
}

void net_handle_join_request(int s,int plr)
{
  struct packet_join_request p;
  struct packet_join_accept r;
  
  if (netserver)
  {
    log(LOG_WARNING,"client got join request");
    return;
  }

  if (read(s,&p,sizeof(p)) != sizeof(p))
  {
    log(LOG_WARNING,"partial packet");
    return;
  }

  log(LOG_INFO,"%s joined",p.player_name);

  strcpy(r.server_motd,netservermotd);
  r.playernum = plr;
  pilot[plr].pilot_type = PILOT_REMOTE;
  pilot[plr].pilot_name = xstrdup(p.player_name);
  player_addrs[plr].sin_port = htons(p.udp_port);

  net_send_packet(s,PACKET_JOIN_ACCEPT,&r,sizeof(r));

  net_send_player_lists();
}

void net_handle_join_accept(int s)
{
  struct packet_join_accept p;

  if (!netserver)
  {
    log(LOG_WARNING,"server got join accept");
    return;
  }

  if (read(s,&p,sizeof(p)) != sizeof(p))
  {
    log(LOG_WARNING,"partial packet");
    return;
  }

  log(LOG_INFO,"Server MOTD: %s",p.server_motd);

  local_player_nr = p.playernum;
  pilot_init();
}

void net_handle_player_list(int s)
{
  struct packet_player_list p;
  int i;

  if (!netserver)
  {
    log(LOG_WARNING,"server got join accept");
    return;
  }

  if (read(s,&p,sizeof(p)) != sizeof(p))
  {
    log(LOG_WARNING,"partial packet");
    return;
  }

  for (i=0; i<MAX_PLAYERS; i++)
    if (p.player_names[i][0] && i != local_player_nr)
    {
      if (pilot[i].pilot_type == PILOT_NONE)
        log(LOG_INFO,"%s joined",p.player_names[i]);

      pilot[i].pilot_type = PILOT_REMOTE;
      pilot[i].pilot_name = xstrdup(p.player_names[i]);
    }
    else if (pilot[i].pilot_type == PILOT_REMOTE)
    {
      log(LOG_INFO,"%s dropped",pilot[i].pilot_name);

      pilot[i].pilot_type = PILOT_NONE;
      free(pilot[i].pilot_name);
    }
}

void net_send_player_lists(void)
{
  struct packet_player_list p;
  int i;

  strcpy(p.player_names[0],pilot[local_player_nr].pilot_name);

  for (i=1; i<MAX_PLAYERS; i++)
    if (player_fds[i] != -1)
      strcpy(p.player_names[i],pilot[i].pilot_name);
    else
      p.player_names[i][0] = '\0';

  for (i=1; i<MAX_PLAYERS; i++)
    if (player_fds[i] != -1)
      net_send_packet(player_fds[i],PACKET_PLAYER_LIST,&p,sizeof(p));
}

void net_send_motion_packet(struct sockaddr_in *a,int plr)
{
  struct packet_position_data p;

  p.playernum = plr;
  memcpy(p.pos,pilot[plr].posn,sizeof(p.pos));
  p.roll = pilot[plr].roll;
  p.pitch = pilot[plr].pitch;
  p.yaw = pilot[plr].yaw;

  sendto(sock_fd,&p,sizeof(p),0,(struct sockaddr *)a,sizeof(*a));
}

void net_send_packet(int s,unsigned char type,void *data,int len)
{
  int r;
  int sent = 0;

  write(s,&type,1);

  do
  {
    r = write(s,data+sent,len-sent);
    if (r > 0)
      sent += r;
    else
      return;
  }
  while (sent < len);
}

void net_begin_game(void)
{
  struct packet_begin_game p;
  int i;

  if (netserver)
    return;

  for (i=1; i<MAX_PLAYERS; i++)
    if (player_fds[i] != -1)
      net_send_packet(player_fds[i],PACKET_BEGIN_GAME,&p,sizeof(p));
}
