/* XRacer (C) 1999 Richard W.M. Jones.
 * $Id: panel.c,v 1.18 1999/09/20 20:01:42 rich Exp $
 */

#include <stdio.h>
#include <stdlib.h>

#include "xracer.h"
#include "globals.h"
#include "sound.h"

/* Fonts used for lap and rank and other things. */
static void *font = 0, *font_large = 0;

/* For the thrust- and shieldometers these are the
 * size parameters:
 *      A       B         C
 *                   __________
 *               ----          |
 * H            /              | I
 *             /            ___|
 *    _____----           --
 *   |                   /       J
 * G |                  /
 *   |_______________---
 *           F           E    D
 * Note that A + B + C = D + E + F and that G + H = I + J.
 * Parameter K is the width of each bar, and parameter
 * L is the separation between each bar. n * (K+L) = A + B + C
 */

struct meter_param
{
  int a, b, c, d, e, f, g, h, i, j, k, l;

  /* The following fields are filled in by meter_init. */
  int width, height;
  int nr_bars;
  int *top_line;		/* METER_WIDTH elements */
  int *bottom_line;		/* METER_WIDTH elements */
  GLfloat *bar_colour;		/* 4*(METER_WIDTH+1) elements */
  GLfloat *bar_fill_colour;	/* 4*(METER_WIDTH+1) elements */
};

static struct meter_param thrust_meter = {
  20, 70, 30, 10, 40, 70,
  20, 80, 15, 85,
  10, 5
};

static struct meter_param shield_meter = {
  20, 70, 30, 10, 40, 70,
  -10, -40, -12, -38,
  10, 5
};

/* Initialize meter structures. */
static void
meter_init (struct meter_param *meter)
{
  int x;

  meter->width = meter->a + meter->b + meter->c;
  meter->height = meter->g + meter->h;
  meter->nr_bars = meter->width / (meter->k + meter->l);
  meter->top_line = xmalloc (sizeof (int) * meter->width);
  meter->bottom_line = xmalloc (sizeof (int) * meter->width);
  meter->bar_colour = xmalloc (4 * sizeof (GLfloat) * (meter->width+1));
  meter->bar_fill_colour = xmalloc (4 * sizeof (GLfloat) * (meter->width+1));

  for (x = 0; x < meter->a; ++x)
    meter->top_line[x] = meter->g;

  for (x = meter->a; x < meter->a + meter->b; ++x)
    meter->top_line[x] =
      meter->g +
      (1 + sin (- M_PI/2 + M_PI * (x - meter->a) / (GLfloat) meter->b))
      * (meter->h/2.);

  for (x = meter->a + meter->b; x < meter->width; ++x)
    meter->top_line[x] = meter->height;

  for (x = 0; x < meter->f; ++x)
    meter->bottom_line[x] = 0;

  for (x = meter->f; x < meter->f + meter->e; ++x)
    meter->bottom_line[x] =
      (1 + sin (- M_PI/2 + M_PI * (x - meter->f) / (GLfloat) meter->e))
      * (meter->j/2.);

  for (x = meter->f + meter->e; x < meter->width; ++x)
    meter->bottom_line[x] = meter->j;

  for (x = 0; x <= meter->width; ++x)
    {
      meter->bar_colour[x*4] = 1;
      meter->bar_colour[x*4+1] = 0;
      meter->bar_colour[x*4+2] = 1;
      meter->bar_colour[x*4+3] = 1;
      meter->bar_fill_colour[x*4] = 1;
      meter->bar_fill_colour[x*4+1] = (GLfloat) x / meter->width;
      meter->bar_fill_colour[x*4+2] = ((GLfloat) x / meter->width)
	                            * ((GLfloat) x / meter->width);
      meter->bar_fill_colour[x*4+3] = 0.8;
    }
}

static void
fill_incomplete_bar (struct meter_param *meter, int i, int w, int x, int y)
{
  glBegin (GL_POLYGON);
  glColor4fv (&meter->bar_fill_colour[i*4]);
  glVertex2i (SX(x + i), SY(y + meter->bottom_line[i]));
  glColor4fv (&meter->bar_fill_colour[(i + w - 1)*4]);
  glVertex2i (SX(x + i + w - 1), SY(y + meter->bottom_line[i + w - 1]));
  glColor4fv (&meter->bar_fill_colour[(i + w - 1)*4]);
  glVertex2i (SX(x + i + w - 1), SY(y + meter->top_line[i + w - 1]));
  glColor4fv (&meter->bar_fill_colour[i*4]);
  glVertex2i (SX(x + i), SY(y + meter->top_line[i]));
  glEnd ();
}

static void
fill_bar (struct meter_param *meter, int i, int x, int y)
{
  fill_incomplete_bar (meter, i, meter->k, x, y);
}

static void
outline_bar (struct meter_param *meter, int i, int x, int y)
{
  glBegin (GL_LINE_LOOP);
  glColor4fv (&meter->bar_colour[i*4]);
  glVertex2i (SX(x + i), SY(y + meter->bottom_line[i]));
  glColor4fv (&meter->bar_colour[(i + meter->k - 1)*4]);
  glVertex2i (SX(x + i + meter->k - 1), SY(y + meter->bottom_line[i + meter->k - 1]));
  glColor4fv (&meter->bar_colour[(i + meter->k - 1)*4]);
  glVertex2i (SX(x + i + meter->k - 1), SY(y + meter->top_line[i + meter->k - 1]));
  glColor4fv (&meter->bar_colour[i*4]);
  glVertex2i (SX(x + i), SY(y + meter->top_line[i]));
  glEnd ();
}

/* Draw the generic meter. */
static void
draw_meter (struct meter_param *meter, GLfloat v, int x, int y)
{
  int i, vi = v * meter->width;

  /* XXX We should be able to make much more use of
   * display lists in this function.
   */

  glShadeModel (GL_SMOOTH);

  for (i = 0; i < vi; i += meter->k + meter->l)
    {
      if (i + meter->k + meter->l < vi)
	fill_bar (meter, i, x, y);
      else
	fill_incomplete_bar (meter,
			     i,
			     (GLfloat) meter->k * (vi - i)
			     / (GLfloat) (meter->k + meter->l),
			     x, y);
    }

  for (i = 0; i < meter->width; i += meter->k + meter->l)
    outline_bar (meter, i, x, y);

  glShadeModel (GL_FLAT);
}

/* Draw the thrust meter. */
static void
thrustometer ()
{
  GLfloat v = pilot[local_player_nr].thrust / craft[local_player_nr].max_thrust;

  draw_meter (&thrust_meter, v, refwidth - 140, 150);
}

/* Draw the shield meter. */
static void
shieldometer ()
{
  draw_meter (&shield_meter, pilot[local_player_nr].shield, refwidth - 140, 130);
}

/* Draw the speedometer. */
static void
speedometer ()
{
  static char buffer [16];

  pilot[local_player_nr].speed =
    magnitude (pilot[local_player_nr].momentum[0]) * 500.;

  /* Since we calculate pilot speed here, we might as well
   * use this opportunity to communicate the speed with the
   * sound driver. Not clean, but there you go.
   */
  sound_set_craft_velocity (pilot[local_player_nr].speed);

  sprintf (buffer, "%3d.%d", pilot[local_player_nr].speed / 10,
    pilot[local_player_nr].speed % 10);
  draw_lcd_string (buffer, refwidth - 200, 16, 48);
}

/* Display the single main time. */
static void
display_time (const char *lap_time)
{
  draw_lcd_string (lap_time, 16, 16, 48);
}

static void
display_lap_time (int lap, const char *lap_time)
{
  static char buffer[64];

  sprintf (buffer, "Lap %d: %s", lap+1, lap_time);
  draw_text (font, buffer, 16, refheight - (80 + (game.nr_laps - lap) * 16));
}

/* Draw the time display. */
static void
timeometer ()
{
  int i;

  display_time (game_get_time_s ());

  for (i = 0; i < 10; ++i)
    if (game.lap_time[i] >= 0)
      display_lap_time (i, game.lap_time_s[i]);
}

/* Draw the lap display. */
static void
lapometer ()
{
  static char buffer[64];
  int x = refwidth - 32, y = 32;

  sprintf (buffer,
	   "Lap %d of %d",
	   pilot[local_player_nr].current_lap,
	   game.nr_laps);
  x -= get_text_width (font, buffer);
  draw_text (font, buffer, x, y);
}

void
calculate_player_rank (int cr, int *rankr, int *nr_playersr)
{
  int rank = 1, nr_players = 0, i;

  /* Work out the player's rank. */
  for (i = 0; i < MAX_PLAYERS; ++i)
    {
      if (pilot[i].pilot_type != PILOT_NONE)
	{
	  if (i != cr)
	    {
	      if (pilot[i].current_lap > pilot[cr].current_lap ||
		  (pilot[i].current_lap == pilot[cr].current_lap &&
		   pilot[i].greatest_segment > pilot[cr].greatest_segment))
		rank ++;
	    }
	  nr_players ++;
	}
    }

  *rankr = rank;
  *nr_playersr = nr_players;
}

/* Draw the player's rank (position). */
static void
rankometer ()
{
  static char buffer[64];
  int x = refwidth - 32, y = 96, rank, nr_players;

  calculate_player_rank (local_player_nr, &rank, &nr_players);

  sprintf (buffer, "%d/%d", rank, nr_players);
  x -= get_text_width (font_large, buffer);
  draw_text (font_large, buffer, x, y);
}

static void
fps_display ()
{
  static char buffer[7];
  double fps;

  fps = 1 / frame_time;
  sprintf (buffer, "%3.1f", fps);
  draw_lcd_string (buffer, refwidth - 100, refheight - 144, 24);
}

/* Powerups are stored as textures, mapped onto a quad.
 * We use the display list when displaying the powerup.
 */
static int powerup_tex[MAX_POWERUP+1];

static int powerup_dl[MAX_POWERUP+1];

/* If there is a powerup, display it. */
static void
powerup_display ()
{
  glEnable (GL_TEXTURE_2D);
  glPushMatrix();
  glScalef(textscale_x, textscale_y, 0);
  glCallList (powerup_dl[pilot[local_player_nr].powerup]);
  glPopMatrix();
  glDisable (GL_TEXTURE_2D);
}

/* Initialize powerups. */
static void
powerup_init ()
{
  int i, x, y;

#if MAX_POWERUP > 0
  powerup_tex[1] = load_texture_with_alpha ("images/powerup-faster.jpg",
					    "images/powerup-faster-alpha.jpg");
#endif
#if MAX_POWERUP > 1
  powerup_tex[2] = load_texture_with_alpha ("images/powerup-energy.jpg",
					    "images/powerup-energy-alpha.jpg");
#endif
#if MAX_POWERUP > 2
  powerup_tex[3] = load_texture_with_alpha ("images/powerup-shield.jpg",
					    "images/powerup-shield-alpha.jpg");
#endif
#if MAX_POWERUP > 3
  powerup_tex[4] = load_texture_with_alpha ("images/powerup-autopilot.jpg",
					    "images/powerup-autopilot-alpha.jpg");
#endif
#if MAX_POWERUP > 4
  powerup_tex[5] = load_texture_with_alpha ("images/powerup-cannon.jpg",
					    "images/powerup-cannon-alpha.jpg");
#endif
#if MAX_POWERUP > 5
  powerup_tex[6] = load_texture_with_alpha ("images/powerup-missile.jpg",
					    "images/powerup-missile-alpha.jpg");
#endif

  x = refwidth/2 - 32;
  y = refheight - 64 - 32;

  for (i = 1; i <= MAX_POWERUP; ++i)
    {
      /* Generate the display list. */
      powerup_dl[i] = glGenLists (1);

      glNewList (powerup_dl[i], GL_COMPILE);

      glBindTexture (GL_TEXTURE_2D, powerup_tex[i]);

      glBegin (GL_QUADS);
      /* glColor4f (0, 0, 0, 1); */
      glTexCoord2f (0, 0);
      glVertex2i (x, y);
      glTexCoord2f (0, 1);
      glVertex2i (x, y + 64);
      glTexCoord2f (1, 1);
      glVertex2i (x + 64, y + 64);
      glTexCoord2f (1, 0);
      glVertex2i (x + 64, y);
      glEnd ();

      glEndList ();
    }
}

void
panel_display ()
{
  GLfloat saved_color[4];
  
  glGetFloatv(GL_CURRENT_COLOR,saved_color);
  
  thrustometer ();
  shieldometer ();
  speedometer ();
  timeometer ();
  lapometer ();
  if (mode != TIME_TRIAL_MODE) rankometer ();

  glColor3fv(saved_color);

  if (pilot[local_player_nr].powerup != NO_POWERUP) powerup_display ();

  if (control[local_player_nr].show_fps) fps_display ();
}

void
panel_init ()
{
  meter_init (&thrust_meter);
  meter_init (&shield_meter);
  powerup_init ();

  font = find_font ("crillee", 14);
  log_assert (font != 0);

  font_large = find_font ("crillee", 48);
  log_assert (font_large != 0);
}
