/* XRacer (C) 1999 Richard W.M. Jones.
 * $Id: pilot.c,v 1.27 1999/07/31 16:53:26 rich Exp $
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include "xracer.h"
#include "xracer-net.h"
#include "profile.h"
#include "globals.h"

/* This matrix gives ideal distance between each point
 * in the craft, maintained by springs.
 */
#if 0
GLfloat spring_dist[3][3] = {
  { 0,   0.5, 0.5 },
  { 0.5, 0,   0.2 },
  { 0.5, 0.2, 0   }
};
#else
GLfloat spring_dist[3][3] = {
  { 0,   2.5, 2.5 },
  { 2.5, 0,   1.0 },
  { 2.5, 1.0, 0   }
};
#endif

#define GRAVITY 0.1
#define SPRING_DAMPING 0.1
#define MOMENTUM_DAMPING 0.9

#define COLLISION_DISTANCE 3
#define COLLISION_FORCE_FACTOR 0.08

struct pilot pilot[8];

struct craft craft[8];

int player_in_view = 0;
int mode = ARCADE_MODE;
int local_player_nr = 0; /* Not used yet, for netgames... */

void
pilot_init ()
{
  int cr;

  pilot[local_player_nr].pilot_name = plr_name;

  for (cr = 0; cr < MAX_PLAYERS; ++cr)
    {
      GLfloat v[3];

      /* XXX Need to work out these positions from the track itself. */
      if (cr > 0)
	{
	  v[0] = ((cr & 1) * 4 - 2);
	  v[1] = cr * 30;
	  v[2] = cr * 2;
	}
      else
	v[0] = v[1] = v[2] = 0;

      /* XXX Craft always point towards +Y at the moment. */
      pilot[cr].posn[0][0] = v[0];
      pilot[cr].posn[0][1] = v[1] + 2.5;
      pilot[cr].posn[0][2] = v[2];
      pilot[cr].posn[1][0] = v[0] - 0.5;
      pilot[cr].posn[1][1] = v[1];
      pilot[cr].posn[1][2] = v[2];
      pilot[cr].posn[2][0] = v[0] + 0.5;
      pilot[cr].posn[2][1] = v[1];
      pilot[cr].posn[2][2] = v[2];

    if (cr != local_player_nr)
    {
      if (mode == ARCADE_MODE)
        pilot[cr].pilot_type = PILOT_ROBOT;
      else
        pilot[cr].pilot_type = PILOT_NONE;
    }
    else
      pilot[cr].pilot_type = PILOT_LOCAL;

      memset (pilot[cr].momentum, 0, sizeof pilot[cr].momentum);
      pilot[cr].thrust = 0;
      pilot[cr].faster = 1;
      pilot[cr].seg[0] = pilot[cr].seg[1] = pilot[cr].seg[2] = 0;
      pilot[cr].roll = pilot[cr].pitch = pilot[cr].yaw = 0;
      pilot[cr].shield = 1;
      pilot[cr].has_external_shield = 0;
      pilot[cr].powerup = NO_POWERUP;
      pilot[cr].greatest_segment = 0;
      pilot[cr].current_lap = 1;
    }

  for (cr = 0; cr < MAX_PLAYERS; ++cr)
    {
      craft[cr].accel = 0.001;
      craft[cr].brake = 0.002;

#if 0
      craft[cr].roll_div = 8192;
#else
      craft[cr].roll_div = 2048;
#endif
      craft[cr].yaw_div = 8192;
      craft[cr].pitch_div = 4096;

      if (pilot[cr].pilot_type == PILOT_ROBOT)
	{
	  /* Robot players. */
	  
	  if( difficulty == EASY)
	  {
	  	craft[cr].max_thrust = 0.05 +  (cr * 0.03);
	  }
	  else if ( difficulty == MEDIUM )
	  {
		craft[cr].max_thrust = 0.10 + (cr * 0.03);
	  }
	  else if ( difficulty == HARD)
	  {
		craft[cr].max_thrust = 0.15 +  (cr * 0.03);
	  } 
	  craft[cr].faster_mult = 1.5;
	}
      else if (pilot[cr].pilot_type == PILOT_LOCAL)
	{
	  /* Local player. */
	  craft[cr].max_thrust = 0.30;
	  craft[cr].faster_mult = 1.75;
	}
    }
}

/* Calculate up vector. */
static inline void
calculate_pilot_up (int cr, GLfloat *up)
{
  GLfloat back_line[3], left_side[3];

  /* Up is a vector at right angles to the back line and the
   * left and right sides. Use a cross product to calculate this.
   */
  back_line[0] = pilot[cr].posn[2][0] - pilot[cr].posn[1][0];
  back_line[1] = pilot[cr].posn[2][1] - pilot[cr].posn[1][1];
  back_line[2] = pilot[cr].posn[2][2] - pilot[cr].posn[1][2];
  left_side[0] = pilot[cr].posn[0][0] - pilot[cr].posn[1][0];
  left_side[1] = pilot[cr].posn[0][1] - pilot[cr].posn[1][1];
  left_side[2] = pilot[cr].posn[0][2] - pilot[cr].posn[1][2];
  cross_product (back_line, left_side, up);
}

static inline void
obey_player_commands (int cr,int pass)
{
  int control_x = 0, control_y = 0;

  PROFILE_START (0);
  if (control[cr].accelerate)
  {
    pilot[cr].thrust += craft[cr].accel;
    if (pilot[cr].thrust > craft[cr].max_thrust)
      pilot[cr].thrust = craft[cr].max_thrust;
    if (pass == 1)
      control[cr].accelerate = (control[cr].mouse_flag ? control[cr].mouse_accelerate : 0);
  }
  if (control[cr].brake)
  {
    pilot[cr].thrust -= craft[cr].brake;
    if (pilot[cr].thrust < 0) pilot[cr].thrust = 0;
    if (pass == 1)
      control[cr].brake = (control[cr].mouse_flag ? control[cr].mouse_decelerate : 0);
  }

  /* If we are held on the start line for the start of the game,
   * then the game_held () function returns true, and we hold
   * the craft where it is.
   */
  if (game_held ()) pilot[cr].thrust = 0;

  /* Work out our current control position, taking controls
   * in the following order: autopilot, keyboard, mouse, joystick.
   */
  if (control[cr].autopilot_flag)
    {
      control_x = control[cr].autopilot_x;
      control_y = control[cr].autopilot_y;
      control[cr].autopilot_flag = 0;
    }
  else if (control[cr].keyboard_flag)
    {
      control_x = control[cr].keyboard_x;
      control_y = control[cr].keyboard_y;
      control[cr].keyboard_flag = 0;
    }
  else if (control[cr].mouse_flag)
    {
      control_x = control[cr].mouse_sensitivity * control[cr].mouse_x;
      control_y = control[cr].mouse_pitch * control[cr].mouse_sensitivity * control[cr].mouse_y;
    }
  else if (control[cr].joystick_flag)
    {
      control_x = control[cr].joystick_x;
      control_y = control[cr].joystick_y;
    }
  PROFILE_STOP (0);

  PROFILE_START (1);
  /* Save x position - we use it in precipitation.c */
  pilot[cr].control_x = control_x;

  /* Banking left and right.
   * NB. x < 0  =>  bank left,
   *     x > 0  =>  bank right.
   */
  pilot[cr].roll = (-control_x / craft[cr].roll_div);
  pilot[cr].yaw = (control_x / craft[cr].yaw_div);

  /* Nose up and down.
   * NB. y < 0  =>  nose down,
   *     y > 0  =>  nose up.
   */
  pilot[cr].pitch = (control_y / craft[cr].pitch_div);
  PROFILE_STOP (1);
}

#define EXTERNAL_SHIELD_TIME 10	/* How many seconds does shield powerup last */
#define AUTOPILOT_TIME 10	/* How many seconds does autopilot last */

/* Various powerup related stuff. */
static inline void
deal_with_powerups (int cr)
{
  /* External shield. */
  if (pilot[cr].has_external_shield &&
    current_time - pilot[cr].external_shield_start_time > EXTERNAL_SHIELD_TIME)
    {
      pilot[cr].has_external_shield = 0;
    }

  /* Autopilot. */
  if (pilot[cr].has_autopilot &&
    current_time - pilot[cr].autopilot_start_time > AUTOPILOT_TIME)
    {
      pilot[cr].has_autopilot = 0;
    }
}

#define SEGMENT_ADJACENT_BY_1(a,b) ({ int _t = (a)-(b); _t = _t >= 0 ? _t : -_t; _t <= 1 || _t == track->nr_segments-1; })
#define SEGMENT_ADJACENT_BY_2(a,b) ({ int _t = (a)-(b); _t = _t >= 0 ? _t : -_t; _t <= 2 || _t >= track->nr_segments-2; })

/* Update pilot position. */
static inline void
pilot_update_n (GLfloat old_posn[8][3][3], int old_seg[8][3],
		int cr)
{
  int p;
  GLfloat c[3][30];		/* XXX Assumes <= 30 sides per segment max. */
  GLfloat mp[3], thrust_vec[3];
  GLfloat roll_m[16], pitch_m[16], yaw_m[16], up[3], backline[3];

  PROFILE_START (2);
  /* In thrust calculation, we require the following
   * vector which runs from the midpoint of the back
   * line to the nose of the craft.
   */
  midpoint (old_posn[cr][1], old_posn[cr][2], mp);
  thrust_vec[0] = old_posn[cr][0][0] - mp[0];
  thrust_vec[1] = old_posn[cr][0][1] - mp[1];
  thrust_vec[2] = old_posn[cr][0][2] - mp[2];
  normalize (thrust_vec, thrust_vec);

  /* Compute midpoint of the craft. */
  midpoint (mp, old_posn[cr][0], mp);
  PROFILE_STOP (2);

  PROFILE_START (3);
  /* Compute rotation matrices for roll, pitch and yaw. */
  calculate_pilot_up (cr, up);

  make_rotation_matrix (pilot[cr].yaw, up[0], up[1], up[2], yaw_m);

  backline[0] = old_posn[cr][1][0] - old_posn[cr][2][0];
  backline[1] = old_posn[cr][1][1] - old_posn[cr][2][1];
  backline[2] = old_posn[cr][1][2] - old_posn[cr][2][2];

  make_rotation_matrix (pilot[cr].pitch,
			backline[0], backline[1], backline[2],
			pitch_m);

  make_rotation_matrix (pilot[cr].roll,
			thrust_vec[0], thrust_vec[1], thrust_vec[2],
			roll_m);
  PROFILE_STOP (3);

  PROFILE_START (4);
  /* Precalculate the distance from each point on the
   * craft to each side of the track.
   */
  for (p = 0; p < 3; ++p)
    {
      int s;
      struct segment *segment = &track->segments[old_seg[cr][p]];

      for (s = 0; s < segment->nr_sides; ++s)
	c[p][s]
	  = distance_point_to_plane (segment->sides[s].plane, old_posn[cr][p]);
    }
  PROFILE_STOP (4);

  PROFILE_START (5);
  /* Any triggers? */
  /* XXX Not quite right - this is slightly dependent on
   * what the texture looks like.
   */
  p = 0;
  {
    int s = old_seg[cr][p];

    if (((s % 10) == 4 || (s % 10) == 5) &&	/* XXX */
	(track->segments[s].trigger & (TRIGGER_LEFT_FAST|
				       TRIGGER_LEFT_FASTER|
				       TRIGGER_LEFT_POWERUP|
				       TRIGGER_RIGHT_FAST|
				       TRIGGER_RIGHT_FASTER|
				       TRIGGER_RIGHT_POWERUP)))
      {
	GLfloat c1 = distance_point_to_plane (track->segments[s].triggers[0],
					      old_posn[cr][p]);
	GLfloat c2 = distance_point_to_plane (track->segments[s].triggers[1],
					      old_posn[cr][p]);
	GLfloat c3 = distance_point_to_plane (track->segments[s].triggers[2],
					      old_posn[cr][p]);

	/*log (LOG_DEBUG, "c1 = %g, c2 = %g, c3 = %g", c1, c2, c3);*/

	if (c1 <= 0 && c2 > 0)
	  {
	    if ((track->segments[s].trigger & TRIGGER_LEFT_FAST))
	      game_trigger_fast (cr);
	    else if ((track->segments[s].trigger & TRIGGER_LEFT_FASTER))
	      game_trigger_faster (cr);
	    else if ((track->segments[s].trigger & TRIGGER_LEFT_POWERUP))
	      game_trigger_powerup (cr);
	  }
	else if (c2 <= 0 && c3 > 0)
	  {
	    if ((track->segments[s].trigger & TRIGGER_RIGHT_FAST))
	      game_trigger_fast (cr);
	    else if ((track->segments[s].trigger & TRIGGER_RIGHT_FASTER))
	      game_trigger_faster (cr);
	    else if ((track->segments[s].trigger & TRIGGER_RIGHT_POWERUP))
	      game_trigger_powerup (cr);
	  }
      }
  }
  PROFILE_STOP (5);

  PROFILE_START (6);
  /* First check if any point has hit one of the sides.
   * If so, then we kill all momentum on all points before
   * doing anything else.
   */
  for (p = 0; p < 3; ++p)
    {
      int s;
      struct segment *segment;

      segment = &track->segments[old_seg[cr][p]];
      for (s = 0; s < segment->nr_sides; ++s)
	{
	  /* Have we passed through this side? */
	  if (
#if 1
	      c[p][s] < 0
#else
	      /* This is an ugly hack (XXX). If the side is the track
	       * bottom, then be lenient.
	       */
	      (segment->sides[s].levitate_factor > 0 &&
	       c[p][s] < -2) ||
	      (segment->sides[s].levitate_factor == 0 &&
	       c[p][s] < 0)
#endif
	      )
	    {
	      game_dunk (cr);
	      goto end_hit_test;
	    }
	}
    }
end_hit_test:
  PROFILE_STOP (6);

  /* We update each of the three points in turn. */
  for (p = 0; p < 3; ++p)
    {
      GLfloat accel[3] = {0, 0, 0}, v[4], w[4], d;
      int s, seg, next_seg;
      struct segment *segment, *next_segment;

      PROFILE_START (7);
      /* The forces acting on each point are:
       *
       * - gravity, straight down
       * - thrust, along a line from midpoint of back to nose
       * - 2 springs, pulling and pushing towards other 2 points
       * - n sides, forces induced by hitting sides of track
       * - contributions from roll, pitch and yaw matrices
       * - collisions with other players, if any
       *
       * We add up all these forces to produce a
       * new acceleration. The acceleration is
       * then added to the momentum, and the momentum
       * is added to the current position.
       */
      /* Gravity. */
      accel[2] -= GRAVITY;

      /* Sides. */
      segment = &track->segments[seg = old_seg[cr][p]];
      for (s = 0; s < segment->nr_sides; ++s)
	{
	  GLfloat force, n[4];

	  /* Compute the magnitude of the force to exert.
	   */
	  if (segment->sides[s].levitate_factor > 0 &&
	      c[p][s] < segment->sides[s].levitate_dist)
	    {
	      GLfloat t = (segment->sides[s].levitate_dist - c[p][s]) /
		segment->sides[s].levitate_dist; /* [0..1] */

	      force = segment->sides[s].levitate_damping * t * t;

	      /* Force is parallel to normal vector. Normal
	       * vector points inwards.
	       */
	      unit_normal_to_side (segment->sides[s].plane, n);
	      accel[0] += n[0] * force;
	      accel[1] += n[1] * force;
	      accel[2] += n[2] * force;
#if 0
	      printf ("contribution from side %d: (%g, %g, %g)\n",
		      s, n[0]*force,n[1]*force,n[2]*force);
#endif
	    }
	}
      PROFILE_STOP (7);

      PROFILE_START (8);
      /* Springs. */
      for (s = 0; s <= 2; ++s)
	if (s != p)
	  {
	    GLfloat ideal, dx, dy, dz, actual, n[3], force;

	    /* Ideal distance. */
	    ideal = spring_dist[p][s];

	    /* Actual distance. */
	    dx = old_posn[cr][p][0] - old_posn[cr][s][0];
	    dy = old_posn[cr][p][1] - old_posn[cr][s][1];
	    dz = old_posn[cr][p][2] - old_posn[cr][s][2];
	    actual = sqrt (dx*dx + dy*dy + dz*dz);

	    /* Compute normal vector of force. */
	    n[0] = old_posn[cr][p][0] - old_posn[cr][s][0];
	    n[1] = old_posn[cr][p][1] - old_posn[cr][s][1];
	    n[2] = old_posn[cr][p][2] - old_posn[cr][s][2];
	    normalize (n, n);

	    /* Calculate force. */
	    force = SPRING_DAMPING * (ideal - actual);

	    /* Apply force. */
	    accel[0] += n[0] * force;
	    accel[1] += n[1] * force;
	    accel[2] += n[2] * force;
	  }
      PROFILE_STOP (8);

      PROFILE_START (9);
    if (mode != TIME_TRIAL_MODE)
	{
	  int ocr, op, seg = old_seg[cr][p];
	  GLfloat dist, v[3], t;

	  /* Collisions.
	   *
	   * Each point on each player is compared against each
	   * point on each other player. If two points are close
	   * to each other, then a force is generated which acts
	   * in a direction so as to push the two points away from
	   * each other.
	   *
	   * The aim of the code below is to ensure that we can
	   * compare all 24 points with all other 23 points as
	   * quickly as possible. To do this, we use the segment
	   * number as a general guide. Only if two points have
	   * adjacent segment numbers do we go further and examine
	   * the actual distance between points. Additionally, we
	   * also know that each craft has three points which are
	   * likely to be in the same or adjacent segments. Therefore
	   * we only need to test the segment number in each craft
	   * once.
	   */
      for (ocr = 0; ocr < MAX_PLAYERS; ++ocr)
      {
        if (pilot[cr].pilot_type == PILOT_NONE
            || pilot[ocr].pilot_type == PILOT_NONE)
          continue;

	    if (ocr != cr)
	      {
		/* Craft in same or adjacent by 1 or 2 segments? */
		if (SEGMENT_ADJACENT_BY_2 (seg, old_seg[ocr][0]))
		  {
		    /* Check each point in the other craft. Are any
		     * in the same or adjacent segment?
		     */
		    for (op = 0; op < 3; ++op)
		      {
			if (SEGMENT_ADJACENT_BY_1 (seg, old_seg[ocr][op]))
			  {
			    /* Do the expensive distance computation for
			     * these two points.
			     */
			    dist = distance (old_posn[cr][p],
					     old_posn[ocr][op]);

			    if (dist <= COLLISION_DISTANCE)
			      {
				/* t = [0..1], where 1 = min distance. */
				t = (COLLISION_DISTANCE - dist)
				  / COLLISION_DISTANCE;

				/* Compute direction of force. */
				v[0] = old_posn[cr][p][0]
				  - old_posn[ocr][op][0];
				v[1] = old_posn[cr][p][1]
				  - old_posn[ocr][op][1];
				v[2] = old_posn[cr][p][2]
				  - old_posn[ocr][op][2];
				normalize (v, v);

				/* Compute magnitude of force. */
				v[0] *= COLLISION_FORCE_FACTOR * t * t;
				v[1] *= COLLISION_FORCE_FACTOR * t * t;
				v[2] *= COLLISION_FORCE_FACTOR * t * t;

				/* Add to acceleration. */
				accel[0] += v[0];
				accel[1] += v[1];
				accel[2] += v[2];

				game_crash (cr);
			      }
			  }
		      }
		  }
	      }
	}
    }
      PROFILE_STOP (9);

      PROFILE_START (10);
      /* Thrust. */
      accel[0] += thrust_vec[0] * pilot[cr].thrust * pilot[cr].faster;
      accel[1] += thrust_vec[1] * pilot[cr].thrust * pilot[cr].faster;
      accel[2] += thrust_vec[2] * pilot[cr].thrust * pilot[cr].faster;

      /* Compute new momentum. */
      pilot[cr].momentum[p][0] += accel[0];
      pilot[cr].momentum[p][1] += accel[1];
      pilot[cr].momentum[p][2] += accel[2];

      /* Compute new position. */
      pilot[cr].posn[p][0] += pilot[cr].momentum[p][0];
      pilot[cr].posn[p][1] += pilot[cr].momentum[p][1];
      pilot[cr].posn[p][2] += pilot[cr].momentum[p][2];

      /* Damp momentum. */
      pilot[cr].momentum[p][0] *= MOMENTUM_DAMPING;
      pilot[cr].momentum[p][1] *= MOMENTUM_DAMPING;
      pilot[cr].momentum[p][2] *= MOMENTUM_DAMPING;

      /* Factor in roll, pitch and yaw. */
      v[0] = pilot[cr].posn[p][0] - mp[0];
      v[1] = pilot[cr].posn[p][1] - mp[1];
      v[2] = pilot[cr].posn[p][2] - mp[2];
      v[3] = 0;
      matrix_vector_multiply (roll_m, v, w);
      matrix_vector_multiply (yaw_m, w, v);
      matrix_vector_multiply (pitch_m, v, w);
      pilot[cr].posn[p][0] = w[0] + mp[0];
      pilot[cr].posn[p][1] = w[1] + mp[1];
      pilot[cr].posn[p][2] = w[2] + mp[2];
      PROFILE_STOP (10);

      PROFILE_START (11);
      /* Gone through enter plane of next segment? */
      next_seg = (seg+1) % track->nr_segments;
      next_segment = &track->segments[next_seg];
      d = next_segment->enter[0] * pilot[cr].posn[p][0]
	+ next_segment->enter[1] * pilot[cr].posn[p][1]
	+ next_segment->enter[2] * pilot[cr].posn[p][2]
	+ next_segment->enter[3];
      if (d >= 0)
	{
	  pilot[cr].seg[p] = next_seg;

	  if (p == 0)
	    {
	      /* Update greatest_segment / lap code. */
	      if (pilot[cr].seg[0] == pilot[cr].greatest_segment + 1)
		{
		  pilot[cr].greatest_segment ++;
		}
	      else if (pilot[cr].seg[0] == 0 &&
		       pilot[cr].greatest_segment == track->nr_segments-1)
		{
		  pilot[cr].greatest_segment = 0;
		  pilot[cr].current_lap ++;

		  if (pilot[cr].current_lap <= game.nr_laps)
		    game_new_lap (cr);
		  else
		    game_end_game (cr);
		}

	      if (cr == local_player_nr)
		{
		  /* Any triggers for this segment? */
		  switch (next_segment->trigger)
		    {
		    default:
		      break;

		    case TRIGGER_START:
		      game_trigger_start ();
		      break;
		    }
		}
	    }
	}
      else
	{
	  /* Gone through enter plane back into previous segment? */
	  d = segment->enter[0] * pilot[cr].posn[p][0]
	    + segment->enter[1] * pilot[cr].posn[p][1]
	    + segment->enter[2] * pilot[cr].posn[p][2]
	    + segment->enter[3];
	  if (d < 0)
	    {
	      pilot[cr].seg[p] --;
	      if (pilot[cr].seg[p] < 0) pilot[cr].seg[p] += track->nr_segments;
	    }
	}
      PROFILE_STOP (11);
    }
}

void
pilot_update ()
{
  int cr;
  int frame_skip;
  static double frame_partial = 0.0;
  GLfloat old_posn[8][3][3];
  int old_seg[8][3];

#define TARGET_FPS 72.0

  frame_skip = (int)((frame_time / (1.0 / TARGET_FPS)) + frame_partial);
  frame_partial = (frame_time / (1.0 / TARGET_FPS)) + frame_partial;
  frame_skip = (int)frame_partial;
  frame_partial -= (double)frame_skip;

  net_run();

  while (frame_skip)
  {
    /* Save old position. */
    for (cr = 0; cr < MAX_PLAYERS; ++cr)
    {
      if (pilot[cr].pilot_type == PILOT_NONE)
        continue;
      
      memcpy (old_posn[cr], pilot[cr].posn, sizeof pilot[cr].posn);
      memcpy (old_seg[cr], pilot[cr].seg, sizeof pilot[cr].seg);
    }

    for (cr = 0; cr < MAX_PLAYERS; ++cr)
    {
      if (pilot[cr].pilot_type == PILOT_NONE)
        continue;

      if (pilot[cr].pilot_type == PILOT_ROBOT
          || pilot[local_player_nr].has_autopilot
	  || game_ended ())
        autopilot(cr);
      deal_with_powerups(cr);
      obey_player_commands(cr,frame_skip);
      pilot_update_n(old_posn,old_seg,cr);
    }

    frame_skip--;
  }
}
