/* XRacer (C) 1999 Richard W.M. Jones.
 * $Id: sound.c,v 1.10 1999/09/12 13:16:24 rich Exp $
 */

/* This is the second attempt at writing a sound driver for
 * XRacer.
 *
 * The sound driver is divided into two parts: the XRacer
 * parent process, and the sound process, which is forked
 * off as a child at initialization.
 *
 * I have tried to address various concerns. (1) If the
 * sound process dies, then the parent process should be
 * able to detect this cleanly and disable sound. (2) The
 * sound subsystem should be as efficient as possible.
 * (3) The sound subsystem should be able to deal with
 * variable quantities (eg. velocity / wind noise)
 * efficiently and smoothly.
 *
 * The XRacer parent process has two methods to communicate
 * with the sound process: using shared memory and using
 * a unidirectional command pipe. Shared memory is used
 * for communicating quantities which change frequently
 * including:
 *
 * - craft velocity
 * - echo parameters
 * - mixer parameters (volume of each channel)
 *
 * The unidirectional command pipe is used to send commands
 * to the sound process, for:
 *
 * - initiating playback of a sample
 * - enabling and disabling sound
 * - changing complex sound configuration (eg. bit rate)
 *
 * The advantage of having a command pipe between the parent
 * and the child is that we can detect an EPIPE error when
 * the child dies. Similarly, when the parent process exits
 * or dies, the child reads 0 bytes, so it can cleanly exit.
 *
 * A note on the structure of the source files.
 *
 * sound.c is the parent process,
 * sound_process.c is the independent child process which
 *   actually generates the sounds,
 * sound.h defines the interface between XRacer and the
 *   parent part of the sound subsystem,
 * sound_process.h defines the interface between the parent
 *   part of the sound subsystem and the child sound process,
 * os_sound.c is the OS-dependent part of the sound driver,
 * os_sound.h is the interface to the OS-dependent routines.
 */

#include "config.h"

#ifdef __CYGWIN32__
#include <windows.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <errno.h>
#include <string.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <sys/mman.h>

#include "xracer.h"
#include "sound.h"
#include "sound_process.h"
#include "os_sound.h"
#include "globals.h"

#define SOUND_OK (sound_state != SOUND_STATE_DEAD)

int sound_state = SOUND_STATE_DEAD;
static struct sound_shm *shm;

#if SOUND_USE_FORK
static int pipe_fd = -1;
#endif

static int sample_rate, nr_channels, bits;

void
sound_init (int init_sample_rate, int init_nr_channels, int init_bits)
{
  FILE *fp;
  int fd, pfd[2], r, audio_fd;
  struct sound_process_data data;

#if !SOUND_USE_FORK && defined(__CYGWIN32__)
  DWORD dwThreadId;
#endif

  if (verbose) printf ("loading samples ...\n");

  /* Convert samples to internal format. */
  load_samples ();

  /* Try to open the audio device. */
  audio_fd = os_sound_open_device ();
  if (audio_fd < 0) return;

#if SOUND_USE_FORK

  /* Create the non-anonymous shared memory mapping. */
  fp = tmpfile ();
  if (fp == NULL)
    {
      log_perror ("temporary file");
      goto audio_exit;
    }
  fd = fileno (fp);
  if (ftruncate (fd, sizeof *shm) < 0)
    {
      log_perror ("ftruncate");
      goto audio_exit;
    }

  shm = (struct sound_shm *) mmap (0, sizeof *shm, PROT_READ|PROT_WRITE,
				   MAP_SHARED, fd, 0);
  if (shm == MAP_FAILED)
    {
      log_perror ("mmap");
      goto mmap_exit;
    }
  memset (shm, 0, sizeof *shm);

  /* Initialize shared memory to reasonable defaults. */
  shm->mixer.wind_noise_amplitude = 0.05;
  shm->mixer.sample_amplitude = 0.65;
  shm->mixer.echo_amplitude = 0.3;
  shm->mixer.output_amplitude = 10000;
  shm->echo.delay = 0.01;
  shm->echo.level = 0.5;
  shm->misc.craft_velocity = 0;

#else /* !SOUND_USE_FORK */

  shm = xmalloc (sizeof *shm);

#endif /* !SOUND_USE_FORK */

#if SOUND_USE_FORK

  /* Create the pipe. */
  if (pipe (pfd) < 0)
    {
      log_perror ("pipe");
      goto mmap_exit;
    }

#endif

#if SOUND_USE_FORK

  /* Fork child process. */
  r = fork ();
  if (r < 0)
    {
      log_perror ("fork");
      goto pipe_exit;
    }
  if (r == 0)
    {
      data.pipe_fd = pfd[0];
      data.shm = shm;
      data.audio_fd = audio_fd;
      data.sample_rate = init_sample_rate;
      data.nr_channels = init_nr_channels;
      data.bits = init_bits;
      data.enabled = 0;

      /* Run child process. */
      close (pfd[1]);
      sound_process (&data);
      exit (0);
    }

#else /* !SOUND_USE_FORK */

#ifdef __CYGWIN32__
  /* Create thread with shared VM. */
  data.pipe_fd = -1;
  data.shm = shm;
  data.audio_fd = audio_fd;
  data.sample_rate = init_sample_rate;
  data.nr_channels = init_nr_channels;
  data.bits = init_bits;
  data.enabled = 0;

  CreateThread (NULL, 0, sound_process, &data, 0, &dwThreadId);
#else
#error "Threaded implementation not implemented for Unix yet."
#endif

#endif /* !SOUND_USE_FORK */

#if SOUND_USE_FORK
  /* Ignore SIGPIPE so we see EPIPE errors. */
  signal (SIGPIPE, SIG_IGN);
#endif

  sample_rate = init_sample_rate;
  nr_channels = init_nr_channels;
  bits = init_bits;

  close (pfd[0]);
  pipe_fd = pfd[1];
  sound_state = SOUND_STATE_DISABLED;
  return;

 pipe_exit:
#if SOUND_USE_FORK
  close (pfd[0]);
  close (pfd[1]);
#endif
 mmap_exit:
#if SOUND_USE_FORK
  munmap ((void *) shm, sizeof *shm);
#endif
 audio_exit:
  close (audio_fd);
}

#if SOUND_USE_FORK

static struct sound_request request_data, *request = &request_data;

#else /* !SOUND_USE_FORK */

static int request_index;
static struct sound_request *request;
static struct sound_request request_buf[SOUND_SHM_MAX_REQUESTS];

#endif /* !SOUND_USE_FORK */

/* Send a simple message over the pipe. */
static void
send_request ()
{
  int r;

  if (!SOUND_OK)
    return;

#if SOUND_USE_FORK

  r = write (pipe_fd, request, sizeof *request);
  if (r < 0)
    {
      if (errno == EPIPE)
	log (LOG_ERROR, "sound process crashed - sound disabled");
      else
	log_perror ("write to sound process");
      sound_state = SOUND_STATE_DEAD;
      return;
    }
  if (r != sizeof *request)
    {
      log (LOG_ERROR, "non-atomic write to sound process - sound disabled");
      sound_state = SOUND_STATE_DEAD;
      return;
    }

#else /* !SOUND_USE_FORK */

  if (request_index == SOUND_SHM_MAX_REQUESTS) {
    if (requests_failed++ > SOUND_SHM_MAX_REQUESTS) {

  /* increment local buffer request index */
  request_index++;

  /* copy local buffer if shared buffer ready */
  if (shm->num_sound_requests == 0) {
    memcpy(shm->request, request_buf, request_index * (sizeof *request_buf));
    shm->num_sound_requests = request_index;
    request_index = 0;
    requests_failed = 0;
  }

  /* set request pointer */
  if (request_index < SOUND_SHM_MAX_REQUESTS)
    request = &request_buf[request_index];

#endif /* !SOUND_USE_FORK */
}

/* Write an integer request. */
static void
int_request (int type, int v)
{
  request->type = type;
  request->u.i = v;
  send_request ();
}

/* Write a float request. */
static void
float_request (int type, float v)
{
  request->type = type;
  request->u.f = v;
  send_request ();
}

/* Enable and disable sound. */
void
sound_enable ()
{
  if (sound_state != SOUND_STATE_DEAD)
    {
      sound_state = SOUND_STATE_ENABLED;
      int_request (SOUND_REQUEST_ENABLE, 0);
    }
}

void
sound_disable ()
{
  if (sound_state != SOUND_STATE_DEAD)
    {
      sound_state = SOUND_STATE_DISABLED;
      int_request (SOUND_REQUEST_DISABLE, 0);
    }
}

/* Set and get parameters. */
void
sound_set_sample_rate (int _sample_rate)
{
  int_request (SOUND_REQUEST_SET_SAMPLE_RATE, sample_rate = _sample_rate);
}

int
sound_get_sample_rate ()
{
  return sample_rate;
}

void
sound_set_nr_channels (int _nr_channels)
{
  int_request (SOUND_REQUEST_SET_NR_CHANNELS, nr_channels = _nr_channels);
}

int
sound_get_nr_channels ()
{
  return nr_channels;
}

void
sound_set_bits (int _bits)
{
  int_request (SOUND_REQUEST_SET_BITS, bits = _bits);
}

int
sound_get_bits ()
{
  return bits;
}

void
sound_set_mixer_wind_noise_amplitude (float _wind_noise_amplitude)
{
  if (!SOUND_OK)
    return;
  
  shm->mixer.wind_noise_amplitude = _wind_noise_amplitude;
}

float
sound_get_mixer_wind_noise_amplitude ()
{
  if (!SOUND_OK)
    return 0.0;
  
  return shm->mixer.wind_noise_amplitude;
}

void
sound_set_mixer_echo_amplitude (float _echo_amplitude)
{
  if (!SOUND_OK)
    return;
  
  shm->mixer.echo_amplitude = _echo_amplitude;
}

float
sound_get_mixer_echo_amplitude ()
{
  if (!SOUND_OK)
    return 0.0;
  
  return shm->mixer.echo_amplitude;
}

void
sound_set_mixer_output_amplitude (float _output_amplitude)
{
  if (!SOUND_OK)
    return;
  
  shm->mixer.output_amplitude = _output_amplitude;
}

float
sound_get_mixer_output_amplitude ()
{
  if (!SOUND_OK)
    return 0.0;
  
  return shm->mixer.output_amplitude;
}

void
sound_set_echo_delay (float _delay)
{
  if (!SOUND_OK)
    return;
  
  shm->echo.delay = _delay;
}

float
sound_get_echo_delay ()
{
  if (!SOUND_OK)
    return 0.0;
  
  return shm->echo.delay;
}

void
sound_set_echo_level (float _level)
{
  if (!SOUND_OK)
    return;
  
  shm->echo.level = _level;
}

float
sound_get_echo_level ()
{
  if (!SOUND_OK)
    return 0.0;
  
  return shm->echo.level;
}

void
sound_set_craft_velocity (float _craft_velocity)
{
  if (!SOUND_OK)
    return;
  
  shm->misc.craft_velocity = _craft_velocity;
}

float
sound_get_craft_velocity ()
{
  if (!SOUND_OK)
    return 0.0;
  
  return shm->misc.craft_velocity;
}

/* Play a sample. */
void
sound_play_sample (int sample_num)
{
  int_request (SOUND_REQUEST_PLAY_SAMPLE, sample_num);
}
