/* XRacer (C) 1999 Richard W.M. Jones.
 * $Id: track.c,v 1.17 1999/09/20 20:01:42 rich Exp $
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <math.h>
#include <sys/time.h>

#include "xracer.h"
#include "jpeg.h"

/* XXX These will be defined in the track file eventually. */
#if 0
#define FOG 0
#define PRECIPITATION 0
#define IS_BACKDROP 0
#else
        /* Try this for a winter wonderland feel ... */
#define FOG 1
#define PRECIPITATION SNOW
#define IS_BACKDROP 1
#define BACKDROP "images/mountains4.jpg"
#define BACKDROP_ALPHA "images/mountains4-alpha.jpg"
#define BACKDROP_X_SCALE 2
#endif

/* This should not be a global variable. */
struct track *track = NULL;

static char **current_t = 0;
static int current_n = 0;
static FILE *fp;
static int nr_textures;
static int *texture;

static int read_line (char ***t);
static void trim (char *str);
static void read_display_list (void);

#define ERR(fs...) { log (LOG_ERROR, fs); exit (1); }

/* Read an XRacer track file. */
void
load_track (const char *filename)
{
  char **t;
  int i, s, n;
  char pipename[2048];

  sprintf (pipename, "gzip -cd < %s", filename);

  /* Open the file. */
  fp = popen (pipename, "r");
  if (fp == NULL)
    {
      log_perror ("cannot open track file `%s'", filename);
      exit (1);
    }

  /* Read the header. */
  n = read_line (&t);
  if (n < 0 || n > 2 || strcasecmp (t[0], "xracer-track-file") != 0)
    ERR ("this is not an XRacer track file");
  /* Check the file format version. */
  if (strcasecmp (t[1], "7") != 0)
    ERR ("wrong version (expected version 7, but this file is version %s)", t[1]);

  /* Read the textures. */
  n = read_line (&t);
  if (n < 1 || strcasecmp (t[0], "nr-textures") != 0)
    ERR ("expected nr-textures, found %s", t[0]);
  sscanf (t[1], "%d", &nr_textures);
  texture = xmalloc (nr_textures * sizeof (int));

  for (i = 0; i < nr_textures; ++i)
    {
      n = read_line (&t);
      texture[i] = load_texture (t[0]);
    }

  /* Allocate track. */
  track = xmalloc (sizeof (struct track));

  /* XXX Eventually these parameters will be read out from the file. */
  track->is_fog = FOG;
#if FOG
  track->fog_colour[0] = 1;
  track->fog_colour[1] = 1;
  track->fog_colour[2] = 1;
  track->fog_colour[3] = 1;
  track->fog_density = 0.01;
  track->fog_mode = GL_EXP;

  glFogi (GL_FOG_MODE, track->fog_mode);
  glFogfv (GL_FOG_COLOR, track->fog_colour);
  glFogf (GL_FOG_DENSITY, track->fog_density);
#endif

#if FOG
  track->background_fill_colour[0] = 1;
  track->background_fill_colour[1] = 1;
  track->background_fill_colour[2] = 1;
  track->background_fill_colour[3] = 0;
#else
  track->background_fill_colour[0] = 0;
  track->background_fill_colour[1] = 0;
  track->background_fill_colour[2] = 0;
  track->background_fill_colour[3] = 0;
#endif

  track->is_precipitation = PRECIPITATION;

  glClearColor (track->background_fill_colour[0],
		track->background_fill_colour[1],
		track->background_fill_colour[2],
		track->background_fill_colour[3]);

#if IS_BACKDROP
  /* XXX Don't use load_texture_*. Load it without building mipmaps instead. */
  track->backdrop_tex
    = load_texture_with_alpha (BACKDROP, BACKDROP_ALPHA);
  track->backdrop_x_scale = BACKDROP_X_SCALE;
#else
  track->backdrop_tex = 0;
#endif

  /* Read the segments. */
  n = read_line (&t);
  if (n < 1 || strcasecmp (t[0], "nr-segments") != 0)
    ERR ("expected nr-segments, found %s", t[0]);
  sscanf (t[1], "%d", &track->nr_segments);
  track->segments = xmalloc (track->nr_segments * sizeof (struct segment));

  for (s = 0; s < track->nr_segments; ++s)
    {
      struct segment *segment = &track->segments[s];

      /* Read the display list. */
      n = read_line (&t);
      if (n < 1 || strcasecmp (t[0], "begin-display-list") != 0)
	ERR ("expected begin-display-list, found %s", t[0]);

      segment->dlist = glGenLists (1);
      if (segment->dlist == 0)
	log_fatal ("glGenLists: cannot allocate display list: %s\n",
		   gluErrorString (glGetError ()));

      glNewList (segment->dlist, GL_COMPILE);

      read_display_list ();

      /* Read the number of sides. */
      n = read_line (&t);
      if (n < 1 || strcasecmp (t[0], "nr-sides") != 0)
	ERR ("expected nr-sides, found %s", t[0]);
      sscanf (t[1], "%d", &segment->nr_sides);

      segment->sides = xmalloc (segment->nr_sides * sizeof (struct side));

      for (i = 0; i < segment->nr_sides; ++i)
	{
	  struct side *side = &segment->sides[i];

	  n = read_line (&t);
	  if (n != 7) ERR ("error in sides description");
	  sscanf (t[0], "%g", &side->plane[0]);
	  sscanf (t[1], "%g", &side->plane[1]);
	  sscanf (t[2], "%g", &side->plane[2]);
	  sscanf (t[3], "%g", &side->plane[3]);
	  sscanf (t[4], "%g", &side->levitate_factor);
	  sscanf (t[5], "%g", &side->levitate_dist);
	  sscanf (t[6], "%g", &side->levitate_damping);
	}

      /* Read the enter plane. */
      n = read_line (&t);
      if (n < 1 || strcasecmp (t[0], "enter-plane") != 0)
	ERR ("expected enter-plane, found %s", t[0]);

      if (n != 5) ERR ("error in enter-plane line");

      sscanf (t[1], "%g", &segment->enter[0]);
      sscanf (t[2], "%g", &segment->enter[1]);
      sscanf (t[3], "%g", &segment->enter[2]);
      sscanf (t[4], "%g", &segment->enter[3]);

      /* Read the trigger flags. */
      n = read_line (&t);
      if (n < 1 || strcasecmp (t[0], "trigger") != 0)
	ERR ("expected trigger, found %s", t[0]);

      segment->trigger = 0;
      for (i = 1; i < n; ++i)
	{
	  if (strcasecmp (t[i], "start") == 0)
	    segment->trigger |= TRIGGER_START;
	  else if (strcasecmp (t[i], "left-fast") == 0)
	    segment->trigger |= TRIGGER_LEFT_FAST;
	  else if (strcasecmp (t[i], "left-faster") == 0)
	    segment->trigger |= TRIGGER_LEFT_FASTER;
	  else if (strcasecmp (t[i], "left-powerup") == 0)
	    segment->trigger |= TRIGGER_LEFT_POWERUP;
	  else if (strcasecmp (t[i], "right-fast") == 0)
	    segment->trigger |= TRIGGER_RIGHT_FAST;
	  else if (strcasecmp (t[i], "right-faster") == 0)
	    segment->trigger |= TRIGGER_RIGHT_FASTER;
	  else if (strcasecmp (t[i], "right-powerup") == 0)
	    segment->trigger |= TRIGGER_RIGHT_POWERUP;
	  else
	    ERR ("unknown trigger flag, %s", t[i]);
	}

      /* Read the trigger planes. */
      n = read_line (&t);
      if (n < 1 || strcasecmp (t[0], "left-trigger-plane") != 0)
	ERR ("expected left-trigger-plane, found %s", t[0]);

      if (n != 5) ERR ("error in left-trigger-plane line");

      sscanf (t[1], "%g", &segment->triggers[0][0]);
      sscanf (t[2], "%g", &segment->triggers[0][1]);
      sscanf (t[3], "%g", &segment->triggers[0][2]);
      sscanf (t[4], "%g", &segment->triggers[0][3]);

      n = read_line (&t);
      if (n < 1 || strcasecmp (t[0], "mid-trigger-plane") != 0)
	ERR ("expected mid-trigger-plane, found %s", t[0]);

      if (n != 5) ERR ("error in mid-trigger-plane line");

      sscanf (t[1], "%g", &segment->triggers[1][0]);
      sscanf (t[2], "%g", &segment->triggers[1][1]);
      sscanf (t[3], "%g", &segment->triggers[1][2]);
      sscanf (t[4], "%g", &segment->triggers[1][3]);

      n = read_line (&t);
      if (n < 1 || strcasecmp (t[0], "right-trigger-plane") != 0)
	ERR ("expected right-trigger-plane, found %s", t[0]);

      if (n != 5) ERR ("error in right-trigger-plane line");

      sscanf (t[1], "%g", &segment->triggers[2][0]);
      sscanf (t[2], "%g", &segment->triggers[2][1]);
      sscanf (t[3], "%g", &segment->triggers[2][2]);
      sscanf (t[4], "%g", &segment->triggers[2][3]);

      /* Read the autopilot stuff. */
      n = read_line (&t);
      if (n < 1 || strcasecmp (t[0], "forwards") != 0)
	ERR ("expected forwards, found %s", t[0]);

      if (n != 4) ERR ("error in forwards line");

      sscanf (t[1], "%g", &segment->forwards[0]);
      sscanf (t[2], "%g", &segment->forwards[1]);
      sscanf (t[3], "%g", &segment->forwards[2]);

      n = read_line (&t);
      if (n < 1 || strcasecmp (t[0], "autopilot-midpoint") != 0)
	ERR ("expected autopilot-midpoint, found %s", t[0]);

      if (n != 4) ERR ("error in autopilot-midpoint line");

      sscanf (t[1], "%g", &segment->autopilot_midpoint[0]);
      sscanf (t[2], "%g", &segment->autopilot_midpoint[1]);
      sscanf (t[3], "%g", &segment->autopilot_midpoint[2]);

      /* Read the list of visible objects. */
      n = read_line (&t);
      if (n < 1 || strcasecmp (t[0], "visible") != 0)
	ERR ("expected visible, found %s", t[0]);

      segment->nr_visible = n-1;
      segment->visible = xmalloc (sizeof (int) * segment->nr_visible);
      for (i = 1; i < n; ++i)
	sscanf (t[i], "%d", &segment->visible[i-1]);

      /* Read the sky visible flag. */
      n = read_line (&t);
      if (n < 1 || strcasecmp (t[0], "sky-visible") != 0)
	ERR ("expected sky-visible, found %s", t[0]);

      sscanf (t[1], "%d", &segment->sky_visible);

      /* Read the lookahead. */
      n = read_line (&t);
      if (n < 1 || strcasecmp (t[0], "lookahead") != 0)
	ERR ("expected lookahead, found %s", t[0]);

      sscanf (t[1], "%d", &segment->lookahead);
    }

  /* Read the objects. */
  n = read_line (&t);
  if (n < 1 || strcasecmp (t[0], "nr-objects") != 0)
    ERR ("expected nr-objects, found %s", t[0]);
  sscanf (t[1], "%d", &track->nr_scenery);
  track->scenery = xmalloc (track->nr_scenery * sizeof (struct segment));

  for (s = 0; s < track->nr_scenery; ++s)
    {
      struct scenery *scenery = &track->scenery[s];

      /* Animated object? */
      n = read_line (&t);
      if (n < 1 || strcasecmp (t[0], "animated") != 0)
	ERR ("expected animated, found %s", t[0]);

      if (strcmp (t[1], "0") == 0)
	{
	  scenery->is_animated = 0;
	  scenery->is_start_lamp = 0;

	  /* Read the display list. */
	  n = read_line (&t);
	  if (n < 1 || strcasecmp (t[0], "begin-display-list") != 0)
	    ERR ("expected begin-display-list, found %s", t[0]);

	  scenery->u.not_animated.dlist = glGenLists (1);
	  if (scenery->u.not_animated.dlist == 0)
	    log_fatal ("glGenLists: cannot allocate display list: %s\n",
		       gluErrorString (glGetError ()));

	  glNewList (scenery->u.not_animated.dlist, GL_COMPILE);

	  read_display_list ();
	}
      else
	{
	  scenery->is_animated = 1;

	  /* Start lamp? */
	  n = read_line (&t);
	  if (n < 1 || strcasecmp (t[0], "start-lamp") != 0)
	    ERR ("expected start-lamp, found %s", t[0]);
	  sscanf (t[1], "%d", &scenery->is_start_lamp);

	  /* Number of frames. */
	  n = read_line (&t);
	  if (n < 1 || strcasecmp (t[0], "nr-frames") != 0)
	    ERR ("expected nr-frames, found %s", t[0]);
	  sscanf (t[1], "%d", &scenery->u.animated.nr_frames);

	  /* Read each frame. */
	  scenery->u.animated.frame = xmalloc (sizeof (int) *
					       scenery->u.animated.nr_frames);
	  memset (scenery->u.animated.frame, 0,
		  sizeof (int) * scenery->u.animated.nr_frames);
	  scenery->u.animated.ftime = xmalloc (sizeof (double) *
					       scenery->u.animated.nr_frames);
	  memset (scenery->u.animated.ftime, 0,
		  sizeof (double) * scenery->u.animated.nr_frames);
	  scenery->u.animated.total_ftime = 0;
	  for (i = 0; i < scenery->u.animated.nr_frames; ++i)
	    {
	      n = read_line (&t);
	      if (n != 2)
		ERR ("expected frame defn, found %s", t[0]);

	      sscanf (t[0], "%d", &scenery->u.animated.frame[i]);
	      sscanf (t[1], "%lf", &scenery->u.animated.ftime[i]);

	      scenery->u.animated.total_ftime +=
		scenery->u.animated.ftime[i];
	    }
	}
    }

  pclose (fp);
}

/* Read a GL display list from a file and run it. */
static void
read_display_list ()
{
  char **t;
  int n;

  glEnableClientState (GL_VERTEX_ARRAY);

  for (;;)
    {
      n = read_line (&t);
      if (n < 0) ERR ("unexpected end of file in display list");
      if (n < 1) ERR ("empty line in display list (probably internal error)");

      if (strcasecmp (t[0], "end-display-list") == 0)
	{
	  glDisableClientState (GL_VERTEX_ARRAY);
	  glEndList ();
	  return;
	}
      else if (strcasecmp (t[0], "vertex-array") == 0)
	{
	  int nv, i;
	  GLfloat *v;

	  sscanf (t[1], "%d", &nv);
	  v = xmalloc (nv * 3 * sizeof (GLfloat));

	  for (i = 0; i < nv; ++i)
	    {
	      n = read_line (&t);
	      sscanf (t[0], "%g", &v[i*3]);
	      sscanf (t[1], "%g", &v[i*3+1]);
	      sscanf (t[2], "%g", &v[i*3+2]);
	    }

	  glVertexPointer (3, GL_FLOAT, 0, v);

	  /* XXX: Memory leak: Must free V later. */
	}
      else if (strcasecmp (t[0], "bind-texture") == 0)
	{
	  int tn;

	  sscanf (t[1], "%d", &tn);

	  if (tn < 0 || tn >= nr_textures)
	    ERR ("texture number out of range (tn = %d, nr_textures = %d)",
		 tn, nr_textures);

	  glBindTexture (GL_TEXTURE_2D, texture[tn]);
	}
      else if (strcasecmp (t[0], "begin-polygon") == 0)
	{
	  glBegin (GL_POLYGON);
	}
      else if (strcasecmp (t[0], "end") == 0)
	{
	  glEnd ();
	}
      else if (strcasecmp (t[0], "tex-coord") == 0)
	{
	  GLfloat r, s;

	  sscanf (t[1], "%g", &r);
	  sscanf (t[2], "%g", &s);
	  glTexCoord2f (r, s);
	}
      else if (strcasecmp (t[0], "array-element") == 0)
	{
	  int i;

	  sscanf (t[1], "%d", &i);
	  glArrayElement (i);
	}
      else
	ERR ("unknown display list token `%s'", t[0]);
    }
}

/* Trim whitespace from the beginning and end of the string. */
static void
trim (char *str)
{
  int len = strlen (str);

  while (len > 0 && 0 < *str && *str <= ' ')
    {
      memmove (str, str+1, len);
      len --;
    }

  while (len > 0 && 0 < str[len-1] && str[len-1] <= ' ')
    {
      str[len-1] = '\0';
      len --;
    }
}

/* Read a single line and return tokens. */
static int
read_line (char ***t)
{
  int i;
  static char line[1024];
  char *token;

  /* Free previous line. */
  if (current_t != 0)
    {
      for (i = 0; i < current_n; ++i)
	free (current_t[i]);
      free (current_t);
      current_t = 0;
      current_n = 0;
    }

 again:
  /* Read in the new line. */
  if (fgets (line, sizeof line, fp) == 0) return -1;

  /* Remove whitespace around the line. */
  trim (line);

  /* Ignore blank lines and comments. */
  if (line[0] == '\0' || line[0] == '#') goto again;

  /* Start to tokenize the line. */
  token = strtok (line, " \t");
  while (token)
    {
      current_t = realloc (current_t, sizeof (char *) * (++current_n));
      current_t[current_n-1] = xstrdup (token);

      token = strtok (NULL, " \t");
    }

  /* Return tokenized line. */
  *t = current_t;
  return current_n;
}
