/* XRacer (C) 1999 Richard W.M. Jones.
 * $Id: mktrack.c,v 1.18 1999/07/31 16:53:26 rich Exp $
 */

#ifdef __CYGWIN32__
#include <windows.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <math.h>

#include "config.h"

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <GL/gl.h>

#include "matrix.h"

static void
usage ()
{
  fprintf (stderr,
	   "Usage:\n"
	   "  mktrack [-options] inputfile [inputfile ...]\n"
	   "Options:\n"
	   "  -d descripfile Specify name of track description file\n"
	   "  -o outputfile  Specify name of output file\n"
	   "  -f format      Specify input file format\n"
	   "  -c             Omit comments from output file\n"
	   "  -v             Verbose\n"
	   "Notes:\n"
	   "  Supported input formats are: ac3d\n"
	   "  If not specified and the first input filename is track.ac, then the output\n"
	   "  file will be called track.trk and the track description file read is called\n"
	   "  track.tdf.\n");
  exit (1);
}

#define FORMAT_AC3D             1

#define TRIGGER_START           1
#define TRIGGER_LEFT_FAST       4
#define TRIGGER_LEFT_FASTER     8
#define TRIGGER_LEFT_POWERUP   16
#define TRIGGER_RIGHT_FAST     32
#define TRIGGER_RIGHT_FASTER   64
#define TRIGGER_RIGHT_POWERUP 128

#define SIDE_TRIGGER_FAST       1
#define SIDE_TRIGGER_FASTER     2
#define SIDE_TRIGGER_POWERUP    4

#define TRACK_TEXTURE          -2

static char *inputfile, *descriptionfile = 0, *outputfile = 0;
static int inputformat = FORMAT_AC3D;
static int verbose = 0, comments = 1;
static FILE *fp, *ofp;

struct segment_list
{
  int nr_segments;
  struct segment *segment;
  int nr_scenery;
  struct segment *scenery;
};

struct segment
{
  int nr_vertices;
  GLfloat *vertex;		/* Array of 3 * nr_vertices. */
  int nr_polys;
  struct poly *poly;
  int nr_sides;
  struct side *side;
  GLfloat enter_plane[4];
  GLfloat left_trigger_plane[4];
  GLfloat mid_trigger_plane[4];
  GLfloat right_trigger_plane[4];
  GLfloat forwards[3];
  GLfloat autopilot_midpoint[3];
  int trigger;
  int *visible;
  int nr_visible;
  int sky_visible;
  int lookahead;
  int is_animated;
  int is_start_lamp;
  int nr_frames;
  int *frame;			/* Array of nr_frames. */
  double *ftime;		/* Array of nr_frames. */
};

struct poly
{
  int nr_vertices;
  int *vertex;			/* Array of nr_vertices. */
  GLfloat *tex;			/* Array of 2 * nr_vertices. */
  int texture_num;		/* This refers to global list of textures. */
};

struct side
{
  GLfloat plane[4];
  GLfloat levitate_factor;
  GLfloat levitate_dist;
  GLfloat levitate_damping;
};

static int nr_textures = 0;
static const char **texture = 0;

static struct segment_list *list = 0;
static int sky_tex = -1;

static void parse_ac3d_file (void);
static void write_segment_list (void);
static void allocate_track_textures (void);
static void parse_description (void);

static struct segment_list *new_segment_list (void);
static struct segment *allocate_segment (struct segment_list *list, int seg);
static struct segment *allocate_scenery (struct segment_list *list, int seg);
static struct poly *get_new_poly_in_segment (struct segment *segment,
					     int nr_vertices);
static int allocate_vertex_in_segment (struct segment *segment,
				       GLfloat *v);
static int allocate_texture (const char *texture_name);
static struct side *get_new_side_in_segment (struct segment *segment);
static void append_visible_scenery_to_segment (int seg, int scen);

int
main (int argc, char *argv[])
{
  int c;
  char pipename[2048];

  while ((c = getopt (argc, argv, "d:f:o:vc")) != -1)
    {
      switch (c)
	{
	case 'd':
	  descriptionfile = optarg;
	  break;
	case 'f':
	  if (strcasecmp (optarg, "ac3d") == 0)
	    inputformat = FORMAT_AC3D;
	  else
	    usage ();
	  break;
	case 'o':
	  outputfile = optarg;
	  break;
	case 'v':
	  verbose = 1;
	  break;
	case 'c':
	  comments = 0;
	  break;
	default:
	  usage ();
	}
    }

  if (optind == argc) usage ();

  inputfile = argv[optind];

  if (outputfile == 0)
    {
      char *t;

      outputfile = malloc (strlen (inputfile) + 20);
      strcpy (outputfile, inputfile);

      /* Truncate inputfile at last '.' and append appropriate
       * designator.
       */
      t = strrchr (outputfile, '.');
      if (t != 0)
	strcpy (t, ".trk");
      else
	strcat (outputfile, ".trk");
    }

  if (descriptionfile == 0)
    {
      char *t;

      descriptionfile = malloc (strlen (inputfile) + 20);
      strcpy (descriptionfile, inputfile);

      /* Truncate inputfile at last '.' and append appropriate
       * designator.
       */
      t = strrchr (descriptionfile, '.');
      if (t != 0)
	strcpy (t, ".tdf");
      else
	strcat (outputfile, ".tdf");
    }

  if (verbose)
    printf ("mktrack (C) 1999 Richard W.M. Jones\n");

  /* Read all the input files. */
  list = new_segment_list ();
  while (optind < argc)
    {
      inputfile = argv [optind++];

      fp = fopen (inputfile, "r");
      if (fp == NULL) { perror (inputfile); exit (1); }

      switch (inputformat)
	{
	case FORMAT_AC3D:
	  parse_ac3d_file ();
	  break;
	}

      fclose (fp);
    }

  /* Open the description file. */
  fp = fopen (descriptionfile, "r");
  if (fp == NULL) { perror (descriptionfile); exit (1); }

  parse_description ();

  fclose (fp);

  /* Allocate track textures, now that we have read all the track
   * and trigger information in.
   */
  allocate_track_textures ();

  /* Open output file. */
  sprintf (pipename, "gzip -c > %s.gz", outputfile);
  ofp = popen (pipename, "w");
  if (ofp == NULL) { perror (pipename); exit (1); }

  write_segment_list ();

  pclose (ofp);

  exit (0);
}

static struct segment_list *
new_segment_list ()
{
  struct segment_list *list = malloc (sizeof (struct segment_list));
  list->nr_segments = 0;
  list->segment = 0;
  list->nr_scenery = 0;
  list->scenery = 0;
  return list;
}

static void
init_segment (struct segment *segment)
{
  segment->nr_vertices = 0;
  segment->vertex = 0;
  segment->nr_polys = 0;
  segment->poly = 0;
  segment->nr_sides = 0;
  segment->side = 0;
  memset (segment->enter_plane, 0, sizeof segment->enter_plane);
  memset (segment->left_trigger_plane, 0, sizeof segment->left_trigger_plane);
  memset (segment->mid_trigger_plane, 0, sizeof segment->mid_trigger_plane);
  memset (segment->right_trigger_plane, 0, sizeof segment->right_trigger_plane);
  memset (segment->forwards, 0, sizeof segment->forwards);
  memset (segment->autopilot_midpoint, 0, sizeof segment->autopilot_midpoint);
  segment->trigger = 0;
  segment->visible = 0;
  segment->nr_visible = 0;
  segment->sky_visible = 0;
  segment->lookahead = 0;
}

static struct segment *
allocate_segment (struct segment_list *list, int seg)
{
  if (seg < list->nr_segments)
    return &list->segment[seg];
  else
    {
      int i = list->nr_segments;

      list->nr_segments = seg+1;
      list->segment = realloc (list->segment,
			       list->nr_segments * sizeof (struct segment));

      /* Initialize newly created segment structures. */
      for (; i < list->nr_segments; ++i)
	init_segment (&list->segment[i]);

      return &list->segment[seg];
    }
}

static struct segment *
allocate_scenery (struct segment_list *list, int seg)
{
  if (seg < list->nr_scenery)
    return &list->scenery[seg];
  else
    {
      int i = list->nr_scenery;

      list->nr_scenery = seg+1;
      list->scenery = realloc (list->scenery,
			       list->nr_scenery * sizeof (struct segment));

      /* Initialize newly created segment structures. */
      for (; i < list->nr_scenery; ++i)
	init_segment (&list->scenery[i]);

      return &list->scenery[seg];
    }
}

/* This function makes a deep copy of scenery object FROM
 * to scenery object TO.
 */
static void
copy_scenery_object (int from, int to)
{
  int i;
  struct segment *to_segment = allocate_scenery (list, to);
  struct segment *from_segment = &list->scenery[from];

  memcpy (to_segment, from_segment, sizeof (struct segment));
  to_segment->vertex = malloc (to_segment->nr_vertices * 3 * sizeof (GLfloat));
  memcpy (to_segment->vertex, from_segment->vertex,
	  to_segment->nr_vertices * 3 * sizeof (GLfloat));
  to_segment->poly = malloc (to_segment->nr_polys * sizeof (struct poly));
  for (i = 0; i < to_segment->nr_polys; ++i)
    {
      to_segment->poly[i].nr_vertices = from_segment->poly[i].nr_vertices;
      to_segment->poly[i].vertex = malloc (to_segment->poly[i].nr_vertices *
					   sizeof (int));
      memcpy (to_segment->poly[i].vertex, from_segment->poly[i].vertex,
	      to_segment->poly[i].nr_vertices * sizeof (int));
      to_segment->poly[i].tex = malloc (to_segment->poly[i].nr_vertices * 2 *
					sizeof (GLfloat));
      memcpy (to_segment->poly[i].tex, from_segment->poly[i].tex,
	      to_segment->poly[i].nr_vertices * 2 * sizeof (GLfloat));
      to_segment->poly[i].texture_num = from_segment->poly[i].texture_num;
    }
  to_segment->side = malloc (to_segment->nr_sides * sizeof (struct side));
  memcpy (to_segment->side, from_segment->side,
	  to_segment->nr_sides * sizeof (struct side));
  to_segment->visible = malloc (to_segment->nr_visible * sizeof (int));
  memcpy (to_segment->visible, from_segment->visible,
	  to_segment->nr_visible * sizeof (int));
}

static struct poly *
get_new_poly_in_segment (struct segment* segment, int nr_vertices)
{
  struct poly *poly;

  segment->nr_polys ++;
  segment->poly = realloc (segment->poly,
			   segment->nr_polys * sizeof (struct poly));
  poly = &segment->poly[segment->nr_polys-1];

  poly->nr_vertices = nr_vertices;
  poly->vertex = malloc (nr_vertices * sizeof (int));
  poly->tex = malloc (nr_vertices * 2 * sizeof (GLfloat));
  poly->texture_num = -1;

  return poly;
}

#define EPSILON 1e-6

static inline int
vertex_equal (GLfloat *v, GLfloat *w)
{
  return
    fabs (v[0] - w[0]) <= EPSILON &&
    fabs (v[1] - w[1]) <= EPSILON &&
    fabs (v[2] - w[2]) <= EPSILON;
}

static int
allocate_vertex_in_segment (struct segment *segment, GLfloat *v)
{
  int i;

  /* Look through the existing vertices for one which is
   * similar. In this way we merge vertices and make the
   * display list much more efficient.
   */
  for (i = 0; i < segment->nr_vertices; ++i)
    {
      if (vertex_equal (&segment->vertex[i*3], v))
	return i;
    }

  /* Need to allocate a new vertex. */
  segment->nr_vertices ++;
  segment->vertex = realloc (segment->vertex,
			     3 * segment->nr_vertices * sizeof (GLfloat));
  i = segment->nr_vertices - 1;
  memcpy (&segment->vertex[i*3], v, 3 * sizeof (GLfloat));
  return i;
}

static int
allocate_texture (const char *texture_name)
{
  int i;

  if (texture_name == 0) return -1;

  /* Look through the existing textures to find one which
   * matches.
   */
  for (i = 0; i < nr_textures; ++i)
    if (strcmp (texture_name, texture[i]) == 0)
      return i;

  /* Allocate a new texture. */
  nr_textures ++;
  texture = realloc (texture, nr_textures * sizeof (char *));
  i = nr_textures - 1;
  texture[i] = texture_name;
  return i;
}

static struct side *
get_new_side_in_segment (struct segment *segment)
{
  struct side *side;

  segment->nr_sides ++;
  segment->side = realloc (segment->side,
			   segment->nr_sides * sizeof (struct side));
  side = &segment->side[segment->nr_sides-1];
  side->plane[0] = 0;
  side->plane[1] = 0;
  side->plane[2] = 0;
  side->plane[3] = 0;
  side->levitate_factor = 0;
  side->levitate_dist = 0;
  side->levitate_damping = 0;
  return side;
}

static int
allocate_track_texture_by_trigger (int trigger)
{
  const char *base_image = "images/track5";
  char image[1024];

  strcpy (image, base_image);

  if (trigger & TRIGGER_LEFT_FAST)
    strcat (image, "+left-fast");
  else if (trigger & TRIGGER_LEFT_FASTER)
    strcat (image, "+left-faster");
  else if (trigger & TRIGGER_LEFT_POWERUP)
    strcat (image, "+left-powerup");
  if (trigger & TRIGGER_RIGHT_FAST)
    strcat (image, "+right-fast");
  else if (trigger & TRIGGER_RIGHT_FASTER)
    strcat (image, "+right-faster");
  else if (trigger & TRIGGER_RIGHT_POWERUP)
    strcat (image, "+right-powerup");

  strcat (image, ".jpg");

  return allocate_texture (strdup (image));
}

/* After we have read the track and the triggers from the
 * description file, we can allocate track textures. This
 * function does that.
 */
static void
allocate_track_textures ()
{
  int s, i;

  for (s = 0; s < list->nr_segments; ++s)
    {
      int trigger = list->segment[s].trigger;

      for (i = 0; i < list->segment[s].nr_polys; ++i)
	{
	  int texture_num = list->segment[s].poly[i].texture_num;

	  if (texture_num == TRACK_TEXTURE)
	    {
	      list->segment[s].poly[i].texture_num =
		allocate_track_texture_by_trigger (trigger);
	    }
	}
    }
}

static void
append_visible_scenery_to_segment (int seg, int scen)
{
  struct segment *segment = &list->segment[seg];

  segment->nr_visible ++;
  segment->visible = realloc (segment->visible,
			      segment->nr_visible * sizeof (int));

  segment->visible[segment->nr_visible-1] = scen;
}

static void
write_segment_list ()
{
  int s, i, j;

  /* Output the header. */
  fprintf (ofp,
	   "xracer-track-file 7\n"
	   "# This is an XRacer track file, generated\n"
	   "# by the mktrack utility.\n"
	   "#\n"
	   "# To find out about XRacer please visit\n"
	   "# http://www.annexia.org/freeware/xracer.html\n"
	   "#\n");

  /* Output the global list of textures. */
  fprintf (ofp, "nr-textures %d\n", nr_textures);
  for (i = 0; i < nr_textures; ++i)
    fprintf (ofp, "%s\n", texture[i]);

  /* Output the list of segments. */
  fprintf (ofp, "nr-segments %d\n", list->nr_segments);

  for (s = 0; s < list->nr_segments; ++s)
    {
      struct segment *segment = &list->segment[s];

      if (comments) fprintf (ofp, "# -- start segment %d\n", s);

      /* Note: I did some tests, and it turns out
       * (with Mesa, at least) that drawing straight
       * polygons is faster than drawing triangle
       * fans and quad strips. Hence, I've no intention
       * of optimizing the code below to search for
       * triangle fans and quad strips.
       */

      /* Start the display list. */
      fprintf (ofp, "begin-display-list\n");

      /* Output the vertices. */
      fprintf (ofp, "vertex-array %d\n", segment->nr_vertices);
      for (i = 0; i < segment->nr_vertices; ++i)
	fprintf (ofp, "%g %g %g\n",
		 segment->vertex[i*3],
		 segment->vertex[i*3+1],
		 segment->vertex[i*3+2]);

      /* Output the polygons. */
      for (i = 0; i < segment->nr_polys; ++i)
	{
	  struct poly *poly = &segment->poly[i];

	  if (comments)
	    fprintf (ofp, "# ---- start segment %d poly %d\n", s, i);
	  if (poly->texture_num >= 0)
	    fprintf (ofp, "bind-texture %d\n", poly->texture_num);
	  fprintf (ofp, "begin-polygon\n");
	  for (j = 0; j < poly->nr_vertices; ++j)
	    fprintf (ofp,
		     "tex-coord %g %g\n"
		     "array-element %d\n",
		     poly->tex[j*2], poly->tex[j*2+1], poly->vertex[j]);
	  fprintf (ofp, "end\n");
	  if (comments)
	    fprintf (ofp, "# ---- end segment %d poly %d\n", s, i);
	}

      /* End the display list. */
      fprintf (ofp, "end-display-list\n");

      /* Output the sides. */
      fprintf (ofp, "nr-sides %d\n", segment->nr_sides);
      for (i = 0; i < segment->nr_sides; ++i)
	{
	  struct side *side = &segment->side[i];

	  fprintf (ofp, "%g %g %g %g %g %g %g\n",
		   side->plane[0],
		   side->plane[1],
		   side->plane[2],
		   side->plane[3],
		   side->levitate_factor,
		   side->levitate_dist,
		   side->levitate_damping);
	}

      /* Output the enter plane. */
      fprintf (ofp, "enter-plane %g %g %g %g\n",
	       segment->enter_plane[0],
	       segment->enter_plane[1],
	       segment->enter_plane[2],
	       segment->enter_plane[3]);

      /* Output the trigger info. */
      fprintf (ofp, "trigger %s %s %s %s %s %s %s\n",
	       (segment->trigger & TRIGGER_START) ? "start" : "",
	       (segment->trigger & TRIGGER_LEFT_FAST) ? "left-fast" : "",
	       (segment->trigger & TRIGGER_LEFT_FASTER) ? "left-faster" : "",
	       (segment->trigger & TRIGGER_LEFT_POWERUP) ? "left-powerup" : "",
	       (segment->trigger & TRIGGER_RIGHT_FAST) ? "right-fast" : "",
	       (segment->trigger & TRIGGER_RIGHT_FASTER) ? "right-faster" : "",
	       (segment->trigger & TRIGGER_RIGHT_POWERUP) ? "right-powerup" : "");

      fprintf (ofp, "left-trigger-plane %g %g %g %g\n",
	       segment->left_trigger_plane[0],
	       segment->left_trigger_plane[1],
	       segment->left_trigger_plane[2],
	       segment->left_trigger_plane[3]);

      fprintf (ofp, "mid-trigger-plane %g %g %g %g\n",
	       segment->mid_trigger_plane[0],
	       segment->mid_trigger_plane[1],
	       segment->mid_trigger_plane[2],
	       segment->mid_trigger_plane[3]);

      fprintf (ofp, "right-trigger-plane %g %g %g %g\n",
	       segment->right_trigger_plane[0],
	       segment->right_trigger_plane[1],
	       segment->right_trigger_plane[2],
	       segment->right_trigger_plane[3]);

      /* Autopilot control points. */
      fprintf (ofp, "forwards %g %g %g\n",
	       segment->forwards[0],
	       segment->forwards[1],
	       segment->forwards[2]);

      fprintf (ofp, "autopilot-midpoint %g %g %g\n",
	       segment->autopilot_midpoint[0],
	       segment->autopilot_midpoint[1],
	       segment->autopilot_midpoint[2]);

      /* Output visible information. */
      fprintf (ofp, "visible");
      for (i = 0; i < segment->nr_visible; ++i)
	fprintf (ofp, " %d", segment->visible[i]);
      fprintf (ofp, "\n");

      fprintf (ofp, "sky-visible %d\n", segment->sky_visible);

      fprintf (ofp, "lookahead %d\n", segment->lookahead);

      if (comments) fprintf (ofp, "# -- end segment %d\n", s);
    }

  /* Output the scenery objects. */
  fprintf (ofp, "nr-objects %d\n", list->nr_scenery);

  for (s = 0; s < list->nr_scenery; ++s)
    {
      struct segment *segment = &list->scenery[s];

      if (comments) fprintf (ofp, "# -- start scenery object %d\n", s);

      /* Animated or not? */
      fprintf (ofp, "animated %d\n", segment->is_animated);

      if (!segment->is_animated)
	{
	  /* XXX Should optimize segment list by finding
	   * triangles, quads, quad strips, triangle fans etc.
	   */

	  /* Start the display list. */
	  fprintf (ofp, "begin-display-list\n");

	  /* Output the vertices. */
	  fprintf (ofp, "vertex-array %d\n", segment->nr_vertices);
	  for (i = 0; i < segment->nr_vertices; ++i)
	    fprintf (ofp, "%g %g %g\n",
		     segment->vertex[i*3],
		     segment->vertex[i*3+1],
		     segment->vertex[i*3+2]);

	  /* Output the polygons. */
	  for (i = 0; i < segment->nr_polys; ++i)
	    {
	      struct poly *poly = &segment->poly[i];

	      if (comments)
		fprintf (ofp, "# ---- start scenery object %d poly %d\n", s, i);
	      if (poly->texture_num >= 0)
		fprintf (ofp, "bind-texture %d\n", poly->texture_num);
	      fprintf (ofp, "begin-polygon\n");
	      for (j = 0; j < poly->nr_vertices; ++j)
		fprintf (ofp,
			 "tex-coord %g %g\n"
			 "array-element %d\n",
			 poly->tex[j*2], poly->tex[j*2+1], poly->vertex[j]);
	      fprintf (ofp, "end\n");
	      if (comments)
		fprintf (ofp, "# ---- end scenery object %d poly %d\n", s, i);
	    }

	  /* End the display list. */
	  fprintf (ofp, "end-display-list\n");
	}
      else
	{
	  /* Animated object. */
	  fprintf (ofp, "start-lamp %d\n", segment->is_start_lamp);

	  fprintf (ofp, "nr-frames %d\n", segment->nr_frames);

	  for (i = 0; i < segment->nr_frames; ++i)
	    fprintf (ofp, "%d %g\n", segment->frame[i], segment->ftime[i]);
	}

      if (comments) fprintf (ofp, "# -- end scenery object %d\n", s);
    }
}

/* AC3D parser. */

static int read_ac3d_line (char ***t);
static void put_back_ac3d_line (char **t, int n);
static struct ac3d_object *parse_ac3d_object (void);
static struct ac3d_surface *parse_ac3d_surface (const struct ac3d_object *parent);
static char *canonicalize_ac3d_texture_path (const char *t);
static void trim (char *str);
static char *get_ac3d_name (char *t);
static void convert_ac3d_object_to_segments (struct segment_list *,struct ac3d_object *);
static void convert_ac3d_surface_to_segments (struct segment_list *list, const char *name, const char *texture_name, int nr_vertices, GLfloat *vertex, struct ac3d_surface *surface);

struct ac3d_object {
  char *name;
  GLfloat loc[3];
  char *texture_name;
  GLfloat texture_repeat[2];
  int nr_vertices;
  GLfloat *vertex;		/* Array of 3 * nr_vertices. */
  int nr_surfaces;
  struct ac3d_surface **surface;
  int nr_kids;
  struct ac3d_object **kid;
};

static inline struct ac3d_object *
new_ac3d_object (void)
{
  struct ac3d_object *p = malloc (sizeof (struct ac3d_object));
  p->name = 0;
  p->loc[0] = p->loc[1] = p->loc[2] = 0;
  p->texture_name = 0;
  p->texture_repeat[0] = p->texture_repeat[1] = 1;
  p->nr_vertices = -1;
  p->vertex = 0;
  p->nr_surfaces = -1;
  p->surface = 0;
  p->nr_kids = -1;
  p->kid = 0;
  return p;
}

struct ac3d_surface {
  unsigned flag;
  int mat;
  int nr_vertices;
  struct ac3d_surf_vertex_ref *vertex;
};

static inline struct ac3d_surface *
new_ac3d_surface (void)
{
  struct ac3d_surface *p = malloc (sizeof (struct ac3d_surface));
  p->flag = -1;
  p->mat = -1;
  p->nr_vertices = -1;
  p->vertex = 0;
  return p;
}

struct ac3d_surf_vertex_ref {
  int v;
  GLfloat tex[2];
};

/* An AC3D file has the general form:
 *    AC3Db
 *    0 or more MATERIAL lines
 *    1 OBJECT definition
 */
static void
parse_ac3d_file ()
{
  char **t;
  int n;
  struct ac3d_object *obj;

  /* Expect header. */
  n = read_ac3d_line (&t);
  if (n != 1 || strcmp (t[0], "AC3Db") != 0)
    {
      fprintf (stderr, "mktrack: input file not in AC3D format or unknown variant of format (n = %d, t[0] = %s)\n", n, t[0]);
      exit (1);
    }

  /* Read the MATERIAL lines. */
  for (;;)
    {
      n = read_ac3d_line (&t);
      if (n < 1)
	{
	  fprintf (stderr, "mktrack: expected MATERIAL line\n");
	  exit (1);
	}
      if (strcmp (t[0], "OBJECT") == 0)
	{
	  put_back_ac3d_line (t, n);
	  goto read_object;
	}
      if (strcmp (t[0], "MATERIAL") != 0)
	{
	  fprintf (stderr, "mktrack: expected MATERIAL line, got %s instead\n",
		   t[0]);
	  exit (1);
	}

      /* Ignore MATERIAL lines for now ... */
    }

 read_object:
  obj = parse_ac3d_object ();

  convert_ac3d_object_to_segments (list, obj);
}

/* An AC3D object definition has the general form:
 *    OBJECT name
 *    [ name "name of object" ]
 *    [ loc x y z ]
 *    [ texture "name of texture" ]
 *    [ numvert nr_vertices
 *      nr_vertices vertices ]
 *    [ numsurf nr_surfaces
 *      nr_surface surface definitions ]
 *    [ kids nr_children
 *      nr_children OBJECT definitions ]
 */
static struct ac3d_object *
parse_ac3d_object ()
{
  char **t;
  int n;
  struct ac3d_object *obj = new_ac3d_object ();

  /* Expect OBJECT line. */
  n = read_ac3d_line (&t);
  if (n < 0) { return 0; }
  if (n < 1 || strcmp (t[0], "OBJECT") != 0)
    {
      fprintf (stderr, "mktrack: expected OBJECT line, got %s instead\n",
	       t[0]);
      exit (1);
    }
  if (strcmp (t[1], "world") != 0 && strcmp (t[1], "poly") != 0)
    {
      fprintf (stderr, "mktrack: can't handle OBJECT type %s != world or poly at the moment\n", t[1]);
      exit (1);
    }

  /* Read the rest of the object. */
  for (;;)
    {
      n = read_ac3d_line (&t);
      if (n < 0) { return obj; }
      if (n < 1) { fprintf (stderr, "mktrack: blank line found\n"); exit (1); }

      if (strcmp (t[0], "name") == 0)
	{
	  obj->name = get_ac3d_name (t[1]);
	}
      else if (strcmp (t[0], "loc") == 0)
	{
	  sscanf (t[1], "%g", &obj->loc[0]);
	  sscanf (t[2], "%g", &obj->loc[1]);
	  sscanf (t[3], "%g", &obj->loc[2]);
	}
      else if (strcmp (t[0], "texture") == 0)
	{
	  char *s = get_ac3d_name (t[1]);
	  obj->texture_name = canonicalize_ac3d_texture_path (s);
	  free (s);
	}
      else if (strcmp (t[0], "texrep") == 0)
	{
	  sscanf (t[1], "%g", &obj->texture_repeat[0]);
	  sscanf (t[2], "%g", &obj->texture_repeat[1]);
	}
      else if (strcmp (t[0], "numvert") == 0)
	{
	  int nv, i;

	  if (sscanf (t[1], "%d", &nv) != 1)
	    {
	      fprintf (stderr, "mktrack: can't read number of vertices\n");
	      exit (1);
	    }

	  obj->nr_vertices = nv;
	  obj->vertex = malloc (sizeof (GLfloat) * 3 * nv);

	  /* Read the vertices. */
	  for (i = 0; i < nv; ++i)
	    {
	      n = read_ac3d_line (&t);
	      sscanf (t[0], "%g", &obj->vertex[i*3]);
	      sscanf (t[1], "%g", &obj->vertex[i*3+1]);
	      sscanf (t[2], "%g", &obj->vertex[i*3+2]);
	      /* Semi-hack: add location to vertices. */
	      obj->vertex[i*3] += obj->loc[0];
	      obj->vertex[i*3+1] += obj->loc[1];
	      obj->vertex[i*3+2] += obj->loc[2];
	    }
	}
      else if (strcmp (t[0], "numsurf") == 0)
	{
	  int ns, i;

	  if (sscanf (t[1], "%d", &ns) != 1)
	    {
	      fprintf (stderr, "mktrack: can't read number of surfaces\n");
	      exit (1);
	    }

	  obj->nr_surfaces = ns;
	  obj->surface = malloc (sizeof (struct ac3d_surface *) * ns);

	  /* Read the surfaces. */
	  for (i = 0; i < ns; ++i)
	    obj->surface[i] = parse_ac3d_surface (obj);
	}
      else if (strcmp (t[0], "kids") == 0)
	{
	  int nk, i;

	  if (sscanf (t[1], "%d", &nk) != 1)
	    {
	      fprintf (stderr, "mktrack: can't read number of kids\n");
	      exit (1);
	    }

	  obj->nr_kids = nk;
	  obj->kid = malloc (sizeof (struct ac3d_object *) * nk);

	  /* Read the kids. */
	  for (i = 0; i < nk; ++i)
	    obj->kid[i] = parse_ac3d_object ();
	}
      else
	{
	  put_back_ac3d_line (t, n);
	  return obj;
	}
    }
}

/* An AC3D surface definition has the general form:
 *    SURF flag
 *    mat material_number
 *    refs nr_vertex_references
 *    nr_vertex_references vertex references
 */
static struct ac3d_surface *
parse_ac3d_surface (const struct ac3d_object *parent)
{
  char **t;
  int n;
  struct ac3d_surface *surf = new_ac3d_surface ();

  n = read_ac3d_line (&t);
  if (n < 0) { return 0; }
  if (n < 1 || strcmp (t[0], "SURF") != 0)
    {
      fprintf (stderr, "mktrack: expected SURF-ace line, got %s instead\n",
	       t[0]);
      exit (1);
    }
  sscanf (t[1], "%x", &surf->flag);

  /* Read the rest of the surface. */
  for (;;)
    {
      n = read_ac3d_line (&t);
      if (n < 0) { return surf; }
      if (n < 1) { fprintf (stderr, "mktrack: blank line found\n"); exit (1); }

      if (strcmp (t[0], "mat") == 0)
	{
	  sscanf (t[1], "%d", &surf->mat);
	}
      else if (strcmp (t[0], "refs") == 0)
	{
	  int nv, i;

	  if (sscanf (t[1], "%d", &nv) != 1)
	    {
	      fprintf (stderr, "mktrack: can't read number of vertices\n");
	      exit (1);
	    }

	  surf->nr_vertices = nv;
	  surf->vertex = malloc (sizeof (struct ac3d_surf_vertex_ref) * nv);

	  /* Read the vertices. */
	  for (i = 0; i < nv; ++i)
	    {
	      n = read_ac3d_line (&t);
	      sscanf (t[0], "%d", &surf->vertex[i].v);
	      sscanf (t[1], "%g", &surf->vertex[i].tex[0]);
	      sscanf (t[2], "%g", &surf->vertex[i].tex[1]);
	      /* Semi-hack. Multiple texture coords by texture repeat
	       * in parent object.
	       */
	      surf->vertex[i].tex[0] *= parent->texture_repeat[0];
	      surf->vertex[i].tex[1] *= parent->texture_repeat[1];
	    }
	}
      else
	{
	  put_back_ac3d_line (t, n);
	  return surf;
	}
    }
}

/* Yuk. AC3D saves texture paths as absolute names, so
 * breaking AC3D files if you move them from one machine
 * to another. Therefore we have to turn the texture
 * path into a relative filename. We do this by searching
 * for the last path component of the form 'xracer*' and
 * removing that component and all preceeding components.
 */
static char *
canonicalize_ac3d_texture_path (const char *t)
{
  const char *p = 0, *q, *r = t;

  while ((q = strstr (r, "/xracer")) != NULL)
    p = q, r = q+1;

  if (p == 0)
    {
      fprintf (stderr, "mktrack: can't canonicalize name %s\n", t);
      exit (1);
    }

  p++;
  while (*p && *p != '/')
    p++;

  if (*p == 0)
    {
      fprintf (stderr, "mktrack: can't canonicalize name %s\n", t);
      exit (1);
    }

  return strdup (p+1);
}

static char **current_t = 0;
static int current_n = 0;
static char **unget_t = 0;
static int unget_n = 0;

static int
read_ac3d_line (char ***t)
{
  int i;
  static char line[1024];
  char *token;

  /* Free previous line. */
  if (current_t != 0)
    {
      for (i = 0; i < current_n; ++i)
	free (current_t[i]);
      free (current_t);
      current_t = 0;
      current_n = 0;
    }

  /* Is there an put back line? If so, return it. */
  if (unget_t != 0)
    {
      current_t = unget_t;
      current_n = unget_n;
      unget_t = 0;
      unget_n = 0;
      *t = current_t;
      return current_n;
    }

 again:
  /* Read in the new line. */
  if (fgets (line, sizeof line, fp) == 0) return -1;

  /* Remove whitespace around the line. */
  trim (line);

  /* I don't know if AC3D files could contain comments and
   * blank lines, but ignore them anyhow.
   */
  if (line[0] == '#' || line[0] == '\0')
    goto again;

  /* Start to tokenize the line. */
  token = strtok (line, " \t");
  while (token)
    {
      current_t = realloc (current_t, sizeof (char *) * (++current_n));
      current_t[current_n-1] = strdup (token);

      token = strtok (NULL, " \t");
    }

  /* Return tokenized line. */
  *t = current_t;
  return current_n;
}

static void
put_back_ac3d_line (char **t, int n)
{
  if (unget_t != 0)
    {
      fprintf (stderr, "mktrack: can't put back more than one line\n");
      exit (1);
    }

  unget_t = current_t;
  unget_n = current_n;
  current_t = 0;
  current_n = 0;
}

/* Remove quotes from t and return a copy. */
static char *
get_ac3d_name (char *t)
{
  int n;

  if (t[0] == '"') t++;
  t = strdup (t);
  n = strlen (t);
  if (t[n-1] == '"') t[n-1] = '\0';
  return t;
}

/* Trim whitespace from the beginning and end of the string. */
static void
trim (char *str)
{
  int len = strlen (str);

  while (len > 0 && 0 < *str && *str <= ' ')
    {
      memmove (str, str+1, len);
      len --;
    }

  while (len > 0 && 0 < str[len-1] && str[len-1] <= ' ')
    {
      str[len-1] = '\0';
      len --;
    }
}

/* This function is somewhat involved. It takes a
 * fully parsed world object (OBJ) and examines it recursively.
 * The complete world object should consist entirely
 * of triangles or quads (assuming the user remembered
 * to triangulate the scene before exporting it :-).
 * We examine each triangle or quad and assign it to
 * a segment based on its name. In this way, we build
 * up a list of segments and for each segment, we
 * build up a list of triangles and quads in that
 * segment.
 */
static void
convert_ac3d_object_to_segments (struct segment_list *list,
				 struct ac3d_object *obj)
{
  int i;

  /* Object is either a group or a low-level object. We
   * can tell which by examining the number of children.
   */
  if (obj->nr_kids <= 0)	/* It's a low-level object. */
    {
      assert (obj->nr_kids <= 0);

      for (i = 0; i < obj->nr_surfaces; ++i)
	convert_ac3d_surface_to_segments (list,
					  obj->name,
					  obj->texture_name,
					  obj->nr_vertices,
					  obj->vertex,
					  obj->surface[i]);
    }
  else				/* It's a group. */
    {
      assert (obj->nr_vertices <= 0);
      assert (obj->nr_surfaces <= 0);

      for (i = 0; i < obj->nr_kids; ++i)
	convert_ac3d_object_to_segments (list, obj->kid[i]);
    }
}

static void
convert_ac3d_surface_to_segments (struct segment_list *list,
				  const char *name,
				  const char *texture_name,
				  int nr_vertices,
				  GLfloat *vertex,
				  struct ac3d_surface *surface)
{
  int seg, i, quad = 0;
  char type = '-';
  struct segment *segment;
  struct poly *poly;
  struct side *side;

  /* Examine the name. It should have one of the following
   * forms:
   *   t_A_B_C
   *      where A = segment number
   *            B = quad number within the segment
   *            C = quad type ('i' = invisible,
   *                           's' = striped track edge,
   *                           'S' = striped track edge (no levitation),
   *                           't' = track bottom,
   *                           'e' = enter plane)
   * or:
   *   tr_A_C
   *      where A = segment number
   *            C = trigger type ('l' = left, 'm' = mid, 'r' = right)
   * or:
   *   pr_A
   *      which is a preview segment (ignored)
   * or:
   *   name_A
   *      where A = object number.
   */
  if (name[0] == 't' && name[1] == '_')
    {
      char c;

      if (sscanf (&name[2], "%d_%d_%c", &seg, &quad, &c) != 3)
	{
	  fprintf (stderr, "mktrack: can't scan track quad name %s\n", name);
	  exit (1);
	}

      type = c;
    }
  else if (name[0] == 't' && name[1] == 'r' && name[2] == '_')
    {
      char c;

      if (sscanf (&name[3], "%d_%c", &seg, &c) != 2)
	{
	  fprintf (stderr, "mktrack: can't scan trigger name %s\n", name);
	  exit (1);
	}

      type = c;
    }
  else if (name[0] == 'p' && name[1] == 'r' && name[2] == '_')
    {
      /* Ignore preview segments. */
      return;
    }
  else
    {
      char *t;

      /* Search name for last _ character. */
      t = strrchr (name, '_');
      if (t == 0)
	{
	  fprintf (stderr, "mktrack: object name must have the format ``name_seg'' where seg is the segment number. The name was in fact: ``%s''\n", name);
	  exit (1);
	}

      /* Parse the number. */
      if (sscanf (t+1, "%d", &seg) != 1)
	{
	  fprintf (stderr, "mktrack: can't scan segment number from name %s\n", name);
	  exit (1);
	}
    }

  if (type != '-')
    segment = allocate_segment (list, seg);
  else
    segment = allocate_scenery (list, seg);

  /* If no texture set for striped or track quad, set one. */
  if ((type == 's' || type == 'S') && texture_name == 0)
    texture_name = "images/beestripes1.jpg";
  else if (type == 't' && texture_name == 0)
    texture_name = ((void *)TRACK_TEXTURE); /* Allocate this later. */

  if (type == '-' || type == 's' || type == 'S' || type == 't')
    {
      poly = get_new_poly_in_segment (segment, surface->nr_vertices);
      for (i = 0; i < surface->nr_vertices; ++i)
	{
	  int vr;

	  vr = surface->vertex[i].v; /* Vertex reference. */
				/* Allocate a vertex. */
	  poly->vertex[i]
	    = allocate_vertex_in_segment (segment, &vertex[vr*3]);
				/* Get the texture coords. */
	  memcpy (&poly->tex[i*2], surface->vertex[i].tex,
		  2 * sizeof (GLfloat));

	  /* Copy over the texture. */
	  if (texture_name != ((void *)TRACK_TEXTURE))
	    poly->texture_num = allocate_texture (texture_name);
	  else
	    poly->texture_num = TRACK_TEXTURE;
	}
    }

  /* Generate a side for this quad. */
  if (type == 's' || type == 't' || type == 'i')
    {
      side = get_new_side_in_segment (segment);
      plane_coefficients (&vertex[3*surface->vertex[0].v],
			  &vertex[3*surface->vertex[1].v],
			  &vertex[3*surface->vertex[2].v], side->plane);
      if (type == 's' || type == 't')
	{
	  side->levitate_factor = 1;
#if 0
	  side->levitate_dist = 12;
	  side->levitate_damping = 0.03;
#elif 0
	  side->levitate_dist = 6;
	  side->levitate_damping = 0.06;
#elif 1
	  side->levitate_dist = 4;
	  side->levitate_damping = 0.06;
#else
	  side->levitate_dist = 3;
	  side->levitate_damping = 0.06;
#endif
	}
      else
	{
	  side->levitate_factor = 0;
	}
    }

  /* Generate the enter plane for this segment. */
  if (type == 'e')
    {
      plane_coefficients (&vertex[3*surface->vertex[0].v],
			  &vertex[3*surface->vertex[1].v],
			  &vertex[3*surface->vertex[2].v],
			  segment->enter_plane);
    }

  /* Generate the trigger planes. */
  if (type == 'l')
    {
      plane_coefficients (&vertex[3*surface->vertex[0].v],
			  &vertex[3*surface->vertex[1].v],
			  &vertex[3*surface->vertex[2].v],
			  segment->left_trigger_plane);
    }
  if (type == 'm')
    {
      plane_coefficients (&vertex[3*surface->vertex[0].v],
			  &vertex[3*surface->vertex[1].v],
			  &vertex[3*surface->vertex[2].v],
			  segment->mid_trigger_plane);

      /* And the forwards vector. */
      segment->forwards[0]
	= vertex[3*surface->vertex[2].v]
	- vertex[3*surface->vertex[1].v];
      segment->forwards[1]
	= vertex[3*surface->vertex[2].v+1]
	- vertex[3*surface->vertex[1].v+1];
      segment->forwards[2]
	= vertex[3*surface->vertex[2].v+2]
	- vertex[3*surface->vertex[1].v+2];

      /* And the autopilot midpoint. */
      segment->autopilot_midpoint[0]
	= vertex[3*surface->vertex[1].v];
      segment->autopilot_midpoint[1]
	= vertex[3*surface->vertex[1].v+1];
      segment->autopilot_midpoint[2]
	= vertex[3*surface->vertex[1].v+2];
    }
  if (type == 'r')
    {
      plane_coefficients (&vertex[3*surface->vertex[0].v],
			  &vertex[3*surface->vertex[1].v],
			  &vertex[3*surface->vertex[2].v],
			  segment->right_trigger_plane);
    }
}

/* These two functions are fairly magical. They parse a
 * list formatted like this: "3,5-6,8", and allow you to
 * read out individual numbers. You call start_parse_list
 * with the list, and you then call next_parse_list, reading
 * each number in turn from the list until this function
 * returns -1.
 *
 * The magical bit is that you can next calls to start_parse_list/
 * next_parse_list and everything still works correctly. To
 * do this, these functions maintain a static stack.
 */
static struct list_parse_stack
{
  struct list_parse_stack *next;
  char *str;
  int current;
  int last_in_list;
} *list_parse_stack = 0;

static void
start_parse_list (char *str)
{
  struct list_parse_stack *p = malloc (sizeof (struct list_parse_stack));
  p->next = list_parse_stack;
  p->str = str;
  p->current = p->last_in_list = -1;
  list_parse_stack = p;
}

static int
next_parse_list ()
{
  struct list_parse_stack *p = list_parse_stack;
  char *q, *r, *s;

  if (p->current == p->last_in_list)
    {
      if (p->str == 0)
	{
	  /*printf ("%p, -1\n", p);*/
	  list_parse_stack = list_parse_stack->next;
	  free (p);
	  return -1;
	}

      /* Chop the string at the next comma and update p->str. */
      r = p->str;
      q = strchr (r, ',');
      if (q != 0) { *q = '\0'; p->str = q+1; } else p->str = 0;

      /* Now parse the next section, namely the string pointed
       * to by `r'.
       */
      s = strchr (r, '-');
      if (s != 0)
	{
	  *s = '\0';
	  sscanf (r, "%d", &p->current);
	  sscanf (s+1, "%d", &p->last_in_list);
	}
      else
	{
	  sscanf (r, "%d", &p->current);
	  p->last_in_list = p->current;
	}
    }
  else
    (p->current) ++;

  /*printf ("%p, %d\n", p, p->current);*/
  return p->current;
}

/* Re-use this useful function for parsing the description file. */
#define read_line read_ac3d_line

/* Parse description file and use it to annotate segment list object. */
static void
parse_description ()
{
  char **t;
  int n, i, s, j, tex;

  for (;;)
    {
      n = read_line (&t);
      if (n < 1)
	{
	  /* End of file. Stop. */
	  return;
	}
      if (strcasecmp (t[0], "segment") == 0)
	{
	  /* Segment description. */
	  start_parse_list (t[1]);
	  while ((s = next_parse_list ()) >= 0)
	    {
	      for (i = 2; i < n; ++i)
		{
		  if (strcasecmp (t[i], "lookahead") == 0)
		    {
		      sscanf (t[i+1], "%d", &list->segment[s].lookahead);
		      i++;
		    }
		  else if (strcasecmp (t[i], "start") == 0)
		    list->segment[s].trigger |= TRIGGER_START;
		  else if (strcasecmp (t[i], "left-fast") == 0)
		    list->segment[s].trigger |= TRIGGER_LEFT_FAST;
		  else if (strcasecmp (t[i], "left-faster") == 0)
		    list->segment[s].trigger |= TRIGGER_LEFT_FASTER;
		  else if (strcasecmp (t[i], "left-powerup") == 0)
		    list->segment[s].trigger |= TRIGGER_LEFT_POWERUP;
		  else if (strcasecmp (t[i], "right-fast") == 0)
		    list->segment[s].trigger |= TRIGGER_RIGHT_FAST;
		  else if (strcasecmp (t[i], "right-faster") == 0)
		    list->segment[s].trigger |= TRIGGER_RIGHT_FASTER;
		  else if (strcasecmp (t[i], "right-powerup") == 0)
		    list->segment[s].trigger |= TRIGGER_RIGHT_POWERUP;
		  else
		    {
		      fprintf (stderr, "mktrack: unknown element %s in segment defn\n",
			       t[i]);
		      exit (1);
		    }
		}
	    }
	}
      else if (strcasecmp (t[0], "scenery") == 0)
	{
	  /* Scenery object description. */
	  start_parse_list (t[1]);
	  while ((s = next_parse_list ()) >= 0)
	    {
	      list->scenery[s].is_animated = 0;

	      for (i = 2; i < n; ++i)
		{
		  if (strcasecmp (t[i], "copy") == 0)
		    {
		      sscanf (t[i+1], "%d", &j);
		      copy_scenery_object (j, s);
		      i++;
		    }
		  else if (strcasecmp (t[i], "visible") == 0)
		    {
		      start_parse_list (t[i+1]);
		      while ((j = next_parse_list ()) >= 0)
			append_visible_scenery_to_segment (j, s);
		      i++;
		    }
		  else if (strcasecmp (t[i], "texture") == 0)
		    {
		      tex = allocate_texture (strdup (t[i+1]));
		      for (j = 0; j < list->scenery[s].nr_polys; ++j)
			list->scenery[s].poly[j].texture_num = tex;
		      i++;
		    }
		  else
		    {
		      fprintf (stderr, "mktrack: unknown element %s in scenery defn\n",
			       t[i]);
		      exit (1);
		    }
		}
	    }
	}
      else if (strcasecmp (t[0], "animator") == 0)
	{
	  /* Animators are similar to scenery objects. */
	  sscanf (t[1], "%d", &s);
	  allocate_scenery (list, s);
	  list->scenery[s].is_animated = 1;
	  list->scenery[s].is_start_lamp = 0;
	  list->scenery[s].nr_frames = -1;
	  for (i = 2; i < n; ++i)
	    {
	      if (strcasecmp (t[i], "start-lamp") == 0)
		{
		  list->scenery[s].is_start_lamp = 1;
		}
	      else if (strcasecmp (t[i], "visible") == 0)
		{
		  start_parse_list (t[i+1]);
		  while ((j = next_parse_list ()) >= 0)
		    append_visible_scenery_to_segment (j, s);
		  i++;
		}
	      else if (strcasecmp (t[i], "nr-frames") == 0)
		{
		  sscanf (t[i+1], "%d", &list->scenery[s].nr_frames);
		  list->scenery[s].frame = malloc (sizeof (int) *
						   list->scenery[s].nr_frames);
		  list->scenery[s].ftime = malloc (sizeof (double) *
						   list->scenery[s].nr_frames);
		  i++;
		}
	      else if (strcasecmp (t[i], "frames") == 0)
		{
		  if (list->scenery[s].nr_frames == -1)
		    {
		      fprintf (stderr, "mktrack: must declare nr-frames first in animator defn\n");
		      exit (1);
		    }
		  for (j = 0; j < list->scenery[s].nr_frames; ++j)
		    sscanf (t[i+j+1], "%d", &list->scenery[s].frame[j]);
		  i += list->scenery[s].nr_frames;
		}
	      else if (strcasecmp (t[i], "frame-times") == 0)
		{
		  if (list->scenery[s].nr_frames == -1)
		    {
		      fprintf (stderr, "mktrack: must declare nr-frames first in animator defn\n");
		      exit (1);
		    }
		  for (j = 0; j < list->scenery[s].nr_frames; ++j)
		    sscanf (t[i+j+1], "%lf", &list->scenery[s].ftime[j]);
		  i += list->scenery[s].nr_frames;
		}
	      else
		{
		  fprintf (stderr, "mktrack: unknown element %s in animator defn\n",
			   t[i]);
		  exit (1);
		}
	    }
	}
      else if (strcasecmp (t[0], "sky") == 0)
	{
	  /* Sky definition. */
	  for (i = 1; i < n; ++i)
	    {
	      if (strcasecmp (t[i], "texture") == 0)
		{
		  sky_tex = allocate_texture (strdup (t[i+1]));
		  i++;
		}
	      else if (strcasecmp (t[i], "visible") == 0)
		{
		  start_parse_list (t[i+1]);
		  while ((s = next_parse_list ()) >= 0)
		    list->segment[s].sky_visible = 1;
		  i++;
		}
	      else
		{
		  fprintf (stderr, "mktrack: unknown element %s in sky defn\n",
			   t[i]);
		  exit (1);
		}
	    }
	}
      else
	{
	  fprintf (stderr, "mktrack: unknown element %s in description file\n",
		   t[0]);
	  exit (1);
	}
    }
}
