/* $Id: error.c,v 1.18 2005/10/11 20:25:04 johans Exp $ */

#include	<sys/types.h>
#include	<sys/stat.h>

#include	<stdio.h>
#include	<unistd.h>
#include	<stdlib.h>
#include	<pwd.h>

#include	"config.h"
#include	"local.h"
#include	"setenv.h"
#include	"path.h"
#include	"mystring.h"
#include	"htconfig.h"

struct virtual			*current;
struct configuration	config;

void	error			(const char *);
void	redirect		(const char *, int);
void	server_error		(const char *, const char *);
static	int	difference		(const char *, const char *);
static	int	check_user		(const struct passwd *);
static	void	user_unknown		(void);
static	void	post_on_non_cgi		(void);
static	void	invalid_path		(void);
static	void	dir_not_avail		(void);
static	void	not_regular		(void);
static	void	permission		(void);
static	void	not_found		(void);
static	void	no_relative_urls	(void);
static	void	bad_request		(void);
static	void	unknown_method		(void);
static	void	unauthorized		(void);
static	void	precondition_failed	(void);
static	void	local_no_page		(void);
static	void	local_invalid_link	(void);
static	void	local_no_pay		(void);

typedef	struct
{
	char		username[XS_USER_MAX];
	int		rank;
} userrank;

static	const	char	*error_code, *error_readable, *error_url,
			*error_url_escaped, *error_url_expanded,
			*local_mode_str;
static	char		buffer[BUFSIZ], *temp;
char			rootdir[XS_PATH_MAX];
int			localmode;

void
error(const char *what)
{
	printf("Content-type: text/html\r\n\r\n");
	printf("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n");
	printf("<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.1//EN\" "
		"\"http://www.w3.org/TR/xhtml11/DTD/xhtml11.dtd\">\n");
	printf("<html xmlns=\"http://www.w3.org/1999/xhtml\">\n");
	printf("<head><title>500 Error occurred</title></head>\n");
	printf("<body><h1>500 Error occurred</h1>\n");
	printf("<p>The <tt>error</tt> utility encountered the following\n");
	printf("error: <b>%s</b></p></body></html>\n", what);
	exit(0);
}

void
redirect(const char *redir, int code)
{
	printf("[redirect() called - transform_user_dir() is broken]\n");
	(void)redir;
	(void)code;
}

void
server_error(const char *readable, const char *code)
{
	printf("[server_error() called - transform_user_dir() is broken]\n");
	(void)readable;
	(void)code;
}

static	int
difference(const char *what1, const char *what2)
{
	int		rank;
	const	char	*search, *follow;
	char		ch;

	rank = 0;
	for (search = what1, follow = what2; (ch = *search); search++)
	{
		if (ch == *follow)
			rank--;
		else if (!strchr(what2, ch))
			rank += 5;
		else
			rank += 2;
		if (*follow)
			follow++;
	}
	rank += strlen(follow);
	if (rank < 0)
		rank = 0;
	return(rank);
}

static	int
check_user(const struct passwd *userinfo)
{
	char		dirname[XS_PATH_MAX], *end;

	if (transform_user_dir(dirname, userinfo, 0))
		return(0);
	strlcat(dirname, "/", XS_PATH_MAX);
	end = dirname + strlen(dirname);
	strlcat(dirname, INDEX_HTML, XS_PATH_MAX);
	if (!access(dirname, F_OK))
		return(1);
	*end = '\0';
	strlcat(dirname, INDEX_HTML_2, XS_PATH_MAX);
	if (!access(dirname, F_OK))
		return(1);
	return(0);
}

static	void
user_unknown()
{
	userrank		top[10];
	const	struct	passwd	*user;
	int			count, count2, rank, said;
	char			filename[XS_PATH_MAX];

	strlcpy(buffer, error_url_escaped + 2, BUFSIZ);
	if ((temp = strchr(buffer, '/')))
		*temp = 0;
	printf("<p>The user <b>%s</b> is unknown to this system.</p>\n",
		buffer);
	strlcpy(buffer, error_url + 2, BUFSIZ);
	if ((temp = strchr(buffer, '/')))
		*(temp++) = 0;
	for (count = 0; count < 10; count++)
	{
		top[count].username[0] = 0;
		top[count].rank = 10000;
	}
	while ((user = getpwent()))
	{
		rank = difference(buffer, user->pw_name);
		count = 0;
		while ((count < 10) && (top[count].rank < rank))
			count++;
		if (count < 10)
		{
			if (!check_user(user))
				continue;
			for (count2 = 9; count2 > count; count2--)
				top[count2] = top[count2 - 1];
			top[count].rank = rank;
			strlcpy(top[count].username, user->pw_name, XS_USER_MAX);
		}
	}
	said = 0;
	for (count = 0; (count < 10) && (top[count].rank <= 20); count++)
	{
		if (!said)
		{
			printf("<p>There are a few usernames that look\n");
			printf("similar to what you typed. Perhaps you\n");
			printf("meant one of these users:</p>\n<ul>\n");
			said = 1;
		}
		printf("<li><a href=\"/%%7E%s/\">%s</a></li>\n",
			top[count].username, top[count].username);
	}
	if (said)
		printf("</ul>\n");
	else
	{
		printf("<p>There are no usernames here that even look like\n");
		printf("what you typed...</p>\n");
	}
	printf("<p>You may look at the <a href=\"/\">main index page</a>");
	snprintf(filename, XS_PATH_MAX, "%s/users.html", HTTPD_DOCUMENT_ROOT);
	if (!access(calcpath(filename), F_OK))
		printf(" or the <a href=\"/users.html\">user list</a>\n");
	printf(".</p>\n");
}

static	void
post_on_non_cgi()
{
	printf("<p>You or your browser has attempted to use the <b>POST</b>\n");
	printf("method on something that is not a CGI binary. <b>POST</b>\n");
	printf("may only be used on CGI binaries. You can try using the\n");
	printf("<b>GET</b> and/or <b>HEAD</b> methods instead.</p>\n");

	printf("<p><a href=\"/\">Get out of here!</a></p>\n");
}

static	void
invalid_path()
{
	printf("<p>You have asked for a URL that the server does not like.\n");
	printf("In particular, the server does not accept paths with\n");
	printf("<b>..</b> in them. Please retry using another URL.</p>\n");
	printf("<p><a href=\"/\">Get out of here!</a></p>\n");
}

static	void
not_found()
{
	char		prefix[BUFSIZ], base[XS_PATH_MAX], filename[XS_PATH_MAX];
	const	char	*begin, *match;
	int		len;
	struct	stat	statbuf;

	printf("<p>The file <b>%s</b> does not exist on this server.</p>\n",
		error_url_escaped);
	if (error_url[1] == '~')
	{
		match = error_url + 2;
		while (*match && (*match != '/'))
			match++;
		begin = match;
		strlcpy(prefix, error_url, BUFSIZ);
		prefix[match - error_url] = 0;
		strlcpy(base, error_url_expanded, BUFSIZ);
		base[(strlen(error_url_expanded) - strlen(error_url) +
			(match - error_url))] = 0;
		strlcat(base, "/", XS_PATH_MAX);
	} else
	{
		prefix[0] = 0;
		snprintf(base, XS_PATH_MAX, "%s/%s/",
			HTTPD_ROOT, HTTPD_DOCUMENT_ROOT);
		begin = error_url;
	}

	len = strlen(begin);
	while (len >= 0)
	{
		snprintf(buffer, BUFSIZ, "%s%*.*s",
			base, -len, len, begin);
		if (!stat(buffer, &statbuf))
		{
			if (S_ISREG(statbuf.st_mode))
			{
				snprintf(buffer, BUFSIZ, "%s%*.*s",
					prefix, -len, len, begin);
				break;
			}
			if (!(S_ISDIR(statbuf.st_mode)))
			{
				len--;
				continue;
			}
			snprintf(buffer, BUFSIZ, "%s%*.*s%s%s",
				base, -len, len, begin,
				(begin[len-1] == '/') ? "" : "/", INDEX_HTML);
			if (!stat(buffer, &statbuf) && S_ISREG(statbuf.st_mode))
			{
				snprintf(buffer, BUFSIZ, "%s%*.*s%s",
					prefix, -len, len, begin,
					(begin[len - 1] == '/') ? "" : "/");
				break;
			}
			snprintf(buffer, BUFSIZ, "%s%*.*s%s%s",
				base, -len, len, begin,
				(begin[len-1] == '/') ? "" : "/", INDEX_HTML_2);
			if (!stat(buffer, &statbuf) && S_ISREG(statbuf.st_mode))
			{
				snprintf(buffer, BUFSIZ, "%s%*.*s%s",
					prefix, -len, len, begin,
					(begin[len - 1] == '/') ? "" : "/");
				break;
			}
		}
		len--;
	}
	if ((len >= 0) && strcmp(buffer, error_url) && strcmp(buffer, "/"))
	{
		printf("<p>The path does seem to partially exist.\n");
		printf("Perhaps the path <a href=\"%s\">%s</a> will\n",
			buffer, buffer);
		printf ("help.</p>\n<p>Alternatively, y");
	} else
		printf("<p>Y");
	printf("ou may take a look at <a href=\"/\">the main index</a>");
	snprintf(filename, XS_PATH_MAX, "%s/users.html", HTTPD_DOCUMENT_ROOT);
	if (!access(calcpath(filename), F_OK))
		printf(" or the <a href=\"/users.html\">user list</a>\n");
	printf(".</p>\n");
}

static	void
not_regular()
{
	printf("<p>What you requested is neither a directory nor a file.\n");
	printf("This error should never occur. Please notify the\n");
	printf("system administration of this machine.</p>\n");
}

static	void
permission()
{
	printf("<p>The file <b>%s</b>, which you tried to retrieve from\n",
		error_url_escaped);
	printf("this server, is protected. You are not allowed to\n");
	printf("retrieve it. If this seems to be in error, please\n");
	printf("contact the person that created the file.</p>\n");
	printf("<p><a href=\"/\">Get out of here!</a></p>\n");
}

static	void
dir_not_avail()
{
	printf("<p>The directory in which the file <b>%s</b> is located\n",
		error_url_escaped);
	printf("is currently not available for retrieval. Perhaps you\n");
	printf("can try later.</p>\n");
	printf("<p><a href=\"/\">Get out of here!</a></p>\n");
}

static	void
no_relative_urls()
{
	printf("<p>Your browser has made a <em>relative</em> request to\n");
	printf("this server. This server, however, does not support\n");
	printf("relative URLs.</p>\n");
	printf("<p><a href=\"/\">Get out of here!</a></p>\n");
}

static	void
bad_request()
{
	const	char	*env;

	env = getenv("SERVER_PROTOCOL");
	printf("<p>Your browser has made a <em>%s</em> request to\n", env);
	printf("this server, which is not valid according to the\n");
	printf ("specification. The server can not possibly give you a\n");
	printf ("sensible answer.</p>\n");
	printf("<p><a href=\"/\">Get out of here!</a></p>\n");
}

static	void
unknown_method()
{
	const	char	*env;

	env = getenv("REQUEST_METHOD");
	printf("<p>Your browser has used a retrieval method other than\n");
	printf("<b>GET</b>, <b>POST</b>, <b>HEAD</b> and <b>OPTIONS</b>.\n");
	printf("In fact it used the method <b>%s</b>,\n",
		env ? env : "(unknown)");
	printf("which this server does not understand.</p>\n");
	printf("<p><a href=\"/\">Get out of here!</a></p>\n");
}

static	void
unauthorized()
{
	printf("<p>You have entered a usercode/password combination\n");
	printf("which is not valid for the URL that you have requested\n");
	printf("Please try again with another usercode and/or password.</p>\n");
}

static	void
precondition_failed()
{
	printf("<p>You have asked for a certain precondition which is\n");
	printf("not met by the requested data. So this data will not be");
	printf("shown.</p>\n");
}

static	void
local_no_page()
{
	const	char	*env;
	char		filename[XS_PATH_MAX];

	strlcpy(buffer, error_url_escaped + 2, BUFSIZ);
	if ((temp = strchr(buffer, '/')))
		*temp = 0;
	printf("<p>The user <b>%s</b>, whom you specified in your URL,\n",
		buffer);
	printf("exists on this system, but has no home page.\n");
	if ((env = getenv("REMOTE_ADDR")) && !strncmp(env, "131.155.140.", 12))
	{
		printf("If you would like to start a home page,\n");
		printf("please mail to <a href=\"mailto:");
		printf("www@stack.nl\">");
		printf("www@stack.nl</a> for details.>");
	}
	printf("</p>\n");
	printf("<p>Perhaps you meant somebody else; in this case, please\n");
	printf("have a look at the <a href=\"/\">main index</a>");
	snprintf(filename, XS_PATH_MAX, "%s/users.html", HTTPD_DOCUMENT_ROOT);
	if (!access(calcpath(filename), F_OK))
		printf(" or the <a href=\"/users.html\">user list</a>\n");
	printf(".</p>\n");
}

static	void
local_invalid_link()
{
	strlcpy(buffer, error_url_escaped + 2, BUFSIZ);
	if ((temp = strchr(buffer, '/')))
		*temp = 0;
	printf("<p>An error has been made in linking <b>/www/%s</b> to\n",
		buffer);
	printf("a correct location. Please contact\n");
	printf("<A HREF=\"mailto:www@stack.nl\">");
	printf("www@stack.nl</A>. The problem will then be corrected as\n");
	printf("soon as possible.</p>\n");
}

static	void
local_no_pay()
{
	char		filename[XS_PATH_MAX];

	strlcpy(buffer, error_url_escaped + 2, BUFSIZ);
	if ((temp = strchr(buffer, '/')))
		*temp = 0;
	printf("<p>The user <b>%s</b>, whom you specified in your URL,\n",
		buffer);
	printf("has not paid his/her member fee to our computer society\n");
	printf("this year. The pages will be online again once the author\n");
	printf("has decided that he/she wants to remain a member.</p>\n");
	printf("<p>Return to the <a href=\"/\">main index</a>\n");
	printf("for more information about our society");
	snprintf(filename, XS_PATH_MAX, "%s/users.html", HTTPD_DOCUMENT_ROOT);
	if (!access(calcpath(filename), F_OK))
		printf(" or the <a href=\"/users.html\">user list</a>\n");
	printf(".</p>\n");
}

int
main(int argc, char **argv)
{
	if (getenv("HTTPD_ROOT"))
		strlcpy(rootdir, getenv("HTTPD_ROOT"), XS_PATH_MAX);
	else
		strlcpy(rootdir, HTTPD_ROOT, XS_PATH_MAX);
	alarm(240);
	if (!(error_code = getenv("ERROR_CODE")) ||
		!(error_readable = getenv("ERROR_READABLE")) ||
		!(error_url = getenv("ERROR_URL")) ||
		!(error_url_expanded = getenv("ERROR_URL_EXPANDED")) ||
		!(error_url_escaped = getenv("ERROR_URL_ESCAPED")) ||
		!(local_mode_str = getenv("LOCALMODE")))
		error("Not called properly - the server must call me");
	localmode = atoi(local_mode_str);
	printf("Content-type: text/html\r\n");
	printf("Status: %s\r\n\r\n", error_readable);
	printf("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n");
	printf("<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.1//EN\" "
		"\"http://www.w3.org/TR/xhtml11/DTD/xhtml11.dtd\">\n");
	printf("<html xmlns=\"http://www.w3.org/1999/xhtml\">\n");
	printf("<head><title>%s</title></head>\n", error_readable);
	printf("<body><h1>%s</h1>\n", error_readable);
	if (!strcmp(error_code, "USER_UNKNOWN"))
		user_unknown();
	else if (!strcmp(error_code, "POST_ON_NON_CGI"))
		post_on_non_cgi();
	else if (!strcmp(error_code, "INVALID_PATH"))
		invalid_path();
	else if (!strcmp(error_code, "DIR_NOT_AVAIL"))
		dir_not_avail();
	else if (!strcmp(error_code, "NOT_REGULAR"))
		not_regular();
	else if (!strcmp(error_code, "PERMISSION"))
		permission();
	else if (!strcmp(error_code, "NOT_FOUND"))
		not_found();
	else if (!strcmp(error_code, "NO_RELATIVE_URLS"))
		no_relative_urls();
	else if (!strcmp(error_code, "BAD_REQUEST"))
		bad_request();
	else if (!strcmp(error_code, "UNKNOWN_METHOD"))
		unknown_method();
	else if (!strcmp(error_code, "UNAUTHORIZED"))
		unauthorized();
	else if (!strcmp(error_code, "PRECONDITION_FAILED"))
		precondition_failed();
	else if (!strcmp(error_code, "LOCAL_NO_PAGE"))
		local_no_page();
	else if (!strcmp(error_code, "LOCAL_INVALID_LINK"))
		local_invalid_link();
	else if (!strcmp(error_code, "LOCAL_NO_PAY"))
		local_no_pay();
	printf("</body></html>\n");
	(void)argc;
	(void)argv;
	return 0;
}
