/* c_sector.hh 1.15 95/12/23 03:11:30 */


// xspacewarp by Greg Walker (gow@math.orst.edu)

// This is free software. Non-profit redistribution and/or modification
// is allowed and welcome.


// describe the class Sector. Sector serves a go-between for the various
// space object classes. Sector coordinates these classes and maintains
// bookkeeping on what space objects are in a sector of the universe and
// where they are located. Sector keeps track of when space objects enter
// or leave the sector, tracks whether a faser/torpedo hit something,
// and calls the update method for each jovian within the sector.
// Sector maintains a singly linked list of the stars inside the sector
// and a doubly linked list of the jovians in the sector.

#ifndef _SECTOR_
#define _SECTOR_

#include <X11/Xlib.h>		// for draw method
#include "common.hh"
#include "c_body.hh"
#include "c_base.hh"
#include "c_blackhole.hh"
#include "c_endever.hh"
#include "c_jovian.hh"
#include "c_star.hh"
#include "space_objects.hh"
#include "messages.hh"


static const int ATTACK_STR_LEN1 = (XtNumber(attack_warning_str_1) - 1);
static const int ATTACK_STR_LEN2 = (XtNumber(attack_warning_str_2) - 1);


class Sector
{
public:
  Sector();
  void erase();
  void setucoors(int ur, int uc);
  bool add_base(Base& base);
  bool add_blackhole(Blackhole& bh);
  void add_endever();
  bool add_jovian(Jovian* j_pt);	// used when initializing universe
  bool add_jovian(Jovian_rec* jr_pt); // used dynamically when jovians leap
  bool add_star(Star* s_pt);
  void rm_base();
  void rm_endever();
  Jovian_rec* rm_jovian(int id);
  int getbasepop() const {return (base_pt? 1:0);}
  int getbhpop() const {return (bh_pt? 1:0);}
  int getendvpop() const {return (endv_pt? 1:0);}
  int getjovpop() const {return (njovians);}
  bool getmask() const {return (mask);}
  void setmask() {mask = true;}
  void draw(Drawable drawable) const;
  Body* occupant(int row, int col) const;
  HitReport detect_hit(const Shot& sh) const;
  void jovian();		// call jovian update method
  void energize();
  Point getbase();		// if sector contains base, return position
  Point getendever();		// if sector contains endever, return position
  bool hasendever() {return (endv_pt? true:false);}

  // "Fed. base at (3,1) is under attack!" stores mesg for flash_to() to use.
  static char attack_msg[ATTACK_STR_LEN1 + ATTACK_STR_LEN2 + 6];
  static Ucoors attack_loc;	// sector (location) where attack is occuring
				// so flash_to() can tell when to stop flashing.
private:
  Ucoors ucoors;		// position of this sector within universe
  bool overlap(int row, int col, int width) const;
  bool near_jovian(int row, int col) const;
  bool near_target(int row, int col) const;
  Base *base_pt;		// at most one base per sector
  Blackhole *bh_pt;		// at most one blackhole per sector
  Endever *endv_pt;		// only one endever
  Jovian_rec *jovrec_pt;	// doubly linked list of jovians
  Star_rec *starrec_pt;		// singly linked list of stars
  int njovians;			// number of jovians in sector
  int nstars;			// number of stars in sector
  bool mask;			// whether sector masks number of jovians
  bool intersect(Point nw, Point ne, Point se, Point sw,
		 Point from, Point to) const;
  Point newto(Point nw, Point ne, Point se, Point sw,
	      Point from, Point to, double sine, double cosine) const;
};


#endif				// _SECTOR_

// end
