/* draw_scanner.cc 1.8 95/12/23 03:11:39 */


// xspacewarp by Greg Walker (gow@math.orst.edu)

// This is free software. Non-profit redistribution and/or modification
// is allowed and welcome.


// draw the endever long-range scanner display


#include <stdio.h>		// sprintf()
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include "common.hh"
#include "params.hh"
#include "c_sector.hh"
#include "c_block.hh"
#include "globals.hh"
#include "messages.hh"


extern void connect_blocks(Drawable, GC, Block&, Block&);

static void draw_grid(void);
static void draw_info(void);
static int read_sector(char *sectorinfo, int row, int col);

// the visual survey of the entire universe

void draw_scanner(void)
{
  // clear previous stuff

  XFillRectangle(DISPLAY, pixmap, defrv_GC, 0, 0, GAMEW, GAMEH);

  // draw scanner on pixmap

  draw_grid();
  draw_info();

  // copy pixmap to window

  XCopyArea(DISPLAY, pixmap, XtWindow(widget),
	    def_GC, 0, 0, GAMEW, GAMEH, 0, 0);
}


// routine for drawing the scanner grid and the numeric row/col labels.
// drawing is all done on the pixmap.

static void draw_grid(void)
{
  Block block1, block2;
  Point origin;
  int i, row, col;
  static const char* nums[9] = {"1", "2", "3", "4", "5", "6", "7", "8", "9"};

  // make horizontal lines

  block1.setcol(SCANCOL+1);
  block2.setcol(SCANCOL+SCANCOLS-1);
  for (row = SCANROW+1; row <= SCANROW+SCANROWS-1; row += 2)
  {
    block1.setrow(row);
    block2.setrow(row);
    connect_blocks(pixmap, def_GC, block1, block2);
  }

  // make vertical lines

  block1.setrow(SCANROW+1);
  block2.setrow(SCANROW+SCANROWS-1);
  for (col = SCANCOL+1; col <= SCANCOL+SCANCOLS-1; col += 5)
  {
    block1.setcol(col);
    block2.setcol(col);
    connect_blocks(pixmap, def_GC, block1, block2);
  }

  // draw numeric labels for columns

  block1.setrow(SCANROW);
  for (col = SCANCOL+3, i = 0; col <= SCANCOL+SCANCOLS-4; col += 5, i++)
  {
    block1.setcol(col);
    origin = block1.origin();
    XDrawString(DISPLAY, pixmap, def_GC, origin.x, origin.y, nums[i], 1);
  }

  // draw numeric labels for rows

  block1.setcol(SCANCOL);
  for (row = SCANROW+2, i = 0; row <= SCANROW+SCANROWS-2; row += 2, i++)
  {
    block1.setrow(row);
    origin = block1.origin();
    XDrawString(DISPLAY, pixmap, def_GC, origin.x, origin.y, nums[i], 1);
  }
}


// routine for writing the sector info onto the scanner display.
// all drawing is done on the pixmap.

static void draw_info(void)
{
  Block block;			// block in drawable at which to draw strings
  Point origin;
  int urow, ucol;		// indices into universe matrix
  int row, col;			// coordinates in graphics drawable
  int n;
  char sectorinfo[5];		// sector summary info to display

  for (urow = 0; urow < UROWS; urow++)
  {
    row = SCANROW + 2*urow + 2;
    block.setrow(row);
    for (ucol = 0; ucol < UCOLS; ucol++)
    {
      if (n = read_sector(sectorinfo, urow, ucol))
      {
	col = SCANCOL + 5*ucol + 3 - n/3;	  // n/3 centers string in box
	block.setcol(col);
	origin = block.origin();
	XDrawString(DISPLAY, pixmap, def_GC, origin.x, origin.y, sectorinfo, n);
      }
    }
  }
}


// pass a char[5] to hold sector information in the form
// [endever][base][number jovians or mask][blackhole].  For
// example, "EB4?", "BM", "B2", "1?", etc where E=endever,
// B=base, M=mask, ?=blackhole, [int]=number of jovians.  Also
// pass the row/col of the sector in the universe matrix.

static int read_sector(char *sectorinfo, int row, int col)
{
  int n = 0;
  static const char nums[9] = {'1', '2', '3', '4', '5', '6', '7', '8', '9'};

  // if sector contains endever

  if (universe[row][col].getendvpop())
  {
    sectorinfo[n] = sector_endever_symbol;
    n++;
  }

  // if sector contains a base

  if (universe[row][col].getbasepop())
  {
    sectorinfo[n] = sector_base_symbol;
    n++;
  }

  // number of jovians or if there is a mask

  if (universe[row][col].getmask())
  {
    sectorinfo[n] = 'M';
    n++;
  }
  else if (universe[row][col].getjovpop())
  {
    sectorinfo[n] = nums[universe[row][col].getjovpop() - 1];
    n++;
  }

  // if sector contains a blackhole
  
  if (universe[row][col].getbhpop())
  {
    sectorinfo[n] = sector_bh_symbol;
    n++;
  }

  sectorinfo[n] = '\000';
  return (n);
}
	       

















// end
