/*****************************************************************************/
/* XStarRoll - Letters fly to far, far, far away.                            */
/*                                                                           */
/* XStarRoll Copyright (c) 1998 Sakai Hiroaki.                               */
/* All Rights Reserved.                                                      */
/*                                                                           */
/* XStarRoll is a simple demonstration program for X11. Letters and pixmaps  */
/* fly to far, far, far away. You can fly optional letters and pixmaps, and  */
/* use XStarRoll as a background picture.                                    */
/*                                                                           */
/* XStarRoll is a free software. You may copy or distribute the original     */
/* XStarRoll freely. But, you may not modify or distribute the original      */
/* XStarRoll without permission of the author.                               */
/*****************************************************************************/

/*****************************************************************************/
/* Include Header Files.                                                     */
/*****************************************************************************/

#include<stdio.h>
#include<stdlib.h>
#include<X11/Xlib.h>
#include<X11/Intrinsic.h>
#include<X11/StringDefs.h>
#include<X11/keysym.h>
#include<X11/xpm.h>
#include<X11/Xaw/Label.h>

/*****************************************************************************/
/* XStarRoll's version.                                                      */
/*****************************************************************************/

#ifndef VERSION
#define VERSION "XStarRoll-unknown-version"
#endif

/*****************************************************************************/
/* If you DEBUG, define below definition.                                    */
/*****************************************************************************/

/*
#define DEBUG
*/

/*****************************************************************************/
/* Font.                                                                     */
/*****************************************************************************/

#define DEFAULT_FONT_NAME "8x16"

/*****************************************************************************/
/* Icon.                                                                     */
/*****************************************************************************/

#include"xstarroll_icon.xpm"
#define XSTARROLL_ICON xstarroll_icon

/*****************************************************************************/
/* Others.                                                                   */
/*****************************************************************************/

#define OFF  0
#define ON   1
#define FILL 2
#define READ_FROM_STRING 0
#define READ_FROM_FILE   1

/*****************************************************************************/
/* Default values.                                                           */
/*****************************************************************************/

#define DEFAULT_CANVAS_WIDTH 640
#define DEFAULT_CANVAS_HEIGHT 480
#define DEFAULT_SOURCE_WIDTH 0
#define DEFAULT_WIDTH 40
#define DEFAULT_LENGTH_X 0
#define DEFAULT_LENGTH_Y 100
#define DEFAULT_SPACE 1
#define DEFAULT_SCROLL 1
#define DEFAULT_START 0
#define DEFAULT_WAIT 0
#define DEFAULT_TITLE_NAME "XStarRoll"

/*****************************************************************************/
/* Default flags.                                                            */
/*****************************************************************************/

#define DEFAULT_ROOT_FLAG 0
#define DEFAULT_DRAW_FLAG 1
#define DEFAULT_QUIT_FLAG 0
#define DEFAULT_BLUR_FLAG 1

/*****************************************************************************/
/* Buffer.                                                                   */
/*****************************************************************************/

#define TMP_BUFFER_SIZE 2048
static char tmp_buffer[TMP_BUFFER_SIZE];

/*****************************************************************************/
/* Structures.                                                               */
/*****************************************************************************/

typedef struct GCBuffer {
  GC gc;
  unsigned long int pixel;
  struct GCBuffer * prev;
  struct GCBuffer * next;
} GCBuffer;

typedef struct GCBufferList {
  GCBuffer * start;
  GCBuffer * end;
} GCBufferList;

typedef struct SourceData {
  short int length;
  GCBuffer * gc_buffer;
  short int x;
  short int y;
  struct SourceData * prev;
  struct SourceData * next;
} SourceData;

typedef struct SourceDataList {
  SourceData * start;
  SourceData * end;
} SourceDataList;

typedef struct FontDataList {
  unsigned char pixel;
  unsigned char length;
  struct FontDataList * next;
} FontDataList;

typedef struct FontData {
  short int width;
  FontDataList ** list_array;
} FontData;

/*****************************************************************************/
/* Structure of XStarRoll.                                                   */
/*****************************************************************************/

typedef struct XStarRoll {
  XtAppContext app_context;
  Display * display;
  int screen, depth;
  Colormap colormap;
  Widget toplevel, canvas;
  Window toplevel_window, root_window, canvas_window;
  Pixmap canvas_pixmap, icon_pixmap;
  Atom atom_1, atom_2;
  unsigned long int foreground_color;
  unsigned long int background_color;
  unsigned long int foreground_pixel;
  unsigned long int background_pixel;
  GC gc_foreground, gc_background, gc_copy;
  char * font_name;
  Font font;
  XFontStruct * font_struct;
  char * title_name;
  int canvas_width, canvas_height, source_width, width;
  int icon_width, icon_height, font_height;
  int root_flag, draw_flag, quit_flag, blur_flag;
  int length_x, length_y, space, scroll, wait;
  long int start;
  GCBufferList * gc_buffer;
  SourceDataList * source_data;
  FontData * font_data;
} XStarRoll;

/*****************************************************************************/
/* Help Message.                                                             */
/*****************************************************************************/

char * help_string =
"\n\t" VERSION "\n"
"\tLetters fly to far, far, far away.\n"
"\n"
"\tDESCRIPTION\n"
"\n"
"    XStarRoll is a simple demonstration program for X11. Letters and pixmaps fly to far, far, far away. You can fly optional letters and pixmaps, and use XStarRoll as a background picture.\n"
"\n"
"\tSYNOPSIS\n"
"\n"
"\txstarroll [parameter1] [parameter2] ...\n"
"\n"
"    If a parameter is a pixmap filename, XStarRoll reads it and flies the pixmap, or, if the parameter is a text filename, XStarRoll reads it and flies letters in it, or else, xstarroll flies the parameter as it is.\n"
"    You can send letters to XStarRoll through the standard input.\n"
"\n"
"\tOPTIONS\n"
"\n"
"(X-ToolKit Options)    -display, -geometry, -name, -font, -fg and -bg are available.\n"
"--    End of options. XStarRoll ignores following options.\n"
"-    Read letters from the standard input.\n"
"-help, -h    Output this Message.\n"
"-root, -r    Draw the picture on the root window of the display.\n"
"-noroot, -nr    Open a new window and draw the picture on it. (Default)\n"
"-quit, -q    Draw the picture and exit.\n"
"-noquit, -nq    XStarRoll does not exit. (Default)\n"
"-blur, -b    Letters fly to the farther, become the more blurred. (Default)\n"
"-noblur, -nb    Letters do not become blurred.\n"
"-fill    Fill all dots entirely.\n"
"-source-width [x], -sw [x]    Specify the width of the source picture for a pixel.\n"
"-width [x], -w [x]    Specify the width of the source picture for a letter. Default value is 40.\n"
"-length-x [x], -lx [x]    Specify the length parameter of X-direction for drawing. Default value is same as the size of the source picture.\n"
"-length-y [x], -ly [x]    Specify the length parameter of Y-direction for drawing. Default value is 100.\n"
"-space [x], -sp [x]    Specify the space between lines. Default value is 1.\n"
"-scroll [x], -sc [x]    Specify the speed and direction of the scroll. Default value is 1.\n"
"-start [x], -st [x]    Specify the start point. Default value is 0.\n"
"-wait [x], -wt [x]    Specify the wait time for a micro second. Default value is 0.\n"
"\n"
"\tKEY OPERATIONS\n"
"\n"
"[Up], [Down], [Right], [Left]    Change the length parameters for drawing.\n"
"[a], [d]    Change the speed of the scroll.\n"
"[p], [n], [PageUp], [PageDown]    Scroll the picture.\n"
"[s]    Return to the start point.\n"
"[q]    Quit.\n"
"\n"
"\tEXAMPLES\n"
"\n"
"% ls -l | xstarroll\n"
"    - XStarRoll flies the output of ls.\n"
"% echo \"This is XStarRoll.\" | xstarroll\n"
"% xstarroll \"This is XStarRoll.\"\n"
"    - XStarRoll flies \"This is XStarRoll.\".\n"
"% echo \"is\" | xstarroll \"This\" - \"XStarRoll.\"\n"
"    - XStarRoll flies \"This\", \"is\" and \"XStarRoll.\".\n"
"% xstarroll sample.txt\n"
"    - XStarRoll flies letters in a file \"sample.txt\".\n"
"% xstarroll sample.xpm\n"
"    - XStarRoll flies a pixmap \"sample.xpm\".\n"
"% xstarroll \"This is XStarRoll.\" sample1.txt sample1.xpm sample2.txt sample2.xpm ...\n"
"    - XStarRoll flies \"This is XStarRoll.\", flies letters in a file sample1.txt, flies a pixmap sample1.xpm, flies letters in a file sample2.txt, flies a pixmap sample2.xpm ...\n"
"\n"
"    If the head of letters is a Tab code or \"#center\", XStarRoll flies letters at the center of the screen.\n"
"\n"
"\tSEE ALSO\n"
"\tX(1)\n"
"\n"
"\tAUTHOR\n"
"\tProgramed by Sakai Hiroaki.\n"
"\tE-Mail:sakai@miya.ee.kagu.sut.ac.jp\n"
"\thttp://wwwmiya.ee.kagu.sut.ac.jp/~sakai/myfreesoft.html\n"
"\n"
"\tCOPYRIGHT\n"
"\n"
"\tXStarRoll Copyright (c) 1998 Sakai Hiroaki.\n"
"\tAll Rights Reserved.\n"
"\n"
"    XStarRoll is a free software. You may copy or distribute the original XStarRoll freely. But, you may not modify or distribute the original XStarRoll without permission of the author.\n"
"\n";

/*****************************************************************************/
/* Functions.                                                                */
/*****************************************************************************/

/*****************************************************************************/
/* Operations of list structures.                                            */
/*****************************************************************************/

/*****************************************************************************/
/* GCBuffer.                                                                 */
/*****************************************************************************/

GCBuffer * CreateGCBuffer(Display * display, Window window,
			  unsigned long int pixel)
{
  GCBuffer * p;

  p = (GCBuffer *)malloc(sizeof(GCBuffer));
  p->gc = XCreateGC(display, window, 0, 0);
  XSetForeground(display, p->gc, pixel);
  XSetFillRule(display, p->gc, WindingRule);
  XSetLineAttributes(display, p->gc, 0, LineSolid, CapNotLast ,JoinMiter);
  p->pixel = pixel;
  p->prev = (GCBuffer *)NULL;
  p->next = (GCBuffer *)NULL;
  return (p);
}

void FreeGCBuffer(Display * display, GCBuffer * p)
{
  XFreeGC(display, p->gc);
  free(p);
}

GCBufferList * InitializeGCBufferList()
{
  GCBufferList * p;

  p = (GCBufferList *)malloc(sizeof(GCBufferList));
  p->start = (GCBuffer *)NULL;
  p->end = (GCBuffer *)NULL;
  return (p);
}

void FreeGCBufferList(Display * display, GCBufferList * p)
{
  GCBuffer * q;

  while (p->start != (GCBuffer *)NULL) {
    q = p->start->next;
    FreeGCBuffer(display, p->start);
    p->start = q;
  }
  free(p);
}

GCBuffer * GetGCBufferFromTop(GCBufferList * p)
{
  return (p->start);
}

GCBuffer * GetGCBufferFromBottom(GCBufferList * p)
{
  return (p->end);
}

GCBuffer * AddGCBufferToTop(GCBufferList * p, GCBuffer * d)
{
  if (d == (GCBuffer *)NULL) return (GCBuffer *)NULL;
  if (p->start == (GCBuffer *)NULL) {
    d->prev = (GCBuffer *)NULL;
    d->next = (GCBuffer *)NULL;
    p->start = d;
    p->end = d;
  } else {
    d->prev = (GCBuffer *)NULL;
    d->next = p->start;
    p->start->prev = d;
    p->start = d;
  }
  return (p->start);
}

GCBuffer * AddGCBufferToBottom(GCBufferList * p, GCBuffer * d)
{
  if (d == (GCBuffer *)NULL) return (GCBuffer *)NULL;
  if (p->start == (GCBuffer *)NULL) {
    d->prev = (GCBuffer *)NULL;
    d->next = (GCBuffer *)NULL;
    p->start = d;
    p->end = d;
  } else {
    d->next = (GCBuffer *)NULL;
    d->prev = p->end;
    p->end->next = d;
    p->end = d;
  }
  return (p->end);
}

GCBuffer * DeleteGCBufferFromTop(GCBufferList * p)
{
  GCBuffer * q;

  if (p->start == (GCBuffer *)NULL) return ((GCBuffer *)NULL);
  q = p->start;
  p->start = p->start->next;
  if (p->start == (GCBuffer *)NULL) p->end = (GCBuffer *)NULL;
  else p->start->prev = (GCBuffer *)NULL;
  q->prev = (GCBuffer *)NULL;
  q->next = (GCBuffer *)NULL;
  return (q);
}

GCBuffer * DeleteGCBufferFromBottom(GCBufferList * p)
{
  GCBuffer * q;

  if (p->start == (GCBuffer *)NULL) return ((GCBuffer *)NULL);
  q = p->end;
  p->end = p->end->prev;
  if (p->end == (GCBuffer *)NULL) p->start = (GCBuffer *)NULL;
  else p->end->next = (GCBuffer *)NULL;
  q->prev = (GCBuffer *)NULL;
  q->next = (GCBuffer *)NULL;
  return (q);
}

GCBuffer * DeleteGCBuffer(GCBufferList * p, GCBuffer * d)
{
  if (p->start == (GCBuffer *)NULL) return ((GCBuffer *)NULL);
  else if (d == (GCBuffer *)NULL) return ((GCBuffer *)NULL);
  else if (d->prev == (GCBuffer *)NULL)
    return (DeleteGCBufferFromTop(p));
  else if (d->next == (GCBuffer *)NULL)
    return (DeleteGCBufferFromBottom(p));
  else {
    d->prev->next = d->next;
    d->next->prev = d->prev;
    d->prev = (GCBuffer *)NULL;
    d->next = (GCBuffer *)NULL;
    return (d);
  }
}

GCBuffer * MoveGCBufferFromBottomToTop(GCBufferList * p)
{
  GCBuffer * q;

  q = DeleteGCBufferFromBottom(p);
  return (AddGCBufferToTop(p, q));
}

GCBuffer * MoveGCBufferFromTopToBottom(GCBufferList * p)
{
  GCBuffer * q;

  q = DeleteGCBufferFromTop(p);
  return (AddGCBufferToBottom(p, q));
}

GCBuffer * MoveGCBufferToTop(GCBufferList * p, GCBuffer * d)
{
  GCBuffer * q;

  q = DeleteGCBuffer(p, d);
  return (AddGCBufferToTop(p, q));
}

GCBuffer * MoveGCBufferToBottom(GCBufferList * p, GCBuffer * d)
{
  GCBuffer * q;

  q = DeleteGCBuffer(p, d);
  return (AddGCBufferToBottom(p, q));
}

/*****************************************************************************/
/* SourceData.                                                               */
/*****************************************************************************/

SourceData * CreateSourceData(int length, GCBuffer * gc_buffer, int x, int y)
{
  SourceData * p;

  p = (SourceData *)malloc(sizeof(SourceData));
  p->length = (short int)length;
  p->gc_buffer = gc_buffer;
  p->x = (short int)x;
  p->y = (short int)y;
  p->prev = (SourceData *)NULL;
  p->next = (SourceData *)NULL;
  return (p);
}

void FreeSourceData(SourceData * p)
{
  free(p);
}

SourceDataList * InitializeSourceDataList()
{
  SourceDataList * p;

  p = (SourceDataList *)malloc(sizeof(SourceDataList));
  p->start = (SourceData *)NULL;
  p->end = (SourceData *)NULL;
  return (p);
}

void FreeSourceDataList(SourceDataList * p)
{
  SourceData * q;

  while (p->start != (SourceData *)NULL) {
    q = p->start->next;
    FreeSourceData(p->start);
    p->start = q;
  }
  free(p);
}

SourceData * GetSourceDataFromTop(SourceDataList * p)
{
  return (p->start);
}

SourceData * GetSourceDataFromBottom(SourceDataList * p)
{
  return (p->end);
}

SourceData * AddSourceDataToTop(SourceDataList * p, SourceData * d)
{
  if (d == (SourceData *)NULL) return (SourceData *)NULL;
  if (p->start == (SourceData *)NULL) {
    d->prev = (SourceData *)NULL;
    d->next = (SourceData *)NULL;
    p->start = d;
    p->end = d;
  } else {
    d->prev = (SourceData *)NULL;
    d->next = p->start;
    p->start->prev = d;
    p->start = d;
  }
  return (p->start);
}

SourceData * AddSourceDataToBottom(SourceDataList * p, SourceData * d)
{
  if (d == (SourceData *)NULL) return (SourceData *)NULL;
  if (p->start == (SourceData *)NULL) {
    d->prev = (SourceData *)NULL;
    d->next = (SourceData *)NULL;
    p->start = d;
    p->end = d;
  } else {
    d->next = (SourceData *)NULL;
    d->prev = p->end;
    p->end->next = d;
    p->end = d;
  }
  return (p->end);
}

SourceData * DeleteSourceDataFromTop(SourceDataList * p)
{
  SourceData * q;

  if (p->start == (SourceData *)NULL) return ((SourceData *)NULL);
  q = p->start;
  p->start = p->start->next;
  if (p->start == (SourceData *)NULL) p->end = (SourceData *)NULL;
  else p->start->prev = (SourceData *)NULL;
  q->prev = (SourceData *)NULL;
  q->next = (SourceData *)NULL;
  return (q);
}

SourceData * DeleteSourceDataFromBottom(SourceDataList * p)
{
  SourceData * q;

  if (p->start == (SourceData *)NULL) return ((SourceData *)NULL);
  q = p->end;
  p->end = p->end->prev;
  if (p->end == (SourceData *)NULL) p->start = (SourceData *)NULL;
  else p->end->next = (SourceData *)NULL;
  q->prev = (SourceData *)NULL;
  q->next = (SourceData *)NULL;
  return (q);
}

SourceData * DeleteSourceData(SourceDataList * p, SourceData * d)
{
  if (p->start == (SourceData *)NULL) return ((SourceData *)NULL);
  else if (d == (SourceData *)NULL) return ((SourceData *)NULL);
  else if (d->prev == (SourceData *)NULL)
    return (DeleteSourceDataFromTop(p));
  else if (d->next == (SourceData *)NULL)
    return (DeleteSourceDataFromBottom(p));
  else {
    d->prev->next = d->next;
    d->next->prev = d->prev;
    d->prev = (SourceData *)NULL;
    d->next = (SourceData *)NULL;
    return (d);
  }
}

SourceData * MoveSourceDataFromBottomToTop(SourceDataList * p)
{
  SourceData * q;

  q = DeleteSourceDataFromBottom(p);
  return (AddSourceDataToTop(p, q));
}

SourceData * MoveSourceDataFromTopToBottom(SourceDataList * p)
{
  SourceData * q;

  q = DeleteSourceDataFromTop(p);
  return (AddSourceDataToBottom(p, q));
}

SourceData * MoveSourceDataToTop(SourceDataList * p, SourceData * d)
{
  SourceData * q;

  q = DeleteSourceData(p, d);
  return (AddSourceDataToTop(p, q));
}

SourceData * MoveSourceDataToBottom(SourceDataList * p, SourceData * d)
{
  SourceData * q;

  q = DeleteSourceData(p, d);
  return (AddSourceDataToBottom(p, q));
}

/*****************************************************************************/
/* Graphic.                                                                  */
/*****************************************************************************/

Pixmap MakePixmapFromFile(XStarRoll * p, char * filename, int * w, int * h)
{
  XpmAttributes attr;
  Pixmap pix, mask;

  attr.colormap = p->colormap;
  attr.closeness = 40000;
  attr.valuemask = XpmSize | XpmReturnPixels | XpmColormap | XpmCloseness;
  XpmReadFileToPixmap(p->display, p->canvas_window, filename,
		      &pix, &mask, &attr);
  *w = attr.width;
  *h = attr.height;
  return (pix);
}

Pixmap MakePixmapFromData(XStarRoll * p, char ** string, int * w, int * h)
{
  XpmAttributes attr;
  Pixmap pix, mask;

  attr.colormap = p->colormap;
  attr.closeness = 40000;
  attr.valuemask = XpmSize | XpmReturnPixels | XpmColormap | XpmCloseness;
  XpmCreatePixmapFromData(p->display, p->canvas_window, string,
			  &pix, &mask, &attr);
  *w = attr.width;
  *h = attr.height;
  return (pix);
}

/*****************************************************************************/
/* Functions for XStarRoll.                                                  */
/*****************************************************************************/

void SetDefaultValues(XStarRoll * p)
{
  p->canvas_width = DEFAULT_CANVAS_WIDTH;
  p->canvas_height = DEFAULT_CANVAS_HEIGHT;
  p->source_width = DEFAULT_SOURCE_WIDTH;
  p->width = DEFAULT_WIDTH;
  p->length_x = DEFAULT_LENGTH_X;
  p->length_y = DEFAULT_LENGTH_Y;
  p->space   = DEFAULT_SPACE;
  p->scroll = DEFAULT_SCROLL;
  p->start = DEFAULT_START;
  p->wait = DEFAULT_WAIT;
  p->root_flag = DEFAULT_ROOT_FLAG;
  p->draw_flag = DEFAULT_DRAW_FLAG;
  p->quit_flag = DEFAULT_QUIT_FLAG;
  p->blur_flag = DEFAULT_BLUR_FLAG;
  p->font_name = DEFAULT_FONT_NAME;
  p->title_name = DEFAULT_TITLE_NAME;
}

void ConnectDisplay(XStarRoll * p, int * argc, char ** argv)
{
  int n;
  Arg args[10];

  n = 0;
  XtSetArg(args[n], XtNwidth,  p->canvas_width); n++;
  XtSetArg(args[n], XtNheight, p->canvas_height); n++;
  p->toplevel = XtAppInitialize(&(p->app_context), "XStarRoll",
				NULL, 0, argc, argv, NULL, args, n);
}

void Help()
{
  printf("%s", help_string);
  exit (0);
}

void DeleteParameter(int * c, char ** v, int i)
{
  (*c)--;
  for (; i < (*c); i++)
    v[i] = v[i + 1];
}

int SetFlagOn(int * c, char ** v, int i)
{
  DeleteParameter(c, v, i);
  return (ON);
}

int SetFlagOff(int * c, char ** v, int i)
{
  DeleteParameter(c, v, i);
  return (OFF);
}

int SetFlag(int * c, char ** v, int i, int f)
{
  DeleteParameter(c, v, i);
  return (f);
}

int GetIntValueFromParameter(int * c, char ** v, int i, int value)
{
  int a;

  DeleteParameter(c, v, i);
  if (i < *c) {
    a = atoi(v[i]);
    DeleteParameter(c, v, i);
    return (a);
  }
  return (value);
}

char * GetStringValueFromParameter(int * c, char ** v, int i, char * value)
{
  char * a;

  DeleteParameter(c, v, i);
  if (i < *c) {
    a = v[i];
    DeleteParameter(c, v, i);
    return (a);
  }
  return (value);
}

void ReadParameters(XStarRoll * p, int * c, char ** v)
{
  int i = 1;

  while (i < *c) {
    if (!strcmp(v[i], "--")) while (i < *c) DeleteParameter(c, v, i);
    else if ( (!strcmp(v[i], "-help")) || (!strcmp(v[i], "-h")) ) Help();
    else if ( (!strcmp(v[i], "-root")) || (!strcmp(v[i], "-r")) )
      p->root_flag = SetFlagOn(c, v, i);
    else if ( (!strcmp(v[i], "-noroot")) || (!strcmp(v[i], "-nr")) )
      p->root_flag = SetFlagOff(c, v, i);
    else if ( (!strcmp(v[i], "-quit")) || (!strcmp(v[i], "-q")) )
      p->quit_flag = SetFlagOn(c, v, i);
    else if ( (!strcmp(v[i], "-noquit")) || (!strcmp(v[i], "-nq")) )
      p->quit_flag = SetFlagOff(c, v, i);
    else if ( (!strcmp(v[i], "-blur")) || (!strcmp(v[i], "-b")) )
      p->blur_flag = SetFlagOn(c, v, i);
    else if ( (!strcmp(v[i], "-noblur")) || (!strcmp(v[i], "-nb")) )
      p->blur_flag = SetFlagOff(c, v, i);
    else if (!strcmp(v[i], "-fill"))
      p->blur_flag = SetFlag(c, v, i, 2);
    else if ( (!strcmp(v[i], "-source-width")) || (!strcmp(v[i], "-sw")) )
      p->source_width = GetIntValueFromParameter(c, v, i, p->source_width);
    else if ( (!strcmp(v[i], "-width")) || (!strcmp(v[i], "-w")) )
      p->width = GetIntValueFromParameter(c, v, i, p->width);
    else if ( (!strcmp(v[i], "-length-x")) || (!strcmp(v[i], "-lx")) )
      p->length_x = GetIntValueFromParameter(c, v, i, p->length_x);
    else if ( (!strcmp(v[i], "-length-y")) || (!strcmp(v[i], "-ly")) )
      p->length_y = GetIntValueFromParameter(c, v, i, p->length_y);
    else if ( (!strcmp(v[i], "-space")) || (!strcmp(v[i], "-sp")) )
      p->space = GetIntValueFromParameter(c, v, i, p->space);
    else if ( (!strcmp(v[i], "-scroll")) || (!strcmp(v[i], "-sc")) )
      p->scroll = GetIntValueFromParameter(c, v, i, p->scroll);
    else if ( (!strcmp(v[i], "-start")) || (!strcmp(v[i], "-st")) )
      p->start = (long int)GetIntValueFromParameter(c, v, i, (int)(p->start));
    else if ( (!strcmp(v[i], "-wait")) || (!strcmp(v[i], "-wt")) )
      p->wait = GetIntValueFromParameter(c, v, i, p->wait);
    else if ( (!strcmp(v[i], "-font")) || (!strcmp(v[i], "-fn")) )
      p->font_name = GetStringValueFromParameter(c, v, i, p->font_name);
    else if ( (!strcmp(v[i], "-name")) || (!strcmp(v[i], "-title")) )
      p->title_name = GetStringValueFromParameter(c, v, i, p->title_name);
    else i++;
  }
}

void InitializeParameters(XStarRoll * p, int * argc, char ** argv)
{
  SetDefaultValues(p);
  ReadParameters(p, argc, argv);
  ConnectDisplay(p, argc, argv);
}

void InitializeDisplay(XStarRoll * p)
{
  int n;
  Arg args[10];

  p->display = XtDisplay(p->toplevel);
  p->screen = DefaultScreen(p->display);
  p->colormap = XDefaultColormap(p->display, p->screen);
  p->root_window = RootWindow(p->display, p->screen);
  p->depth = DefaultDepth(p->display, p->screen);
  p->canvas = XtCreateManagedWidget("", labelWidgetClass, p->toplevel,
				    NULL, 0);
  n = 0;
  XtSetArg(args[n], XtNforeground, &(p->foreground_color)); n++;
  XtSetArg(args[n], XtNbackground, &(p->background_color)); n++;
  XtGetValues(p->canvas, args, n);
  if (p->root_flag == ON) {
    p->toplevel_window = XtWindow(p->toplevel);
    p->canvas_window = p->root_window;
    p->canvas_width = DisplayWidth(p->display, p->screen);
    p->canvas_height = DisplayHeight(p->display, p->screen);
  } else {
    XtRealizeWidget(p->toplevel);
    p->toplevel_window = XtWindow(p->toplevel);
    p->canvas_window = XtWindow(p->canvas);
    n = 0;
    XtSetArg(args[n], XtNwidth,  &(p->canvas_width)); n++;
    XtSetArg(args[n], XtNheight, &(p->canvas_height)); n++;
    XtGetValues(p->canvas, args, n);
  }
  p->canvas_pixmap = XCreatePixmap(p->display, p->canvas_window,
				   p->canvas_width, p->canvas_height,
				   p->depth);
  p->font = XLoadFont(p->display, p->font_name);
  p->font_struct = XQueryFont(p->display, p->font);  
  if (p->source_width == 0)
    p->source_width = p->width *
    ((p->font_struct->max_bounds.width +
      p->font_struct->min_bounds.width) / 2);
  if (p->length_x < 1) p->length_x = p->source_width;
  XFlush(p->display);
}

void InitializeGC(XStarRoll * p)
{
  p->gc_foreground = XCreateGC(p->display, p->canvas_window, 0, 0);
  p->gc_background = XCreateGC(p->display, p->canvas_window, 0, 0);
  p->gc_copy = XCreateGC(p->display, p->canvas_window, 0, 0);
  XSetForeground(p->display, p->gc_foreground, p->foreground_color);
  XSetBackground(p->display, p->gc_foreground, p->background_color);
  XSetForeground(p->display, p->gc_background, p->background_color);
  XSetBackground(p->display, p->gc_background, p->background_color);
  XSetFont(p->display, p->gc_foreground, p->font);
  XSetFunction(p->display, p->gc_copy, GXcopy);
  XSetFillRule(p->display, p->gc_foreground, WindingRule);
  XSetFillRule(p->display, p->gc_background, WindingRule);
  XSetLineAttributes(p->display, p->gc_foreground, 0,
		     LineSolid, CapNotLast ,JoinMiter);
}

int GetCharFromPointer(void ** pp, int flag)
{
  int c;
  char ** cpp;
  FILE ** fpp;

  if (flag == READ_FROM_STRING) {
    cpp = (char **)pp;
    c = **cpp;
    if (c == '\0') c = EOF;
    else (*cpp)++;
  } else if (flag == READ_FROM_FILE) {
    fpp = (FILE **)pp;
    c = fgetc(*fpp);
  }
  return (c);
}

GCBuffer * SearchGCBufferFromPixel(XStarRoll * p, unsigned long int pixel)
{
  GCBuffer * gp;

  gp = p->gc_buffer->start;
  while (gp != (GCBuffer *)NULL)
    if (gp->pixel == pixel) return (gp);
    else gp = gp->next;
  return ((GCBuffer *)NULL);
}

GCBuffer * GetGCBufferFromPixel(XStarRoll * p, unsigned long int pixel)
{
  GCBuffer * gp;

  gp = SearchGCBufferFromPixel(p, pixel);
  if (gp == (GCBuffer *)NULL) {
    gp = CreateGCBuffer(p->display, p->canvas_window, pixel);
    gp = AddGCBufferToTop(p->gc_buffer, gp);
  } else
    gp = MoveGCBufferToTop(p->gc_buffer, gp);
  return (gp);
}

void MakeSourceDataFromPixmap(XStarRoll * p, Pixmap pixmap, int w, int h)
{
  XImage * image;
  SourceData * sp;
  GCBuffer * gp;
  int x, y;
  unsigned long int pixel;

  image = XGetImage(p->display, pixmap, 0, 0, w, h, AllPlanes, XYPixmap);
  for (y = 0; y < h; y++) {
    for (x = 0; x < w; x++) {
      pixel = XGetPixel(image, x, y);
      if ( (x == 0) || (pixel != sp->gc_buffer->pixel) ) {
	gp = GetGCBufferFromPixel(p, pixel);
	sp = CreateSourceData(1, gp, 1, 0);
	sp = AddSourceDataToBottom(p->source_data, sp);
      } else {
	(sp->length)++;
	(sp->x)++;
      }
    }
    sp->x -= w;
    (sp->y)++;
  }
  XDestroyImage(image);
}

void MakeSourceDataFromFontData(XStarRoll * p, char * str, int n, int f)
{
  SourceData * sp;
  GCBuffer * fore_gp;
  GCBuffer * back_gp;
  GCBuffer * gp;
  FontDataList * fp;
  int i, j, off, len;
  unsigned char * string;
  unsigned char pixel;

  string = (unsigned char *)str;
  fore_gp = GetGCBufferFromPixel(p, p->foreground_pixel);
  back_gp = GetGCBufferFromPixel(p, p->background_pixel);

  if (f == ON) {
    off = (p->source_width - XTextWidth(p->font_struct, str, n)) / 2;
    sp = CreateSourceData(0, back_gp, off, 0);
    sp = AddSourceDataToBottom(p->source_data, sp);
  }

  if (n == 0) {
    sp = CreateSourceData(0, back_gp, 0, p->font_height);
    sp = AddSourceDataToBottom(p->source_data, sp);
  } else {
    for (i = 0; i < p->font_height; i++) {
      len = 0;
      sp = (SourceData *)NULL;
      for (j = 0; j < n; j++) {
	fp = p->font_data[(int)(string[j])].list_array[i];
	if ( (sp != (SourceData *)NULL) && (fp->pixel == pixel) ) {
	  sp->x += fp->length;
	  sp->length += fp->length;
	  len += fp->length;
	  fp = fp->next;
	}
	while (fp != (FontDataList *)NULL) {
	  pixel = fp->pixel;
	  if (pixel == ON) gp = fore_gp;
	  else gp = back_gp;
	  sp = CreateSourceData((int)(fp->length), gp, (int)(fp->length), 0);
	  sp = AddSourceDataToBottom(p->source_data, sp);
	  len += fp->length;
	  fp = fp->next;
	}
      }
      sp = CreateSourceData(0, back_gp, - len, 1);
      sp = AddSourceDataToBottom(p->source_data, sp);
    }
  }

  if (f == ON) {
    sp = CreateSourceData(0, back_gp, - off, p->space);
    sp = AddSourceDataToBottom(p->source_data, sp);
  } else if (p->space != 0) {
    sp = CreateSourceData(0, back_gp, 0, p->space);
    sp = AddSourceDataToBottom(p->source_data, sp);
  }
}

void MakeFontData(XStarRoll * p, int c)
{
  int x, y, w, h;
  char string[10];
  Pixmap pixmap;
  XImage * image;
  unsigned char pixel;
  FontDataList * fp;

  string[0] = (char)c;
  string[1] = '\0';
  p->font_data[c].width = w = XTextWidth(p->font_struct, string, 1);
  h = p->font_height;
  p->font_data[c].list_array =
    (FontDataList **)malloc(sizeof(FontDataList *) * h);
  pixmap = XCreatePixmap(p->display, p->canvas_window, w, h, p->depth);
  XFillRectangle(p->display, pixmap, p->gc_background, 0, 0, w, h);
  XDrawString(p->display, pixmap, p->gc_foreground, 0, p->font_struct->ascent,
	      string, 1);
  image = XGetImage(p->display, pixmap, 0, 0, w, h, AllPlanes, XYPixmap);
  for (y = 0; y < h; y++) {
    for (x = 0; x < w; x++) {
      if (XGetPixel(image, x, y) == p->background_pixel) pixel = OFF;
      else pixel = ON;
      if ( (x == 0) || (pixel != fp->pixel) ) {
	if (x == 0)
	  fp = p->font_data[c].list_array[y] =
	  (FontDataList *)malloc(sizeof(FontDataList));
	else {
	  fp->next = (FontDataList *)malloc(sizeof(FontDataList));
	  fp = fp->next;
	}
	fp->pixel = pixel;
	fp->length = 1;
	fp->next = (FontDataList *)NULL;
      } else (fp->length)++;
    }
  }
  XDestroyImage(image);
  XFreePixmap(p->display, pixmap);
}

void WriteLetters(XStarRoll * p, void * pointer, int flag)
{
  int i, c;
  int center_flag = OFF;

  tmp_buffer[i = 0] = '\0';
  while ((c = GetCharFromPointer(&pointer, flag)) != EOF) {
    if (c == '\n') {
      MakeSourceDataFromFontData(p, tmp_buffer, i, center_flag);
      tmp_buffer[i = 0] = '\0';
      center_flag = OFF;
    }
    else if ( (c == '\t') && (i == 0) ) center_flag = ON;
    else if ( (c == '\b') && (i > 0) ) tmp_buffer[--i] = '\0';
    else if (c == 0x1B)
      while ( (((c = GetCharFromPointer(&pointer, flag)) < 'A') || (c > 'Z'))
	     && ((c < 'a') || (c > 'z')) && (c != EOF) );
    else if (c < ' ');
    else {
      if (p->font_data[c].list_array == (FontDataList **)NULL)
	MakeFontData(p, c);
      if (i < TMP_BUFFER_SIZE - 1) {
	tmp_buffer[i++] = (char)c;
	tmp_buffer[i] = '\0';
      }
      if (!strcmp(tmp_buffer, "#center")) {
	tmp_buffer[i = 0] = '\0';
	center_flag = ON;
      } else if (XTextWidth(p->font_struct, tmp_buffer, i) > p->source_width) {
	tmp_buffer[--i] = '\0';
	MakeSourceDataFromFontData(p, tmp_buffer, i, center_flag);
	tmp_buffer[i = 0] = (char)c;
	tmp_buffer[++i] = '\0';
      }
    }
  }
  if (i > 0) MakeSourceDataFromFontData(p, tmp_buffer, i, center_flag);
}

void DrawPixmap(XStarRoll * p, char * filename)
{
  int w, h, off;
  Pixmap pixmap;
  SourceData * sp;
  GCBuffer * back_gp;

  pixmap = MakePixmapFromFile(p, filename, &w, &h);
  off = (p->source_width - w) / 2;
  back_gp = GetGCBufferFromPixel(p, p->background_pixel);
  sp = CreateSourceData(0, back_gp, (short int)off, 0);
  sp = AddSourceDataToBottom(p->source_data, sp);
  MakeSourceDataFromPixmap(p, pixmap, w, h);
  sp = CreateSourceData(0, back_gp, (short int)(- off), 0);
  sp = AddSourceDataToBottom(p->source_data, sp);
  XFreePixmap(p->display, pixmap);
}

void OutputMessage(XStarRoll * p, char * string)
{
  static y = 0;

  y += p->font_struct->ascent + p->font_struct->descent + p->space;
  XDrawString(p->display, p->canvas_pixmap, p->gc_foreground,
	      p->font_struct->max_bounds.width, y, string, strlen(string));
  XCopyArea(p->display, p->canvas_pixmap, p->canvas_window, p->gc_copy,
	    0, 0, p->canvas_width, p->canvas_height, 0, 0);
  XFlush(p->display);
}

void FreeFontData(XStarRoll * p)
{
  int i, j;
  FontDataList * fp;
  FontDataList * fp_next;

  for (i = 0; i < 256; i++) {
    if (p->font_data[i].list_array != (FontDataList **)NULL) {
      for (j = 0; j < p->font_height; j++) {
	fp = p->font_data[i].list_array[j];
	while (fp != (FontDataList *)NULL) {
	  fp_next = fp->next;
	  free(fp);
	  fp = fp_next;
	}
      }
      free(p->font_data[i].list_array);
    }
  }
  free(p->font_data);
}

void GetBlackAndWhitePixel(XStarRoll * p)
{
  Pixmap pix;
  XImage * image;

  pix = XCreatePixmap(p->display, p->canvas_window, 2, 1, p->depth);
  XDrawPoint(p->display, pix, p->gc_foreground, 0, 0);
  XDrawPoint(p->display, pix, p->gc_background, 1, 0);
  image = XGetImage(p->display, pix, 0, 0, 2, 1, AllPlanes, XYPixmap);
  p->foreground_pixel = XGetPixel(image, 0, 0);
  p->background_pixel = XGetPixel(image, 1, 0);
  XDestroyImage(image);
  XFreePixmap(p->display, pix);
}

void InitializeStarRoll(XStarRoll * p, int * argc, char ** argv)
{
  FILE * fp;
  int i;

  p->gc_buffer = InitializeGCBufferList();
  p->source_data = InitializeSourceDataList();
  p->font_data = (FontData *)malloc(sizeof(FontData) * 256);
  for (i = 0; i < 256; i++) p->font_data[i].list_array = (FontDataList **)NULL;
  p->font_height = p->font_struct->ascent + p->font_struct->descent;

  GetBlackAndWhitePixel(p);

  XFillRectangle(p->display, p->canvas_pixmap, p->gc_background, 0, 0,
		 p->canvas_width, p->canvas_height);
  OutputMessage(p, "Now initializing XStarRoll and making data.");
  OutputMessage(p, "Please wait for some minutes...");

  while (*argc > 1) {
    if (!strcmp(argv[1], "-")) {
      OutputMessage(p, "Writing letters from the standard input.");
      if (isatty(fileno(stdin)) != 0)
	OutputMessage(p, "Please input letters and hit Ctrl-D.");
      WriteLetters(p, stdin, READ_FROM_FILE);
      OutputMessage(p, "Done.");
      DeleteParameter(argc, argv, 1);
    } else if ((fp = fopen(argv[1], "rt")) == NULL) {
      OutputMessage(p, "Writing letters.");
      WriteLetters(p, argv[1], READ_FROM_STRING);
      OutputMessage(p, "Done.");
      DeleteParameter(argc, argv, 1);
    } else {
      if ( (fgets(tmp_buffer, 10, fp) != NULL) &&
	   (!strcmp(tmp_buffer, "/* XPM */")) ) {
	fclose(fp);
	strcpy(tmp_buffer, "Drawing a pixmap \"");
	strncat(tmp_buffer, argv[1], TMP_BUFFER_SIZE - 100);
	strcat(tmp_buffer, "\".");
	OutputMessage(p, tmp_buffer);
	DrawPixmap(p, argv[1]);
	OutputMessage(p, "Done.");
	DeleteParameter(argc, argv, 1);
      } else {
	fclose(fp);
	strcpy(tmp_buffer, "Writing letters in a file \"");
	strncat(tmp_buffer, argv[1], TMP_BUFFER_SIZE - 100);
	strcat(tmp_buffer, "\".");
	OutputMessage(p, tmp_buffer);
	fp = fopen(argv[1], "rt");
	WriteLetters(p, fp, READ_FROM_FILE);
	OutputMessage(p, "Done.");
	fclose(fp);
	DeleteParameter(argc, argv, 1);
      }
    }
  }

  if ( (isatty(fileno(stdin)) == 0) && (feof(stdin) == 0) ){
    OutputMessage(p, "Writing letters from the standard input.");
    WriteLetters(p, stdin, READ_FROM_FILE);
    OutputMessage(p, "Done.");
  }
  if (p->source_data->start == (SourceData *)NULL) {
    OutputMessage(p, "Writing default letters.");
    WriteLetters(p, help_string, READ_FROM_STRING);
    OutputMessage(p, "Done.");
  }
  FreeFontData(p);
  OutputMessage(p, "Now starting XStarRoll!");
  sleep(1);
}

void Quit(XStarRoll * p)
{
  FreeSourceDataList(p->source_data);
  FreeGCBufferList(p->display, p->gc_buffer);
  XFreeGC(p->display, p->gc_foreground);
  XFreeGC(p->display, p->gc_background);
  XFreeGC(p->display, p->gc_copy);
  XFreePixmap(p->display, p->canvas_pixmap);
  XFreePixmap(p->display, p->icon_pixmap);
  XtUnmanageChild(p->canvas);
  XtDestroyWidget(p->canvas);
  XtDestroyWidget(p->toplevel);
  exit (0);
}

void PushKey(Widget w, XtPointer pp, XEvent * event, Boolean * dispatch)
{
  XStarRoll * p;
  KeySym ks;

  p = (XStarRoll *)pp;
  if (event->type == KeyPress) {
    ks = XKeycodeToKeysym(event->xkey.display, event->xkey.keycode, 0);
    if      (ks == XK_q        ) Quit(p);
    else if (ks == XK_s        ) p->start = 0;
    else if (ks == XK_n        ) (p->start)++;
    else if (ks == XK_p        ) (p->start)--;
    else if (ks == XK_Page_Down) (p->start) += 10;
    else if (ks == XK_Page_Up  ) (p->start) -= 10;
    else if (ks == XK_a        ) (p->scroll)++;
    else if (ks == XK_d        ) (p->scroll)--;
    else if (ks == XK_Down     ) (p->length_y)++;
    else if (ks == XK_Up       ) {if (p->length_y > 1) (p->length_y)--;}
    else if (ks == XK_Right    ) (p->length_x)++;
    else if (ks == XK_Left     ) {if (p->length_x > 1) (p->length_x)--;}
    p->draw_flag = ON;
  }
}

void ExposeCanvas(Widget w, XtPointer pp, XEvent * event, Boolean * dispatch)
{
  XStarRoll * p;

  p = (XStarRoll *)pp;
  if (event->type == Expose)
    XCopyArea(p->display, p->canvas_pixmap, p->canvas_window, p->gc_copy,
	      event->xexpose.x, event->xexpose.y,
	      event->xexpose.width, event->xexpose.height,
	      event->xexpose.x, event->xexpose.y);
}

void ResizeCanvas(XStarRoll * p)
{
  int i, n;
  Arg args[10];

  n = 0;
  XtSetArg(args[n], XtNwidth,  &(p->canvas_width)); n++;
  XtSetArg(args[n], XtNheight, &(p->canvas_height)); n++;
  XtGetValues(p->toplevel, args, n);
  XResizeWindow(p->display, p->canvas_window,
		p->canvas_width, p->canvas_height);
  XFreePixmap(p->display, p->canvas_pixmap);
  p->canvas_pixmap = XCreatePixmap(p->display, p->canvas_window,
				   p->canvas_width, p->canvas_height,
				   p->depth);
  p->draw_flag = ON;
}

void CanvasEvent(Widget w, XtPointer pp, XEvent * event, Boolean * dispatch)
{
  XStarRoll * p;

  p = (XStarRoll *)pp;
  if (event->type == ConfigureNotify) ResizeCanvas(p);
  else if (event->type == ClientMessage) {
    if ( (event->xclient.message_type == p->atom_1) &&
	 (event->xclient.data.l[0] == p->atom_2) ) Quit(p);
  }
}

void AddEvents(XStarRoll * p)
{
  XtAddEventHandler(p->toplevel, ExposureMask, False, ExposeCanvas, p);
  if (p->root_flag == OFF) {
    XtAddEventHandler(p->toplevel, KeyPressMask, False, PushKey, p);
    XtAddEventHandler(p->toplevel, StructureNotifyMask, True, CanvasEvent, p);
  }
}

void SetIcon(XStarRoll * p)
{
  XWMHints wmh;

  if (p->root_flag == OFF) {
    p->icon_pixmap = MakePixmapFromData(p, XSTARROLL_ICON,
					&(p->icon_width), &(p->icon_height));
    wmh.flags = IconPixmapHint;
    wmh.icon_pixmap = p->icon_pixmap;
    XSetWMHints(p->display, p->toplevel_window, &wmh);
  }
}

void SetWindowAttributes(XStarRoll * p)
{
  XWMHints wmhints;

  if (p->root_flag == OFF) {
    XtSetSensitive(p->toplevel, True);
    XtSetSensitive(p->canvas, True);
    wmhints.input = True;
    XSetWMHints(p->display, p->toplevel_window, &wmhints);
    p->atom_1 = XInternAtom(p->display, "WM_PROTOCOLS", False);
    p->atom_2 = XInternAtom(p->display, "WM_DELETE_WINDOW", False);
    XSetWMProtocols(p->display, p->toplevel_window, &(p->atom_2), 1);
    XStoreName(p->display, p->toplevel_window, p->title_name);
    XSetIconName(p->display, p->toplevel_window, p->title_name);
    SetIcon(p);
  }
}

double GetCanvasPointY(XStarRoll * p, long int y)
{
  return (((double)(p->length_y) * p->canvas_height) / (y + p->length_y));
}

double GetCanvasPointX(XStarRoll * p, long int x, double canvas_y)
{
  return (canvas_y * (x - p->source_width / 2) * p->canvas_width /
	  (p->canvas_height * p->length_x) + p->canvas_width / 2);
}

void DrawCanvasBlurred(XStarRoll * p)
{
  long int x, y;
  short int canvas_iy0, canvas_iy1;
  double canvas_dy0, canvas_dy1;
  XPoint points[4];
  SourceData * sp;

  x = y = 0;
  sp = p->source_data->start;
  canvas_dy0 = GetCanvasPointY(p, p->start - y);
  canvas_dy1 = GetCanvasPointY(p, p->start - y + 1);
  canvas_iy0 = (short int)(canvas_dy0 + 0.5);
  canvas_iy1 = (short int)(canvas_dy1 + 0.5);

  while ( (y <= p->start) && (sp != (SourceData *)NULL) ) {
    if (sp->gc_buffer->pixel != p->background_pixel) {
      if (canvas_iy0 == canvas_iy1);
      else if (canvas_iy0 == canvas_iy1 + 1) {
	points[0].x = (short int)(GetCanvasPointX(p, x, canvas_dy0) + 0.5);
	points[3].x = (short int)(GetCanvasPointX(p, x + sp->length,
						  canvas_dy0) + 0.5);
	if (points[0].x == points[3].x);
	else {
	  points[1].x = (short int)
	    (GetCanvasPointX(p, x, canvas_dy1) + 0.5);
	  points[2].x = (short int)
	    (GetCanvasPointX(p, x + sp->length, canvas_dy1) + 0.5);
	  XDrawLine(p->display, p->canvas_pixmap, sp->gc_buffer->gc,
		    points[1].x, canvas_iy1, points[2].x, canvas_iy1);
	}
      } else {
	points[0].x = (short int)(GetCanvasPointX(p, x, canvas_dy0) + 0.5);
	points[3].x = (short int)(GetCanvasPointX(p, x + sp->length,
						  canvas_dy0) + 0.5);
	if (points[0].x == points[3].x);
	else {
	  points[1].x = (short int)
	    (GetCanvasPointX(p, x, canvas_dy1) + 0.5);
	  points[2].x = (short int)
	    (GetCanvasPointX(p, x + sp->length, canvas_dy1) + 0.5);
	  if (points[1].x == points[2].x) {
	    points[0].y = canvas_iy0;
	    points[1].y = canvas_iy1;
	    points[2].x = points[3].x;
	    points[2].y = canvas_iy0;
	    XFillPolygon(p->display, p->canvas_pixmap, sp->gc_buffer->gc,
			 points, 3, Convex, CoordModeOrigin);
	  } else {
	    points[0].y = points[3].y = canvas_iy0;
	    points[1].y = points[2].y = canvas_iy1;
	    XFillPolygon(p->display, p->canvas_pixmap, sp->gc_buffer->gc,
			 points, 4, Convex, CoordModeOrigin);
	  }
	}
      }
    }
    x += sp->x;
    if (sp->y != 0) {
      y += sp->y;
      if (y <= p->start) {
	canvas_dy0 = GetCanvasPointY(p, p->start - y);
	canvas_dy1 = GetCanvasPointY(p, p->start - y + 1);
	canvas_iy0 = (short int)(canvas_dy0 + 0.5);
	canvas_iy1 = (short int)(canvas_dy1 + 0.5);
      }
    }
    sp = sp->next;
  }
}

void DrawCanvasNoBlurred(XStarRoll * p)
{
  long int x, y;
  short int canvas_iy0, canvas_iy1;
  double canvas_dy0, canvas_dy1;
  XPoint points[4];
  SourceData * sp;

  x = y = 0;
  sp = p->source_data->start;
  canvas_dy0 = GetCanvasPointY(p, p->start - y);
  canvas_dy1 = GetCanvasPointY(p, p->start - y + 1);
  canvas_iy0 = (short int)(canvas_dy0 + 0.5);
  canvas_iy1 = (short int)(canvas_dy1 + 0.5);

  while ( (y <= p->start) && (sp != (SourceData *)NULL) ) {
    if (sp->gc_buffer->pixel != p->background_pixel) {
      if (canvas_iy0 == canvas_iy1) {
	points[0].x = (short int)(GetCanvasPointX(p, x, canvas_dy0) + 0.5);
	points[1].x = (short int)(GetCanvasPointX(p, x, canvas_dy1) + 0.5);
	points[3].x = (short int)(GetCanvasPointX(p, x + sp->length,
						  canvas_dy0) + 0.5);
	if (points[0].x == points[3].x)
	  XDrawPoint(p->display, p->canvas_pixmap, sp->gc_buffer->gc,
		     points[1].x, canvas_iy1);
	else if (points[0].x == points[3].x - 1)
	  XDrawPoint(p->display, p->canvas_pixmap, sp->gc_buffer->gc,
		     points[1].x, canvas_iy1);
	else {
	  points[2].x = (short int)(GetCanvasPointX(p, x + sp->length,
						    canvas_dy1) + 0.5);
	  XDrawLine(p->display, p->canvas_pixmap, sp->gc_buffer->gc,
		    points[1].x, canvas_iy1, points[2].x, canvas_iy1);
	}
      } else if (canvas_iy0 == canvas_iy1 + 1) {
	points[0].x = (short int)(GetCanvasPointX(p, x, canvas_dy0) + 0.5);
	points[1].x = (short int)(GetCanvasPointX(p, x, canvas_dy1) + 0.5);
	points[3].x = (short int)(GetCanvasPointX(p, x + sp->length,
						  canvas_dy0) + 0.5);
	if (points[0].x == points[3].x)
	  XDrawPoint(p->display, p->canvas_pixmap, sp->gc_buffer->gc,
		     points[1].x, canvas_iy1);
	else {
	  points[2].x = (short int)(GetCanvasPointX(p, x + sp->length,
						    canvas_dy1) + 0.5);
	  XDrawLine(p->display, p->canvas_pixmap, sp->gc_buffer->gc,
		    points[1].x, canvas_iy1, points[2].x, canvas_iy1);
	}
      } else {
	points[0].x = (short int)(GetCanvasPointX(p, x, canvas_dy0) + 0.5);
	points[1].x = (short int)(GetCanvasPointX(p, x, canvas_dy1) + 0.5);
	points[3].x = (short int)(GetCanvasPointX(p, x + sp->length,
						  canvas_dy0) + 0.5);
	if (points[0].x == points[3].x)
	  XDrawLine(p->display, p->canvas_pixmap, sp->gc_buffer->gc,
		    points[1].x, canvas_iy1, points[0].x, canvas_iy0);
	else {
	  points[2].x = (short int)(GetCanvasPointX(p, x + sp->length,
						    canvas_dy1) + 0.5);
	  if (points[1].x == points[2].x) {
	    points[0].y = canvas_iy0;
	    points[1].y = canvas_iy1;
	    points[2].x = points[3].x;
	    points[2].y = canvas_iy0;
	    XFillPolygon(p->display, p->canvas_pixmap, sp->gc_buffer->gc,
			 points, 3, Convex, CoordModeOrigin);
	  } else {
	    points[0].y = points[3].y = canvas_iy0;
	    points[1].y = points[2].y = canvas_iy1;
	    XFillPolygon(p->display, p->canvas_pixmap, sp->gc_buffer->gc,
			 points, 4, Convex, CoordModeOrigin);
	  }
	}
      }
    }
    x += sp->x;
    if (sp->y != 0) {
      y += sp->y;
      if (y <= p->start) {
	canvas_dy0 = GetCanvasPointY(p, p->start - y);
	canvas_dy1 = GetCanvasPointY(p, p->start - y + 1);
	canvas_iy0 = (short int)(canvas_dy0 + 0.5);
	canvas_iy1 = (short int)(canvas_dy1 + 0.5);
      }
    }
    sp = sp->next;
  }
}

void DrawCanvasFilled(XStarRoll * p)
{
  long int x, y;
  short int canvas_iy0, canvas_iy1;
  double canvas_dy0, canvas_dy1;
  XPoint points[4];
  SourceData * sp;

  x = y = 0;
  sp = p->source_data->start;
  canvas_dy0 = GetCanvasPointY(p, p->start - y);
  canvas_dy1 = GetCanvasPointY(p, p->start - y + 1);
  canvas_iy0 = (short int)(canvas_dy0 + 0.5);
  canvas_iy1 = (short int)(canvas_dy1 + 0.5);

  while ( (y <= p->start) && (sp != (SourceData *)NULL) ) {
    if (sp->gc_buffer->pixel != p->background_pixel) {
      if (canvas_iy0 == canvas_iy1) {
	points[0].x = (short int)(GetCanvasPointX(p, x, canvas_dy0) + 0.5);
	points[1].x = (short int)(GetCanvasPointX(p, x, canvas_dy1) + 0.5);
	points[2].x = (short int)(GetCanvasPointX(p, x + sp->length,
						  canvas_dy1) + 0.5);
	points[3].x = (short int)(GetCanvasPointX(p, x + sp->length,
						  canvas_dy0) + 0.5);
	if (points[0].x == points[3].x) {
	  XDrawPoint(p->display, p->canvas_pixmap, sp->gc_buffer->gc,
		     points[0].x, canvas_iy0);
	  XDrawPoint(p->display, p->canvas_pixmap, sp->gc_buffer->gc,
		     points[1].x, canvas_iy1);
	  XDrawPoint(p->display, p->canvas_pixmap, sp->gc_buffer->gc,
		     points[2].x, canvas_iy1);
	  XDrawPoint(p->display, p->canvas_pixmap, sp->gc_buffer->gc,
		     points[3].x, canvas_iy0);
	} else {
	  XDrawLine(p->display, p->canvas_pixmap, sp->gc_buffer->gc,
		    points[1].x, canvas_iy1, points[2].x, canvas_iy1);
	  XDrawLine(p->display, p->canvas_pixmap, sp->gc_buffer->gc,
		    points[0].x, canvas_iy0, points[3].x, canvas_iy0);
	  XDrawPoint(p->display, p->canvas_pixmap, sp->gc_buffer->gc,
		     points[2].x, canvas_iy1);
	  XDrawPoint(p->display, p->canvas_pixmap, sp->gc_buffer->gc,
		     points[3].x, canvas_iy0);
	}
      } else if (canvas_iy0 == canvas_iy1 + 1) {
	points[0].x = (short int)(GetCanvasPointX(p, x, canvas_dy0) + 0.5);
	points[1].x = (short int)(GetCanvasPointX(p, x, canvas_dy1) + 0.5);
	points[2].x = (short int)(GetCanvasPointX(p, x + sp->length,
						  canvas_dy1) + 0.5);
	points[3].x = (short int)(GetCanvasPointX(p, x + sp->length,
						  canvas_dy0) + 0.5);
	if (points[0].x == points[3].x) {
	  XDrawPoint(p->display, p->canvas_pixmap, sp->gc_buffer->gc,
		     points[0].x, canvas_iy0);
	  XDrawPoint(p->display, p->canvas_pixmap, sp->gc_buffer->gc,
		     points[1].x, canvas_iy1);
	  XDrawPoint(p->display, p->canvas_pixmap, sp->gc_buffer->gc,
		     points[2].x, canvas_iy1);
	  XDrawPoint(p->display, p->canvas_pixmap, sp->gc_buffer->gc,
		     points[3].x, canvas_iy0);
	} else {
	  XDrawLine(p->display, p->canvas_pixmap, sp->gc_buffer->gc,
		    points[1].x, canvas_iy1, points[2].x, canvas_iy1);
	  XDrawLine(p->display, p->canvas_pixmap, sp->gc_buffer->gc,
		    points[0].x, canvas_iy0, points[3].x, canvas_iy0);
	  XDrawPoint(p->display, p->canvas_pixmap, sp->gc_buffer->gc,
		     points[2].x, canvas_iy1);
	  XDrawPoint(p->display, p->canvas_pixmap, sp->gc_buffer->gc,
		     points[3].x, canvas_iy0);
	}
      } else {
	points[0].x = (short int)(GetCanvasPointX(p, x, canvas_dy0) + 0.5);
	points[1].x = (short int)(GetCanvasPointX(p, x, canvas_dy1) + 0.5);
	points[2].x = (short int)(GetCanvasPointX(p, x + sp->length,
						  canvas_dy1) + 0.5);
	points[3].x = (short int)(GetCanvasPointX(p, x + sp->length,
						  canvas_dy0) + 0.5);
	if (points[0].x == points[3].x) {
	  XDrawLine(p->display, p->canvas_pixmap, sp->gc_buffer->gc,
		    points[1].x, canvas_iy1, points[0].x, canvas_iy0);
	  XDrawLine(p->display, p->canvas_pixmap, sp->gc_buffer->gc,
		    points[2].x, canvas_iy1, points[3].x, canvas_iy0);
	  XDrawPoint(p->display, p->canvas_pixmap, sp->gc_buffer->gc,
		     points[0].x, canvas_iy0);
	  XDrawPoint(p->display, p->canvas_pixmap, sp->gc_buffer->gc,
		     points[3].x, canvas_iy0);
	} else {
	  if (points[1].x == points[2].x) {
	    points[0].y = canvas_iy0;
	    points[1].y = canvas_iy1;
	    points[2].x = points[3].x;
	    points[2].y = canvas_iy0;
	    XFillPolygon(p->display, p->canvas_pixmap, sp->gc_buffer->gc,
			 points, 3, Convex, CoordModeOrigin);
	    XDrawLine(p->display, p->canvas_pixmap, sp->gc_buffer->gc,
		      points[0].x, canvas_iy0, points[2].x, canvas_iy0);
	    XDrawLine(p->display, p->canvas_pixmap, sp->gc_buffer->gc,
		      points[1].x, canvas_iy1, points[2].x, canvas_iy0);
	    XDrawPoint(p->display, p->canvas_pixmap, sp->gc_buffer->gc,
		       points[2].x, canvas_iy0);
	  } else {
	    points[0].y = points[3].y = canvas_iy0;
	    points[1].y = points[2].y = canvas_iy1;
	    XFillPolygon(p->display, p->canvas_pixmap, sp->gc_buffer->gc,
			 points, 4, Convex, CoordModeOrigin);
	    XDrawLine(p->display, p->canvas_pixmap, sp->gc_buffer->gc,
		      points[0].x, canvas_iy0, points[3].x, canvas_iy0);
	    XDrawLine(p->display, p->canvas_pixmap, sp->gc_buffer->gc,
		      points[2].x, canvas_iy1, points[3].x, canvas_iy0);
	    XDrawPoint(p->display, p->canvas_pixmap, sp->gc_buffer->gc,
		       points[3].x, canvas_iy0);
	  }
	}
      }
    }
    x += sp->x;
    if (sp->y != 0) {
      y += sp->y;
      if (y <= p->start) {
	canvas_dy0 = GetCanvasPointY(p, p->start - y);
	canvas_dy1 = GetCanvasPointY(p, p->start - y + 1);
	canvas_iy0 = (short int)(canvas_dy0 + 0.5);
	canvas_iy1 = (short int)(canvas_dy1 + 0.5);
      }
    }
    sp = sp->next;
  }
}

Boolean MainStarRoll(caddr_t pp)
{
  XStarRoll * p;

  p = (XStarRoll *)pp;
  if (p->draw_flag == OFF) {
    usleep(100000);
    if (p->root_flag == ON) {
      sleep(5);
      p->draw_flag = ON;
    }
  } else {
    XFillRectangle(p->display, p->canvas_pixmap, p->gc_background, 0, 0,
		   p->canvas_width, p->canvas_height);
    if (p->blur_flag == ON) DrawCanvasBlurred(p);
    else if (p->blur_flag == OFF) DrawCanvasNoBlurred(p);
    else DrawCanvasFilled(p);
    XCopyArea(p->display, p->canvas_pixmap, p->canvas_window, p->gc_copy,
	      0, 0, p->canvas_width, p->canvas_height, 0, 0);
    XFlush(p->display);
    p->draw_flag = OFF;
    if (p->quit_flag == ON) Quit(p);
  }
  if (p->scroll != 0) {
    p->start += p->scroll;
    p->draw_flag = ON;
  }
  usleep(100);
  usleep(p->wait);
}

int main(int argc, char ** argv)
{
  XStarRoll xstarroll;

  InitializeParameters(&xstarroll, &argc, argv);
  InitializeDisplay(&xstarroll);
  InitializeGC(&xstarroll);
  SetWindowAttributes(&xstarroll);
  AddEvents(&xstarroll);
  InitializeStarRoll(&xstarroll, &argc, argv);
  XtAppAddWorkProc(xstarroll.app_context,(XtWorkProc)MainStarRoll,&xstarroll);
  XtAppMainLoop(xstarroll.app_context);
}

/*****************************************************************************/
/* End of Program.                                                           */
/*****************************************************************************/
