//==============================
// Player.C
// 
// Represents a player, its scoring, ...
// 
// ZNibbles
// Vincent Mallet 1997, 1998, 1999
//==============================

// $Id: Player.C,v 1.5 1999/05/11 02:17:08 vmallet Exp $

/* ZNibbles - a small multiplayer game
 * Copyright (C) 1997, 1998, 1999 Vincent Mallet - vmallet@enst.fr
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>

#include <iostream.h>

#include "Player.H"

#include "BaseInterface.H"
#include "Movable.H"
#include "Trame.H"
#include "World.H"

#include "PlayerObserver.H"



// change player name
void Player::set_name(char *s)
{
  if (s == NULL)
    s = "Anonymous";
  _name = strdup(s);
}


// add player description to frame t
void Player::add_description(Trame &t)
{
  t.put_char(PLAYER_DESC);
  t.put_char(_number);
  t.put_int(id);
  t.put_string(_name);
  t.put_int(_score);
  t.put_int(_frag);
  t.put_short(_best_length);
  t.put_long(playtime);
  t.put_int(worm_id);
  t.put_char(dead_cycle);
}


// update player data from frame t
void Player::read_description(Trame &t)
{
  if (t.get_char() != PLAYER_DESC) {
    cerr << "Player::read_description(): ohhh le bordel!\n";
    exit(1);
  }

  _number       = t.get_char();
  id           = t.get_int();
  set_name(t.get_string());
  _score       = t.get_int();
  _frag        = t.get_int();
  _best_length = t.get_short();
  playtime     = t.get_long();
  starttime    = time(NULL) - playtime;
  worm_id      = t.get_int();
  dead_cycle   = t.get_char();
}


// debug: print a brief representation of the player
void Player::display()
{
  cout << "   ID:" << id << " Player #" << _number \
       << " '" << _name << "' score=" << _score << " frags=" \
       << _frag << " best=" << _best_length << " wid=" \
       << worm_id<< endl;
}


// called when a player dies (actually not the player,
// but its worm).
void Player::die()
{
  char buf[200];
  _frag++;
  if (worm_id) {
    Movable& wr = (Movable&) world.lookup_object(worm_id);
    if (wr.length > _best_length) {
      _best_length = wr.length;
      // notify observers
      notify_best_length_changed();

      if (_best_length > world.longest_worm) {
	world.longest_worm = _best_length;
	sprintf(buf, "Made the LONGEST worm!  size = %d\n", _best_length);
	world.interface->display_system_message(buf, this);
      }
    }
  }
  worm_id = 0;
  dead_cycle = world.players_dead_cycle;

//     if (wr.baby)
//       sprintf(buf, "Died and was still a baby! HAHAHA. (_score=%d) 
// (frags=%d) (best=%d)\n", _score, frag, _best_length);
//     else
//       sprintf(buf, "Died. (score=%d) (_frags=%d) (best=%d)\n", 
// _score, _frag, _best_length);
//  }
//   else
//       sprintf(buf, "Died. (score=%d) (frags=%d) (best=%d)\n", 
//  _score, _frag, _best_length);
//   world.interface->display_system_message(buf, this);
}


void Player::server_cycle()
{
  if (!worm_id) 
    if (!dead_cycle) {
      Movable& worm = * new Movable(world);
      
      worm.direction = D_STOPPED; // don't move...
      worm.player_id = id;
      worm.player    = this;
      worm.two_keys  = two_key;
      worm_id        = worm.id;

      worm.auto_position(world.map);
      worm.add_type(world.map);
      
      world.add_object(worm);
      
      worm.add_description(world.cycle_trame);
    }
    else
      dead_cycle--;
}


// update player score
void Player::add_score(int delta) 
{
  // change score
  _score += delta;
  
  // notify observers
  notify_score_changed();
}


// return length of worm.
int Player::get_worm_length() 
{
  Movable& wr = (Movable&) world.lookup_object(worm_id);
  
  return wr.get_length(); 
}


// Observers stuff

// Add an observer to the list of player observers for this object
void Player::add_observer(PlayerObserver * player)
{
  _observers.append(player);
}


// Remove an observer from the list of player observers for this object
void Player::remove_observer(PlayerObserver * player)
{
  Pix p;

  for (p = _observers.first(); p; _observers.next(p))
    if (_observers(p) == player)
      break;

  if (p != NULL)
    _observers.del(p, -1);
}


// Notify observers that the score just changed
void Player::notify_score_changed()
{
  for (Pix p = _observers.first(); p; _observers.next(p))
    _observers(p)->score_changed(*this);
}


// Notify observers that the best length just changed
void Player::notify_best_length_changed()
{
  for (Pix p = _observers.first(); p; _observers.next(p))
    _observers(p)->best_length_changed(*this);
}
