//======================================
// World.H
//
// ZNibbles World - the real class
// 
// ZNibbles
// Vincent Mallet 1997
//======================================

// $Id: World.H,v 1.10 1999/05/12 01:44:58 vmallet Exp $

/* ZNibbles - a small multiplayer game
 * Copyright (C) 1997, 1998, 1999 Vincent Mallet - vmallet@enst.fr
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef __H_WORLD__
#define __H_WORLD__

#include "DLList.h" // Doubly linked lists

#include "Base.H"

#include "Map.H"
#include "Trame.H"

#include "VoidInterface.H"

class Player;
class _Object;
class BaseInterface;

class World : public Base {
public:
  int    number;                // World Number
  long   playcycle;             // Current play cycle

  int    x_dim;                 // x world dimension
  int    y_dim;                 // y world dimension

  int    players_dead_cycle;    // see Player.H#dead_cycle

  int    maxplayers;            // max number of players (unused: unlimited)
  int    maxobjects;            // max number of objects (unused: unlimited)
  
  int    paused;                // game paused?

  Trame  cycle_trame;           // trame used for cycle notifies

  Map    map;                   // world map (checks, displays etc)

  int    special;               // number of cycles before creating a special nibble
  
  int    longest_worm;          // Length of the longest worm...

  int    nbplayers;             // number of players in world
  DLList<Player*> players;      // list of players

  int    nbobjects;             // number of objects in world
  DLList<_Object*> objects;     // list of objects

  BaseInterface*  interface;    // game interface

  int    debug;                 // boolean flag, enable debug output

  // Constructors
  World(); //  for clients
  World(int num);  // for servers

  // Destructor
  ~World();
  
  // Actual construction of the world, server side
  void server_make(int x, int y, int pmax, int omax);

  // used by server to choose whether we want computer 
  // worms (false) or not (true).
  void server_set_no_computer(bool no_computer) { 
    _server_no_computer = no_computer; 
  }

  // methods for using the user interface
  BaseInterface * get_interface() { return interface; }
  void set_interface(BaseInterface * new_interface) { 
    this->interface = new_interface;
  }
  void init(int argc, char *argv[]) { interface->init(argc, argv); }
  void run() { interface->run(); }
  void end() { interface->end(); }



  void add_object(_Object& obj);

  void add_player(Player& p);
  void server_add_player(int socknum);
  void server_add_player_other(int socknum, Trame& trame, Trame& cycle);

  void remove_object(int obj_id);

  void remove_player(int id);
  void server_remove_player(Player *p, int reason);

  Player& lookup_player(int player_id);
  _Object& lookup_object(int obj_id);
  
  void server_cycle();
  void cycle();
  void own_cycle();

  void get_client_responses();
  void read_player_response(Player& p, Trame& t);  
  void read_player_actions(Player& p, Trame& t);

  void remove_dead();

  void send_description(int socknum);
  void read_description(Trame& t);

  void update_objects();

  void read_new_player(Trame& t);
  _Object* read_new_object(Trame& t);

  void read_changes(Trame& t);

  void build_maptype();
  void draw();
  void display();

  void kill_all_players();
  void quit_game();
  
  void send_to_all(Trame& t);



private:
  Pix lookup_player0(int player_id);
  Pix lookup_object0(int obj_id);
  
  void read_new_object0(_Object& obj, Trame& t);
  void remove_object0(Pix& p);
  int is_free_player_number(int num);
  int first_free_player();
  void close_all_sockets(); // a revoir dans la semantique

  bool _server_no_computer; // set to true for no computer-control worms

  int  _hack_own_player_number; // @@ yes, hack for clients only
  
  
}; // class World


inline World::World() : 
  Base(*this), cycle_trame(0), nbplayers(0), nbobjects(0), debug(0)
{ 
  // default is no interface
  interface = new VoidInterface(*this);
} // for clients


#endif // __H_WORLD__
