#
#  gpsman --- GPS Manager: a manager for GPS receiver data
#
#  Copyright (c) 2003 Miguel Filgueiras (mig@ncc.up.pt) / Universidade do Porto
#
#    This program is free software; you can redistribute it and/or modify
#      it under the terms of the GNU General Public License as published by
#      the Free Software Foundation; either version 2 of the License, or
#      (at your option) any later version.
#
#      This program is distributed in the hope that it will be useful,
#      but WITHOUT ANY WARRANTY; without even the implied warranty of
#      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#      GNU General Public License for more details.
#
#      You should have received a copy of the GNU General Public License
#      along with this program.
#
#  File: command.tcl
#  Last change:  7 March 2003
#

# this file is only source-ed in command-line mode

### change some initial values if needs be

if { $COMMAND(rec) } {
    # use an "accept anything from receiver" setting
    RecCanSendAnything $MYGPS
}

if { $COMMAND(log) } {
    set NotLogging 0
}

##### executing command

proc ExecCommand {} {
    # execute command from command line
    # return 0 on success, 1 on error, "wait" to loop
    global COMMAND GPSProtocol PVTState RECPROTS GPSREC NoImgLib NoGSHP

    switch $COMMAND(name) {
	checkconn {
	    # "is connected" command
	    return [ExecGPSConnectFail]
	}
	show {
	    CmdShow $COMMAND(what)
	}
	haslib {
	    switch $COMMAND(what) {
		gpsmanshp {
		    if { $NoGSHP } { return 1 }
		}
		Img {
		    if { $NoImgLib } { return 1 }
		}
		default { return 1 }
	    }
	}
	getwrite {
	    if { [CmdGetFails $COMMAND(rectypes)] || \
		    [CmdWriteFails $COMMAND(format) $COMMAND(path) \
		                   $COMMAND(filetypes)] } {
		return 1
	    }
	}
	readput {
	    if { [CmdReadFails $COMMAND(format) $COMMAND(path) \
		               $COMMAND(filetypes)] || \
		    [CmdPutFails $COMMAND(rectypes)] } {
		return 1
	    }
	}
	translate {
	    if { [CmdReadFails $COMMAND(infmt) $COMMAND(inpath) \
		               $COMMAND(intypes)] || \
		    [CmdWriteFails $COMMAND(format) $COMMAND(path) \
		                   $COMMAND(filetypes)] } {
		return 1
	    }
	}
	getrtimelog {
	    if { [set prot $COMMAND(prot)] == "" && \
		    [catch {set prot $RECPROTS($GPSREC)}] } {
		return 1
	    }
	    # this depends on the Garmin variant...
	    if { $prot == "garmin" && [ExecGPSConnectFail] } { return 1 }
	    set GPSProtocol $prot	    
	    if { ! [StartRealTimeLog $COMMAND(interv)] || \
		    [CmdRTimeDumpFails $COMMAND(path)] } { return 1 }
	    set PVTState on
	    return wait
	}
    }
    return 0
}

proc ExecGPSConnectFail {} {
    # return 1 on error, 0 on success
    global GPSOpResult

    if { [GPSConnection Ignore Ignore] == 0 } { return 1 }
    vwait GPSOpResult
    return $GPSOpResult
}

proc CmdGetFails {types} {
    # get items of given $types
    # return 0 on success
    global GPSOpResult

    if { [ExecGPSConnectFail] } { return 1 }
    foreach t $types {
	GetGPS $t
	vwait GPSOpResult
	if { $GPSOpResult } { return 1 }
    }
    return 0
}

proc CmdPutFails {types} {
    # put items of given $types
    # return 0 on success
    global GPSOpResult Storage

    if { [ExecGPSConnectFail] } { return 1 }
    foreach wh $types {
	set ids [lindex $Storage($wh) 0]
	global $ids

	PutGPS $wh [array names $ids]
	vwait GPSOpResult
	if { $GPSOpResult } { return 1 }
    }
    return 0
}

proc CmdOpenFile {path mode} {
    # open file
    #   $mode in {in, out, app}
    # if $path in {stdin, stdout} for suitable $mode return $path
    #  otherwise return file descriptor or -1 on error

    if { $mode == "in" } {
	if { $path == "stdin" } { return stdin }
	set mode r
    } elseif { $path == "stdout" } {
	return stdout
    } elseif { $mode == "out" } {
	set mode w
    } else { set mode a }
    if { [catch {set f [open $path $mode]}] } { return -1 }
    return $f
}

proc CmdReadFails {fmt path types} {
    # read file in format $fmt
    # if  $path==stdin  use standard input unless $fmt is Shapefile* in which
    #  case return failure
    # return 0 on success
    global FILEFORMAT

    switch -glob $fmt {
	gpsman {
	    if { [set f [CmdOpenFile $path in]] == -1 || \
		    [catch {set r [LoadFileFrom $f Data]}] || \
		    $r == 0 } { return 1 }
	}
	Shapefile* {
	    if { $path == "stdin" } { return 1 }
	    # a single type per file
	    set dim [string index $fmt end-1]
	    return [ImportShapefileFrom $path $types $dim]
	}
	default {
	    if { $FILEFORMAT($fmt,filetype) != "data" } {
		# a single type per file
		set wh $types
	    } else { set wh Data }
	    if { $fmt == "GPStrans" } {
		set cr 1
	    } else { set cr 0 }
	    if { [set f [CmdOpenFile $path in]] == -1 } { return 1 }
	    InitInputFileVars $f $wh $cr 0
	    if { [catch {set r [ImportFileFrom $f $wh $fmt]}] || $r == 1 } {
		return 1
	    }
	}
    }
    return 0
}	    

proc CmdWriteFails {fmt path types} {
    # write data to file in format $fmt
    # if  $path==stdout  use standard output unless $fmt is Shapefile*
    #  in which case return failure
    # return 0 on success
    global FILEFORMAT

    switch -glob $fmt {
	gpsman {
	    if { [set f [CmdOpenFile $path out]] == -1 } { return 1 }
	    if { [llength $FILEFORMAT(gpsman,types)] == [llength $types] } {
		if { [catch {SaveFileTo $f all Data}] } { return 1 }
	    } else {
		foreach t $types {
		    if { [catch {SaveFileTo $f all $t}] || \
			    [set f [CmdOpenFile $path app]] == -1 } {
			return 1
		    }
		}
		if { $path != "stdout" } { close $f }
	    }
	}
	Shapefile* {
	    if { $path == "stdout" } { return 1 }
	    # a single type per file
	    return [ExportFileTo $path all $fmt $types]
	}
	default {
	    if { [set f [CmdOpenFile $path out]] == -1 } { return 1 }
	    if { $FILEFORMAT($fmt,filetype) != "data" } {
		# a single type per file
		if { [catch {set r [ExportFileTo $f all $fmt $types]}] || \
			$r == 1 } { return 1 }
	    } else {
		# more than 1 type per file
		# assume that file can be written in append mode a type at
		#  a time
		foreach t $types {
		    if { [catch {set r [ExportFileTo $f all $fmt $t]}] || \
			    $r == 1  || \
			    [set f [CmdOpenFile $path app]] == -1 } {
			return 1
		    }
		}
	    }
	    if { $path != "stdout" } { close $f }
	}
    }
    return 0
}

proc CmdRTimeDumpFails {path} {
    # start dumping real-time log continuously
    # return 0 on success
    global PVTDFile PVTDumping

    if { [set PVTDFile [CmdOpenFile $path out]] == -1 } { return 1 }
    set PVTDumping 1
    # SavePVTData $PVTDFile
    return 0
}

proc CmdShow {what} {
    # print information to stdout
    global VERSION FILEFORMAT RECPROTS TXT MESS

    switch $what {
	version { puts $VERSION }
	formats {
	    array set mode [list \
		    in  $TXT(fread) \
		    out $TXT(fwrite) \
		    {in out} $TXT(fread)/$TXT(fwrite)]
	    foreach p [lsort -dictionary [array names FILEFORMAT *,mode]] {
		regsub {,mode$} $p "" f
		if { ! [catch {set dep $FILEFORMAT($f,depends)}] && \
			! [eval $dep] } {
		    puts "$f:\t$MESS(shpwrongfile)"
		    continue
		}
		set m $mode($FILEFORMAT($p))
		puts -nonewline "$f:\t$m, $FILEFORMAT($f,types)"
		if { [set t $FILEFORMAT($f,filetype)] == "data" } {
		    puts ", $TXT(nameData)"
		} else { puts "" }
	    }
	}
	protocols {
	    foreach p [lsort -dictionary [array names RECPROTS *,*]] {
		regexp {^(.+),(.+)$} $p x rec prot
		puts "$prot ($rec)"
	    }
	}
    }
    return
}

##### logo window

proc ShowGPSMan {} {
    # create widgets for displaying information about GPSMan in root window
    global SmallLogo VERSION COLOUR EPOSX EPOSY TXT

    wm protocol . WM_DELETE_WINDOW { exit 1 }
    frame .fr -relief flat -borderwidth 5 -bg $COLOUR(messbg)
    if { "$SmallLogo" != "" } {
	label .fr.im -image $SmallLogo -relief sunken
	pack .fr.im
    }
    label .fr.title -text "$TXT(GMtit) $VERSION" -relief sunken
    bind . <Button-1> About

    pack .fr.title -side top -pady 5
    pack .fr -side top
    update idletasks
    return
}







