#
#  gpsman --- GPS Manager: a manager for GPS receiver data
#
#  Copyright (c) 2003 Miguel Filgueiras (mig@ncc.up.pt) / Universidade do Porto
#
#    This program is free software; you can redistribute it and/or modify
#      it under the terms of the GNU General Public License as published by
#      the Free Software Foundation; either version 2 of the License, or
#      (at your option) any later version.
#
#      This program is distributed in the hope that it will be useful,
#      but WITHOUT ANY WARRANTY; without even the implied warranty of
#      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#      GNU General Public License for more details.
#
#      You should have received a copy of the GNU General Public License
#      along with this program.
#
#  File: command_parse.tcl
#  Last change:  19 March 2003
#
# Includes contributions by
#  - Valere Robin (valere.robin@wanadoo.fr) marked "VR contribution"
#

# this file is only source-ed in command-line mode

##### parsing command line
###    after source-ing options file
###    before source-ing main.tcl and other files!

## properties of known file formats:
#   indices as NAME,PROPERTY  where PROPERTY is
#
#    mode       I/O mode in {in, out, {in out}}
#    depends    optional dependencies as evaluable code (globals must be
#                declared), returning 0 on failure
#    filetype   in {unique, single, data}
#    types      admissible types as a subset of $TYPES-{GR}

# includes VR contribution

array set FILEFORMAT {
    gpsman,mode           {in out}
    gpsman,filetype       data
    gpsman,types          {WP RT TR}
    GPStrans,mode         {in out}
    GPStrans,filetype     single
    GPStrans,types        {WP RT TR}
    Shapefile_2D,mode     {in out}
    Shapefile_2D,depends  { global NoGSHP ; expr ! $NoGSHP }
    Shapefile_2D,filetype single
    Shapefile_2D,types    {WP RT TR}
    Shapefile_3D,mode     {in out}
    Shapefile_3D,depends  { global NoGSHP ; expr ! $NoGSHP }
    Shapefile_3D,filetype single
    Shapefile_3D,types    {WP RT TR}
    EasyGPS,mode          in
    EasyGPS,filetype      unique
    EasyGPS,types         WP
    Fugawi,mode           in
    Fugawi,filetype       unique
    Fugawi,types          WP
    GD2,mode              in
    GD2,filetype          single
    GD2,types             {WP RT TR}
    GTrackMaker,mode      in
    GTrackMaker,filetype  data
    GTrackMaker,types     {WP RT TR}
    MapGuide,mode        in
    MapGuide,filetype    unique
    MapGuide,types       RT
    MapSend,mode          in
    MapSend,depends       { global GPSREC
                            expr ! [string compare $GPSREC "Magellan"] }
    MapSend,filetype      single
    MapSend,types         {WP RT TR}
    Meridian,mode         in
    Meridian,depends      { global GPSREC
                            expr ! [string compare $GPSREC "Magellan"] }
    Meridian,filetype     single
    Meridian,types        {WP RT TR}
}

## receiver protocols with external and internal representations

array set RECPROTS {
    Garmin        garmin
    Garmin,Garmin garmin
    Garmin,NMEA   nmea
    Garmin,SText  stext
    Garmin,simul  simul
}

##### parse command line

proc BadCommandLine {} {
    # parse command line
    # this must be called before source-ing main.tcl and other files!
    # result will be stored on COMMAND array
    # commands accepted
    #   possible prefixes:
    #       -dev DEVICE
    #       -log
    #       -rec Garmin | Lowrance | Magellan   |   -prefs PATH
    #       -prot Garmin | NMEA | SText | simul
    #   commands proper:
    #           is available | connected
    #           show version | formats | protocols
    #           haslib gpsmanshp | Img
    #		getwrite IN-TYPES FORMAT [OUT-TYPES] PATH
    #           readput FORMAT [IN-TYPES] PATH [OUT-TYPES]  (-rec not allowed!)
    #           translate FORMAT [IN-TYPES] PATH FORMAT [OUT-TYPES] PATH
    # return 1 on error
    global argv COMMAND SERIALPORT GPSREC

    array set COMMAND {
	prefsfile ""
	rec 0
	log 0
	prot ""
    }
    set pargs 0
    while 1 {
	switch -- [lindex $argv 0] {
	    -dev {
		set pos 1
		set SERIALPORT [lindex $argv 1]
		set argv [lreplace $argv 0 1]
	    }
	    -log {
		set pos 2
		set COMMAND(log) 1
		set argv [lreplace $argv 0 0]
	    }
	    -rec {
		# not to be used with either  readput  or  -prefs
		#  or after  -prot
		if { $COMMAND(prefsfile) != "" || \
			$COMMAND(prot) != "" } { return 1 }
		set pos 4
		set GPSREC [lindex $argv 1]
		set argv [lreplace $argv 0 1]
		if { [lsearch {Garmin Lowrance Magellan} $GPSREC] == -1 } {
		    return 1
		}
		set COMMAND(rec) 1
	    }
	    -prefs {
		# not to be used with  -rec
		if { $COMMAND(rec) } { return 1 }
		set pos 8
		set COMMAND(prefsfile) [lindex $argv 1]
		set argv [lreplace $argv 0 1]
	    }
	    -prot {
		set pos 16
		set COMMAND(prot) [CmdParseProt [lindex $argv 1]]
		if { $COMMAND(prot) == "" } { return 1 }
		set argv [lreplace $argv 0 1]
	    }
	    default {
		break
	    }
	}
	# no repeated options
	if { $pargs & $pos } { return 1 }
	incr pargs $pos
    }

    if { [set cmd [lindex $argv 0]] == "readput" && $COMMAND(rec) } {
	#  -rec  option not allowed
	return 1
    }	

    set argv [lreplace $argv 0 0]
    switch -- $cmd {
	is {
	    switch -- [lindex $argv 0] {
		available { exit 0 }
		connected {
		    # for the time being this is only supported for
		    #  Garmin receivers
		    if { $GPSREC != "Garmin" } { return 1 }
		    set cmd checkconn
		}
		default { return 1 }
	    }
	}
	show {
	    set what [lindex $argv 0]
	    if { [lsearch "version formats protocols" $what] == -1 } {
		return 1
	    }
	    set COMMAND(what) $what
	}
	haslib {
	    set what [lindex $argv 0]
	    if { [lsearch "gpsmanshp Img" $what] == -1 } {
		return 1
	    }
	    set COMMAND(what) $what
	}
	getwrite {
	    # IN-TYPES FORMAT [OUT-TYPES] PATH
	    # for the time being this is only supported for Garmin receivers
	    if { $GPSREC != "Garmin" } { return 1 }
	    foreach "in_ts tail" [CmdParseTypes $argv] {}
	    if { [llength $tail] < 2 || \
		    [set in_ts [CmdRecTypes $in_ts]] == "" } {
		return 1
	    }
	    set fmt [lindex $tail 0]
	    foreach "out_ts tail" [CmdParseTypes [lreplace $tail 0 0]] {}
	    if { $out_ts == "" } { set out_ts $in_ts }
	    if { [llength $tail] != 1 || \
		    [set COMMAND(path) $tail] == "" || \
		    [CmdFileFmtTypes out $fmt $out_ts] == 0 || \
		    [CmdIncompatibleTypes $in_ts $COMMAND(filetypes)] } {
		return 1
	    }
	    set COMMAND(rectypes) $in_ts
	}
	readput {
	    # FORMAT [IN-TYPES] PATH [OUT-TYPES]
	    # for the time being this is only supported for Garmin receivers
	    if { $GPSREC != "Garmin" } { return 1 }
	    # no  -rec  option
	    if { $COMMAND(rec) } { return 1 }
	    if { [set fmt [lindex $argv 0]] == "" } { return 1 }
	    foreach "in_ts tail" [CmdParseTypes [lreplace $argv 0 0]] {}
	    if { [set COMMAND(path) [lindex $tail 0]] == "" || \
		    [CmdFileFmtTypes in $fmt $in_ts] == 0 } {
		return 1
	    }
	    foreach "out_ts tail" [CmdParseTypes [lreplace $tail 0 0]] {}
	    if { $out_ts == "" } { set out_ts $COMMAND(filetypes) }
	    if { $tail != "" || \
		    [set out_ts [CmdRecTypes $out_ts]] == "" || \
		    [CmdIncompatibleTypes $COMMAND(filetypes) $out_ts] } {
		return 1
	    }
	    set COMMAND(rectypes) $out_ts
	}
	translate {
	    # FORMAT [IN-TYPES] PATH FORMAT [OUT-TYPES] PATH
	    if { [llength $argv] < 4 } { return 1 }
	    if { [set infmt [lindex $argv 0]] == "" } { return 1 }
	    foreach "in_ts tail" [CmdParseTypes [lreplace $argv 0 0]] {}
	    if { [set COMMAND(inpath) [lindex $tail 0]] == "" || \
		    [CmdFileFmtTypes in $infmt $in_ts] == 0 || \
		    [set outfmt [lindex $tail 1]] == "" } {
		return 1
	    }
	    foreach "out_ts tail" [CmdParseTypes [lreplace $tail 0 1]] {}
	    if { $out_ts == "" } { set out_ts $COMMAND(filetypes) }
	    set COMMAND(infmt) $infmt
	    set COMMAND(intypes) $COMMAND(filetypes)
	    if { [llength $tail] != 1 || \
		    [set COMMAND(path) $tail] == "" || \
		    [CmdFileFmtTypes out $outfmt $out_ts] == 0 || \
		    [CmdIncompatibleTypes $COMMAND(intypes) \
					  $COMMAND(filetypes)] } {
	        return 1
	    }
	}
	getrtimelog {
	    # PATH [INTERVAL]
	    # for the time being this is only supported for Garmin receivers
	    if { $GPSREC != "Garmin" } { return 1 }
	    if { [set COMMAND(path) [lindex $argv 0]] == "" || \
		    [set n [llength $argv]] > 2 } {
		return 1
	    }
	    if { $n == 1 } {
		# default is 2 seconds
		set COMMAND(interv) 2
	    } else {
		set COMMAND(interv) [lindex $argv 1]
		if { ! [regexp {^[1-9][0-9]*$} $COMMAND(interv)] } {
		    return 1
		}
	    }	    
	}
	default { return 1 }
    }
    set COMMAND(name) $cmd
    return 0
}

proc CmdParseProt {prot} {
    # translate external form of protocol
    # return "" on error
    global RECPROTS GPSREC

    if { [catch {set p $RECPROTS($GPSREC,$prot)}] } { return "" }
    return $p
}

proc CmdParseTypes {lst} {
    # find element(s) at the beginning of $lst corresponding to item types:
    #   either a single "all", or a subset of {WP, RT, TR}
    # return pair with list of those elements and rest of list

    if { [lindex $lst 0] == "all" } {
	set ts all ; set lst [lreplace $lst 0 0]
    } else {
	set ts ""
	while { [lsearch -exact "WP RT TR" [lindex $lst 0]] != -1 } {
	    lappend ts [lindex $lst 0] ; set lst [lreplace $lst 0 0]
	}
    }
    return [list $ts $lst]
}

proc CmdRecTypes {lst} {
    # interpret list of item types to be get/put from/into receiver
    #  $lst may be either "all", or non-empty subset of $TYPES-{GR}
    # return "" on error, otherwise list of types as non-empty subset
    #  of $TYPES-{GR}

    if { $lst == "all" } { return "WP RT TR" }
    return $lst
}


proc CmdFileFmtTypes {how fmt lst} {
    # check that $fmt is a valid file format for I/O operation on file
    #  with given item types
    #  $how in {in, out}
    #  $lst is either "all", or subset of $TYPES-{GR}
    # set COMMAND(filetypes) and COMMAND(format)
    # return 0 on error
    global FILEFORMAT COMMAND

    # check mode and dependencies
    if { [catch {set mode $FILEFORMAT($fmt,mode)}] || \
	    ! [regexp $how $mode] || \
	    ! ( [catch {set depok $FILEFORMAT($fmt,depends)}] || \
	        [eval $depok] ) } {
	    return 0
    }
    # check types
    set fts $FILEFORMAT($fmt,types)
    switch $FILEFORMAT($fmt,filetype) {
	unique {
	    if { $lst != "" && $lst != "all" && $lst != $fts } {
		return 0
	    }
	}
	single {
	    if { $lst == "all" || [llength $lst] != 1 || \
		    [lsearch -exact $fts $lst] == -1 } { return 0 }
	    set fts $lst
	}
	data {
	    if { $lst != "" && $lst != "all" } {
		set fs ""
		foreach f $lst {
		    if { [lsearch -exact $fts $f] == -1 } { return 0 }
		    lappend fs $f
		}
		set fts $fs
	    }
	}
    }
    set COMMAND(format) $fmt
    set COMMAND(filetypes) $fts
    return 1
}

proc CmdIncompatibleTypes {in_ts out_ts} {
    # check compatibility of input and output types
    # the only possible "conversion" is from RT to WP
    # return 1 on error

    if { $in_ts == $out_ts } { return 0 }
    foreach o $out_ts {
	if { [lsearch $in_ts $o] == -1 && \
		( $o != "WP" || [lsearch $in_ts RT] == -1 ) } {
	    return 1
	}
    }
    return 0
}