#
#  gpsman --- GPS Manager: a manager for GPS receiver data
#
#  Copyright (c) 2003 Miguel Filgueiras (mig@ncc.up.pt) / Universidade do Porto
#
#    This program is free software; you can redistribute it and/or modify
#      it under the terms of the GNU General Public License as published by
#      the Free Software Foundation; either version 2 of the License, or
#      (at your option) any later version.
#
#      This program is distributed in the hope that it will be useful,
#      but WITHOUT ANY WARRANTY; without even the implied warranty of
#      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#      GNU General Public License for more details.
#
#      You should have received a copy of the GNU General Public License
#      along with this program.
#
#  File: files_foreign.tcl
#  Last change:  19 March 2003
#
# See also files.tcl
#  and command_parse.tcl for properties of known file formats
#
# Includes contributions by
#  - Brian Baulch (baulchb@onthenet.com.au) marked "BSB contribution"
#  - Matt Martin (matt.martin@ieee.org) marked "MGM contribution"
#  - Miguel Filgueiras (to code by others) marked "MF contribution"
#  - Valere Robin (valere.robin@wanadoo.fr) marked "VR contribution"
#
# Includes an adaptation of a Perl script by Niki Hammler, http://www.nobaq.net
#  that converts exported FUGAWI data to DDD GPSman data
#

##### reading/writing binary files

## binary coding of files:
#   little- and big-endian integral numbers
#    int, int_big: 16 bits;  long, long_big: 32 bits
#   fixed-length strings: as sequence of bytes with given length
#   variable-length strings: length as int (little-endian), then the
#    sequence of bytes
#   floats and doubles: IEEE little-endian
#   booleans as ints
#   arrays of multiples: e.g., array@3=double,long is an array of pairs
#    whose length is given by element at index 3 (from 0) on the data list
#    (index < index of array); element type cannot be array! a list of lists
#    is used as the internal representation of the array

array set BinConv {
    byte   c    byte,l 1
    int    s    int,l  2
    bool   s    bool,l 2
    int_big S    int_big,l 2
    long   i    long,l 4
    long_big I   long_big,l 4
}

proc ReadBinData {file types} {
    # adapted from proc UnPackData (garmin.tcl)
    # read binary data and convert to list of elements conforming to the
    #  types in the list $types
    # accepted types: byte, int, int_big, long, long_big, float, double,
    #                 charray=*, varstring, unused=*, bool, array@*=*
    #  bool is the same as int
    # delete leading and trailing spaces of strings and char arrays
    # return list of values read in
    global tcl_platform ReadBinError BinConv

    set vals ""
    foreach t $types {
	switch -glob $t {
	    byte {
		set n 1
		binary scan [read $file $n] "c" x
		set x [expr ($x+0x100)%0x100]
	    }
	    bool -   int -   int_big -   long - 
	    long_big {
		set n $BinConv($t,l)
		binary scan [read $file $n] $BinConv($t) x
	    }
	    float {
		# this only works with machines following the
		#  IEEE standard floating point representation
		set n 4
		set bs [read $file $n] 
		if { "$tcl_platform(byteOrder)" == "littleEndian" } {
		    binary scan $bs "f" x
		} else {
		    set id ""
		    foreach k "3 2 1 0" {
			append id [string index $bs $k]
		    }
		    binary scan $id "f" x
		}
	    }
	    double {
		# this only works with machines following the
		#  IEEE standard floating point representation
		set n 8
		set bs [read $file $n]
		if { "$tcl_platform(byteOrder)" == "littleEndian" } {
		    binary scan $bs "d" x
		} else {
		    set id ""
		    foreach k "7 6 5 4 3 2 1 0" {
			append id [string index $bs $k]
		    }
		    binary scan $id "d" x
		}
	    }
	    varstring {
		if { [set lg [ReadBinData $file int]] < 0 } {
		    set ReadBinError 1
		    return $vals
		}
		set x [read $file $lg]
		set n [expr 2+$lg]
		set x [string trim $x " "]
	    }
	    charray=* {
		regsub charray= $t "" n
		set x [read $file $n]
		set x [string trim $x " "]
	    }
	    array*@* {
		if { ! [regexp {array@([0-9]+)=(.+)$} $t m ix lst] || \
			[set lg [lindex $vals $ix]] < 0 || \
			! [regexp {^[0-9]+$} $lg] } {
		    set ReadBinError 1
		    return $vals
		}
		set x "" ; set sts [split $lst ","]
		while { $lg } {
		    incr lg -1
		    lappend x [ReadBinData $file $sts]
		    if { $ReadBinError } { return [lappend vals $x] }
		}
	    }
	    unused=* {
		regsub unused= $t "" n
		read $file $n
		set x UNUSED
	    }
	    default {
		set ReadBinError 1
		return $vals
	    }
	}
	lappend vals $x
    }
    return $vals
}

proc WriteBinData {file types vals} {
    # adapted from proc DataToStr (garmin.tcl)
    # write binary data the result of converting from list of elements
    #  conforming to the types in $types
    # accepted types: byte, int, int_big, long, long_big, float, double,
    #                 charray=*, varstring, unused=*, bool, array@*=*
    #  bool is the same as int
    # return 0 on error
    global tcl_platform BinConv

    foreach t $types v $vals {
	switch -glob $t {
	    byte -  bool -  int -  int_big -  long -
	    long_big {
		puts -nonewline $file [binary format $BinConv($t) $v]
	    }
	    float {
		# this only works with machines following the
		#  IEEE standard floating point representation
		set s [binary format "f" $v]
		if { "$tcl_platform(byteOrder)" != "littleEndian" } {
		    set l [split "$s" ""]
		    set s ""
		    foreach k "3 2 1 0" {
			append s [lindex $l $k]
		    }
		}
		puts -nonewline $file $s
	    }
	    double {
		# this only works with machines following the
		#  IEEE standard floating point representation
		set s [binary format "d" $v]
		if { "$tcl_platform(byteOrder)" != "littleEndian" } {
		    set l [split "$s" ""]
		    set s ""
		    foreach k "7 6 5 4 3 2 1 0" {
			append s [lindex $l $k]
		    }
		}
		puts -nonewline $file $s
	    }
	    varstring {
		WriteBinData $file int [string length $v]
		puts -nonewline $file [binary format "a*" $v]
	    }
	    charray=* {
		regsub charray= $t "" n
		puts -nonewline $file [binary format "A$n" $v]
	    }
	    array*@* {
		if { ! [regexp {array@([0-9]+)=(.+)$} $t m ix lst] || \
			[set lg [lindex $vals $ix]] < 0 || \
			! [regexp {^[0-9]+$} $lg] } {
		    BUG "error in specification to WriteBinData: $t $lg"
		    return 0
		}
		set sts [split $lst ","]
		while { $lg } {
		    incr lg -1
		    if { ! [WriteBinData $file $sts [lindex $v 0]] } {
			return 0
		    }
		    set v [lreplace $v 0 0]
		}
	    }
	    unused=* {
		regsub unused= $t "" n
		puts -nonewline $file [binary format "x$n"]
	    }
	    default {
		BUG "unimplemented data type when converting to binary: $t"
		return 0
	    }
	}
    }
    flush $file
    return 1
}

##### exporting

proc ExportFile {how fmt what} {
    # write data to file in foreign format
    #  $fmt in {GPStrans, Shapefile_2D, Shapefile_3D, MapSend, Meridian}
    #  $how==all: all items of type $what
    #  $how==select: items of type $what chosen from list
    #  $what for GPStrans, Shapefile*: in {WP, RT, TR}
    # return 0 on success, 1 on failure

    return [ExportFileTo "" $how $fmt $what]
}

proc ExportFileTo {file how fmt what} {
    global SFilePFrmt SFilePFType SFileDatum Storage TXT

    switch $how {
	all {
	    set ids [lindex $Storage($what) 0]
	    global $ids
	    set ixs [array names $ids]
	}
	select {
	    if { "[set ixs [ChooseItems $what]]" == "" } { return 1 }
	}
    }
    if { [regexp {^Shapefile_(.+)D$} $fmt m dim] } {
	return [ExportShapefileTo $file $what $ixs $dim]
    }
    if { $file == "" && \
	    [set file [GMOpenFile $TXT(exportto) $what w]] == ".." } {
	return 1
    }
    SetCursor . watch
    switch $fmt {
	GPStrans {
	    ExportHeader $file
	    Export$what $file $ixs 1
	}
	MapSend {
	    # MGM contribution
	    fconfigure $file -translation {binary binary}
	    ExportMapSend$what $file $ixs
	}
	Meridian {
	    # MGM contribution
	    #fconfigure $file -translation {binary binary}
	    ExportMeridian$what $file $ixs
	}
    }
    ResetCursor .
    foreach v "PFrmt PFType Datum" { catch {unset SFile$v($file)} }
    close $file
    return 0
}

proc ExportGREls {how fmt args} {
    # write data on elements of group to file in foreign format
    #  $fmt in {GPStrans, Shapefile_2D, Shapefile_3D}
    #  $how==all: from all groups, element type chosen from list
    #  $how==select: groups and element type chosen from list
    #  $args not used but in call-back
    # in any case only elements of one type in {WP, RT, TR} will be considered
    #  because these formats so require
    global GRName SFilePFrmt SFilePFType SFileDatum MESS TXT

    set types "WP RT TR GR"
    switch $how {
	all {
	    set ixs [array names GRName]
	}
	select {
	    if { "[set ixs [ChooseItems GR]]" == "" } { return }
	}
    }
    set ts ""
    foreach k $types { lappend ts $TXT(name$k) }
    while 1 {
	# select types of items to export
	set whs [GMChooseFrom many $MESS(putwhat) 6 $ts $types]
	if { [set i [lsearch -exact $whs GR]] != -1 } {
	    set whs [lreplace $whs $i $i]
	    set rec 1
	} else { set rec 0 }
	if { [set n [llength $whs]] == 0 } { return }
	if { $n > 1 && ("$fmt" == "GPStrans" || \
		[string first Shapefile $fmt] == 0) } {
	    # GPStrans and Shapefile files have only 1 type of items
	    GMMessage $MESS(exportonly1)
	} else {
	    break
	}
    }
    if { "[set eixs [GRsElements $ixs $rec $whs]]" == "" } { return }
    if { [regexp {^Shapefile_(.+)D$} $fmt m dim] } {
	ExportShapefileTo "" $whs $eixs $dim
	return
    }
    set f [GMOpenFile $TXT(exportto) GR w]
    if { "$f" != ".." } {
	SetCursor . watch
	switch $fmt {
	    GPStrans {
		ExportHeader $f
		Export${whs} $f $eixs 1
	    }
	}
	catch {
	    unset SFilePFrmt($f) ; unset SFilePFType($f) ; unset SFileDatum($f)
	}
	close $f
	ResetCursor .
    }
    return
}

## GPStrans format

proc ExportHeader {file} {
    # write header to file in GPStrans format, using $pformt for positions
    global TimeOffset Datum SFilePFrmt SFilePFType SFileDatum

    set dix [DatumRefId $Datum]
    if { ! [regexp {^[0-9]+$} $dix] } {
	# datum not defined in GPStrans
	set datum "WGS 84"
	set dix [DatumRefId $datum]
    } else { set datum $Datum }
    set h [format {Format: DDD  UTC Offset: %6.2f hrs  Datum[%3d]: %s} \
	    $TimeOffset $dix $datum]
    puts $file $h
    set SFileDatum($file) $datum
            # following values assumed below
    set SFilePFrmt($file) DDD
    set SFilePFType($file) latlong
    return
}

proc ExportWP {file ixs tofile} {
    # write WPs with indices in list $ixs either to file in GPStrans format,
    #  or to auxiliary file for putting data into receiver
    global WPName WPCommt WPPFrmt WPPosn WPDatum WPDate
    global CREATIONDATE SFileDatum

    set d [Today MMDDYYYY]
    foreach i $ixs {
	if { $i != -1 } {
	    if { $CREATIONDATE } {
		set d $WPDate($i)
	    }
	    set latd [lindex $WPPosn($i) 0] ; set longd [lindex $WPPosn($i) 1]
	    if { "$WPDatum($i)" != "$SFileDatum($file)" } {
		set p [ConvertDatum $latd $longd \
			            $WPDatum($i) $SFileDatum($file) DDD]
		set latd [lindex $p 0] ; set longd [lindex $p 1]
	    }
	    if { $tofile } {
		set m [format "W\t%s\t%40s\t%20s\t%03.7f\t%04.7f" \
			      $WPName($i) $WPCommt($i) $d $latd $longd]
	    } else {
		set m [format "W\t%s\t%40s\t%20s\t%03.7f\t%04.7f" \
		       $WPName($i) [JustfLeft 40 $WPCommt($i)] $d $latd $longd]
	    }
	    puts $file $m
	}
    }
    return
}

proc ExportRT {file ixs tofile} {
    # write RTs with indices in list $ixs either to file in GPStrans format,
    #  or to auxiliary file for putting data into receiver
    global RTIdNumber RTCommt RTWPoints MAXROUTES MESS

    set badids 0
    foreach i $ixs {
	if { ! [CheckNumber Ignore $RTIdNumber($i)] } {
	    incr badids
	} else {
	    set wps [Apply "$RTWPoints($i)" IndexNamed WP]
	    if { [Undefined $wps] } {
		GMMessage [format $MESS(undefWP) $RTIdNumber($i)]
	    } elseif { !$tofile && $RTIdNumber($i)>$MAXROUTES } {
		GMMessage [format $MESS(bigRT) $RTIdNumber($i)]
	    } else {
		puts $file [format "R\t%d\t%40s" $RTIdNumber($i) \
			                [JustfLeft 40 $RTCommt($i)]]
		ExportWP $file $wps $tofile
	    }
	}
    }
    if { $badids > 0 } { GMMessage [format $MESS(cantsaveRTid) $badids] }
    return
}

proc ExportTR {file ixs tofile} {
    # write TRs with indices in list $ixs either to file in GPStrans format,
    #  or to auxiliary file for putting data into receiver
    global TRTPoints TRDatum SFileDatum

    foreach i $ixs {
	set c [string compare $TRDatum($i) $SFileDatum($file)]
	foreach tp $TRTPoints($i) {
	    set latd [lindex $tp 0] ; set longd [lindex $tp 1]
	    if { $c } {
		set p [ConvertDatum $latd $longd \
			            $TRDatum($i) $SFileDatum($file) DDD]
		set latd [lindex $p 0] ; set longd [lindex $p 1]
	    }
	    puts $file [format "T\t%s\t%lf\t%lf" [lindex $tp 4] $latd $longd]
	}
	puts $file ""
    }
    return
}

## Shapefile format

set SHPUndef -1e40
set SHPDatum "WGS 84"

proc ExportShapefileTo {fname what ixs dim} {
    # export to Shapefile format items with indices $ixs of type $what
    #  $fname may be empty in which case the user is asked to give a file name
    #  $what in {WP, RT, TR}
    #  $dim in {2, 3} for planar or spatial (with altitude) coordinates
    # all data converted to $SHPDatum (currently set to "WGS 84")
    # return 1 on error, 0 on success
    global WPName WPCommt WPDate WPPosn WPDatum WPAlt RTIdNumber RTCommt \
	    RTWPoints TRName TRObs TRDatum TRTPoints DataIndex MESS TXT \
	    SHPUndef SHPDatum

    if { $fname == "" } {
	set ok 0
	while { "[set fn [GMGetFileName $TXT(exportto) $what  w "" ""]]" \
		!= ".." } {
	    set basename [file join [file dirname $fn] [file rootname $fn]]
	    switch -- [set ext [file extension $fn]] {
		.shp -  .shx -  .dbf -  "" {
		    set ok 1 ; break
		}
		default {
		    if { [GMConfirm [format $MESS(shpext) $ext]] } {
			set ok 1 ; break
		    }
		}
	    }
	}
	if { ! $ok } { return 1 }
    } else {
	set basename [file join [file dirname $fname] [file rootname $fname]]
    }
    if { [set fsid [GSHPCreateFiles $basename $what $dim]] < 1 } {
	switch -- $fsid {
	    0 { set m shpcntopen }
	    -1 -
	    -2 { BUG invalid type or dim }
	    -3 { set m shpcntcrtfs }
	    -4 { set m shpoutmem }
	}
	GMMessage $MESS($m) ; GSHPCloseFiles $fsid
	return 1
    }
    SetCursor . watch
    switch $what {
	WP {
	    foreach ix $ixs {
		set p $WPPosn($ix)
		if { "$WPDatum($ix)" != "$SHPDatum" } {
		    set p [ConvertDatum [lindex $p 0] [lindex $p 1] \
			    $WPDatum($ix) $SHPDatum DDD]
		}
		if { $dim == 3 } {
		    if { "[set alt $WPAlt($ix)]" == "" } { set alt $SHPUndef }
		    set r [GSHPWriteWP $fsid [lindex $p 1] [lindex $p 0] \
			    $alt $WPName($ix) $WPCommt($ix) $WPDate($ix)]
		} else {
		    set r [GSHPWriteWP $fsid [lindex $p 1] [lindex $p 0] \
			    $WPName($ix) $WPCommt($ix) $WPDate($ix)]
		}
		switch -- $r {
		    -3 {
			GMMessage $MESS(shpoutmem)
			return 1
		    }
		    -4 { GMMessage $MESS(shpcntwrtfs) ; return 1 }
		}
	    }
	}
	RT {
	    foreach ix $ixs {
		GSHPCreateRT $dim $RTIdNumber($ix) $RTCommt($ix)
		set wpixs [Apply "$RTWPoints($ix)" IndexNamed WP]
		if { [Undefined $wpixs] } {
		    GMMessage [format $MESS(undefWP) $RTIdNumber($ix)]
		    return 1
		} else {
		    foreach wpix $wpixs {
			set p $WPPosn($wpix)
			if { "$WPDatum($wpix)" != "$SHPDatum" } {
			    set p [ConvertDatum [lindex $p 0] [lindex $p a] \
				    $WPDatum($wpix) $SHPDatum DDD]
			}
			if { $dim == 3 } {
			    if { "[set alt $WPAlt($wpix)]" == "" } {
				set alt $SHPUndef
			    }
			    set r [GSHPAddWPToRT [lindex $p 1] \
				    [lindex $p 0] $alt]
			} else {
			    set r [GSHPAddWPToRT [lindex $p 1] [lindex $p 0]]
			}
			if { $r == -2 } {
			    GMMessage $MESS(shpoutmem)
			    return 1
			}
		    }
		    switch -- [GSHPWriteRT $fsid 1] {
			-5 { GMMessage $MESS(shpoutmem) ; return 1 }
			-6 { GMMessage $MESS(shpcntwrtfs) ; return 1 }
		    }
		}
	    }
	}
	TR {
	    set ilt $DataIndex(TPlatd) ; set ilg $DataIndex(TPlongd)
	    set ial $DataIndex(TPalt)
	    foreach ix $ixs {
		GSHPCreateTR $dim $TRName($ix) $TRObs($ix)
		if { "$TRDatum($ix)" != "$SHPDatum" } {
		    set tps [ChangeTPsDatum $TRTPoints($ix) \
			    $TRDatum($ix) $SHPDatum]
		} else { set tps $TRTPoints($ix) }
		foreach tp $tps {
		    if { $dim == 3 } {
			if { "[set alt [lindex $tp $ial]]" == "" } {
			    set alt $SHPUndef
			}
			set r [GSHPAddTPToTR [lindex $tp $ilg] \
				[lindex $tp $ilt] $alt]
		    } else {
			set r [GSHPAddTPToTR [lindex $tp $ilg] \
				[lindex $tp $ilt]]
		    }
		    if { $r == -2 } {
			GMMessage $MESS(shpoutmem)
			return 1
		    }
		}
		switch -- [GSHPWriteTR $fsid 1] {
		    -5 { GMMessage $MESS(shpoutmem) ; return 1 }
		    -6 { GMMessage $MESS(shpcntwrtfs) ; return 1 }
		}
	    }
	}
    }
    GSHPCloseFiles $fsid
    ResetCursor .
    return 0
}

##### importing

proc OpenImportFileFails {what fmt} {
    # open file in foreign format and set initial values for importing data
    #  $what in $TYPES
    # return 0 unless operation is to be cancelled

    return [OpenInputFileFails $what $fmt]
}

proc ImportFile {what fmt} {
    # open file in foreign format and load data from it
    #  $fmt in {GPStrans, EasyGPS, Fugawi, GD2, GTrackMaker, Shapefile_2D,
    #            Shapefile_3D, MapGuide, MapSend, Meridian}
    #  $what for GPStrans, GD2, Shapefile*: in {WP, RT, TR}
    #        for EasyGPS, Fugawi: WP
    #        for GTrackMaker: Data
    #        for MapGuide: RT

    return [ImportFileFrom "" $what $fmt]
}

proc ImportFileFrom {file what fmt} {
    # if $file=="" then ask user to select a file
    # return 1 on error, 0 on success
    global LChannel GFItemId GFItemCommt GFItemNB

    InitWPRenaming
    if { [regexp {^Shapefile_(.+)D$} $fmt m dim] } {
	return [ImportShapefileFrom $file $what $dim]
    }
    if { $file == "" && [OpenImportFileFails $what $fmt] } { return 1 }
    switch $fmt {
	GPStrans {
	    if { [ImportHeader $LChannel($what)] } {
		Import$what $LChannel($what) normal
	    }
	}
	EasyGPS -  Fugawi -  MapSend -  Meridian {
	    # includes MGM contribution
	    # includes VR contribution
	    Import$fmt $LChannel($what) normal
	}
	GD2 {
	    ImportGD2 $LChannel($what) $what normal
	}
	GTrackMaker {
	    ImportGTrackMaker $LChannel($what)
	}
	MapGuide {
	    ImportMapGuide $LChannel($what) $GFItemId $GFItemCommt $GFItemNB
	}
    }
    CloseInputFile $what
    EndWPRenaming
    return 0
}

proc ImportGREls {what fmt} {
    # load data from file in foreign format according to contents of GR(s)
    #  $fmt in {GPStrans, EasyGPS, Fugawi, GD2}
    #  $what for GPStrans, GD2: WP and RT (no way to identify TRs)
    #        for EasyGPS, Fugawi: WP
    #  $what in $TYPES identifies menu, not being used
    global TYPES

    switch $fmt {
	GPStrans -  GD2 {
	    set ts $TYPES ; set exc TR
	}
	EasyGPS -  Fugawi {
	    # includes VR contribution
	    set ts WP ; set exc ""
	}
    }
    InputToGR $ts $exc OpenImportFileFails ImportGRElsIn CloseInputFile $fmt
    return
}

proc ImportGRElsIn {what ixs fmt} {
    # import items of type $what to replace items with indices in $ixs
    #  or new items if -1 in $ixs from file in foreign format
    #  $fmt in {GPStrans, EasyGPS, Fugawi, GD2}
    #  $what for GPStrans, GD2: in {WP, RT}
    #        for EasyGPS, Fugawi: WP
    global LChannel LFileIxs

    set file $LChannel(GR)
    set LFileIxs($file) $ixs
    switch $fmt {
	GPStrans {
	    if { [ImportHeader $file] } {
		Import$what $file inGR
	    }
	}
	EasyGPS -  Fugawi {
	    # includes VR contribution
	    ImportFugawi $file inGR
	}
	GD2 {
	    ImportGD2 $file $what inGR
	}
    }
    unset LFileIxs($file)
    return
}

## GPStrans format

proc ImportHeader {file} {
    # parse header when reading from file in GPStrans format
    global LFilePFrmt LFilePFType LFileDatum LFileEOF MESS

    set m [ReadFile $file]
    if { $LFileEOF($file) || [string first "Format:" $m]!=0 } {
	GMMessage "$MESS(noformat): $m"
	return 0
    }
    set m [string range $m 8 end]
    set LFilePFrmt($file) [FindPFormat $m]
    if { "$LFilePFrmt($file)" != "BAD" } {
	set LFilePFType($file) [PosType $LFilePFrmt($file)]
	set m [string range $m 17 end]
	if { [scan $m "%f" off] == 1 } {
	    # don't know what to do with time offset
	    set LFileDatum($file) [string range $m 24 end]
	    if { [DatumRefId $LFileDatum($file)] == -1 } {
		GMMessage "$MESS(unkndatum): $LFileDatum($file)"
		return 0
	    }
	    return 1
	}
    }
    GMMessage "$MESS(badformat): $m"
    return 0
}

proc ImportWP {file how} {
    # load WPs from file in GPStrans format
    #  $how in {normal, inGR}: keep all data, keep data on WPs with indices
    #    in $LFileIxs($file)

    LoadWPs $file 1 1 $how
    return
}

proc ImportRT {file how} {
    # load RTs from file in GPStrans format
    #  $how in {normal, inGR}: keep all data, keep data on RTs with
    #     indices in $LFileIxs($file)
    global LFileLNo LFileEOF MESS

    while { 1 } {
	set m [ReadFile $file]
	if { $LFileEOF($file) } { return }
	if { ! [regexp "R\t.*" $m] } {
	    GMMessage "$MESS(badRT) $LFileLNo($file)"
	    return
	}
	if { "[FindArgs RT [string range $m 2 end] $file]" == "BADC" } {
	    GMMessage "$MESS(badRTargs) $LFileLNo($file)"
	    return
	}
	if { [BadRTRead $file 1 $how] } { return }
    }
}

proc ImportTR {file how} {
    # load TRs from file in GPStrans format
    #  $how in {normal, inGR}: keep all data, not used
    global LFileLNo LFileEOF LFileBuffFull MESS

    while { 1 } {
	set m [ReadFile $file]
	if { $LFileEOF($file) } { return }
	if { ! [regexp "T\t.*" $m] } {
	    GMMessage "$MESS(badTR) $LFileLNo($file)"
	    return
	}
	set LFileBuffFull($file) 1
	if { [BadTRRead $file 1 $how] } { return }
    }
}

## Fugawi export format

proc ImportFugawi {file how} {
    # this is a translation and adaptation of the Perl program "convert" under
    #   copyright by Niki Hammler, http://www.nobaq.net
    #   that converts exported FUGAWI data to DDD GPSman data
    global LFileEOF LFileVisible LFileLNo LFileIxs MESS

    set date [Now]
    set dt "" ; set ixs "" ; set ns "" ; set chgns ""
    while { "[set line [ReadFile $file]]" != "" && ! $LFileEOF($file) } {
	set vs [split $line ","]
	if { [set l [llength $vs]] < 5 } {
	    GMMessage "$MESS(nofieldsWP): $LFileLNo($file)" ; return
	} elseif { $l > 5 } {
	    GMMessage "$MESS(excfieldsWP): $LFileLNo($file)" ; return
	}
	set name [lindex $vs 0]
	if { ! [CheckName Ignore $name] } {
	    if { [set nname [AskForName $name]] == "" } { continue }
	    set chgdname $name
	    set name $nname
	} else { set chgdname "" }
	set lat [lindex $vs 2] ; set long [lindex $vs 3]
	if { ! [CheckLat GMMessage $lat DDD] || \
		! [CheckLong GMMessage $long DDD] } { continue }
	set latd [Coord DDD $lat S] ; set longd [Coord DDD $long W]
	set posn [list $latd $longd $lat $long]
	set comment [MakeComment [lindex $vs 1]]
	lappend dt [list $name $comment DDD $posn "WGS 84" $date]
	lappend ixs [IndexNamed WP $name]
	lappend ns $name
	lappend chgns $chgdname
    }
    if { $ns == "" } { return }
    switch $how {
	normal {
	    foreach ix $ixs n $ns d $dt chgdname $chgns {
		set fd [FormData WP "Name Commt PFrmt Posn Datum Date" $d]
		if { $chgdname != "" } {
		    set fd [AddOldNameToObs $fd $chgdname]
		}
		StoreWP $ix $n $fd $LFileVisible($file)
	    }
	}
	inGR {
	    set grixs $LFileIxs($file)
	    foreach ix $ixs n $ns d $dt {
		if { [lsearch -exact $grixs $ix] != -1 } {
		    set fd [FormData WP "Name Commt PFrmt Posn Datum Date" $d]
		    StoreWP $ix $n $fd $LFileVisible($file)
		}
	    }
	}
    }
    return
}

#### import from file in EasyGPS export format
####  VR contribution

proc ImportEasyGPS {file how} {

    global LFileEOF LFileVisible LFileLNo LFileIxs MESS

    set date [Now]
    set dt "" ; set ixs "" ; set ns ""
    while { "[set line [ReadFile $file]]" != "" && ! $LFileEOF($file) } {
	set vs [split $line "<"]
	set l [llength $vs]
	for {set ii 3} {$ii<[expr $l - 2]} {incr ii 10} {
	    set name [lindex $vs [expr $ii + 1]]	    
	    regsub {name id=\"} $name "" name
	    regsub {\">} $name "" name
	    set position [lindex $vs [expr $ii + 4]]
	    regsub {coord lat=\"} $position "" lat
	    regsub {\" lon=.*} $lat "" lat
	    regsub {coord .*lon=\"} $position "" long
 	    regsub {\".*} $long "" long
	    if { ! [CheckName Ignore $name] && \
		    "[set name [AskForName $name]]" == "" } { continue }
	    if { ! [CheckLat GMMessage $lat DDD] || \
		    ! [CheckLong GMMessage $long DDD] } { continue }
	    set latd [Coord DDD $lat S] ; set longd [Coord DDD $long W]
	    set posn [list $latd $longd $lat $long]
	    set obs [lindex $vs [expr $ii + 2]]]
	    regsub {.*CDATA\[} $obs "" obs
	    regsub -all {\]\]\>\]} $obs "" obs
	    set obs1 [lindex $vs [expr $ii + 7]]]
	    regsub {.*>} $obs1 "" obs1
	    #regsub {<} $obs1 "" obs1
	    regsub {\]} $obs1 "" obs1
	    lappend dt [list $name "$obs - $obs1" DDD $posn "WGS 84" $date]
	    lappend ixs [IndexNamed WP $name]
	    lappend ns $name
	}	

    }
    ## MF contribution
    if { $ixs == "" } {
	GMMessage $MESS(shpemptyfile)
	return
    }
    ##

    switch $how {
	normal {
	    foreach ix $ixs n $ns d $dt {
		set fd [FormData WP "Name Obs PFrmt Posn Datum Date" $d]
		StoreWP $ix $n $fd $LFileVisible($file)
	    }
	}
	inGR {
	    set grixs $LFileIxs($file)
	    foreach ix $ixs n $ns d $dt {
		if { [lsearch -exact $grixs $ix] != -1 } {
		    set fd [FormData WP "Name Commt PFrmt Posn Datum Date" $d]
		    StoreWP $ix $n $fd $LFileVisible($file)
		}
	    }
	}
    }
    return
}
### end of VR contribution

#### Map&Guide export format

proc ImportMapGuide {file rtid rtcommt rtrmrk} {
    # this is an adaptation of the script "mg2gpsman.tcl"
    #   under copyright by Heiko Thede (Heiko.Thede@gmx.de)
    #   that converts exported Map&Guide data to GPSman data
    # each file corresponds to one RT that will be split in more than one
    #  if its length exceeds $MAXWPINROUTE
    #  $rtid is the RT identifier/number given by user; it will be
    #   replaced by an automatically generated one if empty
    #  $rtcommt and $rtrmrk are the comment and remark given by the user
    global MAXWPINROUTE TXT MESS PositionFormat LFileVisible

    # get rid of leading/trailing blanks
    foreach v "rtid rtcommt rtrmrk" {
	set $v [string trim [set $v]]
    }
    # generate RT id if none given
    if { $rtid == "" } {
	set rtid [NewName RT] ; set rtix -1
    } else {
	set rtix [IndexNamed RT $rtid]
    }
    set rts "" ; set wps ""

    # WPs: prefix for names, position type
    #  the first $rtid is used for all WPs even if the RT is split
    set prefix "$TXT(RT)$rtid"
    set ptype [PosType $PositionFormat]
    set nwps 0
    # positions of WPs, to avoid creating different WPs in the same place
    set coords "" ; set coordWPs ""
    # read lines
    while { ! [eof $file] } {
	gets $file line
	#select only relevant lines
	if { $line != "" } {
	    set coordstart [string last "(" $line]
	    set coordend [string last ")" $line]
	    set coord [string range $line $coordstart $coordend]
	    if { [set i [lsearch -exact $coords $coord]] != -1 } {
		# use WP in this position
		set wpt [lindex $coordWPs $i]
	    } else {
		# create new WP and remember its position and name
		# generate name using $prefix if possible
		if { [catch { set n [scan $coord (%2d%2d%d,%2d%2d%d) \
			longd longm longs latd latm lats]}] || $n != 6 } {
		    # bad line: ignore WP
		    continue
		}
		set fields [split $line "\t"]
		if { [string length $coord] == 17 } {
		    set longs [expr 0.1*$longs]
		    set lats [expr 0.1*$lats]
		    set wprmrk "[lindex $fields 0] [lrange $fields 2 3]"
		} else {
		    set wprmrk "[lindex $fields 0] [lrange $fields 2 4]"
		}
		set lat [expr $latd+($latm+$lats/60.0)/60.0]
		set long [expr $longd+($longm+$longs/60.0)/60.0]
		set posn [CreatePos $lat $long $PositionFormat $ptype "WGS 84"]
		set wpt [NewName WP $prefix]
		lappend coords $coord ; lappend coordWPs $wpt
		set data [FormData WP "Name Obs PFrmt Posn Datum" \
			[list $wpt $wprmrk $PositionFormat $posn "WGS 84"]]
		StoreWP -1 $wpt $data 0
	    }
	    if { $nwps == $MAXWPINROUTE } {
		# save previous route and start new one
		lappend rts $rtid $wps
		# cannot use proc NewName here as previous RTs were not
		#  stored yet
		set rtid ""
		set nwps 0 ; set wps ""
	    }
	    lappend wps $wpt
	    incr nwps
	}
    }
    if { $wps == "" && $rts == "" } {
	GMMessage $MESS(voidRT)
	return
    }
    if { $nwps > 0 } {
	# save last route
	lappend rts $rtid $wps
    }
    # prepare comment and remark fields if any
    if { $rtcommt != "" && [CheckComment Ignore $rtcommt] } {
	set fields Commt
	set fvals [list $rtcommt]
	set rmrkix 1
    } else {
	set fields "" ; set fvals ""
	set rmrkix 0
    }
    if { $rtrmrk != "" } {
	lappend fields Obs
	lappend fvals $rtrmrk
	# prepare for adding new info to remark
	set norem 0
	set rtrmrk "${rtrmrk}\n"
    } else { set norem 1 }
    lappend fields IdNumber WPoints
    # store all routes
    foreach "rtid wps" $rts {
	if { $rtid == "" } {
	    set rtid [NewName RT] ; set rtix -1
	}
	set fd [FormData RT $fields [linsert $fvals end $rtid $wps]]
	StoreRT $rtix $rtid $fd $wps $LFileVisible($file)

	# add "Insert after $rtid" to remark for use in next route
	set rmk "${rtrmrk}$TXT(insa): $rtid"
	if { $norem } {
	    # add remark field
	    set fields [linsert $fields $rmrkix Obs]
	    set fvals [linsert $fvals $rmrkix $rmk]
	    set norem 0
	} else {
	    # replace remark
	    set fvals [lreplace $fvals $rmrkix $rmrkix $rmk]
	}
    }
    return
}

# gd2 format

proc ImportGD2 {file what how} {
    # import data from $file in gd2 format
    #  $how in {normal, inGR}
    #  $what in {WP, RT, TR}, but not TR if $how==inGR
    # gd2.c is a program by Randolph Bentson (bentson@grieg.seaslug.org)
    #  distributed under GPL
    global LFileEOF LFileVisible LFileBuffFull LFileLNo LFileIxs MESS

    set date [Now]
    set dt "" ; set ixs "" ; set ns "" ; set error 0
    switch $what {
	WP { set ns "" ; set chgns "" }
	RT {
	    set rtid "" ; set dtwps "" ; set lwps "" ; set lchgwps ""
	    set ldwps "" ; set rtwps start
	}
	TR { set tps "" }
    }
    while { "[set line [ReadFile $file]]" != "" && ! $LFileEOF($file) } {
	switch $what {
	    WP {
		if { ! [regexp \
		{^WPT  ? (......) ([-0-9\.]+) ([-0-9\.]+) ([-0-9/:]+) (.*)$} \
			$line x name lat long date commt] || \
		    [set dwp [ImportGD2WP $name $lat $long $date $commt]] == \
		    -1 } {
		    set error 1
		} else {
		    foreach "name chgdn d ix" $dwp {}
		    lappend dt $d ; lappend ixs $ix ; lappend ns $name
		    lappend chgns $chgdn
		}
	    }
	    RT {
		if { "$rtwps" == "" || \
			! [regexp {^RTE ([0-9]+) (.*)$} line rtid rtcommt] } {
		    set error 1
		} else {
		    set rtcommt [MakeComment [string trim $rtcommt]]
		    set what RTWP
		    set rtwps "" ; set dtwps "" ; set rtchgwps ""
		}
	    }
	    RTWP {
		if { ! [regexp \
		       {^ (......) ([-0-9\.]+) ([-0-9\.]+) ([-0-9:/]+) (.*)$} \
			$line x name latd longd date commt] } {
		    set what RT ; set LFileBuffFull($file) 1
		    lappend dt [list $rtid $rtcommt $rtwps]
		    lappend ixs [IndexNamed RT $rtid]
		    lappend ns $rtid
		    lappend lwps $rtwps ; lappend lchgwps $rtchgwps
		    lappend ldwps $dtwps
		    set rtid ""
		} elseif { [set dwp \
			[ImportGD2WP $name $lat $long $date $commt]] == -1 } {
		    set error 1
		} else {
		    foreach "name chgdn d ix" $dwp {}
		    lappend dtwps $d ; lappend rtwps $name
		    lappend rtchgwps $chgdn
		}
	    }
	    TR {
		if { [regexp \
    {^TRK ((S|N)[0-9]+ [0-9\.]+) ((W|E)[0-9]+ [0-9\.]+) ([-0-9:/]+) (0|1)$} \
			$line x latdmm x longdmm x date new] } {
		    set lat [Coord DMM $latdmm S]
		    set long [Coord DMM $longdmm W]
		} elseif { ! [regexp \
			{^TRK ([-0-9\.]+) ([-0-9\.]+) ([-0-9:/]+) (0|1)$} \
			lat long date new] } {
		    set error 1
		}
		if { ! $error } {
		    if { ! [CheckLat GMMessage $lat DDD] || \
			    ! [CheckLong GMMessage $long DDD] || \
			    "[set datesecs [CheckConvDate $date]]" == "" } {
			set error 1
		    } else {
			if { $new && "$tps" != "" } {
			    lappend dt $tps
			    set tps ""
			}
			set p [CreatePos $lat $long DMS latlong "WGS 84"]
			lappend tps [FormData TP \
				       "latd longd latDMS longDMS date secs" \
				       [concat $p $datesecs]]
		    }
		}
	    }
	}
	if { $error } {
	    GMMessage "$MESS(loaderr) $LFileLNo($file)"
	    return
	}
    }
    # terminate pending RT or TR if any
    switch $what {
	RT {
	    if { "$rtid" != "" } {
		if { "$rtwps" == "" } {
		    GMMessage "$MESS(loaderr) $LFileLNo($file)"
		    return
		}
		lappend dt [list $rtid $rtcommt $rtwps]
		lappend ixs [IndexNamed RT $rtid]
		lappend ns $rtid
		lappend lwps $rtwps ; lappend lchgwps $rtchgwps
		lappend ldwps $dtwps
	    }
	}
	TR {
	    if { "$tps" != "" } {
		lappend dt $tps
	    }
	}
    }
    switch $how {
	normal {
	    switch $what {
		WP {
		    foreach ix $ixs n $ns d $dt chgdname $chgns {
			set fd [FormData WP \
				"Name Commt PFrmt Posn Datum Date" $d]
			if { $chgdname != "" } {
			    set fd [AddOldNameToObs $fd $chgdname]
			}
			StoreWP $ix $n $fd $LFileVisible($file)
		    }
		}
		RT {
		    set wpsseen "" ; set wpsseenn ""
		    foreach ix $ixs id $ns d $dt wps $ltwps dwps $ldwps \
			    chgns $lchgwps {
			set wpsn ""
			foreach nwp $wps dwp $dwps chgdname $chgns {
			    if { [set k [lsearch -exact $wpsseen $nwp]] \
				    == -1 } {
				set ix [IndexNamed WP $nwp]
				set fd [FormData WP \
				       "Name Commt PFrmt Posn Datum Date" $dwp]
				if { $chgdname != "" } {
				    set fd [AddOldNameToObs $fd $chgdname]
				}
				set nnwp [StoreWP $ix $nwp $fd 0]
				lappend wpsseen $nwp
				lappend wpsseenn $nnwp
				lappend wpsn $nnwp
			    } else {
				lappend wpsn [lindex $wpsseenn $k]
			    }
			}
			StoreRT $ix $id $d $wpsn $LFileVisible($file)
		    }
		}
		TR {
		    foreach tps $dt {
			set name [NewName TR]
			set fd [FormData TR "Name Datum TPoints" \
				[list $name "WGS 84" $tps]]
			StoreTR -1 $name $fd $LFileVisible($file)
		    }
		}
	    }
	}
	inGR {
	    set grixs $LFileIxs($file)
	    switch $what {
		WP {
		    foreach ix $ixs n $ns d $dt chgdname $chgns {
			if { [lsearch -exact $grixs $ix] != -1 } {
			    set fd [FormData WP \
				    "Name Commt PFrmt Posn Datum Date" $d]
			    if { $chgdname != "" } {
				set fd [AddOldNameToObs $fd $chgdname]
			    }
			    StoreWP $ix $n $fd $LFileVisible($file)
			}
		    }
		}
		RT {
		    set wpsseen "" ; set wpsseenn ""
		    foreach ix $ixs id $ns d $dt wps $ltwps dwps $ldwps \
			    chgns $lchgwps {
			if { [lsearch -exact $grixs $ix] != -1 } {
			    set wpsn ""
			    foreach nwp $wps dwp $dwps chgdname $chgns {
				if { [set k [lsearch -exact $wpsseen $nwp] \
					== -1 } {
				    set ix [IndexNamed WP $nwp]
				    set fd [FormData WP \
					   "Name Commt PFrmt Posn Datum Date" \
					   $dwp]
				    if { $chgdname != "" } {
					set fd [AddOldNameToObs $fd $chgdname]
				    }
				    set nnwp [StoreWP $ix $nwp $fd 0]
				    lappend wpsseen $nwp
				    lappend wpsseenn $nnwp
				    lappend wpsn $nnwp
				} else {
				    lappend wpsn [lindex $wpsseenn $k]
				}
			    }
			    StoreRT $ix $id $d $wpsn $LFileVisible($file)
			}
		    }
		}
	    }
	}
    }
    return
}

proc ImportGD2WP {name lat long date commt} {
    # prepare WP data when importing from file in gd2 format
    # return 0 on error, otherwise list with name, old name (or ""),
    #  list with WP data (see below) and index

    set name [string trim $name]
    if { ! [CheckLat GMMessage $lat DDD] || \
	    ! [CheckLong GMMessage $long DDD] } { return -1 }
    if { ! [CheckName Ignore $name] } {
	if { [set nname [AskForName $name]] == "" } { return -1 }
	set chgdname $name
	set name $nname
    } else { set chgdname "" }
    set posn [CreatePos $lat $long DDD latlong "WGS 84"]
    set commt [MakeComment [string trim $commt]]
    return [list $name $chgdname [list $name $commt DDD $posn "WGS 84" $date] \
	    [IndexNamed WP $name]]
}

## GTrackMaker format

  # datums having the same definition in GTM and GPSMan (possibly under
  #  different names
array set GTMEquivDatum {
    1   "Adindan; B Faso"        2   "Adindan; Cameroon"
    3   "Adindan; Ethiopia"      4   "Adindan; Mali"
    5	"Adindan; Ethiopia+Sudan"    6    "Adindan; Senegal"
    7   "Adindan; Sudan"         8   "Afgooye"
    9   "Ain el Abd 1970; Bahrain"    10  "Ain el Abd 1970; Saudi Arabia"
    11  "American Samoa 1962"    13  "Antigua Island Astro 1943"
    14  "Arc 1950; Botswana"     15  "Arc 1950; Burundi"
    16  "Arc 1950; Lesotho"      17  "Arc 1950; Malawi"
    18  "Arc 1950"               19  "Arc 1950; Swaziland"
    20  "Arc 1950; Zaire"        21  "Arc 1950; Zambia"
    22  "Arc 1950; Zimbabwe"     23  "Arc 1960; Kenya+Tanzania"
    24  "Arc 1960; Kenya"        25  "Arc 1960; Tanzania"
    26  "Ascension Island `58"   27  "Astro Beacon \"E\""
    28  "Astro DOS 71/4"         29  "Astro Tern Island (FRIG)"
    30  "Astronomic Stn `52"     34  "Bellevue (IGN)"
    35  "Bermuda 1957"           36  "Bissau"
    37  "Bogota Observatory"     38  "Bukit Rimpah"
    39  "Camp Area Astro"        40  "Campo Inchauspe"
    41  "Canton Astro 1966"      42  "Cape"
    43  "Cape Canaveral"         44  "Carthage"
    45  "Chatham 1971"           46  "Chua Astro"
    47  "Corrego Alegre"         48  "Dabola"
    49  "Deception Island"       50  "Djakarta (Batavia)"
    51  "DOS 1968"               52  "Easter Island 1967"
    53  "Estonia Coord System 1937"    54  "European 1950; Cyprus"
    55  "European 1950; Egypt"   56  "European 1950; England Channel"
    57  "European 1950; England Channel"
    58  "European 1950; Finland+Norway"
    59  "European 1950; Greece"  60  "European 1950; Iran"
    61  "European 1950; Italy (Sardinia)"
    62  "European 1950; Italy (Sicily)"
    63  "European 1950; Malta"   64  "European 1950"
    65  "European 1950; NW Europe"
    66  "European 1950; Middle East"
    67  "European 1950; Portugal+Spain"
    68  "European 1950; Tunisia"
    69  "European 1979"          70  "Fort Thomas 1955"
    71  "Gan 1970"               72  "Geodetic Datum `49"
    73  "Graciosa Base SW 1948"  74  "Guam 1963"
    75  "Gunung Segara"          76  "GUX 1 Astro"
    77  "Herat North"            78  "Hermannskogel"
    79  "Hjorsey 1955"           80  "Hong Kong 1963"
    81  "Hu-Tzu-Shan"            82  "Indian (Bangladesh)"
    83  "Indian (India, Nepal)"  84  "Indian (Pakistan)"
    85  "Indian 1954"            86  "Indian 1960; Vietnam (Con Son)"
    87  "Indian 1960; Vietnam (N16)"
    88  "Indian 1975"            89  "Indonesian 1974"
    90  "Ireland 1965"           91  "ISTS 061 Astro 1968"
    92  "ISTS 073 Astro `69"     93  "Johnston Island 1961"
    94  "Kandawala"              95  "Kerguelen Island"
    96  "Kertau 1948"            97  "Kusaie Astro 1951"
    98  "NAD83; Canada"          99  "L.C. 5 Astro"
    100 "Leigon"                 101 "Liberia 1964"
    102 "Luzon Philippines"      103 "Luzon Mindanao"
    104 "M'Poraloko"             105 "Mahe 1971"
    106 "Massawa"                107 "Merchich"
    108 "Midway Astro 1961"      109 "Minna; Cameroon"
    110 "Minna"                  111 "Montserrat Island Astro 1958"
    112 "Nahrwn Masirah Ilnd"    113 "Nahrwn Saudi Arbia"
    114 "Nahrwn United Arab"     115 "Naparima BWI"
    116 "NAD27 Alaska"           117 "NAD27 Alaska; Aleutian East"
    118 "NAD27 Alaska; Aleutian West"
    119 "NAD27 Bahamas"          120 "NAD27 San Salvador"
    121 "NAD27 Canada West"      122 "NAD27 Canada Middle"
    123 "NAD27 Canada East"      124 "NAD27 Canada North"
    125 "NAD27 Canada Yukon"     126 "NAD27 Canal Zone"
    127 "NAD27 Cuba"             128 "NAD27 Greenland"
    129 "NAD27 Caribbean"        130 "NAD27 Central"
    131 "NAD27 Canada"           132 "NAD27 CONUS"
    133 "NAD27 CONUS East"       134 "NAD27 CONUS West"
    135 "NAD27 Mexico"           136 "NAD83; Canada"
    137 "NAD83; Aleutian Ids"    138 "NAD83; Canada"
    139 "NAD83; Canada"          140 "NAD83; Hawaii"
    141 "NAD83; Canada"          142 "North Sahara 1959"
    143 "Observatorio 1966"      144 "Old Egyptian"
    145 "Old Hawaiian; Hawaii"   146 "Old Hawaiian; Kauai"
    147 "Old Hawaiian; Maui"     148 "Old Hawaiian"
    149 "Old Hawaiian; Oahu"     150 "Oman"
    151 "Ord Srvy Grt Britn; England"
    152 "Ord Srvy Grt Britn; England+Wales"
    153 "Ord Srvy Grt Britn"     154 "Ord Srvy Grt Britn; Scotland+Shetlands"
    155 "Ord Srvy Grt Britn; Wales"
    156 "Pico De Las Nieves"     157 "Pitcairn Astro 1967"
    158 "Point 58"               159 "Pointe Noire 1948"
    160 "Porto Santo 1936"       161 "Prov So Amrican 56; Bolivia"
    162 "Prov So Amrican 56; Chile North"
    163 "Prov So Amrican 56; Chile South"
    164 "Prov So Amrican 56; Colombia"
    165 "Prov So Amrican 56; Ecuador"
    166 "Prov So Amrican 56; Guyana"
    167 "Prov So Amrican `56"    168 "Prov So Amrican 56; Peru"
    169 "Prov So Amrican 56; Venezuela"
    170 "Prov So Chilean `63"    171 "Puerto Rico"
    172 "Pulkovo 1942"           173 "Qatar National"
    174 "Qornoq"                 175 "Reunion"
    176 "Rome 1940"              177 "S-42 (Pulkovo 1942); Hungary"
    178 "S-42 (Pulkovo 1942); Poland"
    179 "S-42 (Pulkovo 1942); Czechoslavakia"
    180 "S-42 (Pulkovo 1942); Latvia"
    181 "S-42 (Pulkovo 1942); Kazakhstan"
    182 "S-42 (Pulkovo 1942); Albania"
    183 "S-42 (Pulkovo 1942); Hungary"
    184 "S-JTSK"                 185 "Santo (DOS)"
    186 "Sao Braz"               187 "Sapper Hill 1943"
    188 "Schwarzeck"             189 "Selvagem Grande 1938"
    190 "Sierra Leone"           191 "South American 69; Argentina"
    192 "South American 69; Bolivia"
    193 "South American 69; Brazil"
    194 "South American 69; Chile"
    195 "South American 69; Colombia"
    196 "South American 69; Ecuador"
    197 "South American 69; Baltra"
    198 "South American 69; Guyana"
    199 "South American `69"      200 "South American 69; Paraguay"
    201 "South American 69; Peru"
    202 "South American 69; Trinidad+Tobago"
    203 "South American 69; Venezuela"
    204 "South Asia"              205 "Tananarive Observatory 1925"
    206 "Timbalai 1948"           207 "Tokyo"
    208 "Tokyo"                   209 "Tokyo; Okinawa"
    210 "Tokyo; South Korea"      211 "Tristan Astro 1968"
    212 "Viti Levu 1916"          213 "Voirol 1960"
    214 "Wake Island Astro 195"   217 "WGS 84"
    218 "Yacare"                  219 "Zanderij"
    220 "Rijks Driehoeksmeting"   221 "NTF (NTF ellipsoid)"
    224 "CH-1903"                 226 "European 1950; Belgium"
    227 "Israeli"                 228 "Rome 1940; Luxembourg"
    229 "Finland Hayford"         230 "Dionisos"
    231 "South American 69; Brazil/IBGE"
    232 "Potsdam"                 233 "Datum 73"
    234 "WGS 72"
}

  # datums having a different definition in GTM; GPSMan definition will be used
array set GTMEquivDatum {
    12  "Anna 1 Astro 1965"       31  "Australian Geod `66"
    32  "Australian Geod `84"     33  "Ayabelle Lighthouse"
    215 "Wake-Eniwetok 1960"      216 "WGS 1972"
    222 "Potsdam"                 223 "RT 90"
    225 "Austrian (MGI)"
}

set GTMVersions 211

array set GTMTypes {
    header {int charray=10 byte unused=1 byte unused=1 unused=1 byte unused=1
            long long long long long long long float float float float long
            long unused=4 unused=4 bool bool unused=2 unused=2 unused=2
            unused=2 unused=2 unused=2 bool unused=2 varstring varstring
            varstring varstring}
    datum {unused=2 unused=8 unused=8 unused=8 unused=8 int double double int
	   int int}
    image {varstring varstring float float float float long float float byte
           byte}
    wp {double double charray=10 varstring int byte long int float unused=2}
    wpstyle {long varstring byte long long float byte bool long byte byte byte
             byte}
    tr {double double long byte float}
    trstyle {varstring byte long float byte int}
    rt {double double charray=10 varstring varstring int byte byte long int
        float int}
    icon {varstring byte long}
    layer {int varstring long byte byte byte int}
}

array set GTMDescr {
    header {version code maplinewidth unused fontsize unused unused iconcolour
            unused gridcolour bgcolour nwpstyles usercolour nwps ntrs nrts
            maxlong minlong maxlat minlat nimgs ntrnames unused unused
            rectcoords truegrid unused unused unused unused unused unused
            hasmap unused gridfontname unused unused unused}
    datum {unused unused unused unused unused ndatum a f dx dy dz}
    image {NOTUSED}
    wp {lat long name commt symbol style secs txtangle alt unused}
    wpstyle {NOTUSED}
    tr {lat long secs new alt}
    trstyle {NOTUSED}
    rt {lat long wpname wpcommt rtname wpsymbol wpstyle new secs unused unused
        unused}
    icon {NOTUSED}
    layer {NOTUSED}
}

array set GTMConstr {
    header {
	{ if { [lsearch $GTMVersions $version] == -1 } { set m badGTMvers } }
	{ if { "$code" != "TrackMaker" } { set m badGTMfile } }
	{ if { $nwpstyles < 0 || $nwps < 0 || $ntrs < 0 || $nrts < 0 || $nimgs < 0 || $ntrnames < 0 } {
	    set m badGTMcounts } }
        { if { $nwps == 0 } { set nwpstyles 0 } }
	{ if { $ntrs == 0 } { set ntrnames 0 } }
    }
    datum {
	{ if { [catch {set eqdatum $GTMEquivDatum($ndatum)}] } {
	    set m badGTMdatum } }
    }
    wp {
	{ if { $lat < -90 || $lat > 90 } { set m badGTMlat } }
	{ if { $long < -180 || $long > 180 } { set m badGTMlong } }
    }
    tr {
	{ if { $lat < -90 || $lat > 90 } { set m badGTMlat } }
	{ if { $long < -180 || $long > 180 } { set m badGTMlong } }
    }
    rt {
	{ if { $lat < -90 || $lat > 90 } { set m badGTMlat } }
	{ if { $long < -180 || $long > 180 } { set m badGTMlong } }
    }
}

#
# file structure:
#   header, datum, image info, wps, wpstyles, trs, trstyles, rts, layers,
#   symbols, symbol images, map images
#
# header has counts of
#   wpstyles, wps, tps, rtwps, imgs, trnames

proc ImportGTrackMaker {file} {
    # names in GTMDescr lists are implicitly used here as local variables!
    global GTMVersions GTMTypes GTMDescr GTMConstr GTMEquivDatum ReadBinError \
	    PositionFormat DateFormat LFileVisible YEAR0 MESS

    fconfigure $file -translation binary
    set ReadBinError 0
    set m "" ; set one 1
    set wpl "" ; set trl "" ; set trtps "" ; set rtl "" ; set rtwps ""
    foreach b "header datum image wp wpstyle tr trstyle rt" \
	    c "one one nimgs nwps nwpstyles ntrs ntrnames nrts" {
	for { set i 0 } { $i < [set $c] } { incr i } {
	    set vals [ReadBinData $file $GTMTypes($b)]
	    if { $ReadBinError } {
		GMMessage $MESS(errorGTMread)
		return
	    }
	    if { "$GTMDescr($b)" != "NOTUSED" } {
		# assign values to vars
		foreach $GTMDescr($b) $vals {}
		# check constraints that may assign values to other variables
		foreach ct $GTMConstr($b) {
		    eval $ct
		    if { "$m" != "" } {
			GMMessage $MESS($m)
			return
		    }
		}
		# use values in variables corresponding to fields and in
		#  variables assigned during evaluation of constraints
		switch $b {
		    wp {
			# unused: symbol, style, txtangle
			lappend wpl \
				[list $lat $long $name $commt $secs $alt]
		    }
		    tr {
			if { $new } {
			    if { "$trtps" != "" } { lappend trl $trtps }
			    set trtps ""
			}
			lappend trtps [list $lat $long $secs $alt]
		    }
		    rt {
			# unused: wpsymbol, wpstyle
			# assume route name only defined when $new
			if { $new } {
			    if { "$rtwps" != "" } {
				lappend rtl [list $oldroute $rtwps]
			    }
			    set rtwps ""
			    set oldroute $rtname
			}
			lappend rtwps \
				[list $lat $long $wpname $wpcommt $secs]
		    }
		}
	    }
	}
    }
    if { "$trtps" != "" } { lappend trl $trtps }
    if { "$rtwps" != "" } {
	lappend rtl [list $oldroute $rtwps]
    }
    set oldYEAR0 $YEAR0 ; set YEAR0 1990
    set pftype [PosType $PositionFormat]
    foreach wpd $wpl {
	foreach "lat long name commt secs alt" $wpd {}
	# GTM comment saved in remark, along with name if already in use
	if { [set ix [IndexNamed WP $name]] != -1 } {
	    set commt "$name\n$commt"
	}
	set posn [CreatePos $lat $long $PositionFormat $pftype $eqdatum]
	set date [DateFromSecs $secs]
	set data [FormData WP "Name Obs PFrmt Posn Datum Date Alt" \
		[list $name $commt $PositionFormat $posn $eqdatum $date $alt]]
	StoreWP $ix $name $data $LFileVisible($file)
    }
    foreach trtps $trl {
	if { "$trtps" != "" } {
	    set tps ""
	    foreach tpd $trtps {
		foreach "lat long secs alt" $tpd {}
		set dints [DateIntsFromSecs $secs]
		set date [eval FormatDate $DateFormat $dints]
		set secs [eval DateToSecsFrom $dints $oldYEAR0]
		set d [CreatePos $lat $long DMS latlong $eqdatum]
		lappend $d $date $secs $alt
		lappend tps \
		    [FormData TP "latd longd latDMS longDMS date secs alt" $d]
	    }
	    set name [NewName TR]
	    set data [FormData TR "Name Datum TPoints" \
		                  [list $name $eqdatum $tps]]
	    StoreTR -1 $name $data $LFileVisible($file)
	}
    }
    foreach rtd $rtl {
	foreach "rtname rtwps" $rtd {}
	if { "$rtwps" != "" } {
	    set wpns ""
	    foreach wpd $rtwps {
		# GTM wp comment saved in remark, along with name if in use
		foreach "lat long wpname commt secs" $wpd {}
		if { [set ix [IndexNamed WP $wpname]] != -1 } {
		    set commt "$wpname\n$commt"
		}
		set posn \
			[CreatePos $lat $long $PositionFormat $pftype $eqdatum]
		set date [DateFromSecs $secs]
		set data [FormData WP "Name Obs PFrmt Posn Datum Date" \
			[list $name $commt $PositionFormat $posn $eqdatum \
			      $date]]
		set wpname [StoreWP $ix $wpname $data 0]
		lappend wpns $wpname
	    }
	    # GTM route name will be saved in remark
	    set id [NewName RT]
	    set data [FormData RT "IdNumber Obs WPoints" \
		                  [list $id $commt $wps]]
	    StoreRT -1 $id $data $wps $LFileVisible($file)
	}
    }
    set YEAR0 $oldYEAR0
    return
}

## Shapefile format

proc ImportShapefileFrom {fname what dim} {
    # import from Shapefile format items of type $what
    # if $fname=="" ask user to give a file name
    #  $what in {WP, RT, TR}
    #  $dim in {2, 3} for planar or spatial (with altitude) coordinates;
    #   can be lower than that of the data, if greater confirmation is
    #   asked for
    # all data assumed to be in $SHPDatum (currently set to "WGS 84"), x-
    #  and y-coordinates to be longitude and latitude in decimal degrees,
    #  z-coordinate ($dim==3) to be the altitude in metre
    # return 1 on error, 0 on success
    global SHPDatum GFVisible MESS TXT MAXWPINROUTE

    if { $fname == "" } {
	set ok 0
	while { "[set fn [GMGetFileName $TXT(importfrm) $what  r \
		GFVisible [list @$TXT(mapitems)]]]" \
		!= ".." } {
	    set basename [file join [file dirname $fn] [file rootname $fn]]
	    switch -- [set ext [file extension $fn]] {
		.shp -  .shx -  .dbf -  "" {
		    set ok 1 ; break
		}
		default {
		    if { [GMConfirm [format $MESS(shpext) $ext]] } {
			set ok 1 ; break
		    }
		}
	    }
	}
	if { ! $ok } { return 1 }
    } else {
	set basename [file join [file dirname $fname] [file rootname $fname]]
	set GFVisible 0
    }
    if { [set fsid [GSHPOpenInputFiles $basename]] < 1 } {
	switch -- $fsid {
	    0 { set m shpcntopen }
	    -1 { set m shpemptyfile }
	    -2 { set m shpwrongfile }
	    -3 { set m shpoutmem }
	}
	GMMessage $MESS($m)
	GSHPCloseFiles $fsid
	return 1
    }
    if { [set info [GSHPInfoFrom $fsid]] == 0 } { BUG bad channel ; return 1 }
    foreach "fwh fno fdim fix" $info {}
    if { $fno < 1 } {
	GMMessage $MESS(shpemptyfile) ; GSHPCloseFiles $fsid
	return 1
    }
    if { "$fwh" == "UNKNOWN" } {
	if { "$what" == "WP" } {
	    GMMessage $MESS(shpwrongfile) ; GSHPCloseFiles $fsid
	    return 1
	}
    } elseif { "$fwh" != "$what" } {
	GMMessage $MESS(shpwrongfile) ; GSHPCloseFiles $fsid
	return 1
    }
    if { $fdim < $dim && ! [GMConfirm $MESS(shplessdim)] } {
	GSHPCloseFiles $fsid
	return 1
    }
    SetCursor . watch
    set zz [expr $fdim+$dim == 6] ; set alt ""
    set tpfs "latd longd latDMS longDMS"
    switch $fwh {
	WP {
	    set wpfs "Name Commt PFrmt Posn Datum Date"
	    while { $fno } {
		incr fno -1
		if { "[set fd [GSHPGetObj $fsid $fno]]" != "" } {
		    if { $fd == 0 || $fd == -1 } {
			BUG bad GSHPGetObj ; return
		    }
		    if { "[set name [lindex $fd 0]]" == "" } {
			set name [NewName WP]
			set chgdname ""
		    } elseif { ! [CheckName Ignore $name] } {
			if { [set nname [AskForName $name]] == "" } {
			    continue
			}
			set chgdname $name ; set name $nname
		    } else { set chgdname "" }
		    set commt [MakeComment [lindex $fd 1]]
		    if { ! [CheckDate Ignore [set date [lindex $fd 2]]] } {
			set date ""
		    }
		    set long [lindex $fd 3] ; set lat [lindex $fd 4]
		    if { ! [CheckLat GMMessage $lat DDD] || \
			    ! [CheckLong GMMessage $long DDD] } { continue }
		    set posn [CreatePos $lat $long DDD latlong ""]
		    set pd [list $name $commt DDD $posn $SHPDatum $date]
		    if { $zz && ! ([BadAltitude [set alt [lindex $fd 5]]] || \
			    $alt < -1e35) } {
			set fs [linsert $wpfs end Alt]
			lappend pd $alt
		    } else { set fs $wpfs }
		    set pd [FormData WP $wpfs $pd]
		    if { $chgdname != "" } {
			set pd [AddOldNameToObs $pd $chgdname]
		    }
		    StoreWP [IndexNamed WP $name] $name $pd $GFVisible
		}
	    }
	}
	RT {
	    set wpfs "Name PFrmt Posn Datum"
	    while { $fno } {
		incr fno -1
		if { "[set fd [GSHPGetObj $fsid $fno]]" != "" } {
		    if { $fd == 0 || $fd == -1 } {
			BUG bad GSHPGetObj ; return 1
		    }
		    if { "[set name [lindex $fd 0]]" == "" || \
			    ! [CheckNumber Ignore $name] } {
			set name [NewName RT]
		    }
		    set commt [MakeComment [lindex $fd 1]]
		    if { [set np [lindex $fd 2]] < 1 } {
			GMMessage $MESS(voidRT) ; continue
		    }
		    set wpns ""
		    while { $np } {
			incr np -1
			if { [set pd [GSHPReadNextPoint $fsid]] == -2 } {
			    break
			}
			if { $pd == 0 || $pd == -1 } {
			    BUG bad GSHPReadNextPoint ; return 1
			}
			set long [lindex $pd 0] ; set lat [lindex $pd 1]
			if { ! [CheckLat Ignore $lat DDD] || \
				! [CheckLong Ignore $long DDD] } {
			    GMMessage [format $MESS(shpbadWPinRT) \
				               [expr $np+1] $name]
			    continue
			}
			lappend wpns [set wpn [NewName WP]]
			set posn [CreatePos $lat $long DDD latlong ""]
			set wdt [list $wpn DDD $posn $SHPDatum]
			if { $zz && ! ([BadAltitude [set alt [lindex $pd 2]]] \
				|| $alt < -1e35) } {
			    set fs [linsert $wpfs end Alt]
			    lappend wdt $alt
			} else { set fs $wpfs }
			StoreWP -1 $wpn [FormData WP $fs $wdt] 0
		    }
		    if { [set np [llength $wpns]] == 0 } {
			GMMessage $MESS(voidRT) ; continue
		    }
		    if { $np > $MAXWPINROUTE } {
			GMMessage [format $MESS(toomuchWPs) $MAXWPINROUTE]
		    }
		    set fd [FormData RT "IdNumber Commt WPoints" \
			    [list $name $commt $wpns]]
		    StoreRT [IndexNamed RT $name] $name $fd $wpns $GFVisible
		}
	    }
	}
	TR {
	    while { $fno } {
		incr fno -1
		if { "[set fd [GSHPGetObj $fsid $fno]]" != "" } {
		    if { $fd == 0 || $fd == -1 } {
			BUG bad GSHPGetObj ; return 1
		    }
		    if { ! [CheckString Ignore [set name [lindex $fd 0]]] } {
			set name [NewName TR]
		    }
		    set commt [MakeComment [lindex $fd 1]]
		    if { [set np [lindex $fd 2]] < 1 } {
			GMMessage $MESS(voidTR) ; continue
		    }
		    set tps ""
		    while { $np } {
			incr np -1
			if { [set pd [GSHPReadNextPoint $fsid]] == -2 } {
			    break
			}
			if { $pd == 0 || $pd == -1 } {
			    BUG bad GSHPReadNextPoint ; return 1
			}
			set long [lindex $pd 0] ; set lat [lindex $pd 1]
			if { ! [CheckLat GMMessage $lat DDD] || \
				! [CheckLong GMMessage $long DDD] } {
			    continue
			}
			set tpd [CreatePos $lat $long DMS latlong ""]
			if { $zz && ! ([BadAltitude [set alt [lindex $pd 2]]] \
				|| $alt < -1e35) } {
			    set fs [linsert $tpfs end alt]
			    lappend $tpd $alt
			} else { set fs $tpfs }
			lappend tps [FormData TP $fs $tpd]
		    }
		    if { [set np [llength $tps]] == 0 } {
			GMMessage $MESS(voidTR) ; continue
		    }
		    set fd [FormData TR "Name Obs TPoints Datum" \
			    [list $name $commt $tps $SHPDatum]]
		    StoreTR [IndexNamed TR $name] $name $fd $GFVisible
		}
	    }
	}
	UNKNOWN {
	    while { $fno } {
		incr fno -1
		if { "[set fd [GSHPGetObj $fsid $fno]]" != "" } {
		    if { $fd == -1 || $fd == -2 } {
			BUG bad GSHPGetObj ; return 1
		    }
		    set pts ""
		    while { $fd } {
			incr fd -1
			if { [set pd [GSHPReadNextPoint $fsid]] == -2 } {
			    break
			}
			if { $pd == 0 || $pd == -1 } {
			    BUG bad GSHPReadNextPoint ; return 1
			}
			set long [lindex $pd 0] ; set lat [lindex $pd 1]
			if { ! [CheckLat GMMessage $lat DDD] || \
				! [CheckLong GMMessage $long DDD] } {
			    continue
			}
			if { $zz && ([BadAltitude [set alt [lindex $pd 2]]] \
				|| $alt < -1e35) } {
			    set alt ""
			}
			lappend pts [list $lat $long $alt]
		    }
		    if { "$pts" == "" } {
			GMMessage $MESS(void$what) ; continue
		    }
		    set name [NewName $what]
		    if { "$what" == "RT" } {
			set wpfs "Name PFrmt Posn Datum"
			set wpns ""
			foreach pt $pts {
			    lappend wpns [set wpn [NewName WP]]
			    foreach "lat long alt" $pt {}
			    set posn [CreatePos $lat $long DDD latlong ""]
			    set wdt [list $wpn DDD $posn $SHPDatum]
			    if { $zz && "$alt" != "" } {
				set fs [linsert $wpfs end Alt]
				lappend wdt $alt
			    } else { set fs $wpfs }
			    StoreWP -1 $wpn [FormData WP $fs $wdt] 0
			}
			if { [llength $wpns] > $MAXWPINROUTE } {
			    GMMessage [format $MESS(toomuchWPs) $MAXWPINROUTE]
			}
			set fd [FormData RT "IdNumber WPoints" \
				[list $name $wpns]]
			StoreRT -1 $name $fd $wpns $GFVisible
		    } else {
			# TR
			set tps ""
			foreach pt $pts {
			    foreach "lat long alt" $pt {}
			    set tpd [CreatePos $lat $long DMS latlong ""]
			    if { "$alt" != "" } {
				set fs [linsert $tpfs end alt]
				lappend $tpd $alt
			    } else { set fs $tpfs }
			    lappend tps [FormData TP $fs $tpd]
			}
			set fd [FormData TR "Name TPoints Datum" \
				[list $name $tps $SHPDatum]]
			StoreTR -1 $name $fd $GFVisible
		    }
		}
	    }
	}
    }
    GSHPCloseFiles $fsid
    ResetCursor .
    return 0
}

####
# MGM contribution
#### import/export from/to file in MapSend and Meridian formats

proc ReadMapSendHeader {file} {
    global ms_content ms_ver

    set hlen 0
    binary scan [read $file 1] "c" hlen

    set tmp [read $file $hlen]

    set ms_ver 1
    set teststr ""
    scan $tmp "%*s%*2s%d" ms_ver

    set ms_content 0
    set tmp [ReadBinData $file [list long]]
    set ms_content [lindex $tmp 0]

    return $ms_content
}

proc ReadString {file} {
    set slen 0
    binary scan [read $file 1] c slen
    set slen [expr $slen & 0xFF]
    set tmpstr [read $file $slen]
    return $tmpstr
}

proc WriteString {file str} {
    puts -nonewline $file [binary format c [string length $str]]
    puts -nonewline $file $str
    return
}

proc ReadMapSendTP {file} {
    global ms_ver
    set longd 0
    set latd 0
    set alt 0
    set tm 0
    set valid 0
    set cs 0
    # Endianess,word len issues here !!!
    set tmp [ReadBinData $file [list double double long long long]]
    set longd [lindex $tmp 0]
    set latd [lindex $tmp 1]
    set alt [lindex $tmp 2]
    set tm [lindex $tmp 3]
    set valid [lindex $tmp 4]

    # centiseconds
    if { $ms_ver >= 30 } {
	binary scan [read $file 1] "c" cs
    }

    set ns "latd longd latDMS longDMS alt date secs"
    set latd [expr $latd * -1]
    set p [CreatePos $latd $longd DMS latlong "WGS 84"]
    set dt [DateFromSecs [expr $tm - 567648000]]
    set tp ""

    if {$valid} {
	set tp [FormData TP $ns [concat $p $alt [list $dt] [expr $tm%86400]]]
    }

    return $tp
}

proc WriteMapSendTP {file tp} {
    global DataIndex TPlatDMS TPlongDMS TPdate

    set lat [expr [lindex $tp $DataIndex(TPlatd) ] * -1]
    set long [lindex $tp $DataIndex(TPlongd) ]
    set dt [lindex $tp $DataIndex(TPdate)]
    set alt [lindex $tp $DataIndex(TPalt)]
    set secs [lindex $tp $DataIndex(TPsecs)]
    
    WriteBinData $file [list double double long long long byte] \
	    [list $long $lat [expr int($alt)] [expr $secs + 567648000] 1 0]

    return
}

proc ReadMapSendWP {file} {
    global ms_ver LFileVisible SYMBOLS MAG_SYMTAB

    set wpnam [ReadString $file]
    set wpcom [ReadString $file]

    set num 0
    binary scan [read $file 4] i num

    set longd 0
    set latd 0
    set alt 0
    set icon 0
    set stat 0

    binary scan [read $file 1] c icon

    binary scan [read $file 1] c stat

    set tmp [ReadBinData $file [list double double double]]
    set alt [lindex $tmp 0]
    set longd [lindex $tmp 1]
    set latd [lindex $tmp 2]
    set latd [expr $latd * -1]
    set p [CreatePos $latd $longd DMS latlong "WGS 84"]

    set ns "Name Commt Posn Symbol Alt"
    set sym [lindex $MAG_SYMTAB $icon]

    set wp [FormData WP $ns [list $wpnam $wpcom $p $sym $alt]]
    set ix [IndexNamed WP $wpnam]
    StoreWP $ix $wpnam $wp $LFileVisible($file)
    return
}

proc WriteMapSendWP {file widx cnt} {
    global WPPosn WPAlt WPName WPCommt WPSymbol MAG_SYMTAB

    WriteString $file $WPName($widx)
    WriteString $file $WPCommt($widx)

    set latd [lindex $WPPosn($widx) 0]
    set longd [lindex $WPPosn($widx) 1]
    set snum [lsearch -exact $MAG_SYMTAB $WPSymbol($widx)]

    #altitude
    set wpa $WPAlt($widx)
    if {$wpa == "" } {set wpa 0}

    #write index,symbol,status,alt,pos
    WriteBinData $file [list long byte byte double double double]\
	[list $cnt $snum 2 $wpa $longd [expr $latd * -1]]
    return
}

proc ReadMapSendRT {file} {
    global ms_ver LFileVisible
    # Read the route header, blocks

    set rtnam [ReadString $file]

    set tmp [ReadBinData $file [list long long]]
    set num [lindex $tmp 0]
    set block_cnt [lindex $tmp 1]

    set i 0
    set wps ""
    while { $i < $block_cnt } {
	lappend wps [ReadMapSendRTBlock $file]
	# need to accomodate errors here !!!
	incr i
    }

    set stages ""
    set id 1
    set obs ""
    set ix [IndexNamed RT $id]
    set l [FormData RT "IdNumber Commt Obs WPoints Stages" \
	    [list $id "" $obs $wps $stages]]
    StoreRT $ix $id $l $wps $LFileVisible($file)
    return
}

proc ReadMapSendRTBlock {file} {
    # Read  a route block
    global ms_ver LFileVisible SYMBOLS

    set longd 0
    set latd 0
    set icon 0
    set stat 0

    set rtwpnam [ReadString $file]

    set tmp [ReadBinData $file [list long double double]]
    set num [lindex $tmp 0]
    set longd [lindex $tmp 1]
    set latd [expr [lindex $tmp 2] * -1]

    binary scan [read $file 1] c icon

    return $rtwpnam
}

proc ImportMeridian {file how} {
    #   Matt.Martin@ieee.org
    global LFileEOF LFileVisible LFileLNo LFileIxs MESS ms_ver PDTYPE

    # track points
    set tps ""

    set date [Now]
    set dt "" ; set ixs "" ; set ns ""
    set line [gets $file]
    set ftype [string range $line 5 7]

    switch $ftype {
	
	RTE -
	WPL {
	    set block_cnt 0
	    while { [string range $line 5 7] == "WPL" } {
		# Strip checksum and LF/CR
		#regsub {\*..\r?\n?$} [gets $file] "X" line
		regsub "\\*...\*\$" $line "" line
		set thedat [lrange [UnPackData [split $line ""] \
			[concat [list string] $PDTYPE(WPIN)]] 1 7]
		AddMagWPT [lindex [ConvWPData [list $thedat]] 0]
		incr block_cnt
		# get the next line
		set line [gets $file]
	    }

	    # get route info

	    set dat ""
	    while { [string range $line 5 7] == "RTE" } {
		# need to accomodate errors here !!!
		set thedat [lrange [UnPackData [split $line ""] \
			[concat [list string] $PDTYPE(RTIN)]] 1 8]
		lappend dat $thedat
		set line [gets $file]
	    }
	    InDataRT $dat
	}
	TRK {
	    set block_cnt 0
	    while { [string length $line]} {
		# process each line at a time
		set tmp [lrange [UnPackData [split [string range $line 0 \
			[expr [string length $line] -4]] ""] \
			[concat [list string] $PDTYPE(TRIN)]] 1 8]
		lappend tps [lindex [ConvTPData $tmp] 1]
		incr block_cnt
		set line [gets $file]
	    }

	    set tname [NewName TR]
	    set ix [IndexNamed TR $tname]
	    set data [FormData TR "Name Obs Datum TPoints" \
			  [list $tname $block_cnt "WGS 84" $tps]]
	    StoreTR $ix $tname $data $LFileVisible($file)
	}
    }
    return
}

proc ImportMapSend {file how} {
    #   Matt.Martin@ieee.org
    global LFileEOF LFileVisible LFileLNo LFileIxs MESS ms_ver

    # track points
    set tps ""

    set date [Now]
    set dt "" ; set ixs "" ; set ns ""
    fconfigure $file -translation {binary binary}
    set ftype [ReadMapSendHeader $file]

    switch $ftype {
	1 {
	    set wp_cnt 0
	    # waypoint count
	    set tmp [ReadBinData $file [list long]]
	    set block_cnt [lindex $tmp 0]

	    set i 0
	    set wps ""

	    while { $i < $block_cnt } {
		ReadMapSendWP $file
		# need to accomodate errors here !!!
		incr i
	    }

	    # get route count
	    set rt_cnt 0
	    set tmp [ReadBinData $file [list long]]
	    set rt_cnt [lindex $tmp 0]

	    set i 0
	    while { $i < $rt_cnt } {
		ReadMapSendRT $file
		# need to accomodate errors here !!!
		incr i
	    }
	}
	2 {
	    # getting track data
	    #  read track name
	    set tname [ReadString $file]

	    # get block count
	    set block_cnt 0
 	    set tmp [ReadBinData $file [list long]]
 	    set block_cnt [lindex $tmp 0]

	    set i 0
	    while { $i < $block_cnt } {
		lappend tps [ReadMapSendTP $file]
		# need to accomodate errors here !!!
		incr i
	    }

	    set ix [IndexNamed TR $tname]
	    set data [FormData TR "Name Obs Datum TPoints" \
			  [list $tname $block_cnt "WGS 84" $tps]]
	    StoreTR $ix $tname $data $LFileVisible($file)
	}
    }
    return
}

proc MapSendHeader {file type} {
    puts -nonewline $file [binary format c 0xd]
    puts -nonewline $file "4D533334 MS34"
    WriteBinData $file [list long] [list $type]
    return
}


proc ExportMeridianWP {f items} {

    set cnt 0
    foreach i $items {
	incr cnt
	set outdat [PrepMagWPData $i]
	puts -nonewline $f [join [MakeMagPacket WPL $outdat] ""]
    }
    return
}

proc ExportMeridianRT {f items} {
    global RTWPoints

    # First, Dump out all waypoints
    set wplist ""
    foreach i $items {
	set wps [Apply "$RTWPoints($i)" IndexNamed WP]
	foreach w $wps {
	    # only add to list if not there already
	    if { [lsearch -exact $wplist $w] == -1 } {
		lappend wplist $w
	    }
	}
    }
    ExportMeridianWP $f $wplist


    # Then the routes
    foreach i $items {
	set wps [Apply "$RTWPoints($i)" IndexNamed WP]
	set lncnt [expr int(([llength $wps] + 1 )/2)]
	set lnnum 1
	while { [llength $wps]} {
	    set outdat [PrepMagRTdata [lindex $wps 0] [lindex $wps 1] \
		    $lncnt $lnnum $i]
	    set wps [lreplace $wps 0 1]

	    puts -nonewline $f [join [MakeMagPacket RTE $outdat] ""]
	    incr lnnum
	}
    }
    return
}

proc ExportMeridianTR {f items} {
    global TRTPoints TRDatum

    set cnt 0
    foreach i $items {
	incr cnt
	foreach p $TRTPoints($i) {
	    # convert the datum
	    set p [concat [ConvertDatum [lindex $p 0] [lindex $p 1] \
			       $TRDatum($i) "WGS 84" DDD] [lrange $p 4 end]]
	    # setup the packet
	    set outdat [PrepMagTRData $p]
	    # add the checksum and write
	    puts -nonewline $f [join [MakeMagPacket TRK $outdat] ""]
	}
    }
    return
}

proc DumpMapSendWP {f items} {
    WriteBinData $f [list long] [list [llength $items]]
    set cnt 0
    foreach i $items {
	incr cnt
	WriteMapSendWP $f $i $cnt
    }
}

proc ExportMapSendWP {f items} {
    MapSendHeader $f 1
    DumpMapSendWP $f $items
    WriteBinData $f [list long] [list 0]
    return
}

proc ExportMapSendRT {f items} {
    global RTWPoints RTData RTIdNumber WPName WPPosn WPSymbol MAG_SYMTAB

    # MF contribution
    set badrts ""
    ##

    # First, Dump out all waypoints
    set wplist ""
    foreach i $items {
	# MF contribution
	if { ! [CheckNumber Ignore $RTIdNumber($i)] } {
	    lappend badrts $i
	    continue
	}
	##
	set wps [Apply "$RTWPoints($i)" IndexNamed WP]
	foreach w $wps {
	    # only add to list if not there already
	    if { [lsearch -exact $wplist $w] == -1 } {
		lappend wplist $w
	    }
	}
    }
    # MF contribution
    if { [set n [llength $badrts]] > 0 } {
	global MESS
	GMMessage [format $MESS(cantsaveRTid) $n]
    }
    ##

    MapSendHeader $f 1
    DumpMapSendWP $f $wplist

    #############################################################
    # Then the routes
    #############################################################

    # number of routes
    WriteBinData $f [list long] [list [llength $items]]

    foreach i $items {
	# MF contribution
	if { $i == [lindex $badrts 0] } {
	    set badrts [lreplace $badrts 0 0]
	    continue
	}
	##
	set wps [Apply "$RTWPoints($i)" IndexNamed WP]
	set lncnt [expr int(([llength $wps] + 1 )/2)]
	# route name
	WriteString $f $RTIdNumber($i)
	#route num, block cnt
	WriteBinData $f [list long long] [list $RTIdNumber($i) [llength $wps]]

	set wpnum 1
	foreach w $wps {
	    # name
	    WriteString $f $WPName($w)
	    # wp index
	    # long
	    set longd [lindex $WPPosn($w) 1]
	    # lat
	    set latd [lindex $WPPosn($w) 0]
	    # sym
	    set snum [lsearch -exact $MAG_SYMTAB $WPSymbol($w)]
	    incr wpnum
	    WriteBinData $f [list long double double byte] \
		    [list [expr [lsearch $wplist $w]+1] $longd \
		          [expr $latd * -1] $snum]
	}
    }
    return
}

proc ExportMapSendTR {of ixs} {
    global TRName TRTPoints

    MapSendHeader $of 2
    set i [lindex $ixs 0]
    WriteString $of $TRName($i)
    WriteBinData $of [list long] [list [llength $TRTPoints($i)]]
    
    foreach tp $TRTPoints($i) {
	WriteMapSendTP $of $tp
    }
    return
}
