package profxp::screen;

use profxp::config;

BEGIN {
  use Exporter;
  use vars qw($VERSION @ISA @EXPORT);

  $VERSION     = 0.1;
  @ISA         = qw(Exporter);
  @EXPORT      = qw(
                      %cmd
                      &num2char
                      &char2num
                      $lcount
                      $loglevel
                      &do_command
                      &printlist
                      &getkey
                      &parselist
                      &prompt
                      &welcome
                      &psend
                      &precv
                      &error
                      &warning
                      &info
                      &misc
                   );
};

$lcount = 9999;

# 0 - absolutely nothing
# 1 - GUI msgs only
# 2 - normal
# 3 - verbose

$loglevel = 2;

# ----------------------------------------------------------- #
# Command Hash

%cmd = ( 'on' => {
                   desc => 'open new site',
                   help => 'syntax: on <ip/hostname> [port] [user] [pass]',
                   minp => 1,
                   maxp => 4,
                   site => 0
                 },
         'xi' => {
                   desc => 'transfer immediate recursive',
                   help => 'syntax: xi <file/dir> (Wildcards: */?)',
                   minp => 1,
                   maxp => 1,
                   site => 2
                 },        
         'sd' => {
                   desc => 'delete site bookmark',
                   help => 'syntax: sd <sitename>',
                   minp => 1,
                   maxp => 1
                 },
         'cl' => {
                   desc => 'close site',
                   help => 'syntax: cl',
                   minp => 0,
                   maxp => 0,
                   site => 1
                 },
         'rm' => {
                   desc => 'recursive file/dir deletion',
                   help => 'syntax: rm <pattern> (Wildcards: */?)',
                   minp => 1,
                   maxp => 1,
                   site => 1
                 },
         'vf' => {
                   desc => 'view file in viewer program',
                   help => 'syntax: vf <filename>',
                   minp => 1,
                   maxp => 1,
                   site => 1
                 },
         'op' => {
                   desc => 'open site',
                   help => 'syntax: on <sitename>',
                   minp => 1,
                   maxp => 2,
                   site => 0
                 },
         'ss' => {
                   desc => 'save site (bookmark)',
                   help => 'syntax: ss <sitename> [slot]',
                   minp => 1,
                   maxp => 2,
                   site => 1
                 },
         'sl' => {
                   desc => 'site list',
                   help => 'syntax: sl',
                   minp => 0,
                   maxp => 0
                 },
         'cd' => {
                   desc => 'change directory (doh!)',
                   help => 'syntax: cd <path>',
                   minp => 1,
                   maxp => 1,
                   site => 1
                 },
         'ls' => {
                   desc => 'list directory (sorted literally)',
                   help => 'syntax: ls',
                   minp => 0,
                   maxp => 0,
                   site => 1
                 },
         'ld' => {
                   desc => 'list directory (sorted by date)',
                   help => 'syntax: ld',
                   minp => 0,
                   maxp => 0,
                   site => 1
                 },
         'lr' => {
                   desc => 'list RAW directory in viewer',
                   help => 'syntax: lr',
                   minp => 0,
                   maxp => 1,
                   site => 1
                 },
         'lh' => {
                   desc => 'set local interface',
                   help => 'syntax: lh <ip/hostname>',
                   minp => 1,
                   maxp => 1
                 },
         'cf' => {
                   desc => 'set configuration option',
                   help => 'syntax: cf <opt1> [opt2] [opt3] ... <value> (call with no parameters to view options)',
                   minp => 0,
                   maxp => 3
                 },
         'ex' => {
                   desc => 'exit profxp',
                   help => 'exits profxp, tries to disconnect sites',
                   minp => 0,
                   maxp => 0
                 },
         'qa' => {
                   desc => 'add entry to queue',
                   help => 'syntax: qa <entry>',
                   minp => 1,
                   maxp => 1,
                   site => 2
                 },
         'he' => {
                   desc => 'get some help',
                   help => 'doh!',
                   minp => 0,
                   maxp => 1
                 }
       );


# ----------------------------------------------------------
# include the kewl readline support
use Term::ReadKey;
BEGIN{ $ENV{PERL_RL} = 'Perl' };
use Term::ReadLine;
use Term::ReadLine::Perl;
use Term::ReadLine::readline;

# ----------------------------------------------------------
# the command line handler
my $t = new Term::ReadLine 'pfxp';
if (!($t->ReadLine() =~ /Perl/)) {
  error("Please install Term::Readline::Perl",0);
  exit(0);
};
my $tattribs = $t->Attribs;

# ----------------------------------------------------------
# set up TAB completion function

$tattribs->{completion_function} = sub {

  my @tmplist = ();
  my $cand = '';
  my $regex = $_[0];
  $regex =~ s/(\W)/sprintf("\\x%02X",ord($1))/eg;
  
  # trim whitespace from cmdline frag
  $_[1] =~ s/^ +//g;
  #$_[1] =~ s/ +$//g;

  # --------------------------------------------------------
  # config flag completion
  if ($_[1] =~ /^cf/) {
    
    # ------------------------------------------------------
    # 2nd stage completion
    if ($_[1] =~ /.+ +(.+) +/) {
      $flag = $_[1];
      $flag =~ s/cf +(.+) .*/$1/;

      if ($cfg{$flag} =~ /ARRAY/) {
        foreach $cand (@{$cfg{$flag}}) {
          next if (!($cand =~ /^$regex/));
          push @tmplist, $cand;
        };
      }
      elsif ($cfg{$flag} =~ /HASH/) {
        foreach $cand (keys(%{$cfg{$flag}})) {
          next if (!($cand =~ /^$regex/));
          push @tmplist, $cand;
        };
      };
      return @tmplist;
    };
    # ------------------------------------------------------
    # first stage completion
    foreach $cand (keys(%cfg)) {
      next if (!($cand =~ /^$regex/));
      push @tmplist, $cand;
    };
    return @tmplist;
  };
  
  # --------------------------------------------------------
  # sitename completion for the "op", "ss" and "sd" commands
  if ( ($_[1] =~ /^op/) ||
       ($_[1] =~ /^ss/) ||
       ($_[1] =~ /^sd/) ) {
    opendir(CFGDIR,".");
    @sitefiles = grep /\.site$/, readdir CFGDIR;
    closedir(CFGDIR);
    foreach (@sitefiles) {
      $_ =~ s/\.site$//;
      next if (!($_ =~ /^$regex/));
      push @tmplist, ($_);      
    };
    return @tmplist;
  };

  # --------------------------------------------------------
  # file/dir completion
  if ($conn{$active}{sock}) {
    my $oldverbose = $cfg{verbose};
    $cfg{verbose} = 0;
    my $listref = parselist(profxp::command::getlist($conn{$active}{sock}));
    $cfg{verbose} = $oldverbose;
    foreach $cand (keys(%{ $listref })) {
      next if (!($cand =~ /^$regex/i));
      push @tmplist, $cand;
    };
    return @tmplist;
  };
  return(());
};

sub getkey {
  ReadMode 5;
  my $keyclick = ReadKey(-1);
  ReadMode 0;
  return $keyclick;
};

sub waitkey {
  my $key;
  while(!($key = &getkey)) {
    select undef,undef,undef,0.1;
  };
  return $key;
};

sub prompt {
  check_connection($active);
  check_connection($active^1);
  PROMPT:
  if ($lcount > 25) {  
    plain("[".$c{$active^1}.$conn{$active^1}{name}.$c{reset}."|".$c{$active^1}.$conn{$active^1}{last}.$c{reset}."]");
    plain("[".$c{$active}.$conn{$active}{name}.$c{reset}."|".$c{$active}.$conn{$active}{last}.$c{reset}."]");
    $lcount = 0;
  };
  $SIG{ALRM} = \&profxp::command::idle;
  alarm $cfg{noopdelay};
  my $line = $t->readline('[> ');
  alarm 0;
  if ( (check_connection($active)) ||
       (check_connection($active^1)) ) {
    error("aborted",1);
    goto PROMPT;
  };
  return($line);
};

sub check_connection {
  if ( (!($conn{$_[0]}{name} eq "disconnected")) &&
       (!(exists($conn{$_[0]}{sock}))) ) {
    error("connection to $conn{$_[0]}{name} lost",1);
    disconnect($_[0]);
    $lcount = 9999;
    return 1;
  };
  return 0;
};


sub do_command
{
  PARSELINE:
  $_[0] =~ s/^ +//;
  $_[0] =~ s/ +$//;

  # toggle active/inactive on ENTER
  if (!$_[0]) {
    $active = $active^1;
    $lcount = 9999;
    return 1;
  };
  
  # catch RAW commands
  if ($_[0] =~ /^[A-Z]/) {
    if ($conn{$active}{sock}) {
      profxp::command::cmd_raw($_[0]);
    }
    else {
      error("cannot send RAW command, site not open",1);
    };
    return(0);
  };

  my $command = undef;
  if (!($_[0] =~ / /)) {
    $_[0] =~ s/^(.+)$//;
    $command = $1;
  }
  else {
    $_[0] =~ s/^(.+?) //;
    $command = $1;
  };
  $_[0] =~ s/^ +//;

  if (!defined($cmd{$command})) {
    # alias ?
    if (defined($cfg{aliases}{$command})) {
      # glue command to line and restart
      $_[0] = "$cfg{aliases}{$command} ".$_[0];
      goto PARSELINE;
    };
    error("no such command or alias '$command'",1);
    return(0);
  };

  # check if prereq. are fullfilled
  if (exists($cmd{$command}{site})) {
    
    if ( ($cmd{$command}{site} == 0) &&
         (exists($conn{$active}{sock})) ) {
      error("disconnect active site or switch sites first",1);
      return 0;
    };
    
    if ( ($cmd{$command}{site} == 1) &&
         (!(exists($conn{$active}{sock}))) ) {
      error("please connect to a site first",1);
      return 0;
    };
    
    if ( ($cmd{$command}{site} == 2) &&
         (
           !(
              (exists($conn{$active}{sock})) && 
              (exists($conn{$active^1}{sock}))
            )
         )
       ) {
      error("both sites must be connected for this command",1);
      return 0;
    };
    
  };

  # chop minmum parameters
  my @p = split / +/, $_[0], $cmd{$command}{minp};
  # check if we have the min number ...
  if ((($#p)+1) < ($cmd{$command}{minp})) {
    error("not enough parameters for '$command'",1);
    info("$cmd{$command}{help}",1);
    return undef;
  };

  # chop parameters
  @p = split / +/, $_[0], $cmd{$command}{maxp};

  # replace <CODE> with unprintable chars in last parameter (paths/files)
  if ($#p > -1) {
    $p[$#p] = num2char($p[$#p]);
  };
  
  # call handler function
  # fucking unclean but I love such elegant stuff
  # perl rulz
  no strict "refs";
  &{"profxp::command::cmd_$command"}(@p);
};


sub num2char {
  my $str = $_[0];
  my $asciicode = 0;
  my $asciisign = '';
  for ($asciicode = 0; $asciicode < 32; $asciicode++) {
    $asciisign = chr($asciicode);
    $str =~ s/<$asciicode>/$asciisign/g;
  };
  for ($asciicode = 127; $asciicode < 256; $asciicode++) {
    $asciisign = chr($asciicode);
    $str =~ s/<$asciicode>/$asciisign/g;
  };
  return($str);
};


sub char2num {
  my $str = $_[0];
  my $asciicode = 0;
  my $asciisign = '';
  for ($asciicode = 0; $asciicode < 32; $asciicode++) {
    $asciisign = chr($asciicode);
    $str =~ s/$asciisign/<$asciicode>/g;
  };
  for ($asciicode = 127; $asciicode < 256; $asciicode++) {
    $asciisign = chr($asciicode);
    $str =~ s/$asciisign/<$asciicode>/g;
  };
  return($str);
};

sub parselist {
  my $slist = shift;
  my @list = split /\r\n/, $slist;
  my %plist = ();
  foreach $line (@list) {
    next if (!($line =~ /^[b||s||c||p||l||d||\-]/));
		PARSE:
    my ($perms,$number,$uid,$gid,$size,$month,$day,$timename) = split / +/,$line, 8;
		# split time and name at ONE space
		my ($time,$name) = split / /, $timename, 2;
    # fix broken lines when no UID/GID/Number present
    if ($screweddir) {
      ($perms,$number,$uid,$size,$month,$day,$time,$name) = split / +/,$line, 8;
      $gid = "????";
      $uid = "????";
      $number = "?";
    };
    # replace chars < 32 or > 126 with [CODE]
    $name = char2num($name);
    # kick out "." and "..", if present
    next if (($name eq ".") || ($name eq ".."));
    # sanitize month	
    for ($month) {
      /jan/i	and do { $month = "01"; last; };
      /feb/i	and do { $month = "02"; last; };
      /mar/i	and do { $month = "03"; last; };
      /apr/i	and do { $month = "04"; last; };
      /may/i	and do { $month = "05"; last; };
      /jun/i	and do { $month = "06"; last; };
      /jul/i	and do { $month = "07"; last; };
      /aug/i	and do { $month = "08"; last; };
      /sep/i	and do { $month = "09"; last; };
      /oct/i	and do { $month = "10"; last; };
      /nov/i	and do { $month = "11"; last; };
      /dec/i	and do { $month = "12"; last; };
      # at this point, something stinks ..
      # one field may be missing. set the
      # screwed dir flag and start again.
      if (!$screweddir) {
        $screweddir = 1;
        goto PARSE;
      };
    };
    # sanitize day
    $day = sprintf("%02.0ld", $day);
    # sanitize time/year
    my $year;
    if (!($time =~ /\:/)) {
      $year = $time;
      $time = "00:00";
    }
    else {
      my ($tsec,$tmin,$thour,$tmday,$tmon,$tyear,$twday,$tyday,$tisdst) = localtime(time);
      $year = $tyear + 1900;
    };
    if (length($time) == 4) {
      $time = "0".$time;
    };
    # build complete timestamp
    $plist{$name}{name} = $name;
    $plist{$name}{date} = "$year-$month-$day $time";
    
    $plist{$name}{size} = int($size);
    
    # set type
    $perms =~ s/^(.)//;
    $plist{$name}{type} = $1;
    
    # perms/uid/gid
    $plist{$name}{perm} = $perms;
    $plist{$name}{uid} = substr($uid,0,5);
    $plist{$name}{gid} = substr($gid,0,5);
  };
  return \%plist;
};

sub printlist {
  my $pref = shift;
  my $sortmode = shift || 0;
  my %plist = %{ $pref };
  my @sortlist;
  if ($sortmode == 1) {
    # literal, dirs first
    @sortlist = sort {
                       if ( ($plist{$a}{type} eq 'd') && ($plist{$b}{type} eq 'd') ) {
                        return(lc($a) cmp lc($b));
                       };
                       return -1 if ($plist{$a}{type} eq 'd');
                       return 1 if ($plist{$b}{type} eq 'd');
                       return(lc($a) cmp lc($b));
                     }
                     keys(%plist);
  }
  elsif ($sortmode == 2) {
    # date, newest last
    @sortlist = sort {
                       return($plist{$b}{date} cmp $plist{$a}{date});
                     }
                     keys(%plist);
  }
  else {
    # unsorted
    @sortlist = keys(%plist);
  };
  
  info("-/----------------------------------------------------------------->>",1);
  my $linecount = 0;
  my $totalcount = 0;
  foreach $line (@sortlist) {
    my $fcolor = '';
    next if ( (!$cfg{listing}{showdot}) && ($line =~ /^\./) );
    if ($plist{$line}{type} eq '-') {
      my $psize;
      if ($plist{$line}{size} > 0) {
        $psize = int($plist{$line}{size}/1024) + 1;
      }
      else {
        $psize = 0;
        $fcolor = $c{red};
      };
      info("$plist{$line}{perm} $plist{$line}{uid}\t$plist{$line}{gid}\t$plist{$line}{date} ".$psize."k\t".$fcolor."$line".$c{reset},1);
    }
    elsif ($plist{$line}{type} eq 'l') {
      info("$plist{$line}{perm} $plist{$line}{uid}\t$plist{$line}{gid}\t$plist{$line}{date} <LNK>\t$c{linkcolor}$line$c{reset}",1);
    }
    elsif ($plist{$line}{type} eq 'd') {
      info("$plist{$line}{perm} $plist{$line}{uid}\t$plist{$line}{gid}\t$plist{$line}{date} <DIR>\t$c{dircolor}$line$c{reset}",1);
    }
    else {
      info("$plist{$line}{perm} $plist{$line}{uid}\t$plist{$line}{gid}\t$plist{$line}{date} <SPC>\t$c{specialcolor}$line$c{reset}",1);
    };
    $totalcount++;
    $linecount++;
    if ($linecount == $cfg{listing}{pauselines}) {
      $linecount = 0;
      misc("$c{boldwhite}press SPACE for next page, 'a' to abort $c{reset}",1);
      my $key = &waitkey;
      print "\n";
      goto LASTLINE if ($key =~ /a/i);
    };
  };
  if (!$totalcount) {
    info("$c{yellow}empty directory (or all items filtered)$c{reset}",1);
  };
  LASTLINE:
  info("-/----------------------------------------------------------------->>",1);
};


sub welcome {
  info('',1);
  info("-/-------------------------------------------------------------/-",1);
  info(" / $c{red}::$c{reset} $c{boldwhite}profxp $main::version$c{reset} - the final rewrite $c{red}::$c{reset}                       / ",1);
  info(" /                                                             / ",1);
  info(" /      (c) duncanthrax aka Tom Kistner <tom\@duncanthrax.net>  / ",1);
  info("-/-------------------------------------------------------------/-",1);                       
  info('',1);
};

sub psend {
  my $level = $_[2] || 0;
  if ($level <= $cfg{verbose}) {
    my $color = $_[1] || $c{yellow};
    print $color.">>".$c{reset}." $_[0]\n";
    $lcount++;
  };
};

sub precv {
  my $level = $_[2] || 0;
  if ($level <= $cfg{verbose}) {
    my $color = $_[1] || $c{yellow};
    print $color."<<".$c{reset}." $_[0]\n";
    $lcount++;
  };
};

sub plain {
  print "$_[0]\n";
  $lcount++;
};

sub warning {
  my $level = $_[1] || 0;
  if ($level <= $cfg{verbose}) {
    print "$c{yellow}\?\?$c{reset} $_[0]\n";
    $lcount++;
  };
};

sub info {
  my $level = $_[1] || 0;
  if ($level <= $cfg{verbose}) {
    print "$c{green}\:\:$c{reset} $_[0]\n";
    $lcount++;
  };
};

sub error {
  my $level = $_[1] || 0;
  if ($level <= $cfg{verbose}) {
    print "$c{red}\!\!$c{reset} $_[0]\n";
    $lcount++;
  };
};

sub misc {
  print "$c{yellow}->$c{reset} $_[0]";
  $lcount++;
};

1;
