package profxp::tcp;

use profxp::screen;
use profxp::config;

use IO::Socket;
use Net::SOCKS;
use Fcntl;

BEGIN {
  use Exporter;
  use vars qw($VERSION @ISA @EXPORT);

  $VERSION     = 0.1;
  @ISA         = qw(Exporter);
  @EXPORT      = qw(
                      &block
                      &unblock
                      &tcp_open
                      &tcp_close
                      &tcp_listen
                      &tcp_accept
                      &tcp_peek
                      &portstring
                      &abort
                      &abort_finish
                      &tcp_read
                      &resolve
                      &command
                   );
};

my $ctrlc = 0;
my $sigpipe = 0;

sub block {
  fcntl $_[0], F_SETFL, 0;
};

sub unblock {
  fcntl $_[0], F_SETFL, O_NONBLOCK;
};

sub ctrl_c {
  if ($_[0] =~ /INT/) {
    $ctrlc = 1;
    $SIG{'INT'} = IGNORE;
  }
  elsif ($_[0] == 0) {
    $SIG{'INT'} = IGNORE;
  }
  else {
    $ctrlc = 0;
    $SIG{'INT'} = \&ctrl_c;
  };
};

sub sig_pipe {
  if ($_[0] =~ /PIPE/) {
    $sigpipe = 1;
    $SIG{'PIPE'} = IGNORE;
  }
  elsif ($_[0] == 0) {
    $SIG{'PIPE'} = IGNORE;
  }
  else {
    $sigpipe = 0;
    $SIG{'PIPE'} = \&sig_pipe;
  };
};


sub tcp_close
# $_[0] socket
# RET: always 1
{
  eval {
    $_[0]->close();
  };
  return(1);
};


sub tcp_open
# $_[0] target host
# $_[1] target port
# $_[2] local host [0.0.0.0]
# $_[3] local port [0]
# $_[4] timeout [30]
# $_[5] SOCKS version [0]
# $_[6] SOCKS server
# $_[7] SOCKS port
# $_[8] SOCKS user
# $_[9] SOCKS password
# RET: open socket, 0 on failure
{
  my $targethost = $_[0];
  my $targetport = $_[1];
  my $localhost = $_[2] || '0.0.0.0';
  my $localport = $_[3] || 0;
  my $timeout = $_[4] || $cfg{timeout}{connect};
  my $SOCKSversion = $_[5] || 0;
  my $SOCKSserver = $_[6] || '0.0.0.0';
  my $SOCKSport = $_[7] || 1080;
  my $SOCKSuser = $_[8] || '';
  my $SOCKSpass = $_[9] || '';
  
  my $sock;
  
  eval {
  local $SIG{ALRM} = sub { die "alarm\n" };
  alarm $timeout;

  if ($SOCKSversion == 4) {
    # ----------------------------------------------------------- #
    # SOCKS4 method
    my $socks = new Net::SOCKS(socks_addr         => $SOCKSserver,   
                               socks_port         => $SOCKSport,      
                               protocol_version   => 4);

    $sock = $socks->connect(peer_addr =>$targethost, peer_port => $targetport);
    alarm 0;
    return (0) if (!defined($sock));
    
  }
  elsif (SOCKSversion == 5) {
    # ----------------------------------------------------------- #
    # SOCKS5 method
    my $socks = new Net::SOCKS(socks_addr         => $SOCKSserver,
                               socks_port         => $SOCKSport,
                               user_id            => $SOCKSuser,
                               user_password      => $SOCKSpass,
                               force_nonanonymous => 0,
                               protocol_version   => 5);
    $sock = $socks->connect(peer_addr =>$targethost, peer_port => $targetport);
    alarm 0;
    return 0 if (!defined($sock));
  }
  else {
    if ( ($localhost eq '0.0.0.0') ||
         ($localport == 0) ) {
    
      # ----------------------------------------------------------- #
      # direct connection  method w/o local host/port
      $sock = IO::Socket::INET->new(PeerAddr => $targethost,
                                      PeerPort => $targetport,
                                      Timeout  => $timeout,
                                      Proto    => 'tcp');
      alarm 0;                                      
      return 0 if (!defined($sock));
    }
    else {
      # ----------------------------------------------------------- #
      # direct connection  method w/ local host/port
      $sock = IO::Socket::INET->new(PeerAddr => $targethost,
                                      PeerPort => $targetport,
                                      LocalAddr=> $localhost,
                                      LocalPort=> $localport,
                                      Timeout  => $timeout,
                                      Proto    => 'tcp');
      alarm 0;
      return 0 if (!defined($sock));
    };
  };
  };
  if ($@ eq "alarm\n") {
    return 0;
  };
  return($sock);
};


sub tcp_listen
# $_[0] localhost [0.0.0.0]
# $_[1] localport [0]
# RET: listen socket, 0 on failure
{
  if (!defined($tcp_listen_loc_port)) {
    $tcp_listen_loc_port = 30000;
  }
  else {
    $tcp_listen_loc_port++;
  };
  
  my $localport = $_[1] || $tcp_listen_loc_port;
  my $localhost = $_[0] || '0.0.0.0';
  my $listensocket;
  $listensocket = IO::Socket::INET->new(Listen    => 1,
                                        Reuse     => 1,
                                        LocalAddr => $localhost,
                                        LocalPort => $localport,
                                        Timeout   => 5,
                                        Proto     => 'tcp');
  return(0) if (!defined($listensocket));
  return($listensocket);
};


sub tcp_accept
# $_[0] listen socket
# RET: connected socket
{
  my $sock;
  my $listensocket = $_[0];
  return 0 if (!defined($sock = $listensocket->accept()));
  return 0 if (!defined($sock->connected()));
  return($sock);
};

sub tcp_peek {
  if ($_[0]->recv($_,1,MSG_PEEK)) {
    return 1;
  };
  return 0;
};



sub tcp_read
# reads data from socket until it closes
# good for FTP data connections
# $_[0] socket
# $_[1] breakchar ('\n' or 0)
# $_[2] timeout for 'no data' [30]
# $_[3] verbose
# RET: all data as scalar, undef on error
{
  my $sock = $_[0];
  my $breakchar = $_[1] || 0;
  my $timeout = $_[2] || $cfg{timeout}{data};
  my $verbose = $_[3] || 0;
  my $data = '';
  my $counter = 0;
  my $char = "";
  my $alert = 0;
  
  ctrl_c(1);
  sig_pipe(1);

  if ($verbose) {
    my $kbytes = sprintf("%6d",int($counter/1024));
    print ">> Received $kbytes kB <<";
  };

  eval {
    local $SIG{ALRM} = sub { $alert == 1 };
    alarm $timeout;
    
    BYTE: while(1) {
    # set up timeout
      do {
        $char = "";
        recv $sock, $char, 1, 0; 
        last BYTE if ( ($alert) ||
                       ($ctrlc) || 
                       ($sigpipe) || 
                       (ord($char) == $breakchar)
                     );
      }
      while ($char eq "");
      alarm $timeout;
      $data .= $char;
    
      if ($verbose) {
        $counter++;
        if ( ($counter % 1024) == 0) {
          my $oldstate = $|;
          $| = 1;
          print chr(8) x 24;
          my $kbytes = sprintf("%6d",int($counter/1024));
          print ">> Received $kbytes kB <<";
          $| = $oldstate;
        };
      };
    };
  };

  alarm 0;
  
  if ($verbose) {
    my $oldstate = $|;
    $| = 1;
    print chr(8) x 24;
    print ' ' x 24;
    print chr(8) x 24;
    $| = $oldstate;
  };

  ctrl_c(0);
  sig_pipe(0);
  
  if ( ($ctrlc) ||
       ($sigpipe) ||
       ($alert)
     ) {
    return(undef);
  }
  else {
    return($data);
  };
  
};


sub portstring
# $_[0] open listen socket
# RET: PORT string suitable for PORT or PASV purposes
{
  my $myaddr = $_[0]->sockhost();
  my $myport = $_[0]->sockport();
  
  my $highport = int $myport/256;
  my $lowport = $myport - ((int $myport/256)*256);
  my $portstr = $myaddr;
  $portstr =~ s/\./\,/g;
  $portstr .= ",".$highport.",".$lowport;
  return($portstr);
};


sub abort
{
  my $f1 = $_[0];
  my $f2 = $_[1] || undef;
  my $IP = chr(255).chr(244);
  my $SYNCH = chr(255).chr(242);
  # send "Interrupt process"
  print $f1 $IP;
  # send "synch" with TCP Urgent
  $f1->send($SYNCH,MSG_OOB);
  # send ABOR\r\n
  print $f1 "ABOR".chr(13).chr(10);
  if (defined($f2)) {
    # send "Interrupt process"
    print $f2 $IP;
    # send "synch" with TCP Urgent
    $f2->send($SYNCH,MSG_OOB);
    # send ABOR\r\n
    print $f2 "ABOR".chr(13).chr(10);
  };
};

sub abort_finish {
  my $f1 = $_[0];
  my $f2 = $_[1];
  info("$c{yellow}>>>$c{reset} Waiting for ABORT to finish $c{yellow}<<<$c{reset}",1);
  command($f1,undef,"...",2,20);
  command($f1,undef,"2..",2,2);
  if (defined($f2)) {
    command($f2,undef,"...",2,20);
  };
};


sub resolve {
  my $lname = inet_aton("$_[0]");
  return undef if (!$lname);
  my ($a,$b,$c,$d) = unpack('C4', $lname);
  my $dotaddr = $a.".".$b.".".$c.".".$d;
  return $dotaddr;
};


sub command {
  
  $sigpipe = 0;
  $SIG{PIPE} = \&dead_socket;
 
  my $f = $_[0];
  my $timeout = $_[4] || $cfg{timeout}{command};
  my $level = $_[3] || 4;
  
  
  # set up timeout
  eval {
  local $SIG{ALRM} = sub { die "alarm\n" };
  alarm $timeout;
  
  if (defined($_[1])) {
    # clear recieve buffer
    fcntl $f, F_SETFL, O_NONBLOCK;
    my $l;
    while(($f->recv($_,1,MSG_PEEK)) && (!$sigpipe) ) {
      $l = <$f>;
      last if (!$l);
      # remove CRLF
      chop $l;chop $l;
      warning($l,$level);
    };
    fcntl $f, F_SETFL, 0;
    # send command
    print $f "$_[1]".chr(13).chr(10);
    psend("$_[1]",$c{$f},$level);
  };
  
  if ($sigpipe) {
    disrupt($f);
    return 0;
  };
  
  # wait for final response
  while( ($_ = <$f>) && (!$sigpipe) ) {
    # remove CRLF
    chop;chop; 
    precv("$_",$c{$f},$level);
    $lcount++;
    last if ($_ =~ /^[1-9][0-9][0-9] /);
  };

  if ($sigpipe) {
    disrupt($f);
    return 0;
  };

  alarm 0;
  
  $SIG{PIPE} = 'IGNORE';
  };
  
  if ($@ eq "alarm\n") {
    return 0;
  };
  
  if ($_ =~ /^$_[2]/) {
    # return last line of output
    return($_);
  }
  else {
    return 0;
  };
};

sub dead_socket {
  alarm 0;
  $SIG{PIPE} = 'IGNORE';
  $sigpipe = 1;
};

sub disrupt {
  if ($conn{$active}{sock} eq $_[0]) {
    delete $conn{$active}{sock};
  };
  if ($conn{$active^1}{sock} eq $_[0]) {
    delete $conn{$active^1}{sock};
  };
  tcp_close($_[0]);
};

1;
