#ifndef AVIFILE_AVM_MAP_H
#define AVIFILE_AVM_MAP_H

#ifndef AVM_BEGIN_NAMESPACE
#define AVM_BEGIN_NAMESPACE namespace avm {
#define AVM_END_NAMESPACE   };
#endif

#include <assert.h>

AVM_BEGIN_NAMESPACE

// do not use this container in time critical context
// search time is between log(N) and N, depending on order of element insertion
template <class Key, class Value> class avm_map
{
protected:
    template <class Key1, class Value1> struct pair
    {
	Key1 key;
	Value1 value;
	pair() {}
	pair(Key1 k, Value1 v) : key(k), value(v) {}
	pair(const pair<Key1, Value1>& p) : key(p.key), value(p.value) {}
    };
    typedef pair<Key, Value> _Tpair;

    template <class Key1, class Value1> struct binary_tree_node
    {
	avm_map::pair<Key1, Value1>* entry;
	binary_tree_node<Key1, Value1>* left;
	Key1 minval;
	binary_tree_node<Key1, Value1>* right;
	binary_tree_node<Key1, Value1>* parent;
	Key1 maxval;
	binary_tree_node(binary_tree_node<Key1, Value1>* ptr=0) : entry(0), left(0), right(0), parent(ptr), weight(1) {}
	int weight;
	inline void destroy()
	{
	    if(left)
	    {
		left->destroy();
		delete left;
	    }
	    if(right)
	    {
		right->destroy();
		delete right;
	    }
	    delete entry;
	}
    };
    typedef binary_tree_node<Key, Value> _Tnode;

    _Tnode* m_pTree;
    Value* m_pDefaultValue;
    // this funny trick makes old compiler work with this pointer
    static void update_min_max_weights(void* node);
    _Tnode* find_private(const Key& key);
#if 1
    class const_iterator
    {
	_Tnode* p;
    public:
	const_iterator(_Tnode* p1=0) : p(p1) {}
	const_iterator(const const_iterator& c) : p(c.p) {}
	bool operator==(const const_iterator& c) const { return p==c.p; }
	operator const _Tpair*() { return p ? p->entry : 0 ; }
	_Tpair* operator->() { return p ? p->entry : 0 ; }
	const_iterator operator++(int)
	{
	    if(p==0)
		return *this;
	    _Tnode* p1 = p;
	    while(p1)
	    {
		if(p1->parent && (p1==p1->parent->left) && p1->parent->right)
		{
		    p1=p1->parent->right;
		    break;
		}
		if(!p1->parent)
		{
		    p=0;
		    return *this;
		}
		p1=p1->parent;
	    }
	    while(p1->left || p1->right)
		if(p1->left)
		    p1=p1->left;
		else
		    p1=p1->right;
	    p=p1;
	    return *this;
	}
    };
#endif
public:
    avm_map();
    ~avm_map() { m_pTree->destroy(); delete m_pTree; }
#if 1
    const_iterator begin()
    {
	_Tnode* p1=m_pTree;
	    while(p1->left || p1->right)
		if(p1->left)
		    p1=p1->left;
		else
		    p1=p1->right;
	return const_iterator(p1);
    }

    const_iterator end()
    {
	return 0;
    }
#endif
    // inserts a new entry into the map
    Value* insert(const Key& key, Value value);

    // searches for the entry that corresponds to the key. If the search fails, returns 0.
    Value* find(const Key& key)
    {
	_Tnode* tree_node=find_private(key);
	if(!tree_node)
	    return 0;
	else
	{
	    assert(tree_node->entry);
	    return &(tree_node->entry->value);
	}
    }

    // searches for the entry that corresponds to the key. If the search fails, inserts a new entry into the map
    // and returns it ( default STL map behavior ).
    Value* find_insert(const Key& key)
    {
	_Tnode* tree_node=find_private(key);
	if(!tree_node)
	    return insert(key, Value());
	else
	{
	    assert(tree_node->entry);
	    return &(tree_node->entry->value);
	}
    }

    // searches for the entry that corresponds to the key. If the search fails, returns pointer to the default entry
    // ( entry that corresponds to Key=0 ).
    Value* find_default(const Key& key = 0)
    {
	if(key==0)
	    return m_pDefaultValue;
	_Tnode* tree_node=find_private(key);
	if(!tree_node)
	    return m_pDefaultValue;
	else
	{
	    assert(tree_node->entry);
	    return &(tree_node->entry->value);
	}
    }

    // erases entry that corresponds to the key if such entry is present. Returns true on success.
    bool erase(const Key& key);
};


template <class Key, class Value> avm_map<Key, Value>::avm_map()
{
    m_pTree=new _Tnode;
    m_pTree->entry=new pair<Key, Value>;
    m_pDefaultValue=&(m_pTree->entry->value);
}

template <class Key, class Value> void avm_map<Key, Value>::update_min_max_weights(void* n)
{
    // cast to the needed type - used to prevent internal compiler error
    // for old egcc
    avm_map<Key, Value>::_Tnode* node = (avm_map<Key, Value>::_Tnode*) n;

    if(node->entry)
    {
	node->weight=1;
	node->minval=node->maxval=node->entry->key;
	node=node->parent;
    }
    else
    {
	if(node->left && node->left->entry)
	{
	    node->left->weight=0;
	    node->left->minval=node->left->maxval=node->left->entry->key;
	}
	if(node->right && node->right->entry)
	{
	    node->right->weight=0;
	    node->right->minval=node->right->maxval=node->right->entry->key;
	}
    }
    while(node)
    {
	node->weight=0;
	if(node->left)
	{
	    node->minval=node->left->minval;
	    node->weight+=node->left->weight;
	}
	else
	    node->minval=node->right->minval;
	if(node->right)
	{
	    node->maxval=node->right->maxval;
	    node->weight+=node->right->weight;
	}
	else
	    node->maxval=node->left->maxval;
	node=node->parent;
//	while(1)
//	  {
//	    int diff=0;
//	    if(node->left)
//		diff+=node->left->weight;
//	    if(nod->right)
//		diff-=m_pTree->right->weight;
//	    if((diff>-2) && (diff<2))
//		break;
//	}
    }
}

template <class Key, class Value> Value* avm_map<Key, Value>::insert(const Key& key, Value value)
{
    _Tnode* ptr=m_pTree;
    while(1)
    {
	if(ptr->entry)
	{
	    if(ptr->entry->key==key)
	    {
		ptr->entry->value=value;
		return &(ptr->entry->value);
	    }
	    ptr->left=new _Tnode(ptr);
	    ptr->right=new _Tnode(ptr);
	    if(ptr->entry->key<key)
	    {
		ptr->left->entry=ptr->entry;
		ptr->right->entry=new pair<Key, Value>(key, value);
		ptr->entry=0;
		update_min_max_weights(ptr);
		return &(ptr->right->entry->value);
	    }
	    else
	    {
		ptr->left->entry=new pair<Key, Value>(key, value);
		ptr->right->entry=ptr->entry;
		ptr->entry=0;
		update_min_max_weights(ptr);
		return &(ptr->left->entry->value);
	    }
	}

	if(ptr->left && ((key<ptr->left->maxval) || (key==ptr->left->maxval)))
	{
	    ptr=ptr->left;
	    continue;
	}

	if(ptr->right && ((ptr->right->minval<key) || (ptr->right->minval==key)))
	{
	    ptr=ptr->right;
	    continue;
	}
	if(ptr->left && ptr->right)
	{
	    if(ptr->left->weight<ptr->right->weight)
		ptr=ptr->left;
	    else
		ptr=ptr->right;
	    continue;
	}
	if(!ptr->left)
	{
	    ptr->left=new _Tnode(ptr);
	    ptr->left->entry=new pair<Key, Value>(key, value);
	    update_min_max_weights(ptr);
	    return &(ptr->left->entry->value);
	}
	if(!ptr->right)
	{
	    ptr->right=new _Tnode(ptr);
	    ptr->right->entry=new pair<Key, Value>(key, value);
	    update_min_max_weights(ptr);
	    return &(ptr->left->entry->value);
	}
	printf("Should not arrive here\n");
    }
}

template <class Key, class Value> avm_map<Key, Value>::_Tnode* avm_map<Key, Value>::find_private(const Key& key)
{
    _Tnode* ptr=m_pTree;
    while(1)
    {
	if(ptr->entry)
	{
	    if(ptr->entry->key==key)
		return ptr;
	    return 0;
	}

	if(ptr->left && ((key<ptr->left->maxval) || (key==ptr->left->maxval)))
	{
	    ptr=ptr->left;
	    continue;
	}

	if(ptr->right && ((ptr->right->minval<key) || (ptr->right->minval==key)))
	{
	    ptr=ptr->right;
	    continue;
	}
	if(ptr->left && ptr->right)
	{
	    if(ptr->left->weight<ptr->right->weight)
		ptr=ptr->left;
	    else
		ptr=ptr->right;
	    continue;
	}
	return 0;
    }
}

template <class Key, class Value> bool avm_map<Key, Value>::erase(const Key& key)
{
    if(key==0)
	return false; // you may not erase the default entry
    _Tnode* ptr=find_private(key);
    if(!ptr)
	return false;
    _Tnode* parent=ptr->parent;
    assert(parent); // since we have more than one tree node, this can't be root entry
    delete ptr->entry;
    if(parent->left==ptr)
	parent->left=0;
    else
	parent->right=0;
    delete ptr;
    while(!parent->left && !parent->right)
    {
	if(parent->parent->left==parent)
	    parent->parent->left=0;
	else
	    parent->parent->right=0;
	ptr=parent;
	parent=parent->parent;
	delete ptr;
    }
    if(parent->left && parent->left->entry && !parent->right)
    {
	parent->entry=parent->left->entry;
	delete parent->left;
	parent->left=0;
    }
    if(parent->right && parent->right->entry && !parent->left)
    {
	parent->entry=parent->right->entry;
	delete parent->right;
	parent->right=0;
    }
    update_min_max_weights(parent);
    return true;
}

AVM_END_NAMESPACE

#endif
