#ifndef AVIFILE_AVM_OUTPUT_H
#define AVIFILE_AVM_OUTPUT_H
#include <inttypes.h>
#include <avm_stl.h>
#include <avm_map.h>
#include <stdio.h>
#include <stdarg.h>
/***

    -- Design requirements for the output subsystem --
    
    ability to accept information with different debug levels
    ability to associate a special 'stream name' with each message
    ability to output information to console, into files or ignore it, depending on
	debug level or stream name
    being thread-safe
    prinf-like interface
    programmer must be able to install his own information handlers
    
***/

AVM_BEGIN_NAMESPACE

typedef void (*handlerFuncPtr) (const char* s, int opt);

/***

     Each message sent to AvmOutput has two parameters associated with it.
    a) debug level ( non-negative integer )
    b) type or module name ( 0-terminated string or 0 ).
     If its debug level is less than current set debug level of AvmOutput object,
    message is discarded immediately. Otherwise, zero or more of three things
    can happen, depending on its type and AvmObject state.
    1) message can be written to the console,
    2) message can be written into the file,
    3) message can be passed as a parameter to user callback function.
    In each of three cases AvmObject checks if a special setting was set for this 
    type/case and, if so, executes it, otherwise uses default settings for unmarked/unknown
    types.
    
***/

class AvmOutput
{
    string m_sString;
    string m_sCurrentMode;
    int m_iDebugLevel;    
    int m_iWriterDebugLevel;
    struct TextstreamInfo
    {
	int mask;
	handlerFuncPtr handler;
	int opt;
	FILE* file;
	bool console;
	TextstreamInfo() : mask(0), handler(0), file(0), console(true) {}
	void setHandler(handlerFuncPtr newhandler, int newopt) { handler=newhandler; opt=newopt; mask |= 1; }
	void unsetHandler() { mask &= ~1; handler=0; }
	void setFile(FILE* newfile) { file=newfile; mask |= 2; }
	void unsetFile() { mask &= ~2; file=0; }
	void setConsole(bool newconsole) { console=newconsole; mask |= 4; }
	void unsetConsole() { mask &= ~4; console=false; }
    };
    avm_map<const char*, TextstreamInfo> m_sMap;
    void vwrite(const char* format, va_list va);
public:
    void setDebugLevel(int i) { flush(); m_iDebugLevel=i; }
    int getDebugLevel() const { return m_iDebugLevel; }
    void setFileOut(FILE* file, const char* name=0) { flush(); m_sMap.find_insert(name)->setFile(file); }
    void unsetFileOut(const char* name=0) { flush(); m_sMap.find_insert(name)->unsetFile(); }
    void setConOut(bool con, const char* name=0) { flush(); m_sMap.find_insert(name)->setConsole(con); }
    void unsetConOut(const char* name=0) { flush(); m_sMap.find_insert(name)->unsetConsole(); }
    void setHandler(handlerFuncPtr fun, int opt, const char* name=0) { flush(); m_sMap.find_insert(name)->setHandler(fun, opt); }
    void unsetHandler(const char* name=0) { flush(); m_sMap.find_insert(name)->unsetHandler(); }
    
    void write(const char* format, ...);    
    void writei(const char* mode, const char* format, ...);
    void writei(int debuglevel, const char* format, ...);
    void writei(const char* mode, int debuglevel, const char* format, ...);
    void flush();   
};


extern AvmOutput out;

#define AVM_WRITE avm::out.write
#define AVM_WRITEI avm::out.writei
/***
    Usage examples
    
    1. Write lots of stuff to the console ( default behavior )
    
    AVM_WRITE("AVI file %s successfully opened\n", m_sFile.c_str());
    AVM_WRITE("File size %d\n", iFileSize):
    
    2. Write some messages into console and drop others depending on debug level
    
    avmout.setDebugLevel(2); 
    AVM_WRITEI(0, "AVI file %s successfully opened\n", m_sFile.c_str()); // written ( 0 < 2 )
    AVM_WRITEI(1, "File size %d\n", iFileSize); // written ( 1 < 2 )
    AVM_WRITEI(2, "File size %d\n", iFileSize); // discarded ( 2 >= 2 )
    
    3. Dynamically switch between file and console outputs
    
    FILE* f=fopen("debug.log", "w");    
    avmout.setDebugLevel(2);
    avmout.setFileOut(f);
    AVM_WRITE("AVI file %s successfully opened\n", m_sFile.c_str()); // written into the file
    avmout.setConOut();
    AVM_WRITEI(1, "File size %d\n", iFileSize): // written into the file and to the console
    avmout.unsetFileOut();
    fclose(f);
    AVM_WRITE("File size %d\n", iFileSize): // written to the console only
    
    4. Write only output from one module into the file, write everything else to the console
    
    FILE* f1=fopen("aviread.log", "w");    
    FILE* f2=fopen("aviplay.log", "w");    
    avmout.setFileOut(f1, "AviRead");
    avmout.setFileOut(f2, "Aviplay");
    AVM_WRITEI("AviRead", "AVI file %s successfully opened\n", m_sFile.c_str()); // written to aviread.log and to console
    AVM_WRITEI("Aviplay", "Cannot play back this file\n"); // written to aviplay.log and to console
    AVM_WRITE("Aborting"); // written to console only
    
    5. Redirect all output into GUI list box

void myhandlerFunc (const char* s, int opt)
{
    QListBox* box=(QListBox*) opt;
    box.addString(s);
}

    avmout.unsetConOut();
    avmout.setHandler(&myhandlerFunc, &m_ListBox);
    AVM_WRITE("Hello world\n");
    
***/    

AVM_END_NAMESPACE

#endif
