#ifndef AVIFILE_AVM_STL_H
#define AVIFILE_AVM_STL_H

#ifndef __THROW
#define __THROW throw()
#endif

#define AVM_BEGIN_NAMESPACE namespace avm {
#define AVM_END_NAMESPACE   };

// some basic types for our C++ usage
// we are replacing overbloated STL
//#include <stdio.h>
#include <assert.h>

AVM_BEGIN_NAMESPACE;

// simple class for storing char*
// the only reason for this class is about 100% faster compilation
// we do not need overcomplicated  std::string class for our purpose

class string
{
    char* str;
public:
    typedef uint_t size_type;
    static const size_type npos = ~0U;
    string();
    string(char s);
    string(const char* s, uint_t len = 0);
    string(const string& s, uint_t len = 0);
    ~string();
    uint_t size() const;
    char operator[](int i) const { return str[i]; }
    char& operator[](int i) { return str[i]; }
    bool operator==(const char* s) const;
    bool operator==(const string& s) const { return operator==(s.str); }
    bool operator!=(const char* s) const { return !operator==(s); }
    bool operator!=(const string& s) const { return !operator==(s); }
    bool operator<(const string& s) const;
    operator const char*() const { return str; }
    const char* c_str() const { return str; }
    string& operator=(const string& s) { return operator=(s.str); }
    string& operator=(const char* s);
    string& operator+=(const string& s) { return operator+=(s.str); }
    string& operator+=(const char* s);
    string operator+(const string& s) const { return string(str) += s; }
    string operator+(const char* s) const { return string(str) += s; }
    //string operator+(const char* s) { return string(str) += s; }
    string substr(uint_t from = 0, uint_t to = npos) const { return string(str + from, to); };
    size_type find(const string& s, size_type startpos = 0) const;
    size_type find(char c) const;
    size_type rfind(char c) const;
    void insert(uint_t pos, const string& s);
    string& erase(uint_t from = 0, uint_t to = npos);
};

// without this operator attempt to compare const char* with string will give quite unexpected results because
// of implicit usage of operator const char*() with the right operand
inline bool operator==(const char* s1, const string& s2) { return s2==s1; }

// very simple vector class
template <class Type> class vector
{
    Type* m_Type;
    uint_t m_uiSize;
    uint_t m_uiAlloc;
protected:
    void copy(const Type* in, uint_t size, uint_t alloc);
public:
    static const uint_t invalid=(uint_t)(-1);
    typedef Type* iterator;
    typedef const Type* const_iterator;
    vector<Type>(int prealloc = 0);
    // we will not count references - we have to program with this in mind!
    vector<Type>(const vector<Type>& t) :m_Type(0), m_uiSize(0), m_uiAlloc(0)
    {
	operator=(t);
    }
    vector<Type>& operator=(const vector<Type>& t)
    {
	//printf("operator=    %p   %d  %d\n", t.m_Type, t.m_uiSize, t.m_uiAlloc);
	if (this != &t)
	    copy(t.m_Type, t.m_uiSize, t.m_uiAlloc);
	return *this;
    }
    ~vector() { delete[] m_Type; }
    Type& operator[](int i) { return m_Type[i]; }
    const Type& operator[](int i) const { return m_Type[i]; }
    const Type* begin() const { return m_Type; }
    Type* begin() { return m_Type; }
    const Type* end() const { return m_Type + m_uiSize; }
    Type* end() { return m_Type + m_uiSize; }
    uint_t find(const Type& t) const
    {
	for (uint_t i = 0; i < m_uiSize; i++)
	    if (m_Type[i] == t)
		return i;
	return invalid;
    }
    void erase(iterator pos)
    {
        assert(m_uiSize > 0);
	if (m_uiSize > 0)
	{
	    while (pos < end() - 1)
	    {
		pos[0] = pos[1];
		pos++;
	    }
	    pop_back();
	}
    }
    void push_back(const Type& m)
    {
	if (m_uiSize + 1 >= m_uiAlloc)
	    copy(m_Type, m_uiSize, m_uiAlloc * 2);
	m_Type[m_uiSize++] = m;
    }
    void pop_back()
    {
	//printf("vector pop_back %d\n", m_uiSize);
        assert(m_uiSize > 0);
	m_uiSize--;
	if ((m_uiAlloc >= 8) && (m_uiSize < m_uiAlloc / 4))
	    copy(m_Type, m_uiSize, m_uiAlloc / 2);
    }
    Type& front() { return *begin(); }
    const Type& front() const { return *begin(); }
    Type& back() { return *(end() - 1); }
    const Type& back() const { return *(end() - 1); }
    uint_t size() const { return m_uiSize; }
    void resize(uint_t sz)
    {
        m_uiAlloc = sz;
	if (m_uiSize > sz)
	    m_uiSize = sz;
	copy(m_Type, m_uiSize, m_uiAlloc);
	m_uiSize = m_uiAlloc;
	//printf("vector resize %d\n", m_uiSize);
    }
    void clear()
    {
	if (m_uiAlloc > 4)
	{
	    delete[] m_Type;
	    m_uiAlloc = 4;
	    m_Type = new Type[m_uiAlloc];
	}
	m_uiSize = 0;
    }
    void pop_front()
    {
        assert(m_uiSize > 0);
	for (uint_t i = 1; i < m_uiSize; i++)
	    m_Type[i - 1] = m_Type[i];
	pop_back();
    }
    void remove(const Type& t);
    operator const Type*() const { return m_Type; }
};

template <class Type>
vector<Type>::vector<Type>(int prealloc = 0)
    :m_Type(0), m_uiSize(prealloc), m_uiAlloc(prealloc)
{
    if (m_uiAlloc > 0 )
	m_Type = new Type[m_uiAlloc];
    //printf("vector %p   (%d)\n", this, m_uiSize);
}

template <class Type>
void vector<Type>::remove(const Type& t)
{
    int d = 0;
    iterator from = begin();
    for (iterator it = from; it != end(); it++)
    {
	if (t == *it)
	{
	    d++;
	    continue;
	}
	if (from != it)
	    *from++ = *it;
    }
    while (--d >= 0)
	pop_back();
    //printf("REMOVE VECT %d\n", m_uiSize);
}

template <class Type>
void vector<Type>::copy(const Type* in, uint_t sz, uint_t alloc)
{
    Type* tmp = m_Type;
    m_uiAlloc = alloc;
    if (m_uiAlloc <= 0)
        m_uiAlloc = 4;
    m_Type = new Type[m_uiAlloc];
    m_uiSize = sz;
    //printf("COPY VECT %d\n", sz);
    assert(sz <= alloc);
    for (uint_t i = 0; i < sz; i++)
	m_Type[i] = in[i];
    delete[] tmp;
}

AVM_END_NAMESPACE;

#endif
