#ifndef AVIFILE_INFOTYPES_H
#define AVIFILE_INFOTYPES_H

#include "default.h"
#include "avm_stl.h"
#include "formats.h"

class BaseInfo
{
protected:
    avm::string name;		// unique name.
    avm::string about;		// text description.
public:
    BaseInfo();
    BaseInfo(const char* _name, const char* _about);
    virtual ~BaseInfo();

    const char* GetName() const;
    const char* GetAbout() const;
};

/**
 * Describes the attibute of encoder or decoder.
 */
class AttributeInfo : public BaseInfo
{
public:
    /**
     * 'Integer' - attribute is an integer, either limited by i_min and i_max,
     * in which case i_min<i_max, or unlimited.
     * 'String' - attribute can have any string value, such as registration code.
     * 'Select' - value can have one of several string values from 'options'.
     */
    enum Kind { Integer, String, Select };
    Kind kind;
    avm::vector<avm::string> options;
    int i_min, i_max;

    AttributeInfo();
    ~AttributeInfo();
    /* Constructors for unlimited integer attributes */
    AttributeInfo(const char* n);
    /* Constructor for select-type attribute */
    AttributeInfo(const char* n, const char* a, const char** o);
    /* Generic constructor */
    AttributeInfo(const char* n, const char* a, Kind k, int minval=0, int maxval=-1);
};


/**
 * Structure describes video codec, registered in the system.
 */
class CodecInfo : public BaseInfo
{
public:
//
// Fields that loader expects to be filled by plugin
//
    fourcc_t fourcc;		// default fourcc for this codec.
//
// All fourcc's that should be handled with this codec.
//
    avm::vector<fourcc_t> fourcc_array;
    
    enum Kind
    {
        /* On2 Truemotion VP3.x support */
        /* Win32Ex is a little bit far away from 0, but this could
	   help to keep binary compatibility of the libraries. */
	Source = 0, Plugin, Win32, DShow_Dec, DShow_Enc, Win32Ex, Custom=0xFFFFFFFF
    };
    enum Media { Audio, Video };
    enum Direction { Encode = 1, Decode=2, Both=3 };
    
// private name of the codec
// globally unique! i.e. there should be at most one CodecInfo in audio and one CodecInfo in video for a given private name on any system
// assigned to the codec by avifile ( or mplayer ) developers ;^)
    avm::string privatename;
    
//
// Source - structure describes codec that is implemented as a part of library.
// All other values in this field are ignored by the library and can be used within
// plugins.
//
    Kind kind;
    Media media;
    Direction direction;

    avm::vector<AttributeInfo> decoder_info; // acceptable attrs for decoder.
    avm::vector<AttributeInfo> encoder_info; // acceptable attrs for encoder.

//
// These two fields will be filled by plugin loader.
//
    avm::string modulename;
    mutable void* handle;
//
// These fields contain custom data that may be needed by plugin.
// They will be ignored by loader.
//
    GUID guid;		// CLSID ( only needed by DirectShow codecs + Vorbis ).
    avm::string dll;	// Win32 DLL name ( only needed by Win32 DLL codecs ).

    CodecInfo();
    ~CodecInfo();
    CodecInfo(const fourcc_t* array, const char* info, const char* path,
	      const char* about, Kind _kind, const char* privname, Media _media = Video,
	      Direction _direction = Both, const GUID* id = 0,
	      const avm::vector<AttributeInfo>& ei = avm::vector<AttributeInfo>(),
	      const avm::vector<AttributeInfo>& di = avm::vector<AttributeInfo>());
    CodecInfo(const CodecInfo& ci);
    CodecInfo& operator=(const CodecInfo&);
    bool operator==(const CodecInfo& v) const { return (this==&v); }
//
// Searches for codec matching given fourcc, returns NULL if there's no match.
//
    static const CodecInfo* match(fourcc_t codec, Media media=Video, const CodecInfo* start=0, Direction direction=Both);
    static const CodecInfo* match(Media media, const char* privname);
};

struct VideoEncoderInfo
{
    int quality;
    int keyfreq;
    fourcc_t compressor;
    avm::string cname;
    BITMAPINFOHEADER header;
};

extern avm::vector<CodecInfo> video_codecs;
extern avm::vector<CodecInfo> audio_codecs;

#endif // AVIFILE_INFOTYPES_H
