#ifndef AVIFILE_PLUGIN_H
#define AVIFILE_PLUGIN_H

#define PLUGIN_API_VERSION 11107

#include "infotypes.h"
#include "formats.h" // WAVEFORMATEX - slightly more complicated type

class IAudioEncoder;
class IAudioDecoder;
class IVideoEncoder;
class IVideoDecoder;

/*
This library is capable of using external ELF plugins for audio/video
(de)compression. This document describes the interface of such plugin.

Each plugin provides function 
*/    
#ifdef WIN32
#pragma warning ( disable: 4190 )
#endif
extern "C" const avm::vector<CodecInfo>& RegisterPlugin();

extern "C" IAudioDecoder* CreateAudioDecoder(const CodecInfo& info, const WAVEFORMATEX* format);
extern "C" IAudioEncoder* CreateAudioEncoder(const CodecInfo& info, fourcc_t compressor, const WAVEFORMATEX* fmt);

/*
which is used by the library to register formats supported by
this plugin. This function returns reference to array of CodecInfo
objects. Each of them refers to one supported format. See declaration
of CodecInfo type ( include/infotypes.h ) for details.

When plugin supports decoding of at least one video format, it provides
function
*/
extern "C" IVideoDecoder* CreateVideoDecoder(const CodecInfo& info, const BITMAPINFOHEADER& bh, int flip);
/*    
Arguments:
    info: reference to one of entries from array returned with RegisterPlugin().
    bh: reference to format structure describing compressed format.
    flip: normally BITMAPINFOHEADER structure with positive value in
    'biHeight' field refers to upside-down picture. Decoder created with
    flip=0 produces pictures with negative biHeight ( so that they can be 
    memcpy()'ed to the screen ). flip=1 requests picture with positive biHeight.
    
When plugin supports encoding into at least one video format, it provides
function
*/
extern "C" IVideoEncoder* CreateVideoEncoder(const CodecInfo& info, fourcc_t compressor, const BITMAPINFOHEADER& bh);
/*
Arguments:
    info: same as above.
    compressor: fourcc of format to encode into.
    bh: format of uncompressed data.
    
Both functions return pointer to initialized codec on success and 0 on failure.

Codecs can be made configurable through the following interface:
*/    
extern "C" int GetAttrInt(const CodecInfo& info, const char* attribute, int& value);
extern "C" int SetAttrInt(const CodecInfo& info, const char* attribute, int value);
extern "C" int GetAttrString(const CodecInfo& info, const char* attribute, char* value, int size);
extern "C" int SetAttrString(const CodecInfo& info, const char* attribute, const char* value);
/*
Names of attributes and their acceptable values can be found out from
CodecInfo structure.

Plugin can describe why the last operation failed by implementing
function
*/
extern "C" avm::string GetError();
/*

Writing or using these plugins does not require you to have avifile. As an example,
you can take a look at OpenDivX plugin ( http://divx.euro.ru/download.htm ).
It does not depend on existing avifile installation; instead, it includes its
own small set of headers ( ~25k ), which is enough for it to be compiled.
It is completely binary compatible with the library, and still theoretically
can be used on its own from other programs.

To prevent conflicts between different versions of plugins and applications,
each plugin provides function
*/
extern "C" int GetPluginVersion();
/*
Its return value identifies date of last change in its API. For example,
value 10120 corresponds to January 20, 2001.
*/
#endif // AVIFILE_PLUGIN_H
