#ifndef AVIFILE_UTILS_H
#define AVIFILE_UTILS_H

#include "default.h"
/**
 *  Utility functions.
 *    plain C functions are here
 */

/**
 *  These two functions return number of ticks ( processor cycles )
 *   since power-on. Return value of first one wraps around each
 *   few seconds ( depends on processor ), but calculations with
 *   32-bit 'int' are a bit faster than with 64-bit 'int64_t'.
 *  They can be used for very precise and fast measurement of time
 *   intervals - call to localcount() takes less than 0.2us on 500
 *   MHz processor, and its resolution is measured in nanoseconds.
 *   On systems with properly configured Linux kernel they can be 
 *   used as 3-4 times faster alternative to gettimeofday() 
 *   because they do not require a trip to kernel and back.
 *   On other systems they will probably be even more efficient.
 *  For processors without time-stamp counter feature ( a few
 *   old Pentium-compatible models by Cyrix and maybe others )
 *   these functions are emulated using gettimeofday().
 */

extern uint_t (*localcount)(void);
extern int64_t (*longcount)(void);

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

struct _WAVEFORMATEX;

/*
 * function to retrieve/store data in machine endependent format
 * - right now dump implementation - as we could check functionality
 */
static inline uint16_t avm_get_le16(const void* p) __THROW {
    const uint8_t* c = (const uint8_t*) p;
    return c[0] | c[1] << 8;
}

static inline uint8_t* avm_set_le16(void* p, uint16_t v) __THROW {
    uint8_t* b = (uint8_t*) p;
    b[0] = v & 0xff;
    b[1] = (v >> 8) & 0xff;
    return b;
}

static inline uint32_t avm_get_le32(const void* p) __THROW {
    const uint8_t* c = (const uint8_t*) p;
    return c[0] | c[1] << 8 | c[2] << 16 | c[3] << 24;
}

static inline uint64_t avm_get_le64(const void* p) __THROW {
    const uint8_t* c = (const uint8_t*) p;
    return avm_get_le32(c) | (((uint64_t)avm_get_le32(c + 4)) << 32);
}

static inline uint8_t* avm_set_le32(void* p, uint32_t v) __THROW {
    uint8_t* b = (uint8_t*) p;
    b[0] = v & 0xff;
    b[1] = (v >> 8) & 0xff;
    b[2] = (v >> 16) & 0xff;
    b[3] = (v >> 24) & 0xff;
    return b;
}

static inline uint16_t avm_get_be16(const void* p) __THROW {
    const uint8_t* c = (const uint8_t*) p;
    return c[0] << 8 | c[1];
}

static inline uint8_t* avm_set_be16(void* p, uint16_t v) __THROW {
    uint8_t* b = (uint8_t*) p;
    b[0] = (v >> 8) & 0xff;
    b[1] = v & 0xff;
    return b;
}

static inline uint32_t avm_get_be32(const void* p) __THROW  {
    const uint8_t* c = (const uint8_t*) p;
    return c[0] << 24 | c[1] << 16 | c[2] << 8 | c[3];
}

static inline uint64_t avm_get_be64(const void* p) __THROW  {
    const uint8_t* c = (const uint8_t*) p;
    return ((uint64_t) avm_get_be32(c)) << 32 | avm_get_le32(c + 4);
}

static inline uint8_t* avm_set_be32(void* p, uint32_t v) __THROW {
    uint8_t* b = (uint8_t*) p;
    b[0] = (v >> 24) & 0xff;
    b[1] = (v >> 16) & 0xff;
    b[2] = (v >> 8) & 0xff;
    b[3] = v & 0xff;
    return b;
}

/* memory lock for internal usage only - will be removed in future! */
void avm_memory_lock_init(void) __THROW;
void avm_memory_lock(void) __THROW;
void avm_memory_unlock(void) __THROW;

const char* avm_fcc_name(fourcc_t fcc) __THROW;
const char* avm_wave_format_name(short fmt) __THROW; /* not sure - might be special type */
char* avm_wave_format(char* buf, uint_t s, const struct _WAVEFORMATEX* wf) __THROW;

/* wrapping functions */
/* A thread-save usec sleep */
int avm_usleep(unsigned long) __THROW;

/* some system doesn't have setenv/unsetenv */
int avm_setenv(const char *name, const char *value, int overwrite) __THROW;
void avm_unsetenv(const char *name) __THROW;


#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif /* AVIFILE_UTILS_H */
