/* 
 * glass - the openGL Articulated Skeleton System
 *
 * Copyright (C) 2001  Robert Ancell <bob27@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef __glass_h__
#define __glass_h__

#include <stdio.h>
#include <GL/gl.h>

#include "glass_types.h"

#ifdef __cplusplus
extern "C" {
#endif

#ifndef TRUE
#define TRUE 1
#endif
#ifndef FALSE
#define FALSE 0
#endif

/* Return a string for this version of glass */
char *glassGetVersion(void);

/* Load an object */
GlassObject *glassLoadObject(const char *fname);

/* Make a new library */
GlassLibrary *glassLibraryNew(const char *name);

/* Load an object into a library */
GlassObject *glassLibraryLoadObject(GlassLibrary *library, const char *fname);

/* Return the index of a variable with the name `name', return -1 if not found */
int glassFindVariable(GlassObject *object, const char *name);

/* Get the current value of a variable */
GLfloat glassGetVariableValue(GlassObject *object, int variable);

/* Get a pointer to the current value of a variable */
GLfloat *glassGetVariableValuev(GlassObject *object, int variable);

/* Get the minimum value a variable can take */
GLfloat glassGetVariableMin(GlassObject *object, int variable);

/* Get the maximum value a variable can take */
GLfloat glassGetVariableMax(GlassObject *object, int variable);

/* Set a variable, return FALSE if the variable doesn't exist */
int glassSetVariable(GlassObject *object, int variable, GLfloat value);

/* Increment a variable, return FALSE if the variable doesn't exist */
int glassIncVariable(GlassObject *object, int variable, GLfloat dvalue);

/* Find an active point */
int glassFindActivePoint(GlassObject *object, const char *name);

/* Get the position of an active point */
GLfloat *glassGetActivePointPos(GlassObject *object, int apoint);

/* Get the direction of an active point */
GLfloat *glassGetActivePointDir(GlassObject *object, int apoint);

/* Draw the object, return FALSE if error occurs */
int glassDrawObject(GlassObject *object);

/* Return the index of a component with the name `name', return -1 if not found */
int glassFindComponent(GlassObject *object, const char *name);

/* Toggle whether or not this component (triangle list), is drawn or not,
   visible = TRUE = draw component, visible = FALSE = hide component.
   Return FALSE if component doesn't exist */
int glassSetComponentVisibility(GlassObject *object, int component, int visible);

/* Draw one component (triangle list) of the object, return FALSE if component doesn't exist */
int glassDrawComponent(GlassObject *object, int component);

/* Remove an object and all its components etc */
void glassFreeObject(GlassObject *object);

/* Remove a library and all its objects */
void glassFreeLibrary(GlassLibrary *library);

/* glass_texture.c */
GlassTexture      *_glassTextureNew               (void);
void               _glassTextureSetName           (GlassTexture *, const char *);
const char        *_glassTextureGetName           (GlassTexture *);
void               _glassTextureLoadRGB           (GlassTexture *, const char *);
void               _glassFreeTexture              (GlassTexture *);

/* glass_material.c */
GlassMaterial     *_glassMaterialNew              (void);
void               _glassMaterialSetName          (GlassMaterial *, const char *);
const char        *_glassMaterialGetName          (GlassMaterial *);
void               _glassMaterialSetAmbient       (GlassMaterial *,
						   GLfloat, GLfloat, GLfloat, GLfloat);
void               _glassMaterialSetAmbientv      (GlassMaterial *, GLfloat *);
void               _glassMaterialSetDiffuse       (GlassMaterial *,
						   GLfloat, GLfloat, GLfloat, GLfloat);
void               _glassMaterialSetDiffusev      (GlassMaterial *, GLfloat *);
void               _glassMaterialSetSpecular       (GlassMaterial *,
						   GLfloat, GLfloat, GLfloat, GLfloat);
void               _glassMaterialSetSpecularv      (GlassMaterial *, GLfloat *);
void               _glassMaterialSetEmission      (GlassMaterial *,
						   GLfloat, GLfloat, GLfloat, GLfloat);
void               _glassMaterialSetEmissionv     (GlassMaterial *, GLfloat *);
void               _glassMaterialSetShininess     (GlassMaterial *, GLfloat);
void               _glassMaterialSetTexture       (GlassMaterial *, GlassTexture *);
GlassTexture      *_glassMaterialGetTexture       (GlassMaterial *);
void               _glassMaterialTestAlpha        (GlassMaterial *);
void               _glassFreeMaterial             (GlassMaterial *);

/* glass_triangle.c */
GlassTriangle     *_glassTriangleNew              (void);
GlassTriangle     *_glassAlphaTriangleNew         (void);
void               _glassTriangleSetPosition      (GlassTriangle *, int, GLfloat *);
GLfloat           *_glassTriangleGetPosition      (GlassTriangle *, int);
void               _glassTriangleSetColour        (GlassTriangle *, int, GLfloat *);
GLfloat           *_glassTriangleGetColour        (GlassTriangle *, int);
void               _glassTriangleSetNormal        (GlassTriangle *, int, GLfloat *);
GLfloat           *_glassTriangleGetNormal        (GlassTriangle *, int);
void               _glassTriangleSetTexCoord      (GlassTriangle *, int, GLfloat *);
GLfloat           *_glassTriangleGetTexCoord      (GlassTriangle *, int);
void               _glassTriangleSetMaterial      (GlassTriangle *, GlassMaterial *);
GlassMaterial     *_glassTriangleGetMaterial      (GlassTriangle *);
void               _glassFreeTriangle             (GlassTriangle *);

/* glass_tlist.c */
GlassTriangleList *_glassTriangleListNew          (const char *);
void               _glassTriangleListSetName      (GlassTriangleList *, const char *);
const char        *_glassTriangleListGetName      (GlassTriangleList *);
void               _glassTriangleListAddTriangle  (GlassTriangleList *, GlassTriangle *);
int                _glassTriangleListRemoveTriangle   (GlassTriangleList *, GlassTriangle *);
void               _glassTriangleListMakeDisplayLists (GlassTriangleList *);
void               _glassTriangleListSetVisibility    (GlassTriangleList *, int);
void               _glassFreeTriangleList         (GlassTriangleList *);
void               _glassFreeTriangleListChild    (GlassTriangleList *);

/* glass_variable.c */
GlassVariable     *_glassVariableNew              (const char *, GLfloat, GLfloat, GLfloat);
void               _glassVariableSetName          (GlassVariable *, const char *);
const char        *_glassVariableGetName          (GlassVariable *);
void               _glassVariableSetValue         (GlassVariable *, GLfloat);
void               _glassVariableIncValue         (GlassVariable *, GLfloat);
GLfloat            _glassVariableGetValue         (GlassVariable *);
GLfloat           *_glassVariableGetValuev        (GlassVariable *);
void               _glassVariableSetMin           (GlassVariable *, GLfloat);
GLfloat            _glassVariableGetMin           (GlassVariable *);
GLfloat           *_glassVariableGetMinv          (GlassVariable *);
void               _glassVariableSetMax           (GlassVariable *, GLfloat);
GLfloat            _glassVariableGetMax           (GlassVariable *);
GLfloat           *_glassVariableGetMaxv          (GlassVariable *);
void               _glassVariableClip             (GlassVariable *);
void               _glassFreeVariable             (GlassVariable *);

/* glass_apoint.c */
GlassActivePoint  *_glassActivePointNew           (void);
void               _glassActivePointSetName       (GlassActivePoint *, const char *);
const char        *_glassActivePointGetName       (GlassActivePoint *);
void               _glassActivePointSetInitDir    (GlassActivePoint *,
						   GLfloat, GLfloat, GLfloat);
void               _glassActivePointSetInitDirv   (GlassActivePoint *, GLfloat *);
GLfloat           *_glassActivePointGetInitDir    (GlassActivePoint *);
GLfloat           *_glassActivePointGetPos        (GlassActivePoint *);
GLfloat           *_glassActivePointGetDir        (GlassActivePoint *);
void               _glassActivePointUpdate        (GlassActivePoint *);
void               _glassFreeActivePoint          (GlassActivePoint *);

/* glass_transform.c */
GlassTransform    *_glassTransformNew             (GlassTransformType);
int                _glassGetnParams               (GlassTransformType);
void               _glassTransformSetType         (GlassTransform *, GlassTransformType);
GlassTransformType _glassTransformGetType         (GlassTransform *);
void               _glassTransformSetConstant     (GlassTransform *, int, GLfloat);
void               _glassTransformSetVariable     (GlassTransform *, int, GlassVariable *);
void               _glassFreeTransform            (GlassTransform *);

/* glass_component.c */
GlassComponent    *_glassComponentNew             (void);
void               _glassComponentSetTransform    (GlassComponent *, GlassTransform *);
GlassTransform    *_glassComponentGetTransform    (GlassComponent *);
void               _glassComponentSetList         (GlassComponent *, GlassTriangleList *);
GlassTriangleList *_glassComponentGetList         (GlassComponent *);
void               _glassComponentSetActivePoint  (GlassComponent *, GlassActivePoint *);
GlassActivePoint  *_glassComponentGetActivePoint  (GlassComponent *);
void               _glassComponentAddSibling      (GlassComponent *, GlassComponent *);
int                _glassComponentRemoveSibling   (GlassComponent *, GlassComponent *);
void               _glassComponentAddChild        (GlassComponent *, GlassComponent *);
int                _glassComponentRemoveChild     (GlassComponent *, GlassComponent *);
int                _glassComponentTestActivePointr (GlassComponent *);
void               _glassComponentUpdateActivePointsr (GlassComponent *);
void               _glassFreeComponent            (GlassComponent *);
void               _glassFreeComponentr           (GlassComponent *);

/* glass_object.c */
GlassObject       *_glassObjectNew                (const char *);
void               _glassObjectSetName            (GlassObject *, const char *);
const char        *_glassObjectGetName            (GlassObject *);
void               _glassObjectAddTexture         (GlassObject *, GlassTexture *);
GlassTexture      *_glassObjectFindTexture        (GlassObject *, const char *);
int                _glassObjectRemoveTexture      (GlassObject *, GlassTexture *);
void               _glassObjectAddMaterial        (GlassObject *, GlassMaterial *);
GlassMaterial     *_glassObjectFindMaterial       (GlassObject *, const char *);
int                _glassObjectRemoveMaterial     (GlassObject *, GlassMaterial *);
void               _glassObjectAddList            (GlassObject *, GlassTriangleList *);
GlassTriangleList *_glassObjectFindList           (GlassObject *, const char *);
int                _glassObjectRemoveList         (GlassObject *, GlassTriangleList *);
void               _glassObjectAddVariable        (GlassObject *, GlassVariable *);
GlassVariable     *_glassObjectFindVariable       (GlassObject *, const char *);
int                _glassObjectRemoveVariable     (GlassObject *, GlassVariable *);
void               _glassObjectAddActivePoint     (GlassObject *, GlassActivePoint *);
GlassActivePoint  *_glassObjectFindActivePoint    (GlassObject *, const char *);
int                _glassObjectRemoveActivePoint  (GlassObject *, GlassActivePoint *);
void               _glassObjectAddChild           (GlassObject *, GlassComponent *);
int                _glassObjectRemoveChild        (GlassObject *, GlassComponent *);
void               _glassObjectBuildArrays        (GlassObject *);
void               _glassObjectBuildVariableArray (GlassObject *);
void               _glassObjectBuildListArray     (GlassObject *);
void               _glassObjectBuildActivePointArray (GlassObject *);
void               _glassObjectUpdateActivePoints (GlassObject *);
void               _glassObjectMakeDisplayLists   (GlassObject *);
void               _glassFreeObject               (GlassObject *);

/* glass_library.c */
GlassLibrary      *_glassLibraryNew               (const char *name);
int                _glassLibrarySetName           (GlassLibrary *library, const char *name);
char              *_glassLibraryGetName           (GlassLibrary *library);
int                _glassLibraryAddObject         (GlassLibrary *library, GlassObject *object);
GlassObject       *_glassLibraryFindObject        (GlassLibrary *library, const char *name);
GlassTexture      *_glassLibraryFindTexture       (GlassLibrary *library, const char *name);
GlassMaterial     *_glassLibraryFindMaterial      (GlassLibrary *library, const char *name);
GlassTriangleList *_glassLibraryFindList          (GlassLibrary *library, const char *name);
GlassVariable     *_glassLibraryFindVariable      (GlassLibrary *library, const char *name);
GlassActivePoint  *_glassLibraryFindActivePoint   (GlassLibrary *library, const char *name);
int                _glassLibraryRemoveObject      (GlassLibrary *library, GlassObject *object);
int                _glassFreeLibraryEntry         (GlassLibraryEntry *entry);
int                _glassFreeLibraryEntry         (GlassLibraryEntry *entry);
int                _glassFreeLibrary              (GlassLibrary *library);

/* The loading functions (glass_load.c) */
GlassObject       *_glassLoadObject               (GlassLibrary *, const char *);
void               _glassLoadObjectInfo           (GlassObject *, FILE *, const char *);
void               _glassLoadTextures             (GlassObject *, FILE *, const char *);
void               _glassLoadMaterials            (GlassObject *, FILE *, const char *);
void               _glassLoadTriangleLists        (GlassObject *, FILE *, const char *);
void               _glassLoadVariables            (GlassObject *, FILE *, const char *);
void               _glassLoadActivePoints         (GlassObject *, FILE *, const char *);
void               _glassLoadComponentsAndStructure (GlassComponentList **,
						     GlassStructureTree **, FILE *,
						     const char *);
GlassComponentList *_glassComponentListNew        (void);
GlassStructureTree *_glassStructureTreeNew        (void);
char              *_getNextWord                   (FILE *);
float              _getNextFloat                  (FILE *);
int                _glassLoadTexture              (GlassObject *, FILE *, const char *);
int                _glassLoadMaterial             (GlassObject *, FILE *);
int                _glassLoadTriangleList         (GlassObject *, FILE *);
int                _glassLoadTriangle             (GlassObject *, GlassTriangleList *, FILE *);
int                _glassLoadVariable             (GlassObject *, FILE *);
int                _glassLoadActivePoint          (GlassObject *, FILE *);
int                _glassLoadComponent            (GlassComponentList **, FILE *);
int                _glassLoadTransform            (char **, GlassComponentList *, FILE *);
GlassTransformType _glassConvertTransform         (const char *);
int                _glassLoadStructure            (GlassStructureTree **, FILE *);
int                _getRelatives                  (char **, GlassStructureTree **,
						   int, int, int, int);
int                _getInt                        (char **, int, int, int, int);
int                _glassGenerateComponentTree    (GlassObject *,
						   GlassComponentList *,
						   GlassStructureTree *);
GlassComponent    *_linkComponents                (GlassComponentList *);
int                _linkComponentStructure        (GlassStructureTree *, GlassComponentList *);
GlassComponentList *_glassFindComponentList       (GlassComponentList *, int);
int                _glassFreeComponentList        (GlassComponentList *);
int                _glassFreeStructureTree        (GlassStructureTree *);

/* The drawing functions (glass_draw.c) */
int                _glassDrawObject               (GlassObject *);
int                _glassDrawComponent            (GlassComponent *);
int                _glassDrawAlphaComponent       (GlassComponent *);
int                _glassDoTransform              (GlassTransform *);
int                _glassDrawTriangleList         (GlassTriangleList *);
int                _glassDrawAlphaTriangleList    (GlassTriangleList *);
void               _glassSetMaterial              (GlassMaterial *);
void               _glassDrawTriangle             (GlassTriangle *);
void               _glassDrawAlphaTriangle        (GlassTriangle *);

/* The saving functions (glass_save.c) */
int                _glassSaveObject               (GlassObject *, const char *, const char *);
int                _glassWriteTextures            (FILE *, const char *, GlassObject *);
int                _glassWriteTexture             (FILE *, const char *, GlassTexture *);
int                _glassWriteMaterials           (FILE *, GlassObject *);
int                _glassWriteMaterial            (FILE *, GlassMaterial *);
int                _glassWriteTriangleLists       (FILE *, GlassObject *);
int                _glassWriteTriangleList        (FILE *, GlassTriangleList *);
int                _glassWriteTriangleListData    (FILE *, GlassTriangleList *);
int                _glassWriteTriangle            (FILE *, GlassTriangle *);
int                _glassWriteVariables           (FILE *, GlassObject *);
int                _glassWriteVariable            (FILE *, GlassVariable *);
int                _glassWriteActivePoints        (FILE *, GlassObject *);
int                _glassWriteActivePoint         (FILE *, GlassActivePoint *);
int                _glassWriteComponents          (FILE *, GlassObject *);
int                _glassMakeComponentList        (GlassComponentList *, GlassComponent *,
						   int *);
int                _glassGetComponentIndex        (GlassComponentList *, GlassComponent *);
int                _glassWriteComponent           (FILE *, GlassComponent *,
						   int, int, int);
int                _glassWriteTransform           (FILE *, GlassTransform *);
const char        *_glassGetTransformName         (GlassTransformType);

/* glass_rgb.h */
GlassRGBImageTemp *_glassOpenRGBImage             (const char *);
void               _glassCloseRGBImage            (GlassRGBImageTemp *);
void               _glassRGBImageGetRow           (GlassRGBImageTemp *, unsigned char *,
						   int, int);
void               _glassRGBImageGetRawData       (GlassRGBImageTemp *, unsigned char *);
GlassRGBImage     *_glassLoadRGBImage             (const char *);

#ifdef __cplusplus
}
#endif

#endif /* __glass_h__ */
