/* 
 * glass - the openGL Articulated Skeleton System
 *
 * Copyright (C) 2001-2002  Robert Ancell <bob27@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef __glass_h__
#define __glass_h__

#include <GL/gl.h>

#ifdef __cplusplus
extern "C" {
#endif

#ifndef FALSE
#define FALSE 0
#endif
#ifndef TRUE
#define TRUE !FALSE
#endif

/* You don't need to know the internals of these structures! */
typedef struct {} GlassLibrary;
typedef struct {} GlassModel;

/* Return a string for this version of glass */
extern const char *glassGetVersion(void);

/* Load an model */
extern GlassModel *glassLoadModel(const char *fname);

/* Make a new library */
extern GlassLibrary *glassLibraryNew(const char *name);

/* Load an model into a library */
extern GlassModel *glassLibraryLoadModel(GlassLibrary *library, const char *fname);

/* Return the number of variables in this model */
extern int glassGetNumVariables(GlassModel *model);

/* Return the index of a variable with the name `name', return -1 if not found */
extern int glassFindVariable(GlassModel *model, const char *name);

/* Get the current value of a variable */
extern GLfloat glassGetVariableValue(GlassModel *model, int variable);

/* Get a pointer to the current value of a variable */
extern const GLfloat *glassGetVariableValuev(GlassModel *model, int variable);

/* Get the minimum value a variable can take */
extern GLfloat glassGetVariableMin(GlassModel *model, int variable);

/* Get the maximum value a variable can take */
extern GLfloat glassGetVariableMax(GlassModel *model, int variable);

/* Set a variable, return FALSE if the variable doesn't exist */
extern int glassSetVariable(GlassModel *model, int variable, GLfloat value);

/* Increment a variable, return FALSE if the variable doesn't exist */
extern int glassIncVariable(GlassModel *model, int variable, GLfloat dvalue);

/* Return the number of active points in this model */
extern int glassGetNumActivePoints(GlassModel *model);

/* Find an active point */
extern int glassFindActivePoint(GlassModel *model, const char *name);

/* Get the position of an active point */
extern GLfloat *glassGetActivePointPos(GlassModel *model, int apoint);

/* Get the direction of an active point */
extern GLfloat *glassGetActivePointDir(GlassModel *model, int apoint);

/* Draw the model, return FALSE if error occurs */
extern int glassDrawModel(GlassModel *model);

/* Return the number of surfaces in this model */
extern int glassGetNumSurfaces(GlassModel *model);

/* Return the index of a surface with the name `name', return -1 if not found */
extern int glassFindSurface(GlassModel *model, const char *name);

/* Toggle whether or not this surface, is drawn or not,
   visible = TRUE = draw surface, visible = FALSE = hide surface.
   Return FALSE if surface doesn't exist */
extern int glassSetSurfaceVisibility(GlassModel *model, int surface, int visible);

/* Draw one surface of an model, return FALSE if surface doesn't exist */
extern int glassDrawSurface(GlassModel *model, int surface);

/* Remove an model and all its components etc */
extern void glassFreeModel(GlassModel *model);

/* Remove a library and all its models */
extern void glassFreeLibrary(GlassLibrary *library);

#ifdef __cplusplus
}
#endif

#endif /* __glass_h__ */
