/*
 *  Ray++ - Object-oriented ray tracing library
 *  Copyright (C) 1998-2001 Martin Reinecke and others.
 *  See the AUTHORS file for more information.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  See the README file for more information.
 */

#ifndef RAYPP_BSTREAM_H
#define RAYPP_BSTREAM_H

#include "config/config.h"
#include "kernel/byteswap.h"

namespace RAYPP {

const bool file_is_lsb=big_endian, file_is_msb=!big_endian, file_is_natural=false;

template<bool doswap=false> class bostream
  {};

template<> class bostream<false>
  {
  private:
    ofstream os;
    ostream &s;

  public:
    /*! */
    bostream (ostream &st)
      : s (st) {if (!s) throw "file error";}
    bostream (const string &fname)
      : os(fname.c_str()), s (os) {if (!s) throw "file error";}

    template<typename T> bostream &operator<< (const T &data)
      {
      s.write (reinterpret_cast<const char *> (&data), sizeof(data));
      return *this;
      }
    template<typename T> bostream &put (const T *data, int num)
      {
      s.write (reinterpret_cast<const char *> (data), num*sizeof(T));
      return *this;
      }
  };

template<> class bostream<true>
  {
  private:
    ofstream os;
    ostream &s;

  public:
    /*! */
    bostream (ostream &st)
      : s (st) {}
    bostream (const string &fname)
      : os(fname.c_str(),ios::binary), s (os) {}

    template<typename T> bostream &operator<< (const T &data)
      {
      T tmp = data;
      byteswap (tmp);
      s.write (reinterpret_cast<const char *> (&tmp), sizeof(tmp));
      return *this;
      }
    template<typename T> bostream &put (const T *data, int num)
      {
      for (int m=0; m<num; ++m)
	{
	T tmp=data[m];
        byteswap (tmp);
        s.write (reinterpret_cast<const char *> (&tmp), sizeof(tmp));
        }
      return *this;
      }
  };

template<bool doswap=false> class bistream
  {};

template<> class bistream<false>
  {
  private:
    ifstream is;
    istream &s;

  public:
    /*! */
    bistream (istream &st)
      : s (st) {if (!s) throw "file error";}
    bistream (const string &fname)
      : is(fname.c_str(),ios::binary), s (is) {if (!s) throw "file error";}

    template<typename T> bistream &operator>> (T &data)
      {
      s.read (reinterpret_cast<char *> (&data), sizeof(data));
      return *this;
      }
    template<typename T> bistream &get (T *data, int num)
      {
      s.read (reinterpret_cast<char *> (data), num*sizeof(T));
      return *this;
      }
  };

template<> class bistream<true>
  {
  private:
    ifstream is;
    istream &s;

  public:
    /*! */
    bistream (istream &st)
      : s (st) {}
    bistream (const string &fname)
      : is(fname.c_str()), s (is) {}

    template<typename T> bistream &operator>> (T &data)
      {
      s.read (reinterpret_cast<char *> (&data), sizeof(data));
      byteswap (data);
      return *this;
      }
    template<typename T> bistream &get (T *data, int num)
      {
      s.read (reinterpret_cast<char *> (data), num*sizeof(T));
      for (int m=0; m<num; ++m) byteswap (data[m]);
      return *this;
      }
  };

} // namespace RAYPP

#endif
