/*
 *  Ray++ - Object-oriented ray tracing library
 *  Copyright (C) 1998-2001 Martin Reinecke and others.
 *  See the AUTHORS file for more information.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  See the README file for more information.
 */

#ifndef RAYPP_FUZPROJECTOR_H
#define RAYPP_FUZPROJECTOR_H

/*! \file fuzprojector.h
    \brief Declarations for the FUZPROJECTOR class */

/* THIS FILE is copyright (c) 1999 by Ugo Varetto - varetto@computer.org */

#include "lights/projector.h"

namespace RAYPP {

/** \class FUZPROJECTOR lights/fuzprojector.h lights/fuzprojector.h
    A slide projector with adjustable focus.
    FUZPROJECTOR acts as PROJECTOR but has two more parameters for
    simulating an out of focus projector:
    <ul> <li> DeltaAngle </li>
         <li> Samples </li> </ul>
    Each light ray is sampled \a Samples times inside a solid angle
    \a DeltaAngle and intersected with the slide.

    It's the same algorithm used by stochastic raytracing.
    \author Ugo Varetto
*/
class FUZPROJECTOR : public PROJECTOR
  {
  protected:
    //! Aperture of sampling cone
    float8 DeltaAngle;

    //! Number of samples
    int2 Samples;

  public:
    //! Default constructor
    FUZPROJECTOR ();
    //! Constructor
    FUZPROJECTOR (const VECTOR &Loc, const VECTOR &To,
                  const COLOUR &Col,const float8 &e,const float8 &c);
    //! Sets the number of samples
    void Set_Samples(int2 s) {if(s) Samples=s;}
    //! Sets the sampling angle 
    void Set_Angle(float8 a) {DeltaAngle=0.5*radians(a);}
    //! Shading function
    virtual void Cast_Light (const VECTOR &Pos, LIGHT_ARRAY &Arr) const;
  };

} // namespace RAYPP

/** \example programs/fuzproj.cxx
    This is an example of how to use the FUZPROJECTOR class. */

#endif
