/*
 *  Ray++ - Object-oriented ray tracing library
 *  Copyright (C) 1998-2001 Martin Reinecke and others.
 *  See the AUTHORS file for more information.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  See the README file for more information.
 */

#ifndef RAYPP_PROJECTOR_H
#define RAYPP_PROJECTOR_H

/*! \file projector.h
    \brief Declarations for the PROJECTOR class */

/* THIS FILE is copyright (c) 1999 by Ugo Varetto - varetto@computer.org */

#include "lights/spotlight.h"

namespace RAYPP {

//! Helper struct for reading TGA files
struct TGAINFOHEADER
  {
  uint1 nifc;
  uint1 cmapTypes;
  uint1 imgType;
  uint1 cmapOrigin;
  uint2 cmapLength;
  uint2 cmapEntrySize;
  uint2 xOrg;
  uint2 yOrg;
  uint2 imgWidth;
  uint2 imgHeight;
  uint1 pixSize;
  uint1 imgDesc;
  };

//! Constant for performing RGB int to float conversions
const float4 INV_255 = float4(1.0/255.0);
//float4(0.00392156863);

//! Implementation of a slide projector
/*! This class allows for projecting a TGA image onto objects in space.
    The image can be resized as needed and a particular colour can be 
    interpreted as transparent, letting the light beam pass through 
    'transparent' pixels unmodified.
    The blending of light and image colours is controlled by the Mix parameter:
    Mix = 1 means that the light colour will be modulated (multiplied) by the
    image pixel;
    Mix = 0 means no modulation, so the final colour will be the image pixel
    colour.
    \author Ugo Varetto
*/
class PROJECTOR : public SPOTLIGHT
  {
  protected:
    //! Utility variable
    float8 Num;

    //! Colour to be assumed as 'transparent'
    COLOUR Transparent;

    //! If TRUE light will pass through transparent pixels unmodified 
    bool Transparency;

    //! Array of RGB values
    uint1 *Pixels;

    //! Image horizontal size 
    float8 Xs;

    //! Image vertical size
    float8 Ys;

    //! Distance between image centre and light source
    float8 D;

    //! Image - light blending factor
    float8 Mix;

    //! Direction of local X axis
    VECTOR X;

    //! Direction of local Y axis
    VECTOR Y;

    //! Image centre
    VECTOR O;

    //! Image width as read from file
    uint2 Width;

    //! Image original as read from file height
    uint2 Height; 

  public:
    //! Default constructor
    PROJECTOR ();

    //! Constructor
    PROJECTOR (const VECTOR &Loc, const VECTOR &To,
               const COLOUR &Col,const float8 &e,const float8 &c);

    //! Destructor: frees allocated memory
    ~PROJECTOR() {if(Pixels) delete [] Pixels;}
	
    /*! \brief Transforms the light location and direction, and computes
        the new image axes */
    virtual void Transform (const TRANSFORM &trans);

    virtual void Cast_Light (const VECTOR &Pos, LIGHT_ARRAY &Arr) const;

    //! Loads an uncompressed TGA image
    /*! Loads a TGA image and assigns the colour of pixel i,j to 'Transparent'
        Pixel (0,0) is at the lower left. */
    int Set_Image(const char *fname,int i=0,int j=0);

    //! Assigns the position and orientation of the image
    /*! Creates a local coordinate system (x,y,z axes) for the image
        \param up the up vector
        \param d distance from light source to image centre
        The orthonormal axes are:
           *  (Dir - Location).Norm() 
           *  up
           *  (Dir - Location).Norm ^ up
    */
    void Set_ImageAxes(const VECTOR &up,float8 d);

    /*! Creates a local coordinate system (x,y,z axes) for the image
        \param up the up vector
        \param d distance from light source to image centre
        The orthonormal axes are:
           *  (Dir - Location).Norm() 
           *  up
           *  (Dir - Location).Norm ^ up
    */
    //! Sets the image size to the desired values
    void Set_Size(const float8 x,const float8 y);

    //! Enables / Disables transparency
    void Set_Transparency(bool t) {Transparency=t;}

    //! Sets the image - light blending factor
    void Set_Mix(const float8 t);
  };

} // namespace RAYPP

/** \example programs/proj.cxx
    This is an example of how to use the PROJECTOR class. */

#endif
