/*
 *  Ray++ - Object-oriented ray tracing library
 *  Copyright (C) 1998-2001 Martin Reinecke and others.
 *  See the AUTHORS file for more information.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  See the README file for more information.
 */

/*! \file spotlight.h
    \brief Declarations for the SPOTLIGHT class */

/* THIS FILE is copyright (c) 1999 by Ugo Varetto - varetto@computer.org */

#ifndef RAYPP_SPOTLIGHT_H
#define RAYPP_SPOTLIGHT_H

#include "kernel/kernel.h"

namespace RAYPP {

// to be included into utils/math.h
inline float8 radians(float8 a)
 { return ((a*Pi)/180.0); }

//! An OpenGL like spot.
/*! This class implements a spot with a lighting equation that is the
    same used for OpenGL spotlights.
    \author Ugo Varetto */
class SPOTLIGHT: public LIGHT
  {
  protected:
    //! Light volume
    INSIDE_INFO MyVolume;
    //! Light colour
    COLOUR Colour;
    //! Light position
    VECTOR Location;
    //! Point at which the spot looks
    VECTOR To;
    //! Spot orientation
    VECTOR Dir;
    //! Beam attenuation exponent
    float8 Exp;
    //! Beam angle
    float8 CutOff;
    //! Attenuation coefficients
    float8 Attenuation[3];

  public:
    //! Default constructor
    SPOTLIGHT ();
    //! Constructor
    SPOTLIGHT (const VECTOR &loc, const VECTOR &to,
               const COLOUR &col,const float8 &e,const float8 &c);
    /*!
    \param loc the spot position.
    \param to the point toward which the spot is directed
    \param col the light color
    \param e the attenuation exponent
    \param c the beam angle expressed in degrees
    */
    //! Destructor
    virtual ~SPOTLIGHT() {}
    
    //! Ititialization
    virtual void Init ();
	
    //! Transforms position and direction
    virtual void Transform (const TRANSFORM &trans);

    //! Sets the light colour
    void Set_Colour (const COLOUR &new_col);

    //! Sets the light position
    virtual void Set_Location (const VECTOR &new_loc);

    //! Shading function
    virtual void Cast_Light (const VECTOR &Pos, LIGHT_ARRAY &Arr) const;

    //! Sets the 'look at' point
    virtual void Set_To(const VECTOR &new_to);

    //! Sets the beam attenuation exponent
    void Set_Exp(const float8 e);

    //! Sets the beam angle in degrees
    void Set_CutOff(const float8 c);

    //! Sets both the light position and orientation
    virtual void Set_FromTo(const VECTOR &new_loc,const VECTOR &new_to);

    //! Sets the attenuation coefficients
    /*! The total attenuation factor is computed as:
        1/(a2*d2+a1*d1+a0), where:<BR>
        d1 = distance from Point to light<BR>
        d2 = d1*d1. */
    void Set_Attenuation(const float8 a2,const float8 a1, const float8 a0);
  };

} // namespace RAYPP

/** \example programs/spot.cxx
  This is an example of how to use the SPOTLIGHT class.
*/

#endif
