/*
 *  Ray++ - Object-oriented ray tracing library
 *  Copyright (C) 1998-2001 Martin Reinecke and others.
 *  See the AUTHORS file for more information.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  See the README file for more information.
 */

#ifndef RAYPP_IMPLICIT_H
#define RAYPP_IMPLICIT_H

#include "kernel/kernel.h"

namespace RAYPP {

class IMPLICIT: public SHAPE
  {
  public:
    class DENSFUNC
      {
      public:
        virtual ~DENSFUNC () {}

        virtual float8 Density (const VECTOR &Loc) const = 0;
        virtual VECTOR Maxgrad () const = 0;
        virtual VECTOR Mingrad () const = 0;
        virtual VECTOR Normal (const VECTOR &Loc, float8 dens) const
          {
          const float8 eps = Small_dist;
          return VECTOR ((dens - Density (Loc + VECTOR (eps,0,0))) / eps,
                         (dens - Density (Loc + VECTOR (0,eps,0))) / eps,
                         (dens - Density (Loc + VECTOR (0,0,eps))) / eps);
          }
      };

  private:
    STRANSFORM Trans;
    HANDLE<DENSFUNC> Dens;
    AXISBOX Minmax;
    float8 stepwidth;
    bool Inverted, Solid;

    static const float8 Epsilon;

    inline VECTOR Get_Normal (const VECTOR &rawnorm) const;
    float8 Get_Deriv (const VECTOR &dir, float8 density) const;

  public:
    IMPLICIT ();

    virtual void Init ();

    virtual void Transform (const TRANSFORM &trans);

    virtual AXISBOX BBox () const;

    virtual bool Has_Inside () const
      { ci(); return Solid; }
    virtual bool Inside_in_BBox () const
      { ci(); return !(Solid && Inverted); } 

    virtual bool Test (const GEOM_RAY &Ray, float8 &dist, bool &realhit) const;

    virtual bool Inside (const VECTOR &Loc) const;
    virtual bool Intersect
      (const GEOM_RAY &Ray, float8 &dist, VECTOR &Normal) const;
    virtual void All_Intersections (const GEOM_RAY &Ray,
      vector<INTER> &Inter) const;

    void Set_Densfunc (const HANDLE<DENSFUNC> &newdens);
    void Set_Minmax (const AXISBOX &newbox);
    void Set_Stepwidth (float8 st);
    void Set_Solid () { cni(); Solid = true; }
    void Set_Flat () { cni(); Solid = false; }
    void Invert () { cni(); Inverted = !Inverted; }
  };

} // namespace RAYPP

#endif
