/*
 *  Ray++ - Object-oriented ray tracing library
 *  Copyright (C) 1998-2001 Martin Reinecke and others.
 *  See the AUTHORS file for more information.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  See the README file for more information.
 */

#ifndef RAYPP_PARAMETRIC_H
#define RAYPP_PARAMETRIC_H

#include "shapes/flat_shape.h"
#include "utils/utils.h"

namespace RAYPP {

class PARAMETRIC: public FLAT_SHAPE
  {
  public:
    class PARFUNC
      {
      public:
        virtual AXISBOX Get_Extent (const IV8 &u, const IV8 &v) const = 0;
        virtual VECTOR  Get_Point  (float8 u, float8 v) const = 0;
        virtual void    Get_Ranges (IV8 &u, IV8 &v) const = 0;
      };

  private:
    class entry
      {
      public:
        float8 idepth;
        IV8 iu, iv;
        int4 lvl;

        entry () {}
        entry (float8 d, const IV8 &u, const IV8 &v, int4 n)
          : idepth(d), iu(u), iv(v), lvl(n) {}

        bool operator< (const entry &other) const
// YES, that's correct, since we need the NEAREST intersection first
          { return (idepth > other.idepth); }
      };

    STRANSFORM Trans;
    AXISBOX Minmax;
    HANDLE<PARFUNC> Func;
    typedef enum {u_split, v_split} splitdir;
    vector<splitdir> dircache;
    float8 du, dv;
    int4 maxlevel;

    bool Check_Interval (priority_queue<entry> &Queue,
      const IV8 &iu, const IV8 &iv, int4 lvl,
      const GEOM_RAY &Local_Ray, float8 &dist, VECTOR &Normal) const;

  public:
    PARAMETRIC ();
    PARAMETRIC (const HANDLE<PARFUNC> &func);

    virtual void Init ();

    virtual void Transform (const TRANSFORM &trans);

    virtual AXISBOX BBox () const;

    virtual bool Test (const GEOM_RAY &Ray, float8 &dist, bool &realhit) const;

    virtual bool Intersect
      (const GEOM_RAY &Ray, float8 &dist, VECTOR &Normal) const;
    virtual void All_Intersections (const GEOM_RAY &Ray,
      vector<INTER> &Inter) const;

    void Set_Func (const HANDLE<PARFUNC> &newfunc);
    void Set_Resolution (float8 Du, float8 Dv);
  };

} // namespace RAYPP

#endif
